<?php
namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;

/**
 * Cloudflare APO integration for ZiziCache.
 */
class APO extends PluginBase {
    use PluginTrait;

    protected static function checkAvailability(): bool {
        if (!class_exists('CF\WordPress\Hooks')) {
            return false;
        }
        if (!class_exists('CF\API\Plugin')) {
            return false;
        }
        return static::is_apo_enabled();
    }

    public static function init(): void {
        if (!static::isAvailable()) {
            return;
        }
        
        // Security check - ensure user has appropriate permissions
        if (!current_user_can('manage_options')) {
            return;
        }
          static::addAction('zizi_cache_purge_urls:before', 'purgeCloudflareCacheByUrls');
        static::addAction('zizi_cache_purge_pages:before', 'purgeCloudflareCache');
        static::addAction('zizi_cache_purge_everything:before', 'purgeCloudflareCache');
    }

    public static function purgeCloudflareCache(): void {
        try {            // Security check
            if (!current_user_can('manage_options')) {
                \ZiziCache\CacheSys::writeWarning('Insufficient permissions for cache purge', 'APO');
                return;
            }
            
            if (!class_exists('CF\WordPress\Hooks')) {
                \ZiziCache\CacheSys::writeWarning('CF\WordPress\Hooks class not found', 'APO');
                return;
            }
            
            $cfapi = new \CF\WordPress\Hooks();
            $cfapi->purgeCacheEverything();
              } catch (\Exception $e) {
            error_log('ZiZi Cache APO: Error purging Cloudflare cache - ' . $e->getMessage());
        }
    }

    public static function purgeCloudflareCacheByUrls($urls): void {
        try {
            // Security check
            if (!current_user_can('manage_options')) {
                error_log('ZiZi Cache APO: Insufficient permissions for URL cache purge');
                return;
            }
            
            // Input validation and sanitization
            $urls = static::sanitizeInput($urls, 'array');
            if (!is_array($urls) || empty($urls)) {
                return;
            }
            
            if (!class_exists('CF\WordPress\Hooks')) {
                error_log('ZiZi Cache APO: CF\WordPress\Hooks class not found');
                return;
            }
            
            // Convert URLs to post IDs for better Cloudflare compatibility
            $postids = [];
            foreach ($urls as $url) {
                $sanitized_url = esc_url_raw($url);
                if (!empty($sanitized_url)) {
                    $post_id = url_to_postid($sanitized_url);
                    if ($post_id > 0) {
                        $postids[] = $post_id;
                    }
                }
            }
            
            if (empty($postids)) {
                // Fallback to direct URL purging if no post IDs found
                $sanitized_urls = array_filter(array_map('esc_url_raw', $urls));
                if (!empty($sanitized_urls)) {
                    $cfapi = new \CF\WordPress\Hooks();
                    $cfapi->purgeCacheByUrls($sanitized_urls);
                }
                return;
            }
            
            $postids = array_unique($postids);
            $cfapi = new \CF\WordPress\Hooks();
            
            // Use purgeCacheByRelevantURLs for better compatibility
            if (method_exists($cfapi, 'purgeCacheByRelevantURLs')) {
                $cfapi->purgeCacheByRelevantURLs($postids);
            } else {
                // Fallback to URL-based purging
                $sanitized_urls = array_filter(array_map('esc_url_raw', $urls));
                $cfapi->purgeCacheByUrls($sanitized_urls);
            }
              } catch (\Exception $e) {
            error_log('ZiZi Cache APO: Error purging Cloudflare cache by URLs - ' . $e->getMessage());
        }
    }

    private static function is_apo_enabled(): bool {
        try {
            // Check if Cloudflare API Plugin class exists
            if (!class_exists('CF\API\Plugin')) {
                return false;
            }
            
            // Check if WordPress DataStore is available
            if (!class_exists('CF\WordPress\DataStore')) {
                return false;
            }
            
            // Check if DefaultLogger is available
            if (!class_exists('CF\Integration\DefaultLogger')) {
                return false;
            }
            
            // Initialize DataStore with logger
            $datastore = new \CF\WordPress\DataStore(new \CF\Integration\DefaultLogger(false));
            $apo_config = $datastore->get('automatic_platform_optimization');
            
            if (!is_array($apo_config)) {
                return false;
            }
            
            // Check if APO is properly configured and enabled
            if (
                array_key_exists('id', $apo_config) &&
                $apo_config['id'] === 'automatic_platform_optimization' &&
                isset($apo_config['value']) &&
                $apo_config['value'] == '1'
            ) {
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            error_log('ZiZi Cache APO: Error checking APO status - ' . $e->getMessage());
            return false;
        }
    }
    
    public static function getVersion(): string {
        return '2.0.0';
    }
    
    public static function getDescription(): string {
        return 'Enterprise-grade Cloudflare APO (Automatic Platform Optimization) integration with enhanced security, proper APO detection, and error handling. Supports cache purging by URLs and post IDs with fallback mechanisms.';
    }
}
