<?php
namespace ZiziCache\Plugins\Integrations;

use ZiziCache\Plugins\Core\PluginBase;
use ZiziCache\Plugins\Core\PluginTrait;
use ZiziCache\Purge;
use ZiziCache\Preload;

/**
 * Integration for Advanced Custom Fields (ACF) plugin.
 *
 * Handles cache purging and preloading when ACF fields are saved.
 *
 * @package ZiziCache\Plugins\Integrations
 */
class ACF extends PluginBase {
    use PluginTrait;

    /**
     * Performs the actual availability check for ACF plugin.
     *
     * @return bool True if ACF is available, false otherwise
     */
    protected static function checkAvailability(): bool {
        return class_exists('ACF');
    }

    /**
     * Initialize the integration by registering hooks for ACF save events.
     *
     * @return void
     */
    public static function init(): void {
        if (!static::isAvailable()) {
            return;
        }
        static::addAction('acf/save_post', 'handleSave');
        static::addAction('acf/options_page/save', 'handleSave');
    }    /**
     * Handle ACF save events, purge and preload cache for affected content.
     * 
     * Implements security practices according to ACF documentation including
     * input sanitization and URL escaping.
     *
     * @param int|string $post_id The post ID or 'options'
     * @return void
     */
    public static function handleSave($post_id): void {
        // Early return if user doesn't have edit capabilities
        if (!current_user_can('edit_posts')) {
            return;
        }

        // Handle ACF options pages
        if ($post_id === 'options') {
            // Verify user can manage options before purging
            if (current_user_can('manage_options')) {
                Purge::purge_pages();
                Preload::preload_cache(false);
            }
            return;
        }

        // Sanitize post ID according to ACF best practices
        $post_id = static::sanitizeInput($post_id, 'int');
        
        // Validate post ID
        if (empty($post_id) || $post_id <= 0) {
            return;
        }

        // Skip navigation menu items to avoid unnecessary cache operations
        $post_type = get_post_type($post_id);
        if ($post_type === 'nav_menu_item' || $post_type === false) {
            return;
        }

        // Get permalink with error handling
        $url = get_permalink($post_id);
        
        // Only proceed if permalink is valid
        if ($url && $url !== false) {
            // Escape URL according to ACF documentation recommendations
            $safe_url = esc_url($url);
            
            // Purge and preload cache for the affected post URL
            Purge::purge_urls([$safe_url]);
            Preload::preload_url($safe_url);
        }
    }

    /**
     * Returns the version of the ACF integration.
     *
     * @return string Version number
     */
    public static function getVersion(): string {
        return '2.0.0'; // Enhanced security and stability version
    }

    /**
     * Returns description of the ACF integration.
     *
     * @return string Description text
     */
    public static function getDescription(): string {
        return 'Advanced Custom Fields (ACF) integration with enhanced security, stability and cache management for field updates.';
    }
}
