<?php
/**
 * Centralizovaná fronta pro dávkové čištění cache
 * 
 * Shromažďuje URL pro vyčištění a zpracovává je dávkově na konci požadavku,
 * což zlepšuje výkon a snižuje režii při opakovaných operacích čištění cache.
 * 
 * @package ZiziCache\Plugins\Core
 * @author ZiziCache Team
 * @since 0.2.5
 */
namespace ZiziCache\Plugins\Core;

class PurgeQueue {
    /**
     * Global queue of URLs to be purged.
     *
     * @var array
     */
    private static $queue = [];

    /**
     * Add one or more URLs to the global purge queue.
     *
     * This method collects URLs that should be purged and ensures
     * that the queue will be processed at the end of the request.
     *
     * @param string|array $url URL or array of URLs to be purged
     * @return void
     */
    public static function add($url): void {
        if (empty($url)) {
            return;
        }

        if (is_array($url)) {
            self::$queue = array_merge(self::$queue, $url);
        } else {
            self::$queue[] = $url;
        }

        // Ensure the queue is processed at shutdown, but only once per request
        if (!has_action('shutdown', [__CLASS__, 'process'])) {
            add_action('shutdown', [__CLASS__, 'process'], 20);
        }
    }

    /**
     * Process the purge queue and purge all collected URLs at once.
     *
     * This method is called automatically at the end of the request (shutdown)
     * and ensures that all queued URLs are purged in a single batch operation.
     *
     * @return void
     */
    public static function process(): void {
        if (empty(self::$queue)) {
            return;
        }

        // Remove duplicates before processing
        $urls = array_unique(self::$queue);
        self::$queue = [];

        if (function_exists('zizi_cache_purge_urls')) {
            zizi_cache_purge_urls($urls);

            // Log the batch purge if debugging is enabled
            if (defined('WP_DEBUG') && WP_DEBUG && function_exists('zizi_cache_log')) {
                $count = count($urls);
                zizi_cache_log("Batch purged $count URLs from queue", 'info', [
                    'count' => $count,
                    'source' => 'PurgeQueue'
                ]);
            }
        }
    }
}
