<?php
/**
 * Shared functionality for all plugin integrations
 *
 * This trait provides common methods used across plugin integrations
 * to maintain consistent cache handling and reduce code duplication.
 * Using a trait for shared functionality improves maintainability
 * and ensures consistent behavior.
 *
 * @package ZiziCache\Plugins\Core
 * @author ZiziCache Team
 * @since 0.2.0
 */
namespace ZiziCache\Plugins\Core;

trait PluginTrait {    /**
     * Purges cache for specified URLs
     *
     * Safely handles both string and array inputs, verifies permissions,
     * sanitizes URLs and logs the operation.
     *
     * @param array|string $urls One or more URLs to purge
     * @return void
     */
    protected static function purgeCache($urls): void {
        // Ověření oprávnění - operaci mohou provést pouze oprávnění uživatelé
        if (!self::canPurgeCache()) {
            self::log('Pokus o vyčištění cache bez oprávnění byl zamítnut', 'warning');
            return;
        }
        
        if (empty($urls)) {
            return;
        }
        
        $urls = (array) $urls;
        $urls = self::sanitizeUrls($urls);
        
        // Call global cache purging function if available
        if (function_exists('zizi_cache_purge_urls')) {
            zizi_cache_purge_urls($urls);
            self::log('Vyčištěna cache pro URL: ' . implode(', ', $urls), 'info');
        }
    }

    /**
     * Purges all cache with permission verification
     *
     * Triggers the global purge action after verifying the caller has
     * appropriate permissions to perform this significant operation.
     *
     * @return void
     */
    protected static function purgeAll(): void {
        // Ověření oprávnění - operaci mohou provést pouze oprávnění uživatelé
        if (!self::canPurgeCache()) {
            self::log('Attempt to purge all cache without permission was denied', 'warning');
            return;
        }
        
        do_action('zizi_cache_purge_all');
        self::log('Vyčištěna veškerá cache', 'info');
    }
    
    /**
     * Naplánuje URL k vyčištění přes centrální frontu
     * 
     * Místo okamžitého vyčištění naplánuje URL do centrální fronty,
     * která bude zpracována na konci požadavku, což umožňuje
     * dávkové a efektivnější zpracování čištění cache.
     * 
     * @param string|array $url URL adresa nebo pole adres k vyčištění
     * @return void
     */
    protected static function schedulePurge($url): void {
        // Ověření oprávnění - operaci mohou provést pouze oprávnění uživatelé
        if (!self::canPurgeCache()) {
            self::log('Pokus o naplánování vyčištění cache bez oprávnění byl zamítnut', 'warning');
            return;
        }
        
        if (empty($url)) {
            return;
        }
        
        // Sanitizace URL adres
        if (is_array($url)) {
            $url = self::sanitizeUrls($url);
        } else {
            $url = self::sanitizeUrl($url);
        }
        
        // Přidání do centrální fronty
        PurgeQueue::add($url);
        
        self::log('Naplánováno vyčištění cache přes centrální frontu', 'debug', [
            'url' => $url
        ]);
    }
    
    /**
     * Centralizovaná kontrola oprávnění pro operace čištění cache
     * 
     * @return bool
     */
    private static function canPurgeCache(): bool {
        // Povoleno pro WordPress cron
        if (wp_doing_cron()) {
            return true;
        }
        
        // Povoleno pro CLI
        if (defined('WP_CLI') && WP_CLI) {
            return true;
        }
        
        // Povoleno pro administrátory
        if (current_user_can('manage_options')) {
            return true;
        }
        
        // Povoleno pro ověřená API volání
        if (defined('REST_REQUEST') && REST_REQUEST) {
            $nonce = isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '';
            return wp_verify_nonce($nonce, 'wp_rest');
        }
        
        return false;
    }
    
    /**
     * Sanitizace URL pro bezpečné použití v operacích cache
     * 
     * @param string $url URL adresa k sanitizaci
     * @return string Sanitizovaná URL
     */
    protected static function sanitizeUrl($url): string {
        // Použití WordPress funkce pro sanitizaci URL
        $url = esc_url_raw($url);
        
        // Odstranění fragmentů, které nejsou relevantní pro cache
        $url = preg_replace('/#.*$/', '', $url);
        
        return $url;
    }
    
    /**
     * Sanitizuje pole URL adres
     * 
     * @param array $urls Pole URL adres k sanitizaci
     * @return array Sanitizované URL adresy
     */
    protected static function sanitizeUrls(array $urls): array {
        $result = [];
        foreach ($urls as $url) {
            if (!empty($url) && is_string($url)) {
                $result[] = self::sanitizeUrl($url);
            }
        }
        return $result;
    }

    /**
     * Logs a message with integration context
     * 
     * Only logs when WP_DEBUG is enabled, provides a standardized
     * logging interface for all integrations.
     *
     * @param string $message Log message
     * @param string $level Log level (info, warning, error)
     * @param array $context Additional context data
     * @return void
     */
    protected static function log(string $message, string $level = 'info', array $context = []): void {
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            return;
        }
        
        $context['integration'] = static::class;
        
        if (function_exists('zizi_cache_log')) {
            zizi_cache_log($message, $level, $context);
        }
    }
}
