<?php
/**
 * Base abstract class for all plugin integrations
 *
 * This abstract class defines the contract that all plugin integrations must follow.
 * It provides performance-optimized helper methods and consistent interface
 * for checking availability and initializing integrations.
 *
 * @package ZiziCache\Plugins\Core
 * @author ZiziCache Team
 * @since 0.2.0
 */
namespace ZiziCache\Plugins\Core;

abstract class PluginBase {
    /** @var array Statická cache pro výsledky isAvailable() s klíči podle třídy */
    private static $availabilityCache = [];

    /**
     * Determines if the plugin is available and should be initialized
     *
     * Uses caching for better performance to avoid repeated availability checks
     * during the same request. Only performs the actual check once per class.
     *
     * @return bool True if the plugin is available, false otherwise
     */    public static function isAvailable(): bool {
        $class = static::class;
        if (!isset(self::$availabilityCache[$class])) {
            // Nejprve zkontrolujeme, zda není integrace manuálně vypnuta
            if (!static::isEnabled()) {
                self::$availabilityCache[$class] = false;
            } else {
                self::$availabilityCache[$class] = static::checkAvailability();
            }
        }
        return self::$availabilityCache[$class];
    }

    /**
     * Checks if the integration is not manually disabled via filter
     *
     * Allows users to disable specific integrations through code via a filter,
     * providing flexibility without modifying core files.
     *
     * @return bool True if the integration is enabled, false if disabled
     */
    public static function isEnabled(): bool {
        $disabled = apply_filters('zizi_cache_disabled_integrations', []);
        return !in_array(static::class, $disabled, true);
    }

    /**
     * Performs the actual availability check
     * 
     * Implementations should check for the existence of required classes,
     * functions, or other conditions that indicate the integrated plugin
     * is active and available.
     *
     * @return bool True if the plugin is available, false otherwise
     */
    abstract protected static function checkAvailability(): bool;

    /**
     * Initializes the integration by registering hooks and filters
     *
     * This method will only be called if isAvailable() returns true,
     * ensuring optimal performance by not initializing unused integrations.
     *
     * @return void
     */
    abstract public static function init(): void;

    /**
     * Returns the integration name for logging purposes
     *
     * @return string The fully qualified class name
     */
    public static function getName(): string {
        return static::class;
    }    /** 
     * Třídní cache pro registrované hooky
     * Každá třída má vlastní sadu zaregistrovaných hooků
     */
    private static $registeredHooks = [];
    
    /**
     * Registers an action hook with method existence check and duplicate prevention
     *
     * This helper method enhances security by verifying method existence
     * before registering it as a hook callback and prevents duplicate registrations
     * when init() might be called multiple times.
     *
     * @param string $hook The WordPress action hook name
     * @param string $method The method name to call when the hook fires
     * @param int $priority Optional. Hook priority. Default 10
     * @param int $args Optional. Number of arguments. Default 1
     * @return void
     */
    protected static function addAction(string $hook, string $method, int $priority = 10, int $args = 1): void {
        // Použití late static binding pro správné oddělení hooků mezi třídami
        $class = static::class; 
        
        // Inicializace pole pro třídu, pokud ještě neexistuje
        if (!isset(self::$registeredHooks[$class])) {
            self::$registeredHooks[$class] = [];
        }
        
        // Vytvoření unikátního klíče pro tento hook
        $key = "{$hook}::{$method}::{$priority}";
        
        // Kontrola, zda už nebyl zaregistrován
        if (isset(self::$registeredHooks[$class][$key])) {
            return;
        }
        
        // Kontrola, zda metoda existuje před registrací
        if (method_exists($class, $method)) {
            add_action($hook, [$class, $method], $priority, $args);
            self::$registeredHooks[$class][$key] = true;
        }
    }
    
    /**
     * Returns comprehensive metadata about the integration
     * 
     * Used for UI display, diagnostics, and API endpoints to provide
     * detailed information about each integration.
     *
     * @return array Metadata including name, version and description
     */
    public static function getMetadata(): array {
        return [
            'name' => static::getName(),
            'version' => static::getVersion(),
            'description' => static::getDescription(),
        ];
    }
    
    /**
     * Returns the version of the integrated plugin
     *
     * @return string Version number
     */
    public static function getVersion(): string {
        return '1.0.0'; // Should be overridden in concrete classes
    }
    
    /**
     * Returns a description of the integration
     *
     * @return string Description text
     */
    public static function getDescription(): string {
        return ''; // Should be overridden in concrete classes
    }
    
    /**
     * Sanitizuje hodnoty vstupních parametrů - univerzální metoda
     * 
     * @param mixed $data Data k sanitizaci
     * @param string $type Typ dat (text, email, url, int, boolean, html)
     * @return mixed Sanitizovaná data
     */
    protected static function sanitizeInput($data, string $type = 'text') {
        if (is_array($data)) {
            return array_map(function($item) use ($type) {
                return self::sanitizeInput($item, $type);
            }, $data);
        }
        
        switch ($type) {
            case 'email':
                return sanitize_email($data);
            case 'url':
                return esc_url_raw($data);
            case 'int':
                return (int) $data;
            case 'boolean':
                return (bool) $data;
            case 'html':
                return wp_kses_post($data);
            default:
                return sanitize_text_field($data);
        }
    }
}
