<?php

namespace ZiziCache;

/**
 * Performance Profiler for Dynamic Bulk Optimization
 * 
 * Detects server capabilities and suggests optimal settings for bulk image conversion
 * based on hosting type, available resources, and server performance characteristics.
 * 
 * @package ZiziCache
 * @version 1.0.0
 */
class PerformanceProfiler
{
    /**
     * Hosting type constants
     */
    const HOSTING_SHARED = 'shared';
    const HOSTING_MANAGED = 'managed';
    const HOSTING_VPS = 'vps';
    const HOSTING_DEDICATED = 'dedicated';
    
    /**
     * Performance profiles for different hosting types
     */
    private static $performance_profiles = [
        self::HOSTING_SHARED => [
            'batch_size' => 15,
            'batch_delay' => 240,
            'memory_threshold' => 70,
            'concurrent_limit' => 1,
            'description' => 'Shared Hosting (Conservative)'
        ],
        self::HOSTING_MANAGED => [
            'batch_size' => 30,
            'batch_delay' => 120,
            'memory_threshold' => 80,
            'concurrent_limit' => 2,
            'description' => 'Managed Hosting (Balanced)'
        ],
        self::HOSTING_VPS => [
            'batch_size' => 60,
            'batch_delay' => 45,
            'memory_threshold' => 85,
            'concurrent_limit' => 3,
            'description' => 'VPS (Aggressive)'
        ],
        self::HOSTING_DEDICATED => [
            'batch_size' => 100,
            'batch_delay' => 20,
            'memory_threshold' => 90,
            'concurrent_limit' => 5,
            'description' => 'Dedicated Server (Maximum)'
        ]
    ];
    
    /**
     * Cache key for storing detected hosting type
     */
    private const CACHE_KEY = 'zizi_hosting_type_cache';
    
    /**
     * Detect hosting type based on server characteristics
     * 
     * @return string One of the HOSTING_* constants
     */
    public static function detect_hosting_type(): string
    {
        // Check cache first (valid for 24 hours)
        $cached_type = get_transient(self::CACHE_KEY);
        if ($cached_type !== false) {
            return $cached_type;
        }
        
        $score = 0;
        $details = [];
        
        // Memory limit analysis
        $memory_limit = self::get_memory_limit_bytes();
        if ($memory_limit > 0) {
            if ($memory_limit >= 2147483648) { // 2GB+
                $score += 4;
                $details['memory'] = '2GB+ (Dedicated/VPS)';
            } elseif ($memory_limit >= 1073741824) { // 1GB+
                $score += 3;
                $details['memory'] = '1GB+ (VPS)';
            } elseif ($memory_limit >= 536870912) { // 512MB+
                $score += 2;
                $details['memory'] = '512MB+ (Managed)';
            } else {
                $score += 1;
                $details['memory'] = '<512MB (Shared)';
            }
        }
        
        // Execution time analysis
        $max_execution_time = ini_get('max_execution_time');
        if ($max_execution_time == 0) { // Unlimited
            $score += 4;
            $details['execution_time'] = 'Unlimited (Dedicated)';
        } elseif ($max_execution_time >= 300) { // 5+ minutes
            $score += 3;
            $details['execution_time'] = '300s+ (VPS)';
        } elseif ($max_execution_time >= 120) { // 2+ minutes
            $score += 2;
            $details['execution_time'] = '120s+ (Managed)';
        } else {
            $score += 1;
            $details['execution_time'] = '<120s (Shared)';
        }
        
        // Function availability analysis
        $restricted_functions = ['exec', 'shell_exec', 'system', 'proc_open'];
        $available_functions = 0;
        foreach ($restricted_functions as $func) {
            if (function_exists($func) && !in_array($func, array_map('trim', explode(',', ini_get('disable_functions'))))) {
                $available_functions++;
            }
        }
        
        if ($available_functions >= 3) {
            $score += 3;
            $details['functions'] = 'Most system functions available (VPS/Dedicated)';
        } elseif ($available_functions >= 2) {
            $score += 2;
            $details['functions'] = 'Some system functions available (Managed)';
        } elseif ($available_functions >= 1) {
            $score += 1;
            $details['functions'] = 'Limited system functions (Managed/Shared)';
        } else {
            $score += 0;
            $details['functions'] = 'System functions disabled (Shared)';
        }
        
        // File upload limit analysis (indicator of hosting restrictions)
        $upload_max_filesize = self::get_upload_limit_bytes();
        if ($upload_max_filesize >= 134217728) { // 128MB+
            $score += 2;
            $details['upload_limit'] = '128MB+ (VPS/Dedicated)';
        } elseif ($upload_max_filesize >= 67108864) { // 64MB+
            $score += 1;
            $details['upload_limit'] = '64MB+ (Managed)';
        } else {
            $score += 0;
            $details['upload_limit'] = '<64MB (Shared)';
        }
        
        // Determine hosting type based on score
        if ($score >= 12) {
            $hosting_type = self::HOSTING_DEDICATED;
        } elseif ($score >= 9) {
            $hosting_type = self::HOSTING_VPS;
        } elseif ($score >= 6) {
            $hosting_type = self::HOSTING_MANAGED;
        } else {
            $hosting_type = self::HOSTING_SHARED;
        }
        
        // Cache result for 24 hours
        set_transient(self::CACHE_KEY, $hosting_type, 24 * HOUR_IN_SECONDS);
        
        // Log detection for debugging
        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
            \ZiziCache\CacheSys::writeLog(
                "Hosting type detected: {$hosting_type} (score: {$score}/16). Details: " . json_encode($details),
                'PerformanceProfiler'
            );
        }
        
        return $hosting_type;
    }
    
    /**
     * Get performance profile for current or specified hosting type
     * 
     * @param string|null $hosting_type Hosting type (auto-detect if null)
     * @return array Performance configuration
     */
    public static function get_performance_profile(?string $hosting_type = null): array
    {
        if ($hosting_type === null) {
            $hosting_type = self::detect_hosting_type();
        }
        
        $profile = self::$performance_profiles[$hosting_type] ?? self::$performance_profiles[self::HOSTING_SHARED];
        $profile['hosting_type'] = $hosting_type;
        
        return $profile;
    }
    
    /**
     * Get optimized batch size based on server capabilities and current load
     * 
     * @param int $total_images Total number of images to process
     * @param string|null $hosting_type Hosting type override
     * @return int Optimized batch size
     */
    public static function get_optimal_batch_size(int $total_images, ?string $hosting_type = null): int
    {
        $profile = self::get_performance_profile($hosting_type);
        $base_batch_size = $profile['batch_size'];
        
        // Adjust based on total images count
        if ($total_images > 5000) {
            // For very large sets, use smaller batches to avoid timeouts
            $base_batch_size = max(10, intval($base_batch_size * 0.6));
        } elseif ($total_images > 1000) {
            // For large sets, slightly reduce batch size
            $base_batch_size = max(15, intval($base_batch_size * 0.8));
        }
        
        // Check current memory usage
        $memory_usage_percent = self::get_memory_usage_percent();
        if ($memory_usage_percent > 80) {
            $base_batch_size = max(5, intval($base_batch_size * 0.5));
        } elseif ($memory_usage_percent > 60) {
            $base_batch_size = max(10, intval($base_batch_size * 0.75));
        }
        
        return $base_batch_size;
    }
    
    /**
     * Get optimized batch delay based on server performance
     * 
     * @param string|null $hosting_type Hosting type override
     * @return int Delay in seconds
     */
    public static function get_optimal_batch_delay(?string $hosting_type = null): int
    {
        $profile = self::get_performance_profile($hosting_type);
        
        // Check if Action Scheduler queue is getting backed up
        $pending_actions = self::get_pending_actions_count();
        if ($pending_actions > 100) {
            // Increase delay if queue is getting large
            return intval($profile['batch_delay'] * 1.5);
        } elseif ($pending_actions > 50) {
            return intval($profile['batch_delay'] * 1.2);
        }
        
        return $profile['batch_delay'];
    }
    
    /**
     * Check if server can handle additional concurrent processing
     * 
     * @param string|null $hosting_type Hosting type override
     * @return bool True if can handle more load
     */
    public static function can_handle_more_load(?string $hosting_type = null): bool
    {
        $profile = self::get_performance_profile($hosting_type);
        
        // Check memory usage
        $memory_usage_percent = self::get_memory_usage_percent();
        if ($memory_usage_percent > $profile['memory_threshold']) {
            return false;
        }
        
        // Check Action Scheduler queue
        $pending_actions = self::get_pending_actions_count();
        if ($pending_actions > 200) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Get memory limit in bytes
     */
    private static function get_memory_limit_bytes(): int
    {
        $limit = ini_get('memory_limit');
        if ($limit == -1) {
            return -1; // Unlimited
        }
        
        return self::convert_size_to_bytes($limit);
    }
    
    /**
     * Get upload limit in bytes
     */
    private static function get_upload_limit_bytes(): int
    {
        $upload_max = ini_get('upload_max_filesize');
        $post_max = ini_get('post_max_size');
        
        $upload_bytes = self::convert_size_to_bytes($upload_max);
        $post_bytes = self::convert_size_to_bytes($post_max);
        
        return min($upload_bytes, $post_bytes);
    }
    
    /**
     * Convert size string to bytes
     */
    private static function convert_size_to_bytes(string $size): int
    {
        $size = trim($size);
        $last = strtolower($size[strlen($size) - 1]);
        $size = intval($size);
        
        switch ($last) {
            case 'g':
                $size *= 1024 * 1024 * 1024;
                break;
            case 'm':
                $size *= 1024 * 1024;
                break;
            case 'k':
                $size *= 1024;
                break;
        }
        
        return $size;
    }
    
    /**
     * Get current memory usage percentage
     */
    private static function get_memory_usage_percent(): float
    {
        $memory_limit = self::get_memory_limit_bytes();
        if ($memory_limit <= 0) {
            return 0; // Unlimited or unknown
        }
        
        $memory_usage = memory_get_usage(true);
        return ($memory_usage / $memory_limit) * 100;
    }
    
    /**
     * Get count of pending Action Scheduler actions
     */
    private static function get_pending_actions_count(): int
    {
        if (!function_exists('as_get_scheduled_actions')) {
            return 0;
        }
        
        $actions = as_get_scheduled_actions([
            'hook' => 'zizi_image_converter_process_single', // Use string directly instead of private constant
            'status' => 'pending',
            'per_page' => 1
        ]);
        
        return $actions['total'] ?? 0;
    }
    
    /**
     * Clear hosting type cache (for testing/debugging)
     */
    public static function clear_cache(): void
    {
        delete_transient(self::CACHE_KEY);
    }
    
    /**
     * Get detailed server analysis for debugging
     * 
     * @return array Detailed server analysis
     */
    public static function get_server_analysis(): array
    {
        return [
            'hosting_type' => self::detect_hosting_type(),
            'performance_profile' => self::get_performance_profile(),
            'memory_limit' => ini_get('memory_limit'),
            'memory_usage' => memory_get_usage(true),
            'memory_usage_percent' => self::get_memory_usage_percent(),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size'),
            'disabled_functions' => ini_get('disable_functions'),
            'pending_actions' => self::get_pending_actions_count(),
            'can_handle_load' => self::can_handle_more_load()
        ];
    }

    /**
     * Get comprehensive analysis for Vue component
     * 
     * @return array Complete analysis data
     */
    public function get_comprehensive_analysis(): array
    {
        $hosting_type = self::detect_hosting_type();
        $server_info = self::get_server_analysis();
        $current_profile = self::get_performance_profile();
        
        return [
            'hosting_type' => $hosting_type,
            'server_info' => $server_info,
            'current_profile' => $current_profile,
            'recommendations' => $this->get_recommendations($hosting_type, $server_info),
            'analysis_date' => current_time('mysql'),
            'analysis_in_progress' => false
        ];
    }

    /**
     * Run fresh performance analysis with cache clearing
     * 
     * @return array Fresh analysis data
     */
    public function run_fresh_analysis(): array
    {
        // Clear cache to force fresh detection
        self::clear_cache();
        
        return $this->get_comprehensive_analysis();
    }

    /**
     * Get performance recommendations based on current setup
     * 
     * @param string $hosting_type Current hosting type
     * @param array $server_info Server information
     * @return array Performance recommendations
     */
    private function get_recommendations(string $hosting_type, array $server_info): array
    {
        $recommendations = [];
        
        // Memory recommendations
        if ($server_info['memory_usage_percent'] > 80) {
            $recommendations[] = [
                'type' => 'memory',
                'severity' => 'high',
                'message' => 'Memory usage is high. Consider reducing batch size or increasing memory limit.',
                'action' => 'Reduce bulk_max_batch_size to ' . max(10, intval($server_info['current_profile']['batch_size'] * 0.7))
            ];
        }
        
        // Hosting type specific recommendations
        switch ($hosting_type) {
            case self::HOSTING_SHARED:
                $recommendations[] = [
                    'type' => 'hosting',
                    'severity' => 'medium',
                    'message' => 'Shared hosting detected. Conservative settings recommended.',
                    'action' => 'Using optimized settings for shared hosting environment'
                ];
                break;
                
            case self::HOSTING_VPS:
                $recommendations[] = [
                    'type' => 'hosting',
                    'severity' => 'low',
                    'message' => 'VPS hosting detected. Aggressive optimization possible.',
                    'action' => 'Consider enabling high-performance settings for faster processing'
                ];
                break;
                
            case self::HOSTING_DEDICATED:
                $recommendations[] = [
                    'type' => 'hosting',
                    'severity' => 'low',
                    'message' => 'Dedicated server detected. Maximum performance settings recommended.',
                    'action' => 'Enable all performance optimizations for best results'
                ];
                break;
        }
        
        // Action Scheduler recommendations
        if ($server_info['pending_actions'] > 100) {
            $recommendations[] = [
                'type' => 'queue',
                'severity' => 'medium',
                'message' => 'High number of pending actions detected.',
                'action' => 'Consider increasing concurrent limit or reducing batch delay'
            ];
        }
        
        return $recommendations;
    }
}