<?php
namespace ZiziCache;

/**
 * Performance monitoring system for ZiziCache
 * 
 * Tracks and logs performance metrics for REST API requests, AJAX calls, 
 * and plugin bootstrap times to help identify performance bottlenecks.
 *
 * @package ZiziCache
 * @since 1.0.0
 */
class PerformanceMonitor
{
    /** @var array Request start timestamps */
    protected static $start_times = [];
    
    /** @var float Bootstrap start timestamp */
    protected static $bootstrap_start = 0;

    /** @var array Static metrics storage */
    protected static $metrics = [];
    
    /** @var array Database performance metrics */
    protected static $database_metrics = [];
    
    /** @var array HTTP request performance metrics */
    protected static $http_metrics = [];
    
    /** @var array Cache efficiency metrics */
    protected static $cache_metrics = [];
    
    /** @var array WordPress performance metrics */
    protected static $wp_metrics = [];
    
    /** @var bool Debug mode flag */
    protected static $debug_mode = false;

    /**
     * Initialize hooks to measure various performance metrics
     * 
     * Sets up action and filter hooks to monitor performance of:
     * - Plugin bootstrap process
     * - REST API requests
     * - AJAX calls in admin
     *
     * @return void
     */
    public static function init(): void
    {
        // Log plugin bootstrap timing at end of init
        add_action('init', [__CLASS__, 'bootstrap_end'], PHP_INT_MAX);
        // REST API timing
        add_filter('rest_pre_dispatch', [__CLASS__, 'on_rest_pre_dispatch'], 10, 3);
        add_filter('rest_post_dispatch', [__CLASS__, 'on_rest_post_dispatch'], 10, 3);
        // AJAX timing in admin-ajax
        add_action('admin_init', [__CLASS__, 'on_ajax_init'], 1);
        add_action('shutdown', [__CLASS__, 'on_ajax_shutdown'], 0);
        
        // Database performance tracking
        add_action('shutdown', [__CLASS__, 'on_database_shutdown'], 1);
        
        // HTTP performance tracking
        add_filter('pre_http_request', [__CLASS__, 'on_http_pre_request'], 10, 3);
        add_filter('http_response', [__CLASS__, 'on_http_response'], 10, 3);
        add_action('shutdown', [__CLASS__, 'storeHttpMetrics'], 2);
    }

    /**
     * Record REST API request start time
     * 
     * Captures the timestamp when a REST API request begins processing.
     *
     * @param mixed $response The pre-dispatch response, usually null
     * @param \WP_REST_Server $server The REST server instance
     * @param \WP_REST_Request $request The current REST request
     * @return mixed The unmodified response
     */
    public static function on_rest_pre_dispatch($response, $server, $request)
    {
        // mark REST request start
        $key = spl_object_id($request);
        self::$start_times[$key] = microtime(true);
        return $response;
    }

    /**
     * Log REST API request elapsed time
     * 
     * Calculates and logs the time taken to process a REST API request.
     *
     * @param mixed $response The REST API response
     * @param \WP_REST_Server $server The REST server instance
     * @param \WP_REST_Request $request The current REST request
     * @return mixed The unmodified response
     */
    public static function on_rest_post_dispatch($response, $server, $request)
    {
        // compute REST elapsed and write to custom log
        $key = spl_object_id($request);
        if (isset(self::$start_times[$key])) {
            $elapsed = microtime(true) - self::$start_times[$key];
            $method = $request->get_method();
            $route  = $request->get_route();
            \ZiziCache\CacheSys::writeLog(sprintf('[ZiziCache Performance] %s %s took %.4f sec', strtoupper($method), $route, $elapsed));
            unset(self::$start_times[$key]);
        }
        return $response;
    }

    /**
     * Record AJAX request start time in admin-ajax.php
     * 
     * Captures the timestamp when an AJAX action begins processing.
     *
     * @return void
     */
    public static function on_ajax_init(): void
    {
        if (defined('DOING_AJAX') && DOING_AJAX) {
            $action = sanitize_key($_REQUEST['action'] ?? 'unknown');
            $key = 'ajax_' . $action;
            self::$start_times[$key] = microtime(true);
        }
    }

    /**
     * Log AJAX request elapsed time on shutdown
     * 
     * Calculates and logs the time taken to process AJAX requests.
     *
     * @return void
     */
    public static function on_ajax_shutdown(): void
    {
        foreach (self::$start_times as $key => $start) {
            if (strpos($key, 'ajax_') === 0) {
                $action = substr($key, 5);
                $elapsed = microtime(true) - $start;
                \ZiziCache\CacheSys::writeLog(sprintf('[ZiziCache Performance] AJAX %s took %.4f sec', $action, $elapsed));
                unset(self::$start_times[$key]);
            }
        }
    }

    /**
     * Store plugin bootstrap start time
     * 
     * Should be called at the very beginning of the plugin's main file.
     *
     * @return void
     */
    public static function bootstrap_start(): void
    {
        self::$bootstrap_start = microtime(true);
    }

    /**
     * Log plugin bootstrap elapsed time when WordPress init fires
     * 
     * Calculates and logs the time taken to bootstrap the plugin.
     *
     * @return void
     */
    public static function bootstrap_end(): void
    {
        if (self::$bootstrap_start > 0) {
            $elapsed = microtime(true) - self::$bootstrap_start;
            $req = ($_SERVER['REQUEST_METHOD'] ?? 'CLI') . ' ' . ($_SERVER['REQUEST_URI'] ?? '');
            \ZiziCache\CacheSys::writeLog(sprintf('[ZiziCache Bootstrap] %s took %.4f sec', $req, $elapsed));
            self::$bootstrap_start = 0;
        }
    }

    /**
     * Track database performance metrics on shutdown
     * 
     * Collects database statistics including query count, execution time,
     * memory usage, and slow query detection.
     *
     * @return void
     */
    public static function on_database_shutdown(): void
    {
        global $wpdb;
        
        $metrics = [
            'timestamp' => microtime(true),
            'query_count' => $wpdb->num_queries,
            'query_time' => property_exists($wpdb, 'query_time') ? $wpdb->query_time : 0,
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true)
        ];
        
        // Detect slow queries if query logging is active
        if (defined('SAVEQUERIES') && SAVEQUERIES && !empty($wpdb->queries)) {
            $slow_queries = 0;
            foreach ($wpdb->queries as $query) {
                if (isset($query[1]) && $query[1] > 1.0) { // >1 second
                    $slow_queries++;
                }
            }
            $metrics['slow_queries'] = $slow_queries;
        } else {
            $metrics['slow_queries'] = 0;
        }
        
        self::$database_metrics[] = $metrics;
        
        // Log critical situations
        if ($metrics['query_count'] > 100) {
            \ZiziCache\CacheSys::writeLog(sprintf(
                '[ZiziCache Database] High query count: %d queries in %.4f sec',
                $metrics['query_count'],
                $metrics['query_time']
            ));
        }
        
        if ($metrics['slow_queries'] > 0) {
            \ZiziCache\CacheSys::writeLog(sprintf(
                '[ZiziCache Database] Slow queries detected: %d queries >1sec',
                $metrics['slow_queries']
            ));
        }
        
        // Store in SQLite for historical analysis
        if (class_exists('\\ZiziCache\\PreloadSqlite')) {
            \ZiziCache\PreloadSqlite::storePerformanceMetrics('database', $metrics);
        }
    }

    /**
     * Track HTTP request start time
     * 
     * Captures timestamp before HTTP request is made to measure response time.
     *
     * @param mixed $preempt Short-circuit return value
     * @param array $args Request arguments
     * @param string $url Request URL
     * @return mixed Unmodified preempt value
     */
    public static function on_http_pre_request($preempt, $args, $url)
    {
        $key = md5($url);
        self::$start_times['http_' . $key] = microtime(true);
        return $preempt;
    }

    /**
     * Track HTTP request completion and log performance
     * 
     * Calculates response time and logs performance metrics for HTTP requests.
     *
     * @param array|\WP_Error $response HTTP response or WP_Error object
     * @param array $args Request arguments
     * @param string $url Request URL
     * @return array|\WP_Error Unmodified response
     */
    public static function on_http_response($response, $args, $url)
    {
        $key = md5($url);
        $start_key = 'http_' . $key;
        
        if (isset(self::$start_times[$start_key])) {
            $elapsed = microtime(true) - self::$start_times[$start_key];
            
            $metrics = [
                'url' => $url,
                'response_time' => $elapsed,
                'status_code' => is_wp_error($response) ? 0 : wp_remote_retrieve_response_code($response),
                'timestamp' => microtime(true),
                'is_error' => is_wp_error($response),
                'error_message' => is_wp_error($response) ? $response->get_error_message() : ''
            ];
            
            self::$http_metrics[] = $metrics;
            
            // Log slow HTTP requests
            if ($elapsed > 3.0) {
                \ZiziCache\CacheSys::writeLog(sprintf(
                    '[ZiziCache HTTP] Slow request: %s took %.4f sec',
                    $url,
                    $elapsed
                ));
            }
            
            // Log failed requests
            if (is_wp_error($response)) {
                \ZiziCache\CacheSys::writeLog(sprintf(
                    '[ZiziCache HTTP] Failed request: %s - %s',
                    $url,
                    $response->get_error_message()
                ));
            }
            
            unset(self::$start_times[$start_key]);
        }
        
        return $response;
    }

    /**
     * Store HTTP metrics in SQLite at the end of request
     * 
     * @return void
     */
    public static function storeHttpMetrics(): void
    {
        if (!empty(self::$http_metrics) && class_exists('\\ZiziCache\\PreloadSqlite')) {
            $summary = [
                'total_requests' => count(self::$http_metrics),
                'avg_response_time' => array_sum(array_column(self::$http_metrics, 'response_time')) / count(self::$http_metrics),
                'error_count' => count(array_filter(self::$http_metrics, function($m) { return $m['is_error']; })),
                'slow_requests' => count(array_filter(self::$http_metrics, function($m) { return $m['response_time'] > 2.0; })),
                'requests' => self::$http_metrics
            ];
            
            \ZiziCache\PreloadSqlite::storePerformanceMetrics('http', $summary);
        }
    }

    /**
     * Enables debug mode for detailed logging
     *
     * @param bool $enabled
     * @return void
     */
    public static function enableDebugMode(bool $enabled = true): void
    {
        self::$debug_mode = $enabled;
        if ($enabled) {
            self::setupAIErrorHandling();
            // AI Operation logging removed - AI Optimizer disconnected
        }
    }

    /**
     * Analyze database performance metrics
     * 
     * @return array
     */
    private static function getDatabasePerformanceAnalysis(): array
    {
        global $wpdb;
        
        // Get current query count for analysis
        $current_queries = isset($wpdb->num_queries) ? $wpdb->num_queries : 0;
        $current_memory = memory_get_usage(true);
        
        // User-friendly performance analysis
        if ($current_queries < 30) {
            $status = 'excellent';
            $message = 'Database is running excellently';
            $description = 'Query count is optimal. Page loads quickly.';
            $recommendation = '';
            $icon = '✅';
            $color = '#10b981'; // green
        } elseif ($current_queries < 60) {
            $status = 'good';
            $message = 'Database is running well';
            $description = 'Query count is acceptable but can be optimized.';
            $recommendation = 'Consider using a cache plugin or theme optimization.';
            $icon = '👍';
            $color = '#3b82f6'; // blue
        } elseif ($current_queries < 100) {
            $status = 'fair';
            $message = 'Database needs optimization';
            $description = 'High query count may slow down the page.';
            $recommendation = 'We recommend activating cache and checking plugins.';
            $icon = '⚠️';
            $color = '#f59e0b'; // yellow
        } else {
            $status = 'poor';
            $message = 'Database is overloaded';
            $description = 'Too many database queries are significantly slowing down the page.';
            $recommendation = 'Activate cache immediately, check plugins and theme.';
            $icon = '🚨';
            $color = '#ef4444'; // red
        }
        
        return [
            'status' => $status,
            'message' => $message,
            'description' => $description,
            'recommendation' => $recommendation,
            'icon' => $icon,
            'color' => $color,
            'technical_details' => [
                'queries' => $current_queries,
                'memory_mb' => round($current_memory / 1024 / 1024, 2),
                'peak_memory_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2)
            ],
            'last_updated' => current_time('mysql')
        ];
    }

    /**
     * Analyze HTTP performance metrics
     * 
     * @return array
     */
    private static function getHttpPerformanceAnalysis(): array
    {
        if (empty(self::$http_metrics)) {
            return [
                'status' => 'good',
                'message' => 'External communication OK',
                'description' => 'No issues connecting to external services.',
                'recommendation' => '',
                'icon' => '🌐',
                'color' => '#10b981',
                'technical_details' => [
                    'requests' => 0,
                    'avg_response_ms' => 0,
                    'error_rate' => 0
                ],
                'last_updated' => current_time('mysql')
            ];
        }
        
        $total_requests = count(self::$http_metrics);
        $response_times = array_column(self::$http_metrics, 'response_time');
        $avg_response_time = array_sum($response_times) / $total_requests;
        $error_count = count(array_filter(self::$http_metrics, function($m) { return $m['is_error']; }));
        $error_rate = ($error_count / $total_requests) * 100;
        
        // User-friendly HTTP analysis
        if ($error_rate > 20) {
            $status = 'poor';
            $message = 'Connection issues';
            $description = 'High error rate when communicating with external services.';
            $recommendation = 'Check your internet connection and hosting settings.';
            $icon = '🚨';
            $color = '#ef4444';
        } elseif ($avg_response_time > 5) {
            $status = 'fair';
            $message = 'Slow connection';
            $description = 'External services are responding slowly.';
            $recommendation = 'Check your internet connection speed.';
            $icon = '⚠️';
            $color = '#f59e0b';
        } else {
            $status = 'good';
            $message = 'External communication working';
            $description = 'Connection to external services is OK.';
            $recommendation = '';
            $icon = '🌐';
            $color = '#10b981';
        }
        
        return [
            'status' => $status,
            'message' => $message,
            'description' => $description,
            'recommendation' => $recommendation,
            'icon' => $icon,
            'color' => $color,
            'technical_details' => [
                'requests' => $total_requests,
                'avg_response_ms' => round($avg_response_time * 1000, 0),
                'error_rate' => round($error_rate, 1)
            ],
            'last_updated' => current_time('mysql')
        ];
    }

    /**
     * Analyze cache efficiency
     * 
     * @return array
     */
    private static function getCacheEfficiencyAnalysis(): array
    {
        global $wp_object_cache;
        
        $object_cache_enabled = wp_using_ext_object_cache();
        $opcache_enabled = function_exists('opcache_get_status');
        
        // Calculate cache score
        $cache_score = 0;
        $cache_features = [];
        $recommendations = [];
        
        if ($opcache_enabled) {
            $cache_score += 50;
            $cache_features[] = 'OPcache active';
        } else {
            $recommendations[] = 'Activate OPcache for faster PHP';
        }
        
        if ($object_cache_enabled) {
            $cache_score += 30;
            $cache_features[] = 'Object cache active';
        } else {
            $recommendations[] = 'Set up Redis or Memcached';
        }
        
        // Check if ZiziCache is active
        if (class_exists('ZiziCache\CacheSys')) {
            $cache_score += 20;
            $cache_features[] = 'ZiziCache active';
        }
        
        // Determine status based on score
        if ($cache_score >= 80) {
            $status = 'excellent';
            $message = 'Cache optimally configured';
            $description = 'All important cache systems are active.';
            $icon = '🚀';
            $color = '#10b981';
        } elseif ($cache_score >= 50) {
            $status = 'good';
            $message = 'Cache well configured';
            $description = 'Basic cache systems are working but can be improved.';
            $icon = '⚡';
            $color = '#3b82f6';
        } elseif ($cache_score >= 20) {
            $status = 'fair';
            $message = 'Cache partially active';
            $description = 'Some cache systems are missing.';
            $icon = '⚠️';
            $color = '#f59e0b';
        } else {
            $status = 'poor';
            $message = 'Cache not configured';
            $description = 'No cache systems are active.';
            $icon = '🐌';
            $color = '#ef4444';
        }
        
        return [
            'status' => $status,
            'message' => $message,
            'description' => $description,
            'cache_score' => $cache_score,
            'active_features' => $cache_features,
            'recommendation' => implode(' | ', $recommendations),
            'icon' => $icon,
            'color' => $color,
            'technical_details' => [
                'opcache_enabled' => $opcache_enabled,
                'object_cache_enabled' => $object_cache_enabled,
                'zizi_cache_active' => class_exists('ZiziCache\CacheSys')
            ],
            'last_updated' => current_time('mysql')
        ];
    }

    /**
     * Analyze WordPress performance
     * 
     * @return array
     */
    private static function getWordPressPerformanceAnalysis(): array
    {
        return [
            'wp_version' => get_bloginfo('version'),
            'active_plugins' => count(get_option('active_plugins', [])),
            'active_theme' => get_template(),
            'is_multisite' => is_multisite(),
            'wp_debug' => defined('WP_DEBUG') && WP_DEBUG,
            'wp_debug_log' => defined('WP_DEBUG_LOG') && WP_DEBUG_LOG,
            'savequeries' => defined('SAVEQUERIES') && SAVEQUERIES,
            'timezone' => get_option('timezone_string') ?: get_option('gmt_offset')
        ];
    }

    /**
     * Calculate database efficiency rating
     * 
     * @param array $metrics
     * @return string
     */
    private static function calculateDatabaseEfficiency(array $metrics): string
    {
        $query_count = $metrics['query_count'];
        $slow_queries = $metrics['slow_queries'];
        
        if ($query_count <= 20 && $slow_queries === 0) {
            return 'excellent';
        } elseif ($query_count <= 50 && $slow_queries <= 2) {
            return 'good';
        } elseif ($query_count <= 100 && $slow_queries <= 5) {
            return 'fair';
        } else {
            return 'poor';
        }
    }

    /**
     * Get database performance recommendations
     * 
     * @param array $metrics
     * @return array
     */
    private static function getDatabaseRecommendations(array $metrics): array
    {
        $recommendations = [];
        
        if ($metrics['query_count'] > 50) {
            $recommendations[] = 'Consider implementing database query caching';
        }
        
        if ($metrics['slow_queries'] > 0) {
            $recommendations[] = 'Optimize slow database queries (>1 second)';
        }
        
        if ($metrics['query_count'] > 100) {
            $recommendations[] = 'Review and optimize database queries - high query count detected';
        }
        
        if (empty($recommendations)) {
            $recommendations[] = 'Database performance is optimal';
        }
        
        return $recommendations;
    }

    /**
     * Returns all collected metrics including application performance metrics
     *
     * @return array
     */
    public static function getMetrics(): array
    {
        return [
            // Existing metrics
            'static_metrics' => self::$metrics,
            'memory_usage' => [
                'current' => memory_get_usage(true),
                'peak' => memory_get_peak_usage(true),
                'limit' => ini_get('memory_limit')
            ],
            'execution_time' => [
                'current' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'],
                'limit' => ini_get('max_execution_time')
            ],
            
            // NEW Application Performance metrics
            'database_performance' => self::getDatabasePerformanceAnalysis(),
            'http_performance' => self::getHttpPerformanceAnalysis(),
            'cache_efficiency' => self::getCacheEfficiencyAnalysis(),
            'wordpress_performance' => self::getWordPressPerformanceAnalysis(),
            
            'php_info' => [
                'version' => PHP_VERSION,
                'extensions' => get_loaded_extensions()
            ]
        ];
    }

    /**
     * Returns the error severity name
     *
     * @param int $severity
     * @return string
     */
    private static function getErrorSeverityName(int $severity): string
    {
        $names = [
            E_ERROR => 'E_ERROR',
            E_WARNING => 'E_WARNING',
            E_PARSE => 'E_PARSE',
            E_NOTICE => 'E_NOTICE',
            E_CORE_ERROR => 'E_CORE_ERROR',
            E_CORE_WARNING => 'E_CORE_WARNING',
            E_COMPILE_ERROR => 'E_COMPILE_ERROR',
            E_COMPILE_WARNING => 'E_COMPILE_WARNING',
            E_USER_ERROR => 'E_USER_ERROR',
            E_USER_WARNING => 'E_USER_WARNING',
            E_USER_NOTICE => 'E_USER_NOTICE',
            E_STRICT => 'E_STRICT',
            E_RECOVERABLE_ERROR => 'E_RECOVERABLE_ERROR',
            E_DEPRECATED => 'E_DEPRECATED',
            E_USER_DEPRECATED => 'E_USER_DEPRECATED'
        ];
        
        return $names[$severity] ?? "UNKNOWN({$severity})";
    }

    /**
     * Debug method to get current raw metrics data
     * 
     * @return array Raw metrics data for debugging
     */
    public static function getDebugData(): array 
    {
        return [
            'database_metrics' => self::$database_metrics,
            'http_metrics' => self::$http_metrics,
            'cache_metrics' => self::$cache_metrics,
            'wp_metrics' => self::$wp_metrics,
            'static_metrics' => self::$metrics
        ];
    }
}
