<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Page Builder Thumbnail Detector - Specialized detection for page builders
 * 
 * Detects and analyzes thumbnail usage across popular page builders:
 * - Elementor
 * - Divi Builder
 * - Beaver Builder
 * - WPBakery (Visual Composer)
 * - Gutenberg Blocks
 * - Oxygen Builder
 * 
 * Optimized for large datasets with chunked processing
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.5
 */
class PageBuilderThumbnailDetector
{
    /**
     * Supported page builders with their meta keys
     */
    private const BUILDERS = [
        'elementor' => [
            'meta_keys' => ['_elementor_data', '_elementor_page_settings'],
            'name' => 'Elementor',
            'image_patterns' => [
                '/\"image\":\{.*?\"id\":(\d+)/i',
                '/\"background_image\":\{.*?\"id\":(\d+)/i',
                '/\"gallery\":\[.*?\"id\":(\d+)/i'
            ]
        ],
        'divi' => [
            'meta_keys' => ['_et_pb_page_layout', '_et_pb_old_content'],
            'name' => 'Divi Builder',
            'image_patterns' => [
                '/\[et_pb_image.*?src="([^"]*wp-content\/uploads[^"]*)"/i',
                '/\[et_pb_gallery.*?gallery_ids="([^"]*)"/i',
                '/background_image="([^"]*wp-content\/uploads[^"]*)"/i'
            ]
        ],
        'beaver' => [
            'meta_keys' => ['_fl_builder_data', '_fl_builder_draft'],
            'name' => 'Beaver Builder',
            'image_patterns' => [
                '/"photo":"([^"]*wp-content\/uploads[^"]*)"/i',
                '/"gallery":\[.*?"([^"]*wp-content\/uploads[^"]*)"/i',
                '/"bg_image":"([^"]*wp-content\/uploads[^"]*)"/i'
            ]
        ],
        'wpbakery' => [
            'meta_keys' => ['_wpb_vc_js_status', '_wpb_shortcodes_custom_css'],
            'name' => 'WPBakery Page Builder',
            'image_patterns' => [
                '/\[vc_single_image.*?image="(\d+)"/i',
                '/\[vc_gallery.*?images="([^"]*)"/i',
                '/\[vc_row.*?bg_image="(\d+)"/i'
            ]
        ],
        'oxygen' => [
            'meta_keys' => ['ct_builder_shortcodes', 'ct_builder_json'],
            'name' => 'Oxygen Builder',
            'image_patterns' => [
                '/"attachment_id":"(\d+)"/i',
                '/"background-image":"([^"]*wp-content\/uploads[^"]*)"/i'
            ]
        ]
    ];

    /**
     * Gutenberg block patterns for image detection
     */
    private const GUTENBERG_PATTERNS = [
        '/<!-- wp:image.*?"id":(\d+)/i',
        '/<!-- wp:gallery.*?"ids":\[([^\]]*)\]/i',
        '/<!-- wp:cover.*?"id":(\d+)/i',
        '/<!-- wp:media-text.*?"mediaId":(\d+)/i',
        '/"url":"([^"]*wp-content\/uploads[^"]*)"/i'
    ];

    /**
     * Initialize Page Builder Thumbnail Detector
     */
    public static function init(): void
    {
        // No hooks needed - this is an analysis tool
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('PageBuilderThumbnailDetector initialized', 'PBThumbnailDetector');
        }
    }

    /**
     * Detect page builder usage across site
     */
    public static function detect_active_builders(): array
    {
        $active_builders = [];

        foreach (self::BUILDERS as $builder_key => $builder_config) {
            if (self::is_builder_active($builder_key)) {
                $active_builders[$builder_key] = [
                    'name' => $builder_config['name'],
                    'usage_count' => self::count_builder_usage($builder_key),
                    'has_images' => self::builder_has_images($builder_key)
                ];
            }
        }

        // Check for Gutenberg usage
        if (self::has_gutenberg_content()) {
            $active_builders['gutenberg'] = [
                'name' => 'Gutenberg Blocks',
                'usage_count' => self::count_gutenberg_usage(),
                'has_images' => self::gutenberg_has_images()
            ];
        }

        return $active_builders;
    }

    /**
     * Check if specific page builder is active
     */
    private static function is_builder_active(string $builder_key): bool
    {
        switch ($builder_key) {
            case 'elementor':
                return defined('ELEMENTOR_VERSION') || class_exists('\\Elementor\\Plugin');
                
            case 'divi':
                return function_exists('et_divi_fonts_url') || get_template() === 'Divi';
                
            case 'beaver':
                return class_exists('FLBuilder');
                
            case 'wpbakery':
                return defined('WPB_VC_VERSION') || class_exists('Vc_Manager');
                
            case 'oxygen':
                return defined('CT_VERSION') || function_exists('oxygen_vsb_register_condition');
                
            default:
                return false;
        }
    }

    /**
     * Count page builder usage
     */
    private static function count_builder_usage(string $builder_key): int
    {
        if (!isset(self::BUILDERS[$builder_key])) {
            return 0;
        }

        global $wpdb;
        $meta_keys = self::BUILDERS[$builder_key]['meta_keys'];
        $placeholders = implode(',', array_fill(0, count($meta_keys), '%s'));
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} 
             WHERE meta_key IN ({$placeholders})
             AND meta_value != ''",
            ...$meta_keys
        ));

        return (int)$count;
    }

    /**
     * Check if builder has images
     */
    private static function builder_has_images(string $builder_key): bool
    {
        if (!isset(self::BUILDERS[$builder_key])) {
            return false;
        }

        global $wpdb;
        $meta_keys = self::BUILDERS[$builder_key]['meta_keys'];
        $image_patterns = self::BUILDERS[$builder_key]['image_patterns'];
        
        foreach ($meta_keys as $meta_key) {
            $sample_content = $wpdb->get_var($wpdb->prepare(
                "SELECT meta_value FROM {$wpdb->postmeta} 
                 WHERE meta_key = %s 
                 AND meta_value != '' 
                 LIMIT 1",
                $meta_key
            ));

            if ($sample_content) {
                foreach ($image_patterns as $pattern) {
                    if (preg_match($pattern, $sample_content)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    /**
     * Check if site has Gutenberg content
     */
    private static function has_gutenberg_content(): bool
    {
        global $wpdb;
        
        $count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
             WHERE post_content LIKE '%<!-- wp:%' 
             AND post_status = 'publish'"
        );

        return (int)$count > 0;
    }

    /**
     * Count Gutenberg usage
     */
    private static function count_gutenberg_usage(): int
    {
        global $wpdb;
        
        $count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
             WHERE post_content LIKE '%<!-- wp:%' 
             AND post_status = 'publish'"
        );

        return (int)$count;
    }

    /**
     * Check if Gutenberg has images
     */
    private static function gutenberg_has_images(): bool
    {
        global $wpdb;
        
        $sample_content = $wpdb->get_var(
            "SELECT post_content FROM {$wpdb->posts} 
             WHERE post_content LIKE '%<!-- wp:image%' 
             AND post_status = 'publish' 
             LIMIT 1"
        );

        return !empty($sample_content);
    }

    /**
     * Scan page builder content for image usage (optimized for large datasets)
     */
    public static function scan_builder_images(int $batch_size = 50): array
    {
        $results = [
            'total_scanned' => 0,
            'images_found' => [],
            'builder_usage' => [],
            'processing_time' => 0
        ];

        $start_time = microtime(true);
        $active_builders = self::detect_active_builders();

        foreach ($active_builders as $builder_key => $builder_info) {
            if ($builder_key === 'gutenberg') {
                $builder_results = self::scan_gutenberg_images($batch_size);
            } else {
                $builder_results = self::scan_specific_builder_images($builder_key, $batch_size);
            }

            $results['builder_usage'][$builder_key] = $builder_results;
            $results['images_found'] = array_merge($results['images_found'], $builder_results['images']);
            $results['total_scanned'] += $builder_results['posts_scanned'];
        }

        // Remove duplicates
        $results['images_found'] = array_unique($results['images_found']);
        $results['processing_time'] = microtime(true) - $start_time;

        return $results;
    }

    /**
     * Scan specific page builder for images
     */
    private static function scan_specific_builder_images(string $builder_key, int $batch_size): array
    {
        if (!isset(self::BUILDERS[$builder_key])) {
            return ['posts_scanned' => 0, 'images' => []];
        }

        global $wpdb;
        $builder_config = self::BUILDERS[$builder_key];
        $meta_keys = $builder_config['meta_keys'];
        $image_patterns = $builder_config['image_patterns'];
        
        $results = [
            'posts_scanned' => 0,
            'images' => []
        ];

        $offset = 0;
        
        do {
            // Get posts with this builder's meta keys
            $placeholders = implode(',', array_fill(0, count($meta_keys), '%s'));
            $posts = $wpdb->get_results($wpdb->prepare(
                "SELECT DISTINCT pm.post_id, pm.meta_value 
                 FROM {$wpdb->postmeta} pm
                 INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
                 WHERE pm.meta_key IN ({$placeholders})
                 AND pm.meta_value != ''
                 AND p.post_status = 'publish'
                 LIMIT %d OFFSET %d",
                ...array_merge($meta_keys, [$batch_size, $offset])
            ));

            foreach ($posts as $post) {
                $results['posts_scanned']++;
                $content = $post->meta_value;
                
                // Extract images using patterns
                foreach ($image_patterns as $pattern) {
                    if (preg_match_all($pattern, $content, $matches)) {
                        foreach ($matches[1] as $match) {
                            if (is_numeric($match)) {
                                // Attachment ID
                                $results['images'][] = (int)$match;
                            } else {
                                // URL - extract ID from URL
                                $attachment_id = self::extract_attachment_id_from_url($match);
                                if ($attachment_id) {
                                    $results['images'][] = $attachment_id;
                                }
                            }
                        }
                    }
                }
            }

            $offset += $batch_size;
            
        } while (count($posts) === $batch_size);

        return $results;
    }

    /**
     * Scan Gutenberg blocks for images
     */
    private static function scan_gutenberg_images(int $batch_size): array
    {
        global $wpdb;
        
        $results = [
            'posts_scanned' => 0,
            'images' => []
        ];

        $offset = 0;
        
        do {
            $posts = $wpdb->get_results($wpdb->prepare(
                "SELECT ID, post_content FROM {$wpdb->posts} 
                 WHERE post_content LIKE '%<!-- wp:%'
                 AND post_status = 'publish'
                 LIMIT %d OFFSET %d",
                $batch_size,
                $offset
            ));

            foreach ($posts as $post) {
                $results['posts_scanned']++;
                $content = $post->post_content;
                
                // Extract images using Gutenberg patterns
                foreach (self::GUTENBERG_PATTERNS as $pattern) {
                    if (preg_match_all($pattern, $content, $matches)) {
                        foreach ($matches[1] as $match) {
                            if (is_numeric($match)) {
                                $results['images'][] = (int)$match;
                            } elseif (strpos($match, ',') !== false) {
                                // Gallery IDs (comma separated)
                                $ids = explode(',', $match);
                                foreach ($ids as $id) {
                                    $id = trim($id);
                                    if (is_numeric($id)) {
                                        $results['images'][] = (int)$id;
                                    }
                                }
                            } else {
                                // URL
                                $attachment_id = self::extract_attachment_id_from_url($match);
                                if ($attachment_id) {
                                    $results['images'][] = $attachment_id;
                                }
                            }
                        }
                    }
                }
            }

            $offset += $batch_size;
            
        } while (count($posts) === $batch_size);

        return $results;
    }

    /**
     * Extract attachment ID from image URL
     */
    private static function extract_attachment_id_from_url(string $url): ?int
    {
        // Try to get attachment ID from URL
        $attachment_id = attachment_url_to_postid($url);
        
        if ($attachment_id) {
            return $attachment_id;
        }

        // Fallback: extract filename and search
        $filename = basename($url);
        if (empty($filename)) {
            return null;
        }

        global $wpdb;
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} 
             WHERE post_type = 'attachment' 
             AND guid LIKE %s",
            '%' . $filename
        ));

        return $attachment_id ? (int)$attachment_id : null;
    }

    /**
     * Analyze page builder thumbnail usage efficiency
     */
    public static function analyze_builder_efficiency(): array
    {
        $analysis = [
            'active_builders' => self::detect_active_builders(),
            'image_usage' => [],
            'recommendations' => []
        ];

        // Scan for images used in builders
        $builder_images = self::scan_builder_images(100);
        $analysis['image_usage'] = $builder_images;

        // Get all thumbnail sizes
        $all_sizes = ImageThumbnailManager::get_all_thumbnail_sizes();
        
        // Analyze which sizes are actually used by builders
        $size_usage = [];
        foreach ($builder_images['images_found'] as $image_id) {
            $metadata = wp_get_attachment_metadata($image_id);
            if ($metadata && !empty($metadata['sizes'])) {
                foreach ($metadata['sizes'] as $size_name => $size_data) {
                    if (!isset($size_usage[$size_name])) {
                        $size_usage[$size_name] = 0;
                    }
                    $size_usage[$size_name]++;
                }
            }
        }

        // Generate recommendations
        foreach ($all_sizes as $size_name => $size_info) {
            $usage_count = $size_usage[$size_name] ?? 0;
            $total_images = count($builder_images['images_found']);
            
            if ($total_images > 0) {
                $usage_rate = ($usage_count / $total_images) * 100;
                
                if ($usage_rate < 10 && $size_info['source'] !== 'wordpress') {
                    $analysis['recommendations'][] = [
                        'type' => 'optimization',
                        'priority' => 'medium',
                        'title' => "Low Page Builder Usage: {$size_name}",
                        'description' => "Thumbnail size '{$size_name}' is used in only {$usage_rate}% of page builder images",
                        'action' => 'consider_disabling',
                        'size_name' => $size_name,
                        'source' => $size_info['source']
                    ];
                }
            }
        }

        return $analysis;
    }

    /**
     * Get page builder statistics
     */
    public static function get_builder_statistics(): array
    {
        $active_builders = self::detect_active_builders();
        $total_builder_posts = 0;
        $total_builder_images = 0;

        foreach ($active_builders as $builder_data) {
            $total_builder_posts += $builder_data['usage_count'];
        }

        // Quick scan for image count
        if (!empty($active_builders)) {
            $image_scan = self::scan_builder_images(50);
            $total_builder_images = count($image_scan['images_found']);
        }

        return [
            'active_builders' => $active_builders,
            'total_builder_count' => count($active_builders),
            'total_builder_posts' => $total_builder_posts,
            'total_builder_images' => $total_builder_images,
            'scan_performance' => [
                'optimized_scanning' => true,
                'batch_processing' => true,
                'memory_efficient' => true
            ]
        ];
    }
}
