<?php

namespace ZiziCache;

use ZiziCache\CSSEnhanced;

/**
 * Class OptiCore
 *
 * Handles the core optimization pipeline for ZiziCache.
 * Responsible for output buffering, running all optimization steps (HTML, CSS, JS, images, fonts, iframes, CDN, etc.),
 * and injecting cache footprints and headers.
 *
 * @package ZiziCache
 */
class OptiCore
{
    /**
     * Check if a string is valid JSON
     *
     * @param string $string String to check
     * @return bool True if the string is valid JSON
     */
    public static function is_json($string) 
    {
        if (!is_string($string)) {
            return false;
        }
        
        // Trim the string
        $string = trim($string);
        
        // Quick check for JSON prefix
        if ($string === '' || substr($string, 0, 1) !== '{' && substr($string, 0, 1) !== '[') {
            return false;
        }
        
        // Check if it's a valid JSON
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }
    
    /**
     * Check if content is likely HTML
     *
     * @param string $content Content to check
     * @return bool True if content is likely HTML
     */
    public static function is_html($content)
    {
        if (!is_string($content)) {
            return false;
        }
        
        // Trim whitespace
        $content = trim($content);
        
        // Look for common HTML patterns
        return (
            stripos($content, '<!DOCTYPE') === 0 || 
            stripos($content, '<html') !== false ||
            (stripos($content, '<body') !== false && stripos($content, '</body>') !== false) ||
            (stripos($content, '<div') !== false && stripos($content, '</div>') !== false)
        );
    }

    /**
     * Initialize the optimization core.
     *
     * Sets up output buffering and registers hooks for configuration refresh and admin notices.
     *
     * @return void
     */
    public static function init()
    {
        // Do not start output buffering for admin or REST API requests.
        // Buffering modifies HTML and can accidentally corrupt JSON responses used by REST endpoints.
        if (defined('REST_REQUEST') && REST_REQUEST) {
            // Respect REST requests - do not interfere
            return;
        }

        // Also skip when REQUEST_URI clearly targets WP REST API
        if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
            return;
        }

        // Only start output buffering on frontend, not in admin
        if (!is_admin()) {
            ob_start([__CLASS__, 'process_output']);
        }
        
        JavaScript::init();
        Redundancy::init();
        CSS::init();

        // Add a hook to refresh configs after admin changes
        add_action('admin_init', function() {
            // Ensure config is refreshed after admin actions
            if (isset($_POST['zizi_cache_update_settings']) || isset($_GET['zizi_cache_clear_cache'])) {
                SysConfig::$config = get_option('ZIZI_CACHE_CONFIG', []);
            }
        });

        // Note: Removed Font::display_admin_warnings as the method doesn't exist
    }

    /**
     * Process the output buffer and run all optimization steps.
     *
     * Applies multiple optimization techniques to the content including:
     * - Font optimizations (display swap, preloading)
     * - CSS optimizations (minification, removing unused CSS)
     * - IFrame optimizations (lazy loading, placeholders)
     * - Image optimizations (lazy loading, responsive images)
     * - JavaScript optimizations (minification, deferring, self-hosting)
     * - CDN optimization and rewriting
     * - Cache headers configuration
     *
     * @param string $content The HTML content to optimize
     * @param int $phase Output buffering phase (optional, from ob_start callback)
     * @return string Optimized HTML content
     */
    private static function process_output($content, $phase = 0)
    {
        // Skip processing for page builder templates and admin areas
        if (is_embed() || is_feed() || is_preview() || is_customize_preview() || 
            is_singular(['bricks_template', 'elementor_library', 'ct_template', 'et_pb_layout'])) {
            return $content;
        }
        
        // Skip AMP pages
        if (function_exists('is_amp_endpoint') && is_amp_endpoint()) {
            return $content;
        }
        
        // Rychlá kontrola, pokud je obsah prázdný nebo velmi malý
        if (!is_string($content) || strlen(trim($content)) < 10) {
            return $content;
        }
        
        // Skip processing for REST API responses (JSON output)
        if (self::is_json($content)) {
            return $content;
        }
        
        // Skip processing if content is not HTML
        if (!self::is_html($content)) {
            return $content;
        }
        
        $config = SysConfig::$config;
        
        // CRITICAL: Send Early Hints BEFORE any HTML output (HTTP 103)
        // This must happen before any content is sent to browser
        do_action('zizi_cache_before_optimizations');
        
        // Early exit: Check if any optimizations are enabled at all
        $has_optimizations = isset($config) && is_array($config) && (
            ($config['fonts_optimize_google_fonts'] ?? false) ||
            ($config['fonts_display_swap'] ?? false) ||
            ($config['css_minify'] ?? false) ||
            ($config['css_rucss'] ?? false) ||
            ($config['js_minify'] ?? false) ||
            ($config['js_defer'] ?? false) ||
            ($config['js_delay'] ?? false) ||
            ($config['js_user_interaction_delay'] ?? false) ||
            ($config['js_lazy_render'] ?? false) ||
            ($config['img_lazyload'] ?? false) ||
            ($config['img_width_height'] ?? false) ||
            $config['img_preload'] ||
            $config['img_responsive'] ||
            $config['img_priority_hints'] ||
            $config['img_auto_lcp'] ||
            $config['iframe_lazyload'] ||
            $config['iframe_youtube_placeholder'] ||
            $config['iframe_vimeo_placeholder'] ||
            $config['self_host_third_party_css_js'] ||
            $config['redundancy_remove_google_fonts'] ||
            $config['redundancy_remove_dashicons'] ||
            !empty($config['fonts_preload_urls'])
        );
        
        // Early exit: Check if caching is completely disabled
        $cache_enabled = (
            $config['cache_lifespan'] !== 'never' &&
            ($config['cache_logged_in'] || !function_exists('is_user_logged_in') || !is_user_logged_in())
        );
        
        // If no optimizations and no caching, skip everything
        if (!$has_optimizations && !$cache_enabled) {
            return $content;
        }
        
        // For caching, check if content is cacheable
        $is_cacheable = $cache_enabled && CacheSys::is_cacheable($content);
        
        // Apply optimizations if enabled OR if caching (optimizations apply to cached content)
        if ($has_optimizations || $is_cacheable) {
            // Remove redundant Google Fonts - controlled by dedicated config option
            if (!empty($config['redundancy_remove_google_fonts'])) {
                $content = Redundancy::remove_google_fonts($content);
            }

            // Font optimizations (only if fonts are enabled)
            if (!empty($config['fonts_enabled'])) {
                // Font display swap optimization
                if (!empty($config['fonts_display_swap'])) {
                    $content = Font::add_display_swap_to_internal_styles($content);
                    $content = Font::add_display_swap_to_google_fonts($content);
                }
                
                // Inject optimized Google Fonts based on Font Statistics
                if (!empty($config['fonts_optimize_google_fonts'])) {
                    $content = Font::inject_optimized_google_fonts($content);
                }
                
                // Optimize existing Google Fonts links (download and self-host)
                if (!empty($config['fonts_optimize_google_fonts'])) {
                    $content = Font::optimize_google_fonts($content);
                }
                
                // Font preloading
                if (!empty($config['fonts_preload_urls']) || !empty($config['fonts_optimize_google_fonts'])) {
                    $content = Font::preload_fonts($content);
                }
            }

            // CSS optimizations
            // Use enhanced CSS optimization if Sabberworm parser is available
            if (class_exists('Sabberworm\CSS\Parser') && 
                (!empty($config['css_critical']) || !empty($config['css_rucss']) || !empty($config['css_minify']))) {
                $content = CSSEnhanced::optimize($content);
            } else {
                // Fallback to basic CSS optimization
                $content = CSS::minify($content);
                $content = CSS::remove_unused_css($content);
            }
            $content = CSS::self_host_third_party_css($content);

            // IFrame optimizations
            $content = IFrame::add_youtube_placeholder($content);
            $content = IFrame::add_vimeo_placeholder($content);
            $content = IFrame::lazy_load($content);

            // Image optimizations
            Image::parse_images($content);
            Image::add_width_height($content);
            Image::localhost_gravatars($content);
            Image::exclude_above_fold($content);
            Image::lazy_load($content);
            Image::responsive_images($content);

            // Additional image optimizations
            Image::auto_detect_lcp($content); // Automatically detect LCP images
            Image::apply_priority_hints($content); // Apply Priority Hints for images
            $content = Image::write_images($content);
            $content = Image::preload($content); // Preload images
            $content = Image::lazy_load_bg_style($content);
            $content = Image::lazy_load_bg_class($content);

            // JavaScript optimizations
            $content = JavaScript::minify($content);
            $content = JavaScript::self_host_third_party_js($content);
            $content = JavaScript::move_module_scripts($content);
            $content = JavaScript::defer_external($content);
            $content = JavaScript::defer_inline($content);
            $content = JavaScript::delay_scripts($content);
            $content = JavaScript::user_interaction_delay($content); // NEW: Advanced user interaction delay
            $content = JavaScript::enhanced_lazy_render($content);
            $content = JavaScript::replace_lazy_render_markers($content);

            $content = JavaScript::inject_core_lib($content);
            $content = JavaScript::inject_lazy_render_lib($content);

            // CDN optimizations
            $content = CDN::add_preconnect($content);
            $content = CDN::rewrite($content);

            // Add font optimization debug information when debug is enabled
            $content = Font::add_font_optimization_debug_info($content);

            // Remove cache busting query strings
            $content = preg_replace('/\?cache_bust=\d+&#038;/', '?', $content);
            $content = preg_replace('/\?cache_bust=\d+&/', '?', $content);
            $content = preg_replace('/\?cache_bust=\d+/', '', $content);
            $content = preg_replace('/cache_bust%3D\d+(%26)?/', '', $content);
            $content = apply_filters('zizi_cache_optimization:after', $content);

            // Add optimization footprint only if optimizations were applied
            if ($has_optimizations) {
                $date_format = get_option('date_format');
                $time_format = get_option('time_format');
                if (strpos($time_format, 's') === false) {
                    $time_format .= ':s';
                }
                $content = str_replace('</body>', '<!-- ZiziCache: Page optimized ' . current_time("$date_format $time_format") . ' --></body>', $content);
            }

            // If the page is generated as part of a real preload (from PreloadSqlite), add a "warmed" signature
            if (isset($_GET['zizi_warmed']) && $_GET['zizi_warmed'] == '1') {
                $date_format = get_option('date_format');
                $time_format = get_option('time_format');
                if (strpos($time_format, 's') === false) {
                    $time_format .= ':s';
                }
                $datetime = date_i18n("$date_format $time_format", current_time('timestamp'));
                
                $footprint_warmed = "ZiziCache. Cached at: $datetime warmed";
                
                // Only append if the content appears to be HTML (contains HTML tags)
                if (preg_match('/<[^>]+>/', $content)) {
                    $content .= apply_filters('zizi_cache_footprint', "<!-- $footprint_warmed -->");
                }
            }

            // Cache the optimized page only if cacheable OR if this is a warmer request
            $is_warmer_request = isset($_GET['zizi_preload']) && $_GET['zizi_preload'] == '1' && 
                                isset($_GET['zizi_warmed']) && $_GET['zizi_warmed'] == '1';
            
            if ($is_cacheable || $is_warmer_request) {
                CacheSys::cache_page($content);

                // Set cache headers
                header('Cache-Tag: ' . $_SERVER['HTTP_HOST']);
                header('CDN-Cache-Control: max-age=2592000');
                header('Cache-Control: no-cache, must-revalidate');
            }
        }

        // If cache_bust is set and cache_preload is enabled, trigger preload with delay
        if (isset($_GET['cache_bust']) && $config['cache_preload']) {
            // Delay preload. Default is 0.5 seconds.
            $delay = apply_filters('zizi_cache_preload_delay', 0.5);
            usleep($delay * 1000000);
            Preload::preload_available();
        }

        return $content;
    }
}
