<?php

namespace ZiziCache;

use ZiziCache\SysConfig;
use ZiziCache\PerformanceMonitor;
use ZiziCache\Security;

/**
 * Handles the creation and removal of the object-cache.php drop-in for Redis integration.
 *
 * This class manages the object-cache.php drop-in file and updates wp-config.php with the necessary
 * Redis constants based on the plugin configuration. It ensures proper activation and deactivation
 * of Redis object caching for WordPress.
 *
 * @package ZiziCache
 */
class ObjectCacheHandler {

    /**
     * Initializes the ObjectCacheHandler by registering activation and deactivation hooks.
     *
     * On activation and deactivation, ensures the object-cache.php drop-in is removed.
     *
     * @return void
     */
    public static function init() {
        register_deactivation_hook(ZIZI_CACHE_FILE, [__CLASS__, 'on_deactivation']);
        // Remove any existing drop-in on activation; creation is handled via update_config
        register_activation_hook(ZIZI_CACHE_FILE, [__CLASS__, 'remove_dropin']);
        add_action('zizi_cache_update_config:after', [__CLASS__, 'maybe_update_dropin_from_config']);
    }

    /**
     * Creates or updates the object-cache.php drop-in and updates wp-config.php with Redis constants.
     *
     * This method generates the object-cache.php drop-in file using the current Redis configuration
     * and writes the necessary constants to wp-config.php for Redis integration.
     *
     * @return array Result array with status and message
     */
    public static function add_dropin() {
        $fullConfig = SysConfig::$config;
        if (empty($fullConfig['redis_enabled'])) {
            return ['status' => false, 'message' => 'Redis not enabled in configuration'];
        }
        
        $template = file_get_contents(ZIZI_CACHE_PLUGIN_DIR . 'assets/templates/object-cache.php');
        $redisKeys = [
            'redis_enabled',
            'redis_host',
            'redis_password',
            'redis_port',
            'redis_key_salt',
            'redis_timeout',
            'redis_read_timeout',
            'redis_retry_interval',
            'redis_maxttl',
            'redis_database',
            'redis_global_groups',
            'redis_graceful',
        ];
        $redisConfig = array_intersect_key($fullConfig, array_flip($redisKeys));
        
        // Set reasonable default values based on the reference Redis plugin
        // Only set defaults for TCP connections, not Unix sockets
        $isConfigUnixSocket = !empty($redisConfig['redis_host']) && 
                             (strpos($redisConfig['redis_host'], '/') === 0 || 
                              strpos($redisConfig['redis_host'], 'unix:') === 0);
                              
        if (!$isConfigUnixSocket) {
            // Apply TCP defaults only for non-Unix socket connections
            $redisConfig = array_merge([
                'redis_host' => '127.0.0.1',
                'redis_port' => 6379,
                'redis_database' => 0,
                'redis_timeout' => 1,
                'redis_read_timeout' => 1,
                'redis_retry_interval' => 100,
                'redis_maxttl' => 0,
                'redis_graceful' => true,
            ], $redisConfig);
        } else {
            // Apply only common defaults for Unix socket connections (no HOST/PORT defaults)
            $redisConfig = array_merge([
                'redis_database' => 0,
                'redis_timeout' => 1,
                'redis_read_timeout' => 1,
                'redis_retry_interval' => 100,
                'redis_maxttl' => 0,
                'redis_graceful' => true,
            ], $redisConfig);
        }
        
        // Handle Unix socket vs TCP connection
        $isUnixSocket = !empty($redisConfig['redis_host']) && 
                       (strpos($redisConfig['redis_host'], '/') === 0 || 
                        strpos($redisConfig['redis_host'], 'unix:') === 0);
        
        if ($isUnixSocket) {
            // For Unix sockets, ensure port is 0 and host is properly formatted
            $redisConfig['redis_port'] = 0;
            // Remove 'unix:' prefix if present for cleaner socket path
            if (strpos($redisConfig['redis_host'], 'unix:') === 0) {
                $redisConfig['redis_host'] = substr($redisConfig['redis_host'], 5);
            }
        } else {
            // TCP connection - ensure that Redis host is not empty (problem tcp://:6379)
            if (empty($redisConfig['redis_host']) || trim($redisConfig['redis_host']) === '') {
                $redisConfig['redis_host'] = '127.0.0.1';
            }
            
            // Ensure that port is a valid number for TCP connections
            if (empty($redisConfig['redis_port']) || !is_numeric($redisConfig['redis_port'])) {
                $redisConfig['redis_port'] = 6379;
            } else {
                $redisConfig['redis_port'] = (int) $redisConfig['redis_port'];
            }
        }            // Ensure valid timeout values
        if (empty($redisConfig['redis_timeout']) || !is_numeric($redisConfig['redis_timeout'])) {
            $redisConfig['redis_timeout'] = 1;
        }
        
        if (empty($redisConfig['redis_read_timeout']) || !is_numeric($redisConfig['redis_read_timeout'])) {
            $redisConfig['redis_read_timeout'] = 1;
        }
        
        if (!empty($redisConfig['redis_password'])) {
            $redisConfig['redis_password'] = Security::decrypt_password($redisConfig['redis_password']);
        }
        
        $exported = var_export($redisConfig, true);
        // Replace only the CONFIG_PLACEHOLDER in the assignment line
        $content = preg_replace_callback(
            '/(\$config\s*=\s*)CONFIG_PLACEHOLDER(\s*;)/',
            function($matches) use ($exported) {
                return $matches[1] . $exported . $matches[2];
            },
            $template
        );
        
            // Try to create the drop-in file
        try {
            if (!is_dir(WP_CONTENT_DIR)) {
                return ['status' => false, 'message' => 'WP_CONTENT_DIR does not exist or is not accessible'];
            }
            
            if (!is_writable(WP_CONTENT_DIR)) {
                return ['status' => false, 'message' => 'WP_CONTENT_DIR is not writable'];
            }
            
            file_put_contents(WP_CONTENT_DIR . '/object-cache.php', $content);
            
                // Check that the file was actually created
            if (!file_exists(WP_CONTENT_DIR . '/object-cache.php')) {
                return ['status' => false, 'message' => 'Failed to create object-cache.php file'];
            }
        } catch (\Exception $e) {
            return ['status' => false, 'message' => 'Exception occurred: ' . $e->getMessage()];
        }
        
        // Log drop-in creation to plugin log
        $logConfig = $redisConfig;
        if (isset($logConfig['redis_password'])) {
            $logConfig['redis_password'] = !empty($logConfig['redis_password']) ? '********' : '';
        }
        
        CacheSys::writeLog(sprintf(
            '[ZiziCache ObjectCache] object-cache.php created with Redis config: %s',
            print_r($logConfig, true)
        ));
        
        // Update wp-config.php with Redis constants
        $configResult = self::update_wp_config($redisConfig);
        
        return [
            'status' => true, 
            'message' => 'Object cache drop-in created successfully',
            'config_status' => $configResult['status'],
            'config_message' => $configResult['message']
        ];
    }

    /**
     * Updates wp-config.php with Redis constants
     * 
     * @param array $redisConfig Redis configuration
     * @return array Result with status and message
     */
    private static function update_wp_config($redisConfig) {
        $configFile = ABSPATH . 'wp-config.php';
        
        if (!is_file($configFile)) {
            return ['status' => false, 'message' => 'wp-config.php not found'];
        }
        
        if (!is_writable($configFile)) {
            return ['status' => false, 'message' => 'wp-config.php is not writable'];
        }
        
        try {
            $wpConfig = file_get_contents($configFile);
            
                // Detect Unix socket connection
            $redisHost = $redisConfig['redis_host'] ?? '127.0.0.1';
            $redisPort = $redisConfig['redis_port'] ?? 6379;
            $isUnixSocket = false;
            
                // Unix socket detection: path starts with "/" or port is 0
            if (strpos($redisHost, '/') === 0 || $redisPort == 0) {
                $isUnixSocket = true;
            }
            
            $constants = [
                'WP_REDIS_PASSWORD'      => $redisConfig['redis_password'] ?? '',
                'WP_REDIS_DATABASE'      => $redisConfig['redis_database'] ?? 0,
                'WP_REDIS_PREFIX'        => $redisConfig['redis_key_salt'] ?? '',
                'WP_REDIS_TIMEOUT'       => $redisConfig['redis_timeout'] ?? 1,
                'WP_REDIS_READ_TIMEOUT'  => $redisConfig['redis_read_timeout'] ?? 1,
                'WP_REDIS_MAXTTL'        => $redisConfig['redis_maxttl'] ?? 0,
                'WP_REDIS_RETRY_INTERVAL'=> $redisConfig['redis_retry_interval'] ?? 100,
                'WP_REDIS_GRACEFUL'      => $redisConfig['redis_graceful'] ?? true,
            ];
            
                // Set connection type (Unix socket vs TCP)
            if ($isUnixSocket) {
                $constants['WP_REDIS_SCHEME'] = 'unix';
                $constants['WP_REDIS_PATH'] = $redisHost;
                // Port se pro Unix socket nepoužívá
            } else {
                $constants['WP_REDIS_HOST'] = $redisHost;
                $constants['WP_REDIS_PORT'] = $redisPort;
            }
            
            // Remove any existing Redis configuration block
            $wpConfig = preg_replace('/\/\*\*\s*Start Redis Configuration - ZiziCache.*?End Redis Configuration - ZiziCache.*?\*\//s', '', $wpConfig);
            
            // Build Redis configuration block
            $block = "/** Start Redis Configuration - ZiziCache. */\n";
            
            foreach ($constants as $const => $value) {
                $block .= "define('" . $const . "', " . var_export($value, true) . ");\n";
            }
            $block .= "/** End Redis Configuration - ZiziCache. */\n\n";
            
            // Insert block at top after <?php (preserve original formatting)
            $wpConfig = preg_replace('/(<\?php)(\s*\r?\n)/', "$1$2" . $block, $wpConfig, 1);
            file_put_contents($configFile, $wpConfig);
            
            return ['status' => true, 'message' => 'wp-config.php updated successfully'];
        } catch (\Exception $e) {
            return ['status' => false, 'message' => 'Failed to update wp-config.php: ' . $e->getMessage()];
        }
    }

    /**
     * Removes the object-cache.php drop-in and cleans up Redis constants from wp-config.php.
     *
     * This method deletes the object-cache.php drop-in file and removes any Redis-related
     * constants or configuration blocks from wp-config.php.
     *
     * @return array Result with status and message
     */
    public static function remove_dropin() {
        $results = ['status' => true, 'messages' => []];
        
        // Remove object-cache.php file
        $file = WP_CONTENT_DIR . '/object-cache.php';
        if (is_file($file)) {
            try {
                if (!is_writable($file)) {
                    $results['messages'][] = 'object-cache.php is not writable';
                    $results['status'] = false;
                } else {
                    unlink($file);
                    $results['messages'][] = 'object-cache.php removed successfully';
                }
            } catch (\Exception $e) {
                $results['messages'][] = 'Failed to remove object-cache.php: ' . $e->getMessage();
                $results['status'] = false;
            }
        } else {
            $results['messages'][] = 'object-cache.php not found';
        }
        
        // Remove Redis constants from wp-config.php
        $configFile = ABSPATH . 'wp-config.php';
        if (is_file($configFile)) {
            if (!is_writable($configFile)) {
                $results['messages'][] = 'wp-config.php is not writable';
                $results['status'] = false;
            } else {
                try {
                    $wpConfig = file_get_contents($configFile);
                    $originalConfig = $wpConfig;
                    
                    // Multiple cleanup strategies for thorough removal
                    
                    // 1. Remove complete Redis block with improved regex
                    $wpConfig = preg_replace('/\/\*\*[\s\*]*Start Redis Configuration - ZiziCache.*?End Redis Configuration - ZiziCache.*?\*\/\s*(\r?\n)*/s', '', $wpConfig);
                    
                    // 2. Remove any remaining individual Redis constants (more comprehensive)
                    $wpConfig = preg_replace("/^\s*define\\(\s*['\"]WP_REDIS_[A-Z_]+['\"]\\s*,.*?\\);\s*(\r?\n)?/m", '', $wpConfig);
                    
                    // 3. Remove any Redis-related comments that might remain
                    $wpConfig = preg_replace('/^\s*\/\/.*Redis.*$/m', '', $wpConfig);
                    
                    // 4. Clean up excessive empty lines
                    $wpConfig = preg_replace('/(\r?\n){3,}/', "\n\n", $wpConfig);
                    
                    if ($wpConfig !== $originalConfig) {
                        file_put_contents($configFile, $wpConfig);
                        $results['messages'][] = 'Redis constants completely removed from wp-config.php';
                    } else {
                        $results['messages'][] = 'No Redis constants found in wp-config.php';
                    }
                } catch (\Exception $e) {
                    $results['messages'][] = 'Failed to update wp-config.php: ' . $e->getMessage();
                    $results['status'] = false;
                }
            }
        } else {
            $results['messages'][] = 'wp-config.php not found';
        }
        
        return $results;
    }

    /**
     * Conditionally updates the object-cache.php drop-in based on new configuration.
     *
     * If Redis is enabled in the new configuration, the drop-in is created or updated.
     * Otherwise, the drop-in is removed.
     *
     * @param array $newConfig The new plugin configuration array.
     * @return array Result with status and message
     */
    public static function maybe_update_dropin($newConfig) {
        if (!empty($newConfig['redis_enabled'])) {
            return static::add_dropin();
        } else {
            return static::remove_dropin();
        }
    }
    
    /**
     * Checks Redis connection and returns diagnostics.
     * 
     * @return array Diagnostic results
     */
    public static function check_redis_connection() {
        // Zkontrolujeme, zda existuje object-cache.php
        if (!file_exists(WP_CONTENT_DIR . '/object-cache.php')) {
            return [
                'status' => false,
                'message' => 'Redis object cache drop-in not installed',
                'diagnostics' => []
            ];
        }
        
        // Zkontrolujeme, zda je Redis aktivní (TCP nebo Unix socket)
        $hasRedisConfig = (defined('WP_REDIS_HOST') && !empty(WP_REDIS_HOST)) ||
                         (defined('WP_REDIS_SCHEME') && WP_REDIS_SCHEME === 'unix' && defined('WP_REDIS_PATH') && !empty(WP_REDIS_PATH));
        
        if (!$hasRedisConfig) {
            return [
                'status' => false,
                'message' => 'Redis not properly configured in wp-config.php (missing host or Unix socket path)',
                'diagnostics' => []
            ];
        }
        
        $diagnostics = [];
        
        // Základní diagnostika konfigurace
        $diagnostics['config'] = [
            'host' => defined('WP_REDIS_HOST') ? WP_REDIS_HOST : 'not set',
            'port' => defined('WP_REDIS_PORT') ? WP_REDIS_PORT : 'not set',
            'scheme' => defined('WP_REDIS_SCHEME') ? WP_REDIS_SCHEME : 'tcp',
            'path' => defined('WP_REDIS_PATH') ? WP_REDIS_PATH : 'not set',
            'database' => defined('WP_REDIS_DATABASE') ? WP_REDIS_DATABASE : 'not set',
            'timeout' => defined('WP_REDIS_TIMEOUT') ? WP_REDIS_TIMEOUT : 'not set',
            'read_timeout' => defined('WP_REDIS_READ_TIMEOUT') ? WP_REDIS_READ_TIMEOUT : 'not set',
            'retry_interval' => defined('WP_REDIS_RETRY_INTERVAL') ? WP_REDIS_RETRY_INTERVAL : 'not set',
        ];
        
        // Zkontrolujeme přítomnost rozšíření PHP pro Redis
        $diagnostics['extensions'] = [
            'phpredis' => extension_loaded('redis'),
            'predis' => class_exists('Predis\\Client'),
            'relay' => extension_loaded('relay'),
        ];
        
        // Zkontrolujeme připojení k Redis serveru
        try {
            // Pokusíme se získat instanci redis objektu
            global $wp_object_cache;
            
            if (!isset($wp_object_cache) || !method_exists($wp_object_cache, 'redis_status')) {
                return [
                    'status' => false,
                    'message' => 'WordPress object cache not properly initialized',
                    'diagnostics' => $diagnostics
                ];
            }
            
            $redis_status = $wp_object_cache->redis_status();
            $diagnostics['connection'] = [
                'status' => $redis_status,
                'client' => $wp_object_cache->redis_instance() ? get_class($wp_object_cache->redis_instance()) : 'unknown',
                'version' => $wp_object_cache->redis_version(),
            ];
            
            if (isset($wp_object_cache->diagnostics)) {
                $diagnostics['details'] = $wp_object_cache->diagnostics;
            }
            
            if (isset($wp_object_cache->errors) && !empty($wp_object_cache->errors)) {
                $diagnostics['errors'] = $wp_object_cache->errors;
            }
            
            return [
                'status' => $redis_status,
                'message' => $redis_status ? 'Redis connection successful' : 'Redis connection failed',
                'diagnostics' => $diagnostics
            ];
            
        } catch (\Exception $e) {
            return [
                'status' => false,
                'message' => 'Exception occurred during Redis connection check: ' . $e->getMessage(),
                'diagnostics' => $diagnostics
            ];
        }
    }

    /**
     * Plugin deactivation hook - handles Redis cleanup and configuration reset
     *
     * @param string $plugin Plugin basename
     * @return void
     */
    public static function on_deactivation($plugin) {
        if ($plugin === ZIZI_CACHE_FILE_NAME) {
            // Log deactivation start
            CacheSys::writeLog('Redis deactivation hook triggered for plugin: ' . $plugin);
            
            // Flush Redis if available
            try {
                if (file_exists(WP_CONTENT_DIR . '/object-cache.php')) {
                    global $wp_object_cache;
                    if (isset($wp_object_cache) && method_exists($wp_object_cache, 'flush')) {
                        $wp_object_cache->flush();
                        CacheSys::writeLog('Redis cache flushed during deactivation');
                    }
                }
            } catch (\Exception $e) {
                CacheSys::writeLog('Redis flush during deactivation failed: ' . $e->getMessage());
            }
            
            // Always remove drop-in
            $removeResult = static::remove_dropin();
            CacheSys::writeLog('Object cache drop-in removal result: ' . json_encode($removeResult));
            
            // Clear Redis config to prevent reactivation (KEY FIX!)
            $config = SysConfig::$config;
            if (!empty($config['redis_enabled'])) {
                $config['redis_enabled'] = false;
                SysConfig::update_config($config);
                CacheSys::writeLog('Redis disabled in config during plugin deactivation');
            } else {
                CacheSys::writeLog('Redis was already disabled in config');
            }
        } else {
            CacheSys::writeLog('Deactivation hook called for different plugin: ' . $plugin . ', expected: ' . ZIZI_CACHE_FILE_NAME);
        }
    }

    /**
     * Action hook handler for config updates
     *
     * @return array Result with status and message
     */
    public static function maybe_update_dropin_from_config() {
        return static::maybe_update_dropin(SysConfig::$config);
    }
}