<?php
/**
 * Handles OPcache management for ZiziCache: statistics, flush functionality, and integration with WP-CLI.
 *
 * This class provides methods to manage PHP OPcache, including statistics retrieval, flushing, and integration with
 * WP-CLI and AJAX. It also supports automatic OPcache flush after WordPress upgrades.
 *
 * @package ZiziCache
 */

namespace ZiziCache;

class OPcache
{
    /**
     * Initializes hooks for OPcache management: CLI commands and auto-flush after upgrade.
     *
     * Registers WP-CLI commands and hooks for automatic OPcache flush after WordPress updates.
     * Also registers an AJAX handler for OPcache domain usage.
     *
     * @return void
     */
    public static function init()
    {
        // Register WP-CLI commands
        if (defined('WP_CLI') && WP_CLI) {
            \WP_CLI::add_command('zizi-cache opcache flush', [__CLASS__, 'cli_flush']);
            \WP_CLI::add_command('zizi-cache opcache stats', [__CLASS__, 'cli_stats']);
        }

        // Automatically flush after WordPress update if enabled in config
        add_action('upgrader_process_complete', [__CLASS__, 'maybe_auto_flush'], 10, 2);

        // AJAX handler for OPcache domain usage (for admin-ajax.php)
        add_action('wp_ajax_zizi_opcache_domain_usage', function() {
            try {
                if (!current_user_can('manage_options')) {
                    \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache: Unauthorized AJAX call');
                    wp_send_json_error('Unauthorized', 403);
                }
                if (!class_exists(__CLASS__)) {
                    \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache: OPcache class missing!');
                    wp_send_json_error('OPcache class missing', 500);
                }
                // FIX: use self:: instead of ZiziCache\OPcache::
                $data = self::get_domain_memory_usage();
                if (isset($data['error'])) {
                    \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache: OPcache error: ' . $data['error']);
                }
                wp_send_json_success($data);
            } catch (\Throwable $e) {
                \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache AJAX fatal: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
                wp_send_json_error('Fatal: ' . $e->getMessage(), 500);
            }
        });
    }

    /**
     * Registers REST API routes for OPcache management.
     *
     * Adds endpoints for OPcache statistics, flush, and domain usage.
     *
     * @return void
     */
    public static function register_rest_routes()
    {
        $namespace = basename(dirname(ZIZI_CACHE_FILE));
        register_rest_route($namespace, '/opcache/stats', [
            'methods'             => 'GET',
            'callback'            => [__CLASS__, 'rest_stats'],
            'permission_callback' => [Authority::class, 'is_allowed'],
        ]);
        register_rest_route($namespace, '/opcache/flush', [
            'methods'             => 'POST',
            'callback'            => [__CLASS__, 'rest_flush'],
            'permission_callback' => [Authority::class, 'is_allowed'],
        ]);
        register_rest_route($namespace, '/opcache/domain-usage', [
            'methods'             => 'GET',
            'callback'            => [__CLASS__, 'rest_domain_usage'],
            'permission_callback' => [Authority::class, 'is_allowed'],
        ]);
    }

    /**
     * REST callback: Return basic OPcache statistics.
     *
     * @param \WP_REST_Request $request The REST request object.
     * @return \WP_REST_Response The response containing OPcache statistics.
     */
    public static function rest_stats(\WP_REST_Request $request)
    {
        return rest_ensure_response(self::get_basic_stats());
    }

    /**
     * REST callback: Flush PHP OPcache.
     *
     * @param \WP_REST_Request $request The REST request object.
     * @return \WP_REST_Response|\WP_Error The response or error after attempting to flush OPcache.
     */
    public static function rest_flush(\WP_REST_Request $request)
    {
        if (self::flush()) {
            return rest_ensure_response(['success' => true]);
        }
        return new \WP_Error('opcache_flush_failed', 'OPcache flush failed.');
    }

    /**
     * Conditionally flushes OPcache after a WordPress update if enabled in configuration.
     *
     * @param object $upgrader The upgrader object.
     * @param array  $options  The options array for the upgrade process.
     * @return void
     */
    public static function maybe_auto_flush($upgrader, $options)
    {
        if (!empty($options['action']) && $options['action'] === 'update' &&
            !empty($options['type']) && in_array($options['type'], ['core','plugin','theme'], true) &&
            SysConfig::$config['opcache_auto_flush_after_upgrade']
        ) {
            self::flush();
        }
    }

    /**
     * Retrieves basic OPcache statistics.
     *
     * @return array Array of OPcache statistics and memory usage.
     */
    public static function get_basic_stats(): array
    {
        $status = function_exists('opcache_get_status') ? opcache_get_status() : [];
        $config = function_exists('opcache_get_configuration') ? opcache_get_configuration() : ['directives' => []];
        $memory = $status['memory_usage'] ?? [];
        $stats  = $status['opcache_statistics'] ?? [];
        $mem_limit = $config['directives']['opcache.memory_consumption'] ?? 0;

        $opcache_version = '';
        if (function_exists('opcache_get_configuration')) {
            $config_data = opcache_get_configuration();
            if (isset($config_data['version']['opcache_version'])) {
                $opcache_version = $config_data['version']['opcache_version'];
            } elseif (extension_loaded('Zend OPcache')) {
                $opcache_version = phpversion('Zend OPcache');
            }
        } elseif (extension_loaded('Zend OPcache')) {
            $opcache_version = phpversion('Zend OPcache');
        }

        return [
            'total_memory'    => round($mem_limit / 1024 / 1024, 2),
            'used_memory'     => round(($memory['used_memory'] ?? 0) / 1024 / 1024, 2),
            'free_memory'     => round(($memory['free_memory'] ?? 0) / 1024 / 1024, 2),
            'wasted_memory'   => round(($memory['wasted_memory'] ?? 0) / 1024 / 1024, 2),
            'hit_rate'        => round($stats['opcache_hit_rate'] ?? 0, 2),
            'num_cached_keys' => $stats['num_cached_keys'] ?? 0,
            'max_cached_keys' => $stats['max_cached_keys'] ?? 0,
            'opcache_version' => $opcache_version,
            'php_version'     => phpversion(),
            'host'            => function_exists('gethostname') ? gethostname() : php_uname('n'),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? '',
            // --- Additional detailed statistics ---
            'cached_files'        => $stats['num_cached_scripts'] ?? 0,
            'hits'                => $stats['hits'] ?? 0,
            'misses'              => $stats['misses'] ?? 0,
            'blacklist_misses'    => $stats['blacklist_misses'] ?? 0,
            'interned_strings'    => [
                'buffer_size'       => $status['interned_strings_usage']['buffer_size'] ?? 0,
                'used_memory'       => $status['interned_strings_usage']['used_memory'] ?? 0,
                'free_memory'       => $status['interned_strings_usage']['free_memory'] ?? 0,
                'number_of_strings' => $status['interned_strings_usage']['number_of_strings'] ?? 0,
            ],
        ];
    }

    /**
     * Flushes the PHP OPcache.
     *
     * @return bool True if OPcache was flushed successfully, false otherwise.
     */
    public static function flush(): bool
    {
        try {
            // Check if OPcache is enabled and available
            if (!extension_loaded('Zend OPcache')) {
                CacheSys::writeLog('[ERROR] ZiziCache: OPcache flush failed: Zend OPcache extension is not loaded');
                return false;
            }
            
            if (!function_exists('opcache_reset')) {
                CacheSys::writeLog('[ERROR] ZiziCache: OPcache flush failed: opcache_reset function does not exist');
                return false;
            }
            
            // Check if OPcache is restricted by API configuration
            if (ini_get('opcache.restrict_api') && 
                !in_array($_SERVER['SERVER_ADDR'] ?? '', explode(',', ini_get('opcache.restrict_api')))) {
                CacheSys::writeLog('[ERROR] ZiziCache: OPcache flush failed: OPcache is restricted via opcache.restrict_api setting');
                return false;
            }
            
            // Try to flush OPcache
            $result = opcache_reset();
            
            if ($result) {
                CacheSys::writeLog('[INFO] ZiziCache: OPcache flush succeeded');
            } else {
                CacheSys::writeLog('[ERROR] ZiziCache: OPcache flush failed: Unknown reason');
            }
            
            return $result;
        } catch (\Throwable $e) {
            CacheSys::writeLog('[ERROR] ZiziCache: Exception during OPcache flush: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
            return false;
        }
    }

    /**
     * WP-CLI handler: Flushes OPcache via command line.
     *
     * @param array $args      Positional arguments.
     * @param array $assoc_args Associative arguments.
     * @return void
     */
    public static function cli_flush($args, $assoc_args)
    {
        if (self::flush()) {
            \WP_CLI::success('PHP OPcache flushed.');
        } else {
            \WP_CLI::error('Failed to flush PHP OPcache.');
        }
    }

    /**
     * WP-CLI handler: Prints OPcache statistics via command line.
     *
     * @param array $args      Positional arguments.
     * @param array $assoc_args Associative arguments.
     * @return void
     */
    public static function cli_stats($args, $assoc_args)
    {
        $stats = self::get_basic_stats();
        \WP_CLI::print_value($stats);
    }

    /**
     * Gets OPcache memory usage grouped by domain and type (API source).
     * Falls back to single-domain analysis if multi-domain data is unavailable.
     *
     * @return array Array of OPcache memory usage grouped by domain and type.
     */
    public static function get_domain_memory_usage(): array
    {
        if (!extension_loaded('Zend OPcache') || !function_exists('opcache_get_status')) {
            return ['error' => 'Zend OPcache is not available on this server.'];
        }

        // Try multi-domain analysis first
        $opcache_status = @opcache_get_status();
        if ($opcache_status !== false && isset($opcache_status['scripts']) && is_array($opcache_status['scripts'])) {
            $grouped_domains = self::analyze_multi_domain($opcache_status['scripts']);
            if (!empty($grouped_domains) && !isset($grouped_domains['Unknown domain'])) {
                return $grouped_domains;
            }
        }

        // Fallback to single-domain analysis
        return self::get_single_domain_usage();
    }

    /**
     * Analyzes OPcache scripts for multi-domain hosting environments.
     *
     * @param array $scripts Scripts array from opcache_get_status().
     * @return array Grouped domain usage data.
     */
    private static function analyze_multi_domain(array $scripts): array
    {
        $grouped_domains = [];
        foreach ($scripts as $script) {
            if (!isset($script['full_path'], $script['memory_consumption'])) {
                continue;
            }

            $script_name = $script['full_path'];
            $domain = self::extract_domain_from_path($script_name);
            $plugin_type = self::get_plugin_type_from_path($script_name);

            if (!isset($grouped_domains[$domain])) {
                $grouped_domains[$domain] = [
                    'WordPress Core' => 0,
                    'Theme' => 0,
                    'Plugins' => [],
                ];
            }

            if ($plugin_type === 'Plugin') {
                $plugin_name = 'Unknown plugin';
                if (preg_match('~wp-content/plugins/([^/]+)/~', $script_name, $matches)) {
                    $plugin_name = $matches[1];
                }
                if (!isset($grouped_domains[$domain]['Plugins'][$plugin_name])) {
                    $grouped_domains[$domain]['Plugins'][$plugin_name] = 0;
                }
                $grouped_domains[$domain]['Plugins'][$plugin_name] += $script['memory_consumption'];
            } elseif ($plugin_type === 'Theme' || $plugin_type === 'WordPress Core') {
                $grouped_domains[$domain][$plugin_type] += $script['memory_consumption'];
            }
        }

        // Convert to MB
        foreach ($grouped_domains as $domain => &$types) {
            $types['WordPress Core'] = round($types['WordPress Core'] / 1024 / 1024, 2);
            $types['Theme'] = round($types['Theme'] / 1024 / 1024, 2);
            foreach ($types['Plugins'] as $plugin => &$mem) {
                $mem = round($mem / 1024 / 1024, 2);
            }
        }

        return $grouped_domains;
    }

    /**
     * Gets OPcache usage for current WordPress installation only.
     * Used as fallback when multi-domain analysis is not available.
     *
     * @return array Single domain usage data or error.
     */
    public static function get_single_domain_usage(): array
    {
        $cache_key = 'zizi_opcache_single_domain';
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        try {
            $status = @opcache_get_status(true);
            if (!is_array($status) || empty($status['scripts'])) {
                return ['error' => 'No detailed OPcache scripts available.'];
            }

            CacheSys::writeLog('[INFO] Starting single domain OPcache analysis with ' . count($status['scripts']) . ' scripts');
            
            $result = self::analyze_single_domain_scripts($status['scripts']);
            
            // Cache for 2 minutes (shorter for better responsiveness)
            set_transient($cache_key, $result, 120);
            return $result;

        } catch (\Throwable $e) {
            CacheSys::writeLog('[ERROR] Single domain OPcache analysis failed: ' . $e->getMessage());
            return ['error' => 'Failed to analyze OPcache data.'];
        }
    }

    /**
     * Analyzes scripts for current WordPress installation.
     *
     * @param array $scripts Scripts from opcache_get_status(true).
     * @return array Analyzed usage data.
     */
    private static function analyze_single_domain_scripts(array $scripts): array
    {
        $wp_path = defined('ABSPATH') ? realpath(ABSPATH) : '';
        if (!$wp_path) {
            return ['error' => 'Cannot determine WordPress path.'];
        }

        $wp_path = rtrim(str_replace('\\', '/', $wp_path), '/');
        $core_total = 0;
        $theme_total = 0;
        $other_total = 0;
        $plugin_usage = [];

        // Get active plugins - simplified approach for better compatibility
        $plugin_paths = [];
        $wp_plugins_dir = defined('WP_PLUGIN_DIR') ? str_replace('\\', '/', WP_PLUGIN_DIR) : $wp_path . '/wp-content/plugins';
        
        if (function_exists('get_option')) {
            $active_plugins = get_option('active_plugins', []);
            foreach ($active_plugins as $plugin) {
                $plugin_slug = dirname($plugin);
                if ($plugin_slug && $plugin_slug !== '.') {
                    $plugin_dir = $wp_plugins_dir . '/' . $plugin_slug;
                    $plugin_paths[$plugin_dir] = $plugin_slug;
                }
            }
        }

        // Get theme info
        $theme_name = 'Unknown Theme';
        $theme_path = '';
        if (function_exists('get_stylesheet_directory')) {
            $theme_path = str_replace('\\', '/', get_stylesheet_directory());
        }
        if (function_exists('wp_get_theme')) {
            $theme = wp_get_theme();
            $theme_name = $theme->get('Name') ?: $theme_name;
        }

        // Debug: Log first few scripts for analysis
        $debug_scripts = array_slice($scripts, 0, 5);
        CacheSys::writeLog('[DEBUG] OPcache first 5 scripts: ' . print_r(array_column($debug_scripts, 'full_path'), true));
        CacheSys::writeLog('[DEBUG] WP Path: ' . $wp_path);
        CacheSys::writeLog('[DEBUG] Plugin paths: ' . print_r(array_keys($plugin_paths), true));

        // Process all scripts
        foreach ($scripts as $script) {
            if (!isset($script['full_path'], $script['memory_consumption'])) {
                continue;
            }

            $path = str_replace('\\', '/', $script['full_path']);
            $mem = $script['memory_consumption'];

            // WordPress Core detection (more flexible)
            if (strpos($path, $wp_path . '/wp-includes/') === 0 ||
                strpos($path, $wp_path . '/wp-admin/') === 0 ||
                (strpos($path, $wp_path . '/') === 0 && 
                 preg_match('~/wp-[^/]*\.php$~', $path) && 
                 substr_count($path, '/') === substr_count($wp_path, '/') + 1)) {
                $core_total += $mem;
                continue;
            }

            // Plugin detection (improved)
            $matched_plugin = false;
            foreach ($plugin_paths as $plugin_dir => $plugin_slug) {
                if (strpos($path, $plugin_dir . '/') === 0) {
                    if (!isset($plugin_usage[$plugin_slug])) {
                        $plugin_usage[$plugin_slug] = 0;
                    }
                    $plugin_usage[$plugin_slug] += $mem;
                    $matched_plugin = true;
                    break;
                }
            }
            if ($matched_plugin) continue;

            // Generic plugin detection as fallback
            if (strpos($path, '/wp-content/plugins/') !== false) {
                if (preg_match('~/wp-content/plugins/([^/]+)/~', $path, $matches)) {
                    $plugin_slug = $matches[1];
                    if (!isset($plugin_usage[$plugin_slug])) {
                        $plugin_usage[$plugin_slug] = 0;
                    }
                    $plugin_usage[$plugin_slug] += $mem;
                    continue;
                }
            }

            // Theme detection
            if ($theme_path && strpos($path, $theme_path . '/') === 0) {
                $theme_total += $mem;
                continue;
            }

            // Generic theme detection as fallback
            if (strpos($path, '/wp-content/themes/') !== false) {
                $theme_total += $mem;
                continue;
            }

            $other_total += $mem;
        }

        // Get domain name
        $domain_name = 'localhost';
        if (function_exists('get_site_url')) {
            $site_url = get_site_url();
            if ($site_url) {
                $parsed = parse_url($site_url);
                $domain_name = $parsed['host'] ?? 'localhost';
            }
        }

        // Build result structure
        $result = [
            $domain_name => [
                'WordPress Core' => round($core_total / 1024 / 1024, 2),
                'Theme' => round($theme_total / 1024 / 1024, 2),
                'Plugins' => [],
            ]
        ];

        // Add plugins with memory usage > 0, with proper names
        foreach ($plugin_usage as $plugin_slug => $mem) {
            if ($mem > 0) {
                // Try to get nice plugin name
                $plugin_name = $plugin_slug;
                if (function_exists('get_plugins')) {
                    $all_plugins = get_plugins();
                    foreach ($all_plugins as $plugin_file => $plugin_data) {
                        if (dirname($plugin_file) === $plugin_slug) {
                            $plugin_name = $plugin_data['Name'] ?? $plugin_slug;
                            break;
                        }
                    }
                }
                $result[$domain_name]['Plugins'][$plugin_name] = round($mem / 1024 / 1024, 2);
            }
        }

        // Add other if significant
        if ($other_total > 0) {
            $result[$domain_name]['Other'] = round($other_total / 1024 / 1024, 2);
        }

        // Debug log final result
        CacheSys::writeLog('[DEBUG] OPcache single domain result: ' . print_r($result, true));

        return $result;
    }

    /**
     * Extracts the domain name from a file path.
     *
     * @param string $path The file path.
     * @return string The extracted domain name or 'Unknown domain'.
     */
    private static function extract_domain_from_path($path)
    {
        if (preg_match('~/([^/]+\.[^/]+)/~', $path, $matches)) {
            return $matches[1];
        }
        return 'Unknown domain';
    }

    /**
     * Determines the plugin type from a script file path.
     *
     * @param string $scriptPath The script file path.
     * @return string Returns 'Plugin', 'Theme', 'WordPress Core', or 'Unknown'.
     */
    private static function get_plugin_type_from_path($scriptPath)
    {
        if (strpos($scriptPath, '/wp-content/plugins/') !== false) {
            return 'Plugin';
        } elseif (strpos($scriptPath, '/wp-content/themes/') !== false) {
            return 'Theme';
        } elseif (strpos($scriptPath, '/wp-includes/') !== false) {
            return 'WordPress Core';
        } else {
            return 'Unknown';
        }
    }
    }

