<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Media Library Indexer - Automatic FTP Upload Detection
 * 
 * Provides automatic detection and indexing of manually uploaded images
 * when WordPress Media Library is accessed.
 * 
 * @package ZiziCache
 * @since 0.5.2-beta.7
 */
class MediaLibraryIndexer
{
    /**
     * In-memory cache for indexed files to avoid repeated SQL queries
     */
    private static $indexed_files_cache = null;
    
    /**
     * Cache timestamp to ensure cache freshness
     */
    private static $cache_timestamp = null;
    
    /**
     * Cache validity duration (5 minutes)
     */
    private const CACHE_VALIDITY = 300;

    /**
     * Supported image MIME types
     */
    private const SUPPORTED_MIME_TYPES = [
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg', 
        'png' => 'image/png',
        'gif' => 'image/gif',
        'webp' => 'image/webp',
        'avif' => 'image/avif',
        'bmp' => 'image/bmp',
        'tiff' => 'image/tiff',
        'tif' => 'image/tiff'
    ];

    /**
     * Initialize the automatic indexer
     */
    public static function init(): void
    {
        // Automatic indexing when Media Library is accessed
        add_action('load-upload.php', [__CLASS__, 'maybe_auto_index_uploads']);
        
        // REMOVED: wp_ajax_query-attachments hook to prevent interference with upload process
        // The indexer will only run when user actually visits Media Library page
        
        // Additional hook for media-related admin pages and dashboard widgets  
        add_action('load-index.php', [__CLASS__, 'maybe_auto_index_uploads']); // Dashboard
        
        // OPTIMIZED: Less aggressive admin scanning - only on specific pages
        add_action('load-post.php', [__CLASS__, 'maybe_auto_index_uploads_admin']); // Edit post
        add_action('load-post-new.php', [__CLASS__, 'maybe_auto_index_uploads_admin']); // New post
        // REMOVED: General admin_init hook to reduce query load
        
        // Admin action for manual indexing (keep for API compatibility)
        add_action('wp_ajax_zizi_index_uploads', [__CLASS__, 'ajax_index_uploads']);
        
        // Initialize comprehensive media deletion handler
        if (class_exists('\\ZiziCache\\MediaDeletionHandler')) {
            \ZiziCache\MediaDeletionHandler::init();
        }
        
        // Log initialization
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('MediaLibraryIndexer initialized with comprehensive MediaDeletionHandler', 'MediaLibraryIndexer');
        }
    }

    /**
     * Maybe run automatic indexing - only if needed and not too recently
     */
    public static function maybe_auto_index_uploads(): void
    {
        // Don't run if not admin or insufficient permissions
        if (!current_user_can('upload_files')) {
            return;
        }

        // CRITICAL FIX: Check throttling FIRST before any file scanning
        // This prevents unnecessary SQL queries when indexing is throttled
        $last_scan = get_transient('zizi_media_indexer_last_auto_scan');
        if ($last_scan && (time() - $last_scan) < 300) { // 5 minutes
            return; // Exit early - no SQL queries executed
        }

        // Quick check: are there any unindexed files?
        // Now with optimized cache this is much faster
        $quick_stats = self::get_unindexed_stats(true); // Light scan mode
        if (!$quick_stats || $quick_stats['unindexed_files'] === 0) {
            // Update throttle even if no files to prevent frequent rechecks
            set_transient('zizi_media_indexer_last_auto_scan', time(), 3600);
            return; // No files to process
        }

        // We have files to process and throttling allows it
        // Log automatic indexing start
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Automatic indexing started: {$quick_stats['unindexed_files']} unindexed files found",
                'MediaLibraryIndexer'
            );
        }

        // Run scan with limited batch size to avoid performance issues
        $result = self::scan_and_index_uploads(50); // Limit to 50 files per auto-scan

        // Update last scan time
        set_transient('zizi_media_indexer_last_auto_scan', time(), 3600);

        // Log results
        if (class_exists('\\ZiziCache\\CacheSys') && $result['indexed_files'] > 0) {
            \ZiziCache\CacheSys::writeLog(
                "Automatic indexing completed: {$result['indexed_files']} files indexed",
                'MediaLibraryIndexer'
            );
        }
    }

    /**
     * Maybe run automatic indexing in admin with longer throttling
     */
    public static function maybe_auto_index_uploads_admin(): void
    {
        // Don't run if not admin or insufficient permissions
        if (!current_user_can('upload_files') || !is_admin()) {
            return;
        }

        // CRITICAL FIX: Check longer throttle FIRST before any file scanning
        $last_admin_scan = get_transient('zizi_media_indexer_last_admin_scan');
        if ($last_admin_scan && (time() - $last_admin_scan) < 900) { // 15 minutes
            return; // Exit early - no SQL queries executed
        }

        // Quick check: are there any unindexed files?
        $quick_stats = self::get_unindexed_stats(true); // Light scan mode
        if (!$quick_stats || $quick_stats['unindexed_files'] === 0) {
            // Update throttle even if no files to prevent frequent rechecks
            set_transient('zizi_media_indexer_last_admin_scan', time(), 7200);
            return; // No files to process
        }

        // We have files to process and throttling allows it
        // Log automatic indexing start
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Admin automatic indexing started: {$quick_stats['unindexed_files']} unindexed files found",
                'MediaLibraryIndexer'
            );
        }

        // Run scan with smaller batch size to avoid admin slowdown
        $result = self::scan_and_index_uploads(25); // Limit to 25 files per admin auto-scan

        // Update last admin scan time
        set_transient('zizi_media_indexer_last_admin_scan', time(), 7200);

        // Log results
        if (class_exists('\\ZiziCache\\CacheSys') && $result['indexed_files'] > 0) {
            \ZiziCache\CacheSys::writeLog(
                "Admin automatic indexing completed: {$result['indexed_files']} files indexed",
                'MediaLibraryIndexer'
            );
        }
    }

    /**
     * AJAX handler for indexing uploads (legacy API compatibility)
     */
    public static function ajax_index_uploads(): void
    {
        // Security check
        if (!current_user_can('upload_files')) {
            wp_die(__('Insufficient permissions'));
        }

        check_ajax_referer('zizi_media_indexer', 'nonce');

        $result = self::scan_and_index_uploads();
        
        wp_send_json_success($result);
    }

    /**
     * Scan uploads directory and index unregistered images
     * 
     * @param int $limit Maximum number of files to process per run (0 = no limit)
     * @return array Results with statistics
     */
    public static function scan_and_index_uploads(int $limit = 0): array
    {
        $upload_dir = wp_upload_dir();
        if (empty($upload_dir['basedir']) || !is_dir($upload_dir['basedir'])) {
            return [
                'success' => false,
                'message' => 'Uploads directory not accessible',
                'scanned_files' => 0,
                'indexed_files' => 0,
                'skipped_files' => 0
            ];
        }

        $result = [
            'success' => true,
            'message' => '',
            'scanned_files' => 0,
            'indexed_files' => 0,
            'skipped_files' => 0,
            'indexed_file_list' => [],
            'errors' => []
        ];

        try {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator(
                    $upload_dir['basedir'], 
                    \RecursiveDirectoryIterator::SKIP_DOTS
                ),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );

            $processed = 0;
            foreach ($iterator as $file) {
                if (!$file->isFile()) continue;

                $file_path = $file->getRealPath();
                $extension = strtolower($file->getExtension());

                // Skip if not a supported image type
                if (!isset(self::SUPPORTED_MIME_TYPES[$extension])) {
                    continue;
                }

                $result['scanned_files']++;

                // Skip if already indexed
                if (self::is_file_indexed($file_path)) {
                    $result['skipped_files']++;
                    continue;
                }

                // Stop if we hit the limit AFTER checking if file needs indexing
                if ($limit > 0 && $processed >= $limit) {
                    break;
                }
                
                $processed++;

                // Try to index the file
                try {
                    $attachment_id = self::index_file($file_path);
                    if ($attachment_id) {
                        $result['indexed_files']++;
                        $result['indexed_file_list'][] = basename($file_path);
                        
                        // Log successful indexing
                        if (class_exists('\\ZiziCache\\CacheSys')) {
                            \ZiziCache\CacheSys::writeLog(
                                "Indexed file: " . basename($file_path) . " as attachment ID: $attachment_id",
                                'MediaLibraryIndexer'
                            );
                        }
                    } else {
                        $result['skipped_files']++;
                        $result['errors'][] = "Failed to index: " . basename($file_path);
                    }
                } catch (\Exception $e) {
                    $result['skipped_files']++;
                    $result['errors'][] = "Error indexing " . basename($file_path) . ": " . $e->getMessage();
                    
                    // Log errors
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeError(
                            "MediaLibraryIndexer error: " . $e->getMessage(),
                            'MediaLibraryIndexer'
                        );
                    }
                }
            }

            // Create result message
            $result['message'] = sprintf(
                'Scan completed: %d files scanned, %d indexed, %d skipped',
                $result['scanned_files'],
                $result['indexed_files'],
                $result['skipped_files']
            );

        } catch (\Exception $e) {
            $result['success'] = false;
            $result['message'] = 'Error during scan: ' . $e->getMessage();
            $result['errors'][] = $e->getMessage();
        }

        return $result;
    }

    /**
     * Load all indexed files into cache using single SQL query
     * This replaces hundreds of individual queries with one batch operation
     */
    private static function load_indexed_files_cache(): void
    {
        global $wpdb;
        
        // Load all indexed files with single query
        $results = $wpdb->get_results(
            "SELECT meta_value 
             FROM {$wpdb->postmeta} 
             WHERE meta_key = '_wp_attached_file'",
            ARRAY_A
        );
        
        // Create lookup array for O(1) access
        $indexed_files = [];
        foreach ($results as $row) {
            if (!empty($row['meta_value'])) {
                // Normalize path for consistent lookup
                $normalized_path = ltrim(str_replace('\\', '/', $row['meta_value']), '/');
                $indexed_files[$normalized_path] = true;
            }
        }
        
        self::$indexed_files_cache = $indexed_files;
        self::$cache_timestamp = time();
        
        // Log cache loading
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Loaded " . count($indexed_files) . " indexed files into cache (batch operation)",
                'MediaLibraryIndexer'
            );
        }
    }

    /**
     * Check if cache is valid and fresh
     */
    private static function is_cache_valid(): bool
    {
        return self::$indexed_files_cache !== null 
               && self::$cache_timestamp !== null 
               && (time() - self::$cache_timestamp) < self::CACHE_VALIDITY;
    }

    /**
     * Clear the indexed files cache
     */
    private static function clear_cache(): void
    {
        self::$indexed_files_cache = null;
        self::$cache_timestamp = null;
    }

    /**
     * Check if file is already indexed in WordPress Media Library
     */
    private static function is_file_indexed(string $file_path): bool
    {
        // Ensure cache is loaded and valid
        if (!self::is_cache_valid()) {
            self::load_indexed_files_cache();
        }
        
        // Get relative path from uploads base directory
        $upload_dir = wp_upload_dir();
        $relative_path = str_replace($upload_dir['basedir'] . '/', '', $file_path);
        $relative_path = str_replace('\\', '/', $relative_path);
        
        // Normalize the relative path (remove leading slashes)
        $normalized_relative_path = ltrim($relative_path, '/\\');
        
        // O(1) cache lookup instead of SQL query
        return isset(self::$indexed_files_cache[$normalized_relative_path]);
    }

    /**
     * Index a single file into WordPress Media Library
     */
    private static function index_file(string $file_path): ?int
    {
        if (!file_exists($file_path) || !is_readable($file_path)) {
            return null;
        }

        // Get file info
        $file_info = pathinfo($file_path);
        $extension = strtolower($file_info['extension']);
        
        // Validate MIME type
        if (!isset(self::SUPPORTED_MIME_TYPES[$extension])) {
            return null;
        }

        // Generate relative path early for locking
        $upload_dir = wp_upload_dir();
        
        // Normalize both paths to forward slashes for comparison
        $normalized_basedir = str_replace('\\', '/', $upload_dir['basedir']);
        $normalized_file_path = str_replace('\\', '/', $file_path);
        
        $relative_path = str_replace($normalized_basedir . '/', '', $normalized_file_path);
        $relative_path = ltrim($relative_path, '/\\');
        
        // Skip thumbnail files - WordPress generates thumbnails with -WIDTHxHEIGHT pattern
        // Examples: image-150x150.jpg, image-300x200.png, image-1024x768.webp
        if (preg_match('/-\d+x\d+\.(' . implode('|', array_keys(self::SUPPORTED_MIME_TYPES)) . ')$/i', basename($file_path))) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Skipped thumbnail file: " . basename($file_path),
                    'MediaLibraryIndexer'
                );
            }
            return null; // This is a WordPress thumbnail, skip it
        }
        
        // Skip scaled images (WordPress 5.3+) - these have -scaled suffix
        if (preg_match('/-scaled\.(' . implode('|', array_keys(self::SUPPORTED_MIME_TYPES)) . ')$/i', basename($file_path))) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Skipped scaled image: " . basename($file_path),
                    'MediaLibraryIndexer'
                );
            }
            return null; // This is a WordPress scaled image, skip it  
        }
        
        // Skip rotated images - these have -rotated suffix  
        if (preg_match('/-rotated\.(' . implode('|', array_keys(self::SUPPORTED_MIME_TYPES)) . ')$/i', basename($file_path))) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Skipped rotated image: " . basename($file_path),
                    'MediaLibraryIndexer'
                );
            }
            return null; // This is a WordPress rotated image, skip it
        }

        // Skip converted files - these should never be registered as separate attachments
        $original_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'tiff', 'tif'];
        $converted_extensions = ['webp', 'avif'];
        
        $file_extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        $filename_without_ext = pathinfo($file_path, PATHINFO_FILENAME);
        
        // If this is a converted file (.webp, .avif), ALWAYS check if original exists
        if (in_array($file_extension, $converted_extensions)) {
            // For converted files, the original should have same base name but different extension
            // Examples:
            // - "image.jpg.webp" -> check for "image.jpg"  
            // - "image-150x150.jpg.webp" -> check for "image-150x150.jpg"
            // - "image.png.avif" -> check for "image.png"
            
            foreach ($original_extensions as $orig_ext) {
                $original_file = dirname($file_path) . '/' . $filename_without_ext . '.' . $orig_ext;
                if (file_exists($original_file)) {
                    return null; // Skip converted file - original exists
                }
            }
            
            // Also skip if there's ANY image file with same base name (without converted extension)
            // This handles cases where plugin converts files but original might have different extension
            $base_name = $filename_without_ext;
            
            // Remove any trailing extension from base name (e.g., "image.jpg" -> "image")
            if (preg_match('/^(.+)\.(jpg|jpeg|png|gif|bmp|tiff|tif)$/i', $base_name, $matches)) {
                $true_base_name = $matches[1];
                
                foreach ($original_extensions as $orig_ext) {
                    $possible_original = dirname($file_path) . '/' . $true_base_name . '.' . $orig_ext;
                    if (file_exists($possible_original)) {
                        return null; // Skip converted file - corresponding original exists
                    }
                }
            }
        }
        
        // ATOMIC CHECK: Use transient locking to prevent race conditions
        $lock_key = 'zizi_indexing_' . md5($relative_path);
        if (get_transient($lock_key)) {
            // Another process is already indexing this file
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "File already being indexed (locked): " . basename($file_path),
                    'MediaLibraryIndexer'
                );
            }
            return null;
        }
        
        // Set lock for 30 seconds
        set_transient($lock_key, time(), 30);

        $mime_type = self::SUPPORTED_MIME_TYPES[$extension];
        
        // Check if file is already indexed (using relative path generated earlier)
        $existing = get_posts([
            'post_type' => 'attachment',
            'meta_query' => [
                [
                    'key' => '_wp_attached_file',
                    'value' => $relative_path,
                    'compare' => '='
                ]
            ],
            'posts_per_page' => 1,
            'fields' => 'ids'
        ]);

        if (!empty($existing)) {
            return $existing[0]; // File already indexed
        }
        
        // Create attachment post
        $attachment_data = [
            'guid' => $upload_dir['baseurl'] . '/' . $relative_path,
            'post_mime_type' => $mime_type,
            'post_title' => sanitize_file_name($file_info['filename']),
            'post_content' => '',
            'post_status' => 'inherit'
        ];

        // Insert attachment
        $attachment_id = wp_insert_attachment($attachment_data, $file_path);
        
        if (is_wp_error($attachment_id)) {
            delete_transient($lock_key);
            return null;
        }

        // CRITICAL FOR DELETION: Update attachment file path with normalized relative path
        // This ensures WordPress can properly locate and delete the file later
        update_post_meta($attachment_id, '_wp_attached_file', $relative_path);

        // CACHE INVALIDATION: Clear cache since we added a new indexed file
        self::clear_cache();

        // Get image dimensions to update basic metadata (but don't generate thumbnails)
        $image_size = @getimagesize($file_path);
        if ($image_size) {
            $metadata = [
                'width' => $image_size[0],
                'height' => $image_size[1],
                'file' => $relative_path
            ];
            
            // Update metadata - critical for proper deletion
            wp_update_attachment_metadata($attachment_id, $metadata);
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Indexed file with dimensions: " . basename($file_path) . 
                    " ({$image_size[0]}x{$image_size[1]})",
                    'MediaLibraryIndexer'
                );
            }
        }
        
        // Clean up lock
        delete_transient($lock_key);
        
        return $attachment_id;
    }

    /**
     * Get statistics about unindexed media files
     * 
     * @param bool $light_scan Whether to do a quick light scan (sample only)
     * @return array Statistics
     */
    public static function get_unindexed_stats(bool $light_scan = false): array
    {
        $upload_dir = wp_upload_dir();
        if (empty($upload_dir['basedir']) || !is_dir($upload_dir['basedir'])) {
            return ['error' => 'Uploads directory not accessible'];
        }

        $stats = [
            'total_files' => 0,
            'unindexed_files' => 0,
            'converted_files' => 0,
            'total_size' => 0
        ];

        try {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator(
                    $upload_dir['basedir'], 
                    \RecursiveDirectoryIterator::SKIP_DOTS
                ),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );

            $scanned = 0;
            $max_scan = $light_scan ? 20 : 0; // OPTIMIZED: Reduced from 200 to 20 files for light scan

            foreach ($iterator as $file) {
                if (!$file->isFile()) continue;

                // For light scan, limit the number of files checked
                if ($light_scan && $scanned >= $max_scan) {
                    break;
                }

                $file_path = $file->getRealPath();
                $extension = strtolower($file->getExtension());

                // Check if it's a supported image type
                if (isset(self::SUPPORTED_MIME_TYPES[$extension])) {
                    $stats['total_files']++;
                    $stats['total_size'] += $file->getSize();
                    
                    // Check if indexed
                    if (!self::is_file_indexed($file_path)) {
                        $stats['unindexed_files']++;
                    }
                    
                    $scanned++;
                }

                // Check for converted files (webp/avif)
                if (in_array($extension, ['webp', 'avif'])) {
                    $stats['converted_files']++;
                }
            }

        } catch (\Exception $e) {
            return ['error' => 'Error scanning directory: ' . $e->getMessage()];
        }

        return $stats;
    }

    /**
     * Validate and repair attachment file metadata
     * 
     * @param int $attachment_id Attachment ID to validate
     * @return array Results of validation/repair
     */
    public static function validate_attachment_metadata(int $attachment_id): array
    {
        return \ZiziCache\MediaDeletionHandler::validate_and_repair_attachment_metadata($attachment_id);
    }
}