<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Media Deletion Handler - Simplified Version
 * 
 * Ensures WordPress can properly delete physical files when attachments 
 * are deleted manually from Media Library, especially for FTP uploaded files.
 * 
 * This handler only fixes metadata to enable WordPress's built-in deletion,
 * without adding any automatic behavior.
 * 
 * @package ZiziCache
 * @since 0.5.3-beta.6
 */
class MediaDeletionHandler
{
    /**
     * Initialize the media deletion handler
     */
    public static function init(): void
    {
        error_log("MediaDeletionHandler: init() called");
        
        // Add admin action handler for media deletion - runs early in admin_init
        add_action('admin_init', [__CLASS__, 'handle_admin_deletion'], 5);
        
        // Keep the direct deletion hooks as backup
        add_action('before_delete_post', [__CLASS__, 'before_delete_attachment'], 0);
        add_action('delete_attachment', [__CLASS__, 'ensure_proper_metadata'], 0, 2);
    }
    
    /**
     * Handle admin deletion requests to fix metadata before WordPress processes them
     */
    public static function handle_admin_deletion(): void
    {
        // Check if this is a delete request from media library
        if (!is_admin() || !current_user_can('delete_posts')) {
            return;
        }
        
        // Check for delete action
        if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['post'])) {
            $attachment_id = intval($_GET['post']);
            
            // Verify this is an attachment
            if (get_post_type($attachment_id) === 'attachment') {
                error_log("MediaDeletionHandler: Intercepting admin deletion for attachment $attachment_id");
                
                // Verify nonce
                if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], "delete-post_$attachment_id")) {
                    wp_die('Security check failed');
                }
                
                // Fix metadata BEFORE WordPress processes deletion
                self::ensure_proper_metadata($attachment_id);
                
                // Now let WordPress handle the actual deletion normally
                // We don't need to redirect - WordPress will continue with its normal flow
            }
        }
    }

    /**
     * Handles deletion of attachments before WordPress starts the process
     */
    public static function before_delete_attachment($post_id): void
    {
        // Only handle attachments
        if (get_post_type($post_id) !== 'attachment') {
            return;
        }

        error_log("MediaDeletionHandler: before_delete_attachment called for post $post_id");
        self::ensure_proper_metadata($post_id);
    }

    /**
     * Ensure attachment has proper metadata for deletion
     */
    public static function ensure_proper_metadata(int $attachment_id): void
    {
        error_log("MediaDeletionHandler: ensure_proper_metadata called for $attachment_id");

        // Get current file path from metadata DIRECTLY from database
        global $wpdb;
        $file = $wpdb->get_var($wpdb->prepare(
            "SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = '_wp_attached_file'",
            $attachment_id
        ));
        
        if (!$file) {
            error_log("MediaDeletionHandler: No file path found for attachment $attachment_id");
            return;
        }

        error_log("MediaDeletionHandler: Current file path from DB: $file");

        // Check if file path is absolute and needs conversion
        if (self::is_absolute_path($file)) {
            error_log("MediaDeletionHandler: Converting absolute path to relative");
            
            $upload_dir = wp_upload_dir();
            $base_dir = rtrim($upload_dir['basedir'], '/\\');
            
            error_log("MediaDeletionHandler: Upload base dir: $base_dir");
            
            // Normalize both paths for comparison (fix Windows path/case differences)
            $normalized_file = str_replace('\\', '/', strtolower($file));
            $normalized_base = str_replace('\\', '/', strtolower($base_dir));
            
            // Convert absolute path to relative
            if (strpos($normalized_file, $normalized_base) === 0) {
                $relative_path = ltrim(str_replace($normalized_base, '', $normalized_file), '/\\');
                $relative_path = str_replace('\\', '/', $relative_path); // Normalize slashes
                
                error_log("MediaDeletionHandler: New relative path: $relative_path");
                
                // Update the metadata directly in database
                $result = $wpdb->update(
                    $wpdb->postmeta,
                    ['meta_value' => $relative_path],
                    ['post_id' => $attachment_id, 'meta_key' => '_wp_attached_file'],
                    ['%s'],
                    ['%d', '%s']
                );
                
                error_log("MediaDeletionHandler: Database update result: $result");
                
                // Clear any cached metadata
                wp_cache_delete($attachment_id, 'post_meta');
                clean_post_cache($attachment_id);
                
                error_log("MediaDeletionHandler: Cache cleared for attachment $attachment_id");
                
                // Verify the change
                $verify_file = get_attached_file($attachment_id);
                error_log("MediaDeletionHandler: Verified new path: $verify_file");
            } else {
                error_log("MediaDeletionHandler: Absolute path not in uploads directory: $file");
            }
        } else {
            error_log("MediaDeletionHandler: Path is already relative, no conversion needed");
        }
        
        error_log("MediaDeletionHandler: Processing complete for attachment $attachment_id");
    }

    /**
     * Check if a path is absolute
     */
    private static function is_absolute_path(string $path): bool
    {
        // Windows absolute paths start with drive letter or UNC path
        if (PHP_OS_FAMILY === 'Windows') {
            return preg_match('/^[a-zA-Z]:[\\\\\\/]/', $path) || strpos($path, '\\\\') === 0;
        }
        
        // Unix absolute paths start with /
        return strpos($path, '/') === 0;
    }

    /**
     * Get the WordPress upload directory base path
     */
    private static function get_upload_base_dir(): string
    {
        $upload_dir = wp_upload_dir();
        return $upload_dir['basedir'];
    }

    /**
     * Normalize a file path (convert absolute to relative if needed)
     */
    private static function normalize_file_path(string $file_path, int $attachment_id): string
    {
        // If already relative, return as-is
        if (!self::is_absolute_path($file_path)) {
            return $file_path;
        }

        $upload_dir = self::get_upload_base_dir();
        
        // If absolute path is within uploads directory, convert to relative
        if (strpos($file_path, $upload_dir) === 0) {
            $relative_path = ltrim(str_replace($upload_dir, '', $file_path), '/\\');
            error_log("MediaDeletionHandler: Normalized absolute path to relative: $relative_path");
            return $relative_path;
        }

        // If absolute path is outside uploads directory, keep as absolute
        error_log("MediaDeletionHandler: Keeping absolute path (outside uploads): $file_path");
        return $file_path;
    }
}