<?php
namespace ZiziCache;

use ZiziCache\Security;

/**
 * ZiziCache License Manager - Production Ready
 * 
 * Handles license activation, validation, and storage via ZiziCache proxy API
 * No direct API keys - all communication through secure proxy
 * 
 * @package ZiziCache
 * @version 2.0
 * @security Proxy-based communication, encrypted local storage
 */
class LicenseManager 
{
    // ZiziCache Proxy API endpoint (secure proxy to Lemon Squeezy)
    const ZIZICACHE_API_URL = 'https://api.zizicache.com/v1/license-manager';
    
    // Cache validation for 24 hours (daily license check)
    const VALIDATION_CACHE_TIME = 86400; // 24 hours in seconds
    
    /**
     * In-memory cache for license data (per-request optimization)
     * @var array|null Cached license data
     */
    private static $license_cache = null;
    
    /**
     * In-memory cache for license validation result (per-request optimization)
     * @var array|null Cached validation result
     */
    private static $validation_cache = null;
    
    /**
     * In-memory cache for is_active result (per-request optimization)
     * @var bool|null Cached is_active result
     */
    private static $is_active_cache = null;
    
    /**
     * Activates a license through ZiziCache proxy API
     * 
     * @param string $license_key License key from customer
     * @return array Success/error response with message
     */
    public static function activate(string $license_key): array 
    {
        // Input validation
        if (strlen($license_key) < 20 || !self::validate_license_format($license_key)) {
            CacheSys::writeLog('License activation failed: Invalid license key format', 'License');
            return ['success' => false, 'message' => 'Invalid license key format'];
        }
        
        // Check if already activated
        if (self::is_key_already_stored($license_key)) {
            CacheSys::writeLog('License activation failed: Key already activated', 'License');
            return ['success' => false, 'message' => 'License key is already activated on this site'];
        }
        
        // Log activation attempt
        CacheSys::writeLog('Attempting license activation via ZiziCache proxy API', 'License');
        
        // Call ZiziCache proxy API for activation
        $response = wp_remote_post(self::ZIZICACHE_API_URL, [
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'User-Agent' => 'ZiziCache/' . ZIZI_CACHE_VERSION . ' WordPress/' . get_bloginfo('version')
            ],
            'body' => json_encode([
                'action' => 'activate',
                'license_key' => $license_key,
                'instance_name' => self::get_instance_name(),
                'site_url' => get_site_url()
            ]),
            'timeout' => 30,
            'sslverify' => true
        ]);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            CacheSys::writeError('License API connection failed: ' . $error_message, 'License');
            return ['success' => false, 'message' => 'API connection failed: ' . $error_message];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Přidání validace JSON před parsováním
        $body = self::ensure_clean_json($body);
        $data = json_decode($body, true);
        
        // Handle rate limiting (429)
        if ($status_code === 429) {
            CacheSys::writeError('License API rate limit exceeded', 'License');
            return ['success' => false, 'message' => 'Rate limit exceeded. Please try again later.'];
        }
        
        // Handle successful activation
        if ($status_code === 200 && isset($data['success']) && $data['success']) {
            if (isset($data['license_data'])) {
                self::store_license_securely($license_key, $data['license_data']);
            }
            CacheSys::writeLog('License activated successfully via proxy', 'License');
            return ['success' => true, 'message' => 'License activated successfully'];
        }
        
        // Handle API errors with specific messages
        $error = $data['error'] ?? 'Unknown activation error';
        CacheSys::writeError('License activation failed: ' . $error, 'License');
        
        // Provide user-friendly messages for common errors
        if (strpos($error, 'activation limit') !== false) {
            return [
                'success' => false, 
                'message' => 'This license has reached its activation limit. Please deactivate it from another site first, or contact support if you need more activations.',
                'error_type' => 'activation_limit'
            ];
        }
        
        if (strpos($error, 'expired') !== false) {
            return [
                'success' => false, 
                'message' => 'This license has expired. Please renew your subscription or purchase a new license.',
                'error_type' => 'expired'
            ];
        }
        
        if (strpos($error, 'disabled') !== false) {
            return [
                'success' => false, 
                'message' => 'This license has been disabled. Please contact support for assistance.',
                'error_type' => 'disabled'
            ];
        }
        
        if (strpos($error, 'not found') !== false || strpos($error, 'invalid') !== false) {
            return [
                'success' => false, 
                'message' => 'Invalid license key. Please check the key and try again.',
                'error_type' => 'invalid'
            ];
        }
        
        return ['success' => false, 'message' => $error];
    }
    
    /**
     * Validates license through ZiziCache proxy API with caching
     * 
     * @param bool $force_remote Force remote validation bypass cache
     * @return array Validation result with status
     */
    public static function validate(bool $force_remote = false): array 
    {
        // Return cached validation result if available and not forcing remote
        if (!$force_remote && self::$validation_cache !== null) {
            CacheSys::writeLog('License validation: Using in-memory cache', 'License');
            return self::$validation_cache;
        }
        
        $license_data = self::get_stored_license();
        if (!$license_data) {
            $result = ['valid' => false, 'message' => 'No license found'];
            if (!$force_remote) {
                self::$validation_cache = $result;
            }
            return $result;
        }
        
        // Check cached validation first (rate limiting)
        if (!$force_remote && self::is_validation_cached_from_data($license_data)) {
            $result = ['valid' => true, 'message' => 'License valid (cached)', 'cached' => true];
            self::$validation_cache = $result;
            CacheSys::writeLog('License validation: Using database cache (24h)', 'License');
            return $result;
        }
        
        // Decrypt license key securely
        $license_key = Security::decrypt_password($license_data['license_key_encrypted']);
        if (!$license_key) {
            CacheSys::writeError('Failed to decrypt license key', 'License');
            return ['valid' => false, 'message' => 'Failed to decrypt license key'];
        }
        
        CacheSys::writeLog('Validating license via ZiziCache proxy API', 'License');
        
        // Remote validation through proxy
        $response = wp_remote_post(self::ZIZICACHE_API_URL, [
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'User-Agent' => 'ZiziCache/' . ZIZI_CACHE_VERSION . ' WordPress/' . get_bloginfo('version')
            ],
            'body' => json_encode([
                'action' => 'validate',
                'license_key' => $license_key,
                'instance_id' => $license_data['instance_id'],
                'site_url' => get_site_url()
            ]),
            'timeout' => 30,
            'sslverify' => true
        ]);
        
        if (is_wp_error($response)) {
            // Fallback to cached validation if API fails
            if (self::has_recent_validation_from_data($license_data)) {
                CacheSys::writeLog('License validation API failed, using fallback cache', 'License');
                $result = ['valid' => true, 'message' => 'License valid (API unavailable)', 'fallback' => true];
                if (!$force_remote) {
                    self::$validation_cache = $result;
                }
                return $result;
            }
            
            $error_message = $response->get_error_message();
            CacheSys::writeError('License validation failed: ' . $error_message, 'License');
            $result = ['valid' => false, 'message' => 'Validation failed: ' . $error_message];
            if (!$force_remote) {
                self::$validation_cache = $result;
            }
            return $result;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Přidání validace JSON před parsováním
        $body = self::ensure_clean_json($body);
        $data = json_decode($body, true);
        
        if ($status_code === 200 && isset($data['valid'])) {
            // Cache validation result
            self::cache_validation_result($license_data['id'], $data['valid']);
            
            // Update license status if provided
            if ($data['valid'] && isset($data['status'])) {
                self::update_license_status($license_data['id'], $data['status']);
            }
            
            $message = $data['valid'] ? 'License is valid' : ($data['error'] ?? 'License is invalid');
            CacheSys::writeLog('License validation result: ' . $message, 'License');
            
            $result = [
                'valid' => $data['valid'],
                'message' => $message,
                'remote' => true
            ];
            
            // Cache the result for the rest of this request (unless forced remote)
            if (!$force_remote) {
                self::$validation_cache = $result;
            }
            
            return $result;
        }
        
        CacheSys::writeError('Invalid license validation API response', 'License');
        $result = ['valid' => false, 'message' => 'Invalid API response'];
        
        // Cache negative result for the rest of this request
        if (!$force_remote) {
            self::$validation_cache = $result;
        }
        
        return $result;
    }
    
    /**
     * Deactivates license through ZiziCache proxy API
     * Always clears local data regardless of API response
     *
     * @return array Success/error response with message
     */
    public static function deactivate(): array 
    {
        $license_data = self::get_stored_license();
        if (!$license_data) {
            return ['success' => true, 'message' => 'No active license found'];
        }
        
        // Decrypt license key securely
        $license_key = Security::decrypt_password($license_data['license_key_encrypted']);
        $instance_id = $license_data['instance_id'] ?? null;
        
        if (!$license_key) {
            CacheSys::writeError('License deactivation failed: Could not decrypt license key', 'License');
            // Even if we couldn't decrypt the key, we'll remove the local data
            self::clear_license_data();
            return ['success' => true, 'message' => 'License removed locally'];
        }
        
        // If we don't have instance_id, don't send API request
        if (empty($instance_id)) {
            CacheSys::writeLog('License deactivation: No instance_id found, removing local data only', 'License');
            self::clear_license_data();
            return ['success' => true, 'message' => 'License removed locally'];
        }
        
        CacheSys::writeLog('Attempting license deactivation via ZiziCache proxy API', 'License');
        
        // ZiziCache proxy API deactivation
        $response = wp_remote_post(self::ZIZICACHE_API_URL, [
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'User-Agent' => 'ZiziCache/' . ZIZI_CACHE_VERSION . ' WordPress/' . get_bloginfo('version')
            ],
            'body' => json_encode([
                'action' => 'deactivate',
                'license_key' => $license_key,
                'instance_id' => $instance_id,
                'site_url' => get_site_url()
            ]),
            'timeout' => 30,
            'sslverify' => true
        ]);
        
        // Regardless of the API call result, always clean up local data
        self::clear_license_data();
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            CacheSys::writeError('License API deactivation failed: ' . $error_message, 'License');
            return ['success' => true, 'message' => 'License removed locally (API error: ' . $error_message . ')'];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Přidání validace JSON před parsováním
        $body = self::ensure_clean_json($body);
        $data = json_decode($body, true);
        
        // Successful deactivation in API
        if ($status_code === 200 && isset($data['success']) && $data['success']) {
            CacheSys::writeLog('License deactivated successfully via proxy', 'License');
            return ['success' => true, 'message' => 'License deactivated successfully'];
        }
        
        // Even in case of API error, we return success because the license is already deleted locally
        $error = $data['error'] ?? 'Unknown deactivation error';
        CacheSys::writeLog('License API deactivation issue: ' . $error, 'License');
        return ['success' => true, 'message' => 'License removed locally (API status: ' . $error . ')'];
    }
    
    /**
     * Gets the status of the active license
     * 
     * @return array|null License data or null if not found
     */
    public static function get_status(): ?array 
    {
        $license_data = self::get_stored_license();
        
        if (!$license_data) {
            return null;
        }
        
        // Transform data to format expected by LemonSqueezyUpdater
        if (!empty($license_data['product_id']) || !empty($license_data['variant_id'])) {
            $license_data['meta'] = [
                'product_id' => $license_data['product_id'] ?? null,
                'variant_id' => $license_data['variant_id'] ?? null,
                'product_name' => $license_data['product_name'] ?? null,
                'customer_email' => $license_data['customer_email'] ?? null,
                'customer_name' => $license_data['customer_name'] ?? null,
            ];
        }
        
        return $license_data;
    }
    
    /**
     * Is the license active and valid? (with in-memory caching)
     * 
     * @return bool True if license is active and valid
     */
    public static function is_active(): bool 
    {
        // Return cached result if available (per-request optimization)
        if (self::$is_active_cache !== null) {
            CacheSys::writeLog('License is_active: Using in-memory cache', 'License');
            return self::$is_active_cache;
        }
        
        $validation = self::validate();
        $result = $validation['valid'] ?? false;
        
        // Cache the result for the rest of this request
        self::$is_active_cache = $result;
        CacheSys::writeLog('License is_active: Computed and cached result = ' . ($result ? 'true' : 'false'), 'License');
        
        return $result;
    }
    
    /**
     * Clear in-memory cache (useful for testing or forced refresh)
     */
    public static function clear_memory_cache(): void 
    {
        self::$license_cache = null;
        self::$validation_cache = null;
        self::$is_active_cache = null;
        CacheSys::writeLog('In-memory license cache cleared', 'License');
    }
    
    /**
     * Get cache statistics for debugging
     * 
     * @return array Cache statistics
     */
    public static function get_cache_stats(): array 
    {
        return [
            'license_cache_set' => self::$license_cache !== null,
            'validation_cache_set' => self::$validation_cache !== null,
            'is_active_cache_set' => self::$is_active_cache !== null,
            'license_cache_data' => self::$license_cache ? 'Present (hidden for security)' : 'Not cached',
            'validation_cache_data' => self::$validation_cache,
            'is_active_cache_data' => self::$is_active_cache
        ];
    }
    
    /**
     * Securely stores the license with encryption
     * 
     * @param string $license_key Original license key
     * @param array $api_data Data from ZiziCache proxy API
     */
    private static function store_license_securely(string $license_key, array $api_data): void 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        // Delete any existing active license
        $wpdb->delete($table, ['status' => 'active']);
        
        // Extract data from proxy API response (adapted format)
        $license_info = $api_data['license_key'] ?? $api_data;
        $instance_info = $api_data['instance'] ?? [];
        $meta_info = $api_data['meta'] ?? $api_data;
        
        // Insert new license with encryption
        $result = $wpdb->insert($table, [
            'license_key_hash' => hash('sha256', $license_key),
            'license_key_encrypted' => Security::encrypt_password($license_key),
            'instance_id' => $instance_info['id'] ?? $api_data['instance_id'] ?? null,
            'instance_name' => $instance_info['name'] ?? self::get_instance_name(),
            'status' => 'active',
            'activation_limit' => $license_info['activation_limit'] ?? 1,
            'activation_usage' => $license_info['activation_usage'] ?? 1,
            'expires_at' => $license_info['expires_at'] ?? null,
            'customer_email' => $meta_info['customer_email'] ?? null,
            'customer_name' => $meta_info['customer_name'] ?? null,
            'product_name' => $meta_info['product_name'] ?? 'ZiziCache',
            'product_id' => $meta_info['product_id'] ?? null,
            'variant_id' => $meta_info['variant_id'] ?? null,
            'site_url' => get_site_url(),
            'last_validated' => current_time('mysql')
        ]);
        
        if ($result === false) {
            CacheSys::writeError('Failed to store license data in database: ' . $wpdb->last_error, 'License');
        }
        
        // Clear in-memory cache after storing new license
        self::$license_cache = null;
        self::$validation_cache = null;
        self::$is_active_cache = null;
    }
    
    /**
     * Retrieves the stored license from the database with in-memory caching
     * 
     * @return array|null License data or null
     */
    private static function get_stored_license(): ?array 
    {
        // Return cached result if available (per-request optimization)
        if (self::$license_cache !== null) {
            CacheSys::writeLog('License data: Using in-memory cache', 'License');
            return self::$license_cache;
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        $license_data = $wpdb->get_row(
            "SELECT * FROM {$table} WHERE status = 'active' ORDER BY created_at DESC LIMIT 1",
            ARRAY_A
        );
        
        // Cache the result for the rest of this request
        self::$license_cache = $license_data;
        CacheSys::writeLog('License data: Fetched from database and cached in memory', 'License');
        
        return $license_data;
    }
    
    /**
     * Checks if the license is already stored
     * 
     * @param string $license_key License key to check
     * @return bool True if already stored
     */
    private static function is_key_already_stored(string $license_key): bool 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        $hash = hash('sha256', $license_key);
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE license_key_hash = %s AND status = 'active'",
            $hash
        ));
        
        return $count > 0;
    }
    
    /**
     * Validates the license key format (UUID format)
     * 
     * @param string $key License key to validate
     * @return bool True if valid format
     */
    private static function validate_license_format(string $key): bool 
    {
        // Lemon Squeezy license keys are UUIDs
        return preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $key);
    }
    
    /**
     * Gets the instance name for the current website
     * 
     * @return string Instance name for current site
     */
    private static function get_instance_name(): string 
    {
        return parse_url(get_site_url(), PHP_URL_HOST) ?: 'Unknown Site';
    }
    
    /**
     * Caches validation result for rate limiting
     * 
     * @param int $license_id License ID
     * @param bool $is_valid Validation result
     */
    private static function cache_validation_result(int $license_id, bool $is_valid): void 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        $wpdb->update(
            $table,
            ['last_validated' => current_time('mysql')],
            ['id' => $license_id],
            ['%s'],
            ['%d']
        );
    }
    
    /**
     * Is the validation cache still valid? (optimized version using existing data)
     * 
     * @param array $license_data License data already fetched from database
     * @return bool True if cache is valid
     */
    private static function is_validation_cached_from_data(array $license_data): bool 
    {
        if (!isset($license_data['last_validated']) || !$license_data['last_validated']) {
            return false;
        }
        
        $cache_expiry = strtotime($license_data['last_validated']) + self::VALIDATION_CACHE_TIME;
        return time() < $cache_expiry;
    }
    
    /**
     * Is the validation cache still valid? (legacy method - for backward compatibility)
     * 
     * @param int $license_id License ID
     * @return bool True if cache is valid
     */
    private static function is_validation_cached(int $license_id): bool 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        $last_validated = $wpdb->get_var($wpdb->prepare(
            "SELECT last_validated FROM {$table} WHERE id = %d",
            $license_id
        ));
        
        if (!$last_validated) return false;
        
        $cache_expiry = strtotime($last_validated) + self::VALIDATION_CACHE_TIME;
        return time() < $cache_expiry;
    }
    
    /**
     * Has recent validation? (fallback for API outage) - optimized version using existing data
     * 
     * @param array $license_data License data already fetched from database
     * @return bool True if has recent validation
     */
    private static function has_recent_validation_from_data(array $license_data): bool 
    {
        if (!isset($license_data['last_validated']) || !$license_data['last_validated']) {
            return false;
        }
        
        // 7 days grace period when API is unavailable
        $grace_expiry = strtotime($license_data['last_validated']) + (7 * 24 * 3600);
        return time() < $grace_expiry;
    }
    
    /**
     * Has recent validation? (fallback for API outage) - legacy method for backward compatibility
     * 
     * @param int $license_id License ID
     * @return bool True if has recent validation
     */
    private static function has_recent_validation(int $license_id): bool 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        $last_validated = $wpdb->get_var($wpdb->prepare(
            "SELECT last_validated FROM {$table} WHERE id = %d",
            $license_id
        ));
        
        if (!$last_validated) return false;
        
        // 7 days grace period when API is unavailable
        $grace_expiry = strtotime($last_validated) + (7 * 24 * 3600);
        return time() < $grace_expiry;
    }
    
    /**
     * Update license status from API response
     * 
     * @param int $license_id License ID
     * @param string $status New status
     */
    private static function update_license_status(int $license_id, string $status): void 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        $allowed_statuses = ['inactive', 'active', 'expired', 'disabled'];
        if (in_array($status, $allowed_statuses)) {
            $wpdb->update(
                $table,
                ['status' => $status],
                ['id' => $license_id],
                ['%s'],
                ['%d']
            );
        }
    }
    
    /**
     * Clears all license data and related caches
     */
    private static function clear_license_data(): void 
    {
        global $wpdb;
        $table = $wpdb->prefix . 'zizi_cache_license';
        
        // Clear license data from database (all license records)
        $wpdb->query("TRUNCATE TABLE {$table}");
        
        // Clear all related transients
        $update_cache_key = 'zizi_cache_ls_update_' . md5(plugin_basename(ZIZI_CACHE_FILE));
        delete_transient(self::get_validation_transient_key());
        delete_transient($update_cache_key);
        
        // Clear object cache for transients if it exists
        if (function_exists('wp_cache_delete')) {
            wp_cache_delete(self::get_validation_transient_key(), 'transient');
            wp_cache_delete($update_cache_key, 'transient');
        }
        
        // Clear all other caches that might store license data
        wp_cache_delete('zizi_cache_license_status', 'options');
        delete_option('zizi_cache_last_license_check');
        
        // Clear in-memory cache
        self::$license_cache = null;
        self::$validation_cache = null;
        self::$is_active_cache = null;
        
        CacheSys::writeLog('License data and related caches completely cleared', 'License');
    }
    
    /**
     * Gets the transient key for license validation
     * 
     * @param string $license_id License ID
     * @return string Transient key
     */
    private static function get_validation_transient_key(string $license_id = ''): string 
    {
        if (!$license_id) {
            $license_data = self::get_stored_license();
            $license_id = $license_data['id'] ?? 'default';
        }
        
        return 'zizi_cache_ls_validation_' . md5($license_id . get_site_url());
    }
    
    /**
     * Automatic daily license validation check
     * Called during plugin initialization to ensure license validity
     */
    public static function auto_daily_validation(): void 
    {
        // Only check if we have an active license
        if (!self::is_active()) {
            return;
        }
        
        $license_data = self::get_stored_license();
        if (!$license_data) {
            return;
        }
        
        // Check if last validation was more than 24 hours ago
        $last_validated = $license_data['last_validated'];
        if ($last_validated) {
            $last_check_timestamp = strtotime($last_validated);
            $now = current_time('timestamp');
            
            // If last check was less than 24 hours ago, skip
            if (($now - $last_check_timestamp) < self::VALIDATION_CACHE_TIME) {
                return;
            }
        }
        
        // Perform background validation (don't block page load)
        wp_schedule_single_event(time() + 30, 'zizi_cache_background_license_validation');
        CacheSys::writeLog('Scheduled background license validation', 'License');
    }
    
    /**
     * Background license validation (runs via cron)
     */
    public static function background_license_validation(): void 
    {
        CacheSys::writeLog('Running background license validation', 'License');
        
        $result = self::validate(true); // Force remote check
        
        if (!$result['valid']) {
            CacheSys::writeLog('Background license validation failed: ' . $result['message'], 'License');
            
            // If license is invalid, mark as inactive
            if (strpos($result['message'], 'expired') !== false || 
                strpos($result['message'], 'invalid') !== false) {
                
                $license_data = self::get_stored_license();
                if ($license_data && isset($license_data['id'])) {
                    global $wpdb;
                    $table = $wpdb->prefix . 'zizi_cache_license';
                    $wpdb->update($table, 
                        ['status' => 'expired'], 
                        ['id' => $license_data['id']],
                        ['%s'],
                        ['%d']
                    );
                    
                    CacheSys::writeLog('License marked as expired due to validation failure', 'License');
                }
            }
        } else {
            CacheSys::writeLog('Background license validation successful', 'License');
        }
    }
    
    /**
     * Zajistí čistý JSON bez nechtěných znaků
     * 
     * @param string $json_string JSON řetězec k vyčištění
     * @return string Vyčištěný JSON řetězec
     */
    private static function ensure_clean_json(string $json_string): string 
    {
        // Kontrola, jestli string není prázdný
        if (empty($json_string)) {
            CacheSys::writeLog('Prázdná JSON odpověď z API', 'License');
            return '{}'; // Vrátit prázdný JSON objekt
        }
        
        // Odstranění BOM znaků, které mohou být na začátku
        $json_string = preg_replace('/^\xEF\xBB\xBF/', '', $json_string);
        
        // Odstranění bílých znaků před a po JSON
        $json_string = trim($json_string);
        
        // Pokud je řetězec validní JSON, vrátíme ho přímo
        if (self::is_valid_json($json_string)) {
            return $json_string;
        }
        
        // Pokud JSON není validní, pokusíme se najít a extrahovat validní JSON část
        if (preg_match('/(\{.*\}|\[.*\])/', $json_string, $matches)) {
            $potential_json = $matches[0];
            if (self::is_valid_json($potential_json)) {
                CacheSys::writeLog('Nalezen validní JSON v nevalidní odpovědi', 'License');
                return $potential_json;
            }
        }
        
        // Log problému pro diagnostiku
        CacheSys::writeError('Nepodařilo se najít validní JSON v odpovědi: ' . substr($json_string, 0, 200), 'License');
        
        // Vrátíme původní string - json_decode() selže, ale aspoň máme záznam o problému
        return $json_string;
    }
    
    /**
     * Kontrola zda je string validní JSON
     * 
     * @param string $string String ke kontrole
     * @return bool True pokud je validní JSON
     */
    private static function is_valid_json(string $string): bool 
    {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }
}
