<?php
namespace ZiziCache;

use ZiziCache\LicenseManager;
use ZiziCache\Security;
use ZiziCache\CacheSys;

/**
 * ZiziCache WordPress Plugin Updater - Production Ready
 * 
 * Provides secure auto-updates for licensed ZiziCache users via ZiziCache API proxy
 * Communicates with https://api.zizicache.com instead of directly with Lemon Squeezy
 * Features atomic updates with comprehensive backup/restore system
 * 
 * @package ZiziCache
 * @version 2.0
 * @security License validation through ZiziCache API proxy, SSL verification, atomic updates
 */
class LemonSqueezyUpdater 
{
    private $plugin_slug;
    private $plugin_file;
    private $version;
    private $api_url;
    private $cache_key;
    private $cache_allowed;
    
    // Security & Performance Constants
    private const DOWNLOAD_TIMEOUT = 45; // seconds for download timeout
    private const API_TIMEOUT = 15; // seconds for API calls
    private const CACHE_DURATION = 300; // 5 minutes cache
    private const MIN_FILE_SIZE = 1024; // minimum plugin file size (1KB)
      // ZiziCache API URL
    private const ZIZICACHE_API_URL = 'https://api.zizicache.com/v1/update-checker';
      
    /**
     * Initialize the updater with ZiziCache API proxy
     *
     * @param string $plugin_file Path to the main plugin file
     * @param string $version Current plugin version
     * @param string $api_url Optional custom API URL (defaults to ZiziCache API)
     */
    public function __construct($plugin_file, $version, $api_url = null) 
    {
        $this->plugin_file = $plugin_file;
        $this->plugin_slug = plugin_basename($plugin_file);
        $this->version = $version;
        $this->api_url = $api_url ?: self::ZIZICACHE_API_URL;
        $this->cache_key = 'zizi_cache_update_' . md5($this->plugin_slug);
        $this->cache_allowed = true;
        
        $this->init();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init() 
    {
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_plugin_update']);
        add_filter('plugins_api', [$this, 'plugins_api_call'], 10, 3);
        add_filter('upgrader_pre_download', [$this, 'download_package'], 10, 3);
        
        // Cleanup maintenance mode after update completion
        add_action('upgrader_process_complete', [$this, 'cleanup_maintenance_mode'], 10, 2);
        
        // Add changelog links to plugin page
        add_filter('plugin_action_links', [$this, 'add_changelog_link'], 10, 2);
        add_filter('plugin_row_meta', [$this, 'add_changelog_meta_link'], 10, 4);
        
        // Enqueue admin styles
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_styles']);
        
        // Enqueue admin styles for changelog links
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_styles']);
        
          // Debug logging
        if (defined('ZIZI_CACHE_DEBUG') && ZIZI_CACHE_DEBUG) {
            CacheSys::writeLog('ZiziCache Updater initialized for plugin: ' . $this->plugin_slug, 'DEBUG', 'Updater');
            CacheSys::writeLog('Using ZiziCache API: ' . $this->api_url, 'DEBUG', 'Updater');
        }
    }

    /**
     * Check for plugin updates via ZiziCache API
     * 
     * @param object $transient WordPress update transient
     * @return object Modified transient
     */
    public function check_for_plugin_update($transient)
    {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        // Check if user has active license
        if (!LicenseManager::is_active()) {
            CacheSys::writeLog('Update check skipped: No active license', 'Updater');
            return $transient;
        }
        
        // Get remote version info from ZiziCache API
        $remote_version = $this->get_remote_version();
        
        if (!$remote_version) {
            CacheSys::writeLog('Update check failed: Could not fetch remote version from ZiziCache API', 'Updater');
            return $transient;
        }

        // Compare versions
        $local = $this->normalize_version($this->version);
        $remote = $this->normalize_version($remote_version->version);
        
        CacheSys::writeLog("Version comparison: Local '{$this->version}' (normalized: '{$local}') vs Remote '{$remote_version->version}' (normalized: '{$remote}')", 'Updater');

        if (version_compare($local, $remote, '<')) {
            CacheSys::writeLog("Update available: {$local} -> {$remote} (raw: {$this->version} -> {$remote_version->version})", 'Updater');
            
            $transient->response[$this->plugin_slug] = (object) [
                'slug' => dirname($this->plugin_slug),
                'plugin' => $this->plugin_slug,
                'new_version' => $remote_version->version,
                'tested' => $remote_version->tested ?? '6.5',
                'package' => $remote_version->download_url ?? false,
                'url' => 'https://zizicache.com/changelog/',
                'icons' => [
                    '1x' => ZIZI_CACHE_PLUGIN_URL . 'assets/images/icon-128x128.png',
                    '2x' => ZIZI_CACHE_PLUGIN_URL . 'assets/images/icon-256x256.png'
                ],
                'upgrade_notice' => 'New version available with improvements and bug fixes'
            ];
        } else {
            CacheSys::writeLog("Plugin is up to date: {$local} (raw: {$this->version}) - Remote: {$remote} (raw: {$remote_version->version})", 'Updater');
        }
        
        return $transient;
    }

    /**
     * Handle plugin information popup requests
     * 
     * @param false|object|array $result The result object or array
     * @param string $action The type of information being requested
     * @param object $args Plugin API arguments
     * @return false|object Plugin information or false
     */
    public function plugins_api_call($result, $action, $args) 
    {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if (!isset($args->slug) || $args->slug !== dirname($this->plugin_slug)) {
            return $result;
        }        // Only for licensed users
        if (!LicenseManager::is_active()) {
            return $result;
        }
        
        // Get remote version from ZiziCache API
        $remote_version = $this->get_remote_version();
        if (!$remote_version) {
            return $result;
        }

        // Create plugin info object
        $plugin_info = new \stdClass();
        $plugin_info->name = 'ZiziCache';
        $plugin_info->slug = dirname($this->plugin_slug);
        $plugin_info->version = $remote_version->version;
        $plugin_info->author = '<a href="https://zizicache.com/">ZiziCache.com</a>';
        $plugin_info->homepage = 'https://zizicache.com/';
        $plugin_info->requires = '4.7';
        $plugin_info->tested = '6.5';
        $plugin_info->downloaded = 1000;
        $plugin_info->last_updated = current_time('Y-m-d');
        
        // Add download link if available
        if (!empty($remote_version->download_url)) {
            $plugin_info->download_link = $remote_version->download_url;
        }
        
        // Sections for popup
        $plugin_info->sections = [
            'description' => '<h3>ZiziCache Premium</h3><p>Advanced WordPress caching and optimization plugin with automatic updates.</p>',
            'changelog' => $this->parse_local_changelog(),
            'installation' => '<p>This plugin updates automatically for licensed users.</p>',
        ];
        
        CacheSys::writeLog('Plugin info popup generated for version: ' . $remote_version->version, 'Updater');
        return $plugin_info;
    }    /**
     * Handle secure plugin download with atomic update protection
     * 
     * @param bool $reply Whether to bail without returning the package
     * @param string $package The package URL
     * @param object $upgrader WP_Upgrader instance
     * @return bool|string Download result or package URL
     */
    public function download_package($reply, $package, $upgrader) 
    {
        // Detailed logging for debug
        CacheSys::writeLog('Received package for download: ' . $package, 'Updater');

        // Detect our plugin for atomic download
        $plugin_detected = false;
        
        // Check normal plugin update
        if (isset($upgrader->skin->plugin) && $upgrader->skin->plugin === $this->plugin_slug) {
            $plugin_detected = true;
            CacheSys::writeLog('Detected plugin via skin->plugin property', 'Updater');
        }
        // Check hook_extra from upgrader_package_options
        elseif (isset($upgrader->skin->options['hook_extra']['plugin']) && $upgrader->skin->options['hook_extra']['plugin'] === $this->plugin_slug) {
            CacheSys::writeLog('Detected plugin via hook_extra', 'Updater');
            $plugin_detected = true;
        }
        // Check by package URL pattern for ZiziCache API
        elseif (strpos($package, 'api.zizicache.com') !== false || strpos($package, 'lemonsqueezy.com') !== false) {
            CacheSys::writeLog('Detected plugin via package URL pattern', 'Updater');
            $plugin_detected = true;
        }
        
        if (!$plugin_detected) {
            CacheSys::writeLog('Plugin slug mismatch or not our package - skipping custom download', 'Updater');
            return $reply;
        }

        CacheSys::writeLog('Starting secure ZiziCache plugin download process', 'Updater');
        
        // Verify license for download
        if (!$this->verify_license_for_download()) {
            CacheSys::writeLog('Download denied: License verification failed', 'Updater');
            return new \WP_Error('no_license', 'A valid license is required to download updates.');
        }
        
        // Create comprehensive backup before download
        $backup_result = $this->create_comprehensive_backup();
        if (!$backup_result['success']) {
            CacheSys::writeLog('Warning: Could not create plugin backup: ' . $backup_result['message'], 'Updater');
        }
        
        // Download the package with enhanced error handling
        CacheSys::writeLog('Initiating secure download with ' . self::DOWNLOAD_TIMEOUT . 's timeout: ' . $package, 'Updater');
        
        $temp_file = download_url($package, self::DOWNLOAD_TIMEOUT);
        
        if (is_wp_error($temp_file)) {
            $this->restore_from_backup();
            CacheSys::writeLog('Download error: ' . $temp_file->get_error_message(), 'Updater');
            return $temp_file;
        }
        
        // Verify download integrity with enhanced checks
        $integrity_result = $this->verify_download_integrity($temp_file);
        if (!$integrity_result['valid']) {
            @unlink($temp_file);
            $this->restore_from_backup();
            CacheSys::writeLog('Download failed: ' . $integrity_result['message'], 'Updater');
            return new \WP_Error('integrity_error', $integrity_result['message']);
        }
        
        CacheSys::writeLog('Plugin update downloaded and verified successfully', 'Updater');
        
        // Store download info for post-update cleanup
        update_option('zizi_cache_update_info', [
            'temp_file' => $temp_file,
            'package_url' => $package,
            'backup_created' => $backup_result['success'],
            'backup_path' => $backup_result['path'] ?? null,
            'download_time' => current_time('mysql')        ], false);
        
        return $temp_file;
    }/**
     * Get remote version information from ZiziCache API
     * 
     * @return object|false Remote version data or false
     */
    public function get_remote_version()
    {
        // Check cache first
        if ($this->cache_allowed) {
            $cached = get_transient($this->cache_key);
            if ($cached !== false) {
                CacheSys::writeLog('Using cached version data', 'Updater');
                return $cached;
            }
        }
        
        // Get license data for API request
        $license_data = LicenseManager::get_status();
        if (!$license_data || !isset($license_data['license_key_encrypted'])) {
            CacheSys::writeLog('No license data available for version check', 'Updater');
            return false;
        }
        
        // Decrypt license key for proxy API
        $license_key = Security::decrypt_password($license_data['license_key_encrypted']);
        if (!$license_key) {
            CacheSys::writeLog('Failed to decrypt license key for version check', 'Updater');
            return false;
        }
        
        CacheSys::writeLog('Checking for updates via ZiziCache API', 'Updater');
        
        // Prepare request to ZiziCache API
        $request_body = [
            'license_key' => $license_key,
            'current_version' => $this->version,
            'plugin_slug' => dirname($this->plugin_slug),
            'site_url' => get_site_url()
        ];
        
        $response = wp_remote_post($this->api_url, [
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'User-Agent' => 'ZiziCache/' . $this->version . '; ' . get_site_url()
            ],
            'body' => json_encode($request_body),
            'timeout' => self::API_TIMEOUT,
            'sslverify' => true
        ]);
        
        if (is_wp_error($response)) {
            CacheSys::writeLog('ZiziCache API error: ' . $response->get_error_message(), 'Updater');
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            CacheSys::writeLog("ZiziCache API returned error code: {$response_code}", 'Updater');
            return false;
        }
        
        $data = json_decode($response_body, true);
        
        if (!$data) {
            CacheSys::writeLog('Invalid JSON response from ZiziCache API', 'Updater');
            return false;
        }
        
        if (isset($data['error'])) {
            CacheSys::writeLog('ZiziCache API error: ' . $data['error'], 'Updater');
            return false;
        }
        
        // Check if we have valid version data
        if (!isset($data['version']) || !isset($data['name'])) {
            CacheSys::writeLog('Invalid version data from ZiziCache API', 'Updater');
            return false;
        }
        
        $version_data = (object) [
            'version' => $data['version'],
            'name' => $data['name'],
            'download_url' => $data['download_url'] ?? false,
            'tested' => $data['tested'] ?? '6.5',
            'requires' => $data['requires'] ?? '4.7',
            'last_updated' => $data['last_updated'] ?? current_time('Y-m-d')
        ];

        CacheSys::writeLog('Version check completed. Current: ' . $this->version . ', Latest: ' . $version_data->version, 'Updater');
        
        // Cache the result
        if ($this->cache_allowed) {
            set_transient($this->cache_key, $version_data, self::CACHE_DURATION);
            CacheSys::writeLog('Version data cached for ' . (self::CACHE_DURATION / 60) . ' minutes', 'Updater');
        }
        
        return $version_data;
    }    
    /**
     * Verify license for download via ZiziCache proxy
     * 
     * @return bool
     */
    private function verify_license_for_download()
    {
        // Require active license
        if (!LicenseManager::is_active()) {
            return false;
        }
        
        $license_data = LicenseManager::get_status();
        if (!$license_data || !isset($license_data['license_key_encrypted'])) {
            return false;
        }
        
        return true;
    }
      /**
     * Create comprehensive backup before update with atomic protection
     * 
     * @return array Success status and backup information
     */
    private function create_comprehensive_backup(): array 
    {
        $plugin_dir = dirname(dirname(__FILE__)); // src/ -> plugin root
        $backup_dir = $plugin_dir . '_atomic_backup_' . time();
        
        try {
            // Create backup directory
            if (!wp_mkdir_p($backup_dir)) {
                return ['success' => false, 'message' => 'Could not create backup directory'];
            }
            
            // Copy entire plugin directory with verification
            if (!$this->atomic_copy_directory($plugin_dir, $backup_dir)) {
                $this->recursive_delete($backup_dir);
                return ['success' => false, 'message' => 'Could not copy plugin files to backup'];
            }
            
            // Verify backup integrity
            if (!$this->verify_backup_integrity($plugin_dir, $backup_dir)) {
                $this->recursive_delete($backup_dir);
                return ['success' => false, 'message' => 'Backup integrity verification failed'];
            }
            
            // Store backup metadata
            $backup_info = [
                'path' => $backup_dir,
                'timestamp' => time(),
                'original_path' => $plugin_dir,
                'version' => $this->version,
                'checksum' => $this->calculate_directory_checksum($plugin_dir)
            ];
            
            update_option('zizi_cache_atomic_backup', $backup_info, false);
            CacheSys::writeLog('Comprehensive backup created successfully: ' . $backup_dir, 'Updater');
            
            return ['success' => true, 'message' => 'Backup created successfully', 'path' => $backup_dir];
            
        } catch (Exception $e) {
            CacheSys::writeError('Backup creation failed: ' . $e->getMessage(), 'Updater');
            return ['success' => false, 'message' => 'Backup creation exception: ' . $e->getMessage()];
        }
    }
    
    /**
     * Atomic directory copy with progress tracking
     * 
     * @param string $src Source directory
     * @param string $dst Destination directory
     * @return bool Success
     */
    private function atomic_copy_directory(string $src, string $dst): bool 
    {
        if (!is_dir($src)) {
            return false;
        }
        
        if (!wp_mkdir_p($dst)) {
            return false;
        }
          $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($src, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $item) {
            $target = $dst . DIRECTORY_SEPARATOR . $iterator->getSubPathName();
            
            if ($item->isDir()) {
                if (!wp_mkdir_p($target)) {
                    return false;
                }
            } else {
                // Skip temporary and cache files
                if ($this->should_skip_file($item->getPathname())) {
                    continue;
                }
                
                if (!copy($item, $target)) {
                    CacheSys::writeError('Failed to copy file: ' . $item->getPathname(), 'Updater');
                    return false;
                }
                
                // Verify file was copied correctly
                if (filesize($item) !== filesize($target)) {
                    CacheSys::writeError('File size mismatch during copy: ' . $item->getPathname(), 'Updater');
                    return false;
                }
            }
        }
        
        return true;
    }
    
    /**
     * Verify backup integrity by comparing checksums
     * 
     * @param string $original_dir Original directory
     * @param string $backup_dir Backup directory
     * @return bool Integrity check result
     */
    private function verify_backup_integrity(string $original_dir, string $backup_dir): bool 
    {
        $original_checksum = $this->calculate_directory_checksum($original_dir);
        $backup_checksum = $this->calculate_directory_checksum($backup_dir);
        
        $integrity_valid = ($original_checksum === $backup_checksum);
        
        if ($integrity_valid) {
            CacheSys::writeLog('Backup integrity verified successfully', 'Updater');
        } else {
            CacheSys::writeError('Backup integrity verification failed - checksums do not match', 'Updater');
        }
        
        return $integrity_valid;
    }
    
    /**
     * Calculate directory checksum for integrity verification
     * 
     * @param string $dir Directory path
     * @return string MD5 checksum of directory contents
     */
    private function calculate_directory_checksum(string $dir): string 
    {
        $files = [];
        
        if (is_dir($dir)) {            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($dir, \RecursiveDirectoryIterator::SKIP_DOTS)
            );
            
            foreach ($iterator as $file) {
                if ($file->isFile() && !$this->should_skip_file($file->getPathname())) {
                    $relative_path = str_replace($dir, '', $file->getPathname());
                    $files[$relative_path] = md5_file($file->getPathname());
                }
            }
        }
        
        ksort($files);
        return md5(serialize($files));
    }
    
    /**
     * Check if file should be skipped during backup
     * 
     * @param string $file_path File path
     * @return bool True if file should be skipped
     */
    private function should_skip_file(string $file_path): bool 
    {
        $skip_patterns = [
            '/\.tmp$/',
            '/\.log$/',
            '/\.cache$/',
            '/_backup_/',
        ];

        foreach ($skip_patterns as $pattern) {
            if (preg_match($pattern, $file_path)) {
                return true;
            }
        }

        return false;
    }
    
    /**
     * Cleanup backup after successful update
     */
    private function cleanup_backup() 
    {
        $backup_info = get_option('zizi_cache_atomic_backup');
        if ($backup_info && isset($backup_info['path']) && is_dir($backup_info['path'])) {
            $this->recursive_delete($backup_info['path']);
            delete_option('zizi_cache_atomic_backup');
            CacheSys::writeLog('Atomic backup cleaned up: ' . $backup_info['path'], 'Updater');
        }
        
        // Also cleanup legacy backup
        $legacy_backup = get_option('zizi_cache_backup_path');
        if ($legacy_backup && is_dir($legacy_backup)) {
            $this->recursive_delete($legacy_backup);
            delete_option('zizi_cache_backup_path');
            CacheSys::writeLog('Legacy backup cleaned up: ' . $legacy_backup, 'Updater');
        }
    }
    
    /**
     * Recursive directory delete
     * 
     * @param string $dir Directory to delete
     * @return bool Success
     */
    private function recursive_delete($dir) 
    {
        if (!is_dir($dir)) {
            return @unlink($dir);
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . DIRECTORY_SEPARATOR . $file;
            is_dir($path) ? $this->recursive_delete($path) : @unlink($path);
        }
        
        return @rmdir($dir);
    }
    
    /**
     * Cleanup maintenance mode after update
     * 
     * @param object $upgrader WP_Upgrader instance
     * @param array $hook_extra Hook extra data
     */
    public function cleanup_maintenance_mode($upgrader, $hook_extra) 
    {
        // Only for our plugin
        if (isset($hook_extra['plugin']) && $hook_extra['plugin'] === $this->plugin_slug) {
            $maintenance_file = ABSPATH . '.maintenance';
            if (file_exists($maintenance_file)) {
                @unlink($maintenance_file);
                CacheSys::writeLog('Maintenance file cleaned up after update', 'Updater');
            }
            
            // Cleanup backup after successful update
            $this->cleanup_backup();
        }
    }
    
    /**
     * Normalize version string to x.y.z, strip pre-release suffixes
     *
     * @param string $version Version string
     * @return string Normalized version
     */
    private function normalize_version(string $version): string
    {
        // Extract core semantic version (major.minor.patch)
        if (preg_match('/^(\d+\.\d+\.\d+)/', $version, $matches)) {
            return $matches[1];
        }
        return $version;
    }
    
    /**
     * Clear update cache
     */
    public function clear_cache() 
    {
        delete_transient($this->cache_key);
        CacheSys::writeLog('Update cache cleared', 'Updater');
    }
    
    /**
     * Force clear update cache for immediate update check
     * Useful for testing new versions
     */
    public function force_clear_cache() 
    {
        delete_transient($this->cache_key);
        
        // Clear also WordPress core update cache to force recheck
        delete_site_transient('update_plugins');
        
        // Trigger immediate WordPress update check
        wp_update_plugins();
        
        CacheSys::writeLog('Update cache forcefully cleared', 'Updater');
    }
    
    /**
     * Get cache status
     * 
     * @return array Cache information
     */
    public function get_cache_status() 
    {
        $cached_data = get_transient($this->cache_key);
        return [
            'cached' => $cached_data !== false,
            'cache_key' => $this->cache_key,
            'data' => $cached_data
        ];
    }
    
    /**
     * Get configured API URL (for testing)
     *
     * @return string API URL
     */
    public function get_api_url(): string
    {
        return $this->api_url;
    }

    /**
     * Verify download integrity with comprehensive checks
     * 
     * @param string $temp_file Path to downloaded file
     * @return array Verification result with success status and message
     */
    private function verify_download_integrity($temp_file): array 
    {
        // 1. Check file exists and has reasonable size
        if (!file_exists($temp_file)) {
            return ['valid' => false, 'message' => 'Downloaded file does not exist'];
        }
        
        $file_size = filesize($temp_file);
        if ($file_size < self::MIN_FILE_SIZE) {
            return ['valid' => false, 'message' => 'Downloaded file too small: ' . $file_size . ' bytes'];
        }
        
        // 2. Verify ZIP archive integrity
        if (!class_exists('ZipArchive')) {
            CacheSys::writeLog('Warning: ZipArchive class not available, skipping ZIP integrity check', 'Updater');
        } else {
            $zip = new \ZipArchive();
            $result = $zip->open($temp_file, \ZipArchive::CHECKCONS);
            if ($result !== TRUE) {
                return ['valid' => false, 'message' => 'ZIP integrity check failed: ' . $result];
            }
            $zip->close();
        }
        
        // 3. Verify plugin structure
        $structure_check = $this->verify_plugin_structure($temp_file);
        if (!$structure_check['valid']) {
            return $structure_check;
        }
        
        CacheSys::writeLog('Download integrity verification passed', 'Updater');
        return ['valid' => true, 'message' => 'Download integrity verified successfully'];
    }
    
    /**
     * Verify plugin structure in ZIP file
     * 
     * @param string $zip_file Path to ZIP file
     * @return array Verification result
     */
    private function verify_plugin_structure($zip_file): array 
    {
        if (!class_exists('ZipArchive')) {
            return ['valid' => true, 'message' => 'ZipArchive not available, skipping structure check'];
        }
        
        $zip = new \ZipArchive();
        if ($zip->open($zip_file) !== TRUE) {
            return ['valid' => false, 'message' => 'Cannot open ZIP file for structure verification'];
        }
        
        $main_file_found = false;
        $has_src_directory = false;
        
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $file_name = $zip->getNameIndex($i);
            
            // Check for main plugin file
            if (basename($file_name) === 'zizi-cache.php') {
                $main_file_found = true;
            }
            
            // Check for src directory
            if (strpos($file_name, 'src/') !== false) {
                $has_src_directory = true;
            }
        }
        
        $zip->close();
        
        if (!$main_file_found) {
            return ['valid' => false, 'message' => 'Main plugin file (zizi-cache.php) not found in ZIP'];
        }
        
        if (!$has_src_directory) {
            return ['valid' => false, 'message' => 'Plugin src directory not found in ZIP'];
        }
        
        return ['valid' => true, 'message' => 'Plugin structure verified successfully'];
    }
    
    /**
     * Restore plugin from backup in case of failure
     * 
     * @return bool Success status
     */
    private function restore_from_backup(): bool 
    {
        $backup_info = get_option('zizi_cache_atomic_backup');
        
        if (!$backup_info || !isset($backup_info['path'])) {
            CacheSys::writeError('No backup information found for restore', 'Updater');
            return false;
        }
        
        $backup_path = $backup_info['path'];
        $original_path = $backup_info['original_path'];
        
        if (!is_dir($backup_path)) {
            CacheSys::writeError('Backup directory does not exist: ' . $backup_path, 'Updater');
            return false;
        }
        
        try {
            // Remove current plugin directory
            if (is_dir($original_path)) {
                $this->recursive_delete($original_path);
            }
            
            // Restore from backup
            if (!$this->atomic_copy_directory($backup_path, $original_path)) {
                CacheSys::writeError('Failed to restore plugin from backup', 'Updater');
                return false;
            }
            
            // Verify restore integrity
            $original_checksum = $this->calculate_directory_checksum($original_path);
            $expected_checksum = $backup_info['checksum'] ?? '';
            
            if ($original_checksum !== $expected_checksum) {
                CacheSys::writeError('Restore integrity verification failed', 'Updater');
                return false;
            }
            
            CacheSys::writeLog('Plugin successfully restored from backup', 'Updater');
            return true;
            
        } catch (Exception $e) {
            CacheSys::writeError('Exception during backup restore: ' . $e->getMessage(), 'Updater');
            return false;
        }
    }

    /**
     * Parse changelog from local readme.txt file
     * 
     * @return string Formatted changelog HTML
     */
    private function parse_local_changelog(): string 
    {
        $readme_path = dirname($this->plugin_file) . '/readme.txt';
        
        if (!file_exists($readme_path)) {
            CacheSys::writeLog('readme.txt not found for changelog parsing', 'Updater');
            return '<p>Changelog not available.</p>';
        }
        
        $readme_content = file_get_contents($readme_path);
        if (!$readme_content) {
            return '<p>Could not read changelog.</p>';
        }
        
        // Extract changelog section
        if (!preg_match('/== Changelog ==(.*?)(?=== |$)/s', $readme_content, $matches)) {
            return '<p>No changelog section found.</p>';
        }
        
        $changelog_raw = trim($matches[1]);
        
        // Parse version entries
        $versions = [];
        $lines = explode("\n", $changelog_raw);
        $current_version = null;
        $current_changes = [];
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            // Version header (= 1.0.0 =)
            if (preg_match('/^= ([0-9]+\.[0-9]+\.[0-9]+.*?) =$/', $line, $version_match)) {
                // Save previous version
                if ($current_version && !empty($current_changes)) {
                    $versions[$current_version] = $current_changes;
                }
                
                $current_version = $version_match[1];
                $current_changes = [];
            }
            // Change entry (* Change description)
            elseif (preg_match('/^\* (.+)$/', $line, $change_match)) {
                if ($current_version) {
                    $current_changes[] = $change_match[1];
                }
            }
            // Sub-entries or multi-line changes
            elseif (!empty($line) && $current_version && !preg_match('/^=/', $line)) {
                if (!empty($current_changes)) {
                    $last_index = count($current_changes) - 1;
                    $current_changes[$last_index] .= ' ' . $line;
                }
            }
        }
        
        // Save last version
        if ($current_version && !empty($current_changes)) {
            $versions[$current_version] = $current_changes;
        }
        
        // Generate HTML
        if (empty($versions)) {
            return '<p>No version information found in changelog.</p>';
        }
        
        $changelog_html = '';
        $version_count = 0;
        
        foreach ($versions as $version => $changes) {
            $version_count++;
            
            // Show only first 5 versions to keep popup manageable
            if ($version_count > 5) {
                $remaining_versions = count($versions) - 5;
                $changelog_html .= "<p><em>... and {$remaining_versions} more versions. See full changelog in readme.txt</em></p>";
                break;
            }
            
            $changelog_html .= "<h4>Version {$version}</h4>";
            $changelog_html .= "<ul>";
            
            foreach ($changes as $change) {
                // Clean up and format change text
                $change = htmlspecialchars(trim($change));
                
                // Add styling for different types of changes
                if (preg_match('/^(NEW|ADDED):/i', $change)) {
                    $change = '<span style="color: #0073aa; font-weight: bold;">🆕</span> ' . $change;
                } elseif (preg_match('/^(FIX|FIXED|SECURITY):/i', $change)) {
                    $change = '<span style="color: #d63638; font-weight: bold;">🔧</span> ' . $change;
                } elseif (preg_match('/^(REFACTOR|IMPROVED|ENHANCEMENT):/i', $change)) {
                    $change = '<span style="color: #00a32a; font-weight: bold;">⚡</span> ' . $change;
                } else {
                    $change = '<span style="color: #646970;">•</span> ' . $change;
                }
                
                $changelog_html .= "<li>{$change}</li>";
            }
            
            $changelog_html .= "</ul>";
        }
        
        CacheSys::writeLog('Local changelog parsed successfully for ' . count($versions) . ' versions', 'Updater');
        return $changelog_html;
    }

    /**
     * Add changelog link to plugin action links
     * This adds a "View Changelog" link next to Deactivate, Docs etc.
     * 
     * @param array $links Existing plugin action links
     * @param string $plugin_file Plugin file path
     * @return array Modified action links
     */
    public function add_changelog_link($links, $plugin_file) 
    {
        // Only add for our plugin
        if ($plugin_file !== $this->plugin_slug) {
            return $links;
        }
        
        // Only for licensed users
        if (!LicenseManager::is_active()) {
            return $links;
        }
        
        // Create changelog link that opens the plugin details popup
        $plugin_slug = dirname($this->plugin_slug);
        $changelog_url = admin_url('plugin-install.php?tab=plugin-information&plugin=' . $plugin_slug . '&section=changelog&TB_iframe=true&width=600&height=550');
        
        // Add changelog link
        $changelog_link = '<a href="' . esc_url($changelog_url) . '" class="thickbox open-plugin-details-modal changelog-link" aria-label="' . esc_attr__('View ZiziCache changelog') . '">View Changelog</a>';
        
        // Insert changelog link before deactivate link
        $new_links = [];
        foreach ($links as $key => $link) {
            if ($key === 'deactivate') {
                $new_links['changelog'] = $changelog_link;
            }
            $new_links[$key] = $link;
        }
        
        return $new_links;
    }

    /**
     * Add changelog link to plugin row meta (below description)
     * This adds additional meta information in the plugin description area
     * 
     * @param array $plugin_meta Plugin meta links
     * @param string $plugin_file Plugin file path
     * @param array $plugin_data Plugin data
     * @param string $status Plugin status
     * @return array Modified meta links
     */
    public function add_changelog_meta_link($plugin_meta, $plugin_file, $plugin_data, $status) 
    {
        // Only add for our plugin
        if ($plugin_file !== $this->plugin_slug) {
            return $plugin_meta;
        }
        
        // Only for licensed users
        if (!LicenseManager::is_active()) {
            return $plugin_meta;
        }
        
        // Create changelog meta link
        $plugin_slug = dirname($this->plugin_slug);
        $changelog_url = admin_url('plugin-install.php?tab=plugin-information&plugin=' . $plugin_slug . '&section=changelog&TB_iframe=true&width=600&height=550');
        
        $changelog_meta = '<a href="' . esc_url($changelog_url) . '" class="thickbox open-plugin-details-modal changelog-link" aria-label="' . esc_attr__('View changelog') . '">Changelog</a>';
        
        // Add to meta links
        $plugin_meta[] = $changelog_meta;
        
        return $plugin_meta;
    }

    /**
     * Enqueue admin styles for changelog links
     */
    public function enqueue_admin_styles() 
    {
        // Only on plugins page
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'plugins') {
            return;
        }
        
        // Add inline CSS for changelog links
        $css = '
            .plugins .changelog-link {
                color: #0073aa;
                text-decoration: none;
                font-weight: 500;
            }
            .plugins .changelog-link:hover {
                color: #005a87;
                text-decoration: underline;
            }
            .plugins .changelog-link:before {
                content: "📋";
                margin-right: 3px;
                font-size: 12px;
            }
        ';
        
        wp_add_inline_style('wp-admin', $css);
    }
}