<?php
namespace ZiziCache;

/**
 * LCP (Largest Contentful Paint) Handler for ZiziCache plugin
 * 
 * Provides REST API endpoints for storing and managing Largest Contentful Paint data
 * from client-side performance monitoring. Handles automatic detection of LCP elements
 * and stores performance metrics for optimization purposes.
 * 
 * Features:
 * - REST API endpoints for LCP data collection
 * - AJAX fallback for environments where REST API isn't working
 * - Client-side LCP detection script injection
 * - Visual highlighting of LCP elements for debugging
 * - Data persistence using WordPress transients
 * - CORS support for cross-origin requests
 * 
 * @package    ZiziCache
 * @subpackage Performance
 * @since      1.0.0
 * @author     ZiziCache Team
 */
class LcpHandler
{    
    /**
     * Initialize the LCP handler and register hooks
     * 
     * Sets up REST API endpoints, enqueues client-side scripts, and configures
     * AJAX handlers for LCP data collection. This method should be called during
     * plugin initialization.
     * 
     * @since 1.0.0
     * @return void
     */
    public static function init(): void
    {
        add_action('rest_api_init', [__CLASS__, 'register_rest_endpoints']);
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_lcp_detector']);
        
        // We add CORS headers directly in register_rest_endpoints() instead of a separate method
        // Removed: add_action('rest_pre_serve_request', [__CLASS__, 'add_cors_headers']);
        // Add AJAX endpoint as fallback for environments where REST API isn't working
        add_action('wp_ajax_zizi_lcp_data', [__CLASS__, 'ajax_handle_lcp_data']);
        add_action('wp_ajax_nopriv_zizi_lcp_data', [__CLASS__, 'ajax_handle_lcp_data']);
    }

    /**
     * Registers REST API endpoints for LCP data collection
     * 
     * Creates multiple endpoint variants to ensure maximum compatibility across
     * different WordPress configurations. Handles CORS preflight requests and
     * provides fallback endpoints with different namespace formats.
     * 
     * Registered endpoints:
     * - /{plugin-dir}/lcp (dynamic namespace)
     * - /zizi-cache/lcp (fixed namespace)
     * - /zizi-cache/v1/lcp (versioned namespace)
     * - /zizicache/v1/lcp (alternative namespace)
     * 
     * @since 1.0.0
     * @return void
     */    
    public static function register_rest_endpoints(): void
    {
        // Add CORS headers directly here instead of a separate add_cors_headers method
        // This resolves the issue with a non-existent add_cors_headers method
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type, X-WP-Nonce');
            status_header(200);
            exit;
        }

        // Get current REST API namespace based on the plugin directory name
        $plugin_dir_name = basename(dirname(ZIZI_CACHE_FILE));
        
        // Register multiple endpoint variants to ensure compatibility
        
        // 1. Dynamic namespace (current plugin directory)
        register_rest_route($plugin_dir_name, '/lcp', [
            'methods' => ['POST', 'OPTIONS'],  // Added OPTIONS for CORS preflight
            'callback' => [__CLASS__, 'handle_lcp_data'],
            'permission_callback' => '__return_true'  // Allow all users (including non-logged in)
        ]);
        
        // 2. Fixed namespace 'zizi-cache'
        register_rest_route('zizi-cache', '/lcp', [
            'methods' => ['POST', 'OPTIONS'],  // Added OPTIONS for CORS preflight
            'callback' => [__CLASS__, 'handle_lcp_data'],
            'permission_callback' => '__return_true'  // Allow all users (including non-logged in)
        ]);

        // 3. Fixed namespace with version 'zizi-cache/v1'
        register_rest_route('zizi-cache/v1', '/lcp', [
            'methods' => ['POST', 'OPTIONS'],  // Added OPTIONS for CORS preflight
            'callback' => [__CLASS__, 'handle_lcp_data'],
            'permission_callback' => '__return_true'  // Allow all users (including non-logged in)
        ]);
        
        // 4. Let's also register an endpoint in the format zizicache/v1
        register_rest_route('zizicache/v1', '/lcp', [
            'methods' => ['POST', 'OPTIONS'],  // Added OPTIONS for CORS preflight
            'callback' => [__CLASS__, 'handle_lcp_data'],
            'permission_callback' => '__return_true'  // Allow all users (including non-logged in)
        ]);
    }

    /**     * Enqueues the LCP detection script and related assets
     * 
     * Loads the client-side JavaScript for detecting Largest Contentful Paint elements
     * and conditionally injects CSS for visual highlighting. Only loads when automatic LCP 
     * detection is enabled and not on admin pages.
     * 
     * Features:
     * - Conditional loading based on configuration
     * - CSS highlighting only loaded when detection should be shown (admins or public debug)
     * - Multiple REST API endpoint fallbacks
     * - AJAX fallback for REST API issues
     * - Visual highlighting styles for LCP elements (conditional)
     * - Admin/debug mode detection
     * - Responsive CSS for different screen sizes
     * 
     * @since 1.0.0
     * @return void
     */    
    public static function enqueue_lcp_detector(): void
    {
        // Skip on admin pages and page builder editors
        if (is_admin() || 
            (class_exists('ZiziCache\Plugins\Integrations\PageBuilders') && 
             \ZiziCache\Plugins\Integrations\PageBuilders::is_page_builder_active())) {
            return;
        }

        // Check if automatic LCP detection is enabled        
        if (empty(SysConfig::$config['img_auto_lcp'])) {
            return;
        }
        
        // Loading LCP detector - adding unique version with timestamp to prevent cache
        wp_enqueue_script(
            'zizicache-lcp-detector',
            ZIZI_CACHE_PLUGIN_URL . 'assets/js/zizi-lcp-detector.js',
            [],
            ZIZI_CACHE_VERSION . '.' . time(), // Adding timestamp to prevent caching
            true // Load in footer
        );
        
        // Get AJAX URL for use in case of REST API issues
        $ajax_url = admin_url('admin-ajax.php');
        
        // Try different endpoint versions for maximum compatibility
        $plugin_dir_name = basename(dirname(ZIZI_CACHE_FILE));
        
        // Preferred endpoint for LCP data
        // Try several possible formats from most likely to less likely
        $possible_endpoints = [
            rest_url('zizi-cache/v1/lcp'),      // Most common format 
            rest_url($plugin_dir_name . '/lcp'), // Dynamic namespace            
            rest_url('zizi-cache/lcp'),         // Simpler without version
            rest_url('zizicache/v1/lcp'),       // Possible variant without hyphen
        ];
          // Create nonce for logged-in users only, as it doesn't make sense for non-logged in users
        // (WP verifies nonce against user ID)
        $nonce = is_user_logged_in() ? wp_create_nonce('wp_rest') : '';        // Check if LCP detection should be shown for administrators and/or public users
        $show_admin = !empty(SysConfig::$config['img_auto_lcp_admin_debug']);
        $show_public = !empty(SysConfig::$config['img_auto_lcp_public_debug']);
        $is_admin = Authority::isAllowedSilent(); // Use silent check to avoid access denied logs for guests
        
        // Determine if LCP detection should be displayed
        $show_lcp_detection = ($is_admin && $show_admin) || (!$is_admin && $show_public);

        // Add configuration for LCP detector
        wp_localize_script('zizicache-lcp-detector', 'ZiziCacheConfig', [
            'api' => [
                // Provide all possible endpoints
                'lcpEndpoint' => $possible_endpoints[0],  // Primary endpoint
                'fallbackEndpoints' => $possible_endpoints, // All possible endpoints for fallback 
                'ajaxUrl' => $ajax_url,          // AJAX URL for potential fallback 
                'nonce' => $nonce                // Nonce for logged-in users
            ],
            // Indicator whether to display detailed information about LCP
            'isAdmin' => $show_lcp_detection,
            'debugMode' => defined('WP_DEBUG') && WP_DEBUG // Information about debug mode
        ]);        
        
        // CSS for highlighting LCP elements is loaded ONLY when LCP detection display is enabled
        if ($show_lcp_detection) {
            // Inline CSS for highlighting LCP element - first create the handle
            wp_register_style('zizicache-lcp-detector-css', false);
            wp_enqueue_style('zizicache-lcp-detector-css');        

            // Inline CSS for LCP element highlighting - NEW APPROACH
            // Using CSS for outlining without changing DOM structure
            $lcp_highlight_css = "
                /* Basic styles for highlighting LCP element - outline only, no positioning changes */
                .zizicache-lcp-element {
                    outline: 3px dashed #22c55e !important;
                    outline-offset: 3px !important;
                    /* No CSS position changes */
                }
                
                /* Badge for LCP - as a standalone overlay element */
                .zizicache-lcp-badge {
                    position: absolute !important;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, Arial, sans-serif !important;
                    font-size: 11px !important;
                    line-height: 1.4 !important;
                    padding: 2px 6px !important;
                    background-color: #22c55e !important;
                    color: white !important;
                    border-radius: 4px !important;
                    z-index: 2147483647 !important; /* Maximum possible value */
                    pointer-events: none !important;
                    box-shadow: 0 1px 3px rgba(0,0,0,0.3) !important;
                    white-space: nowrap !important;
                    user-select: none !important;
                    font-weight: bold !important;
                    transform: translateZ(0) !important;
                }
                
                /* Overlay for highlighting background images */
                .zizicache-lcp-outline {
                    box-sizing: border-box !important;
                    border: 3px dashed #22c55e !important;
                    pointer-events: none !important;
                    z-index: 2147483646 !important;
                }
                
                /* Adjusted position for small screens */
                @media screen and (max-width: 768px) {
                    .zizicache-lcp-badge {
                        font-size: 9px !important;
                        padding: 1px 4px !important;
                    }
                }
            ";
            
            wp_add_inline_style('zizicache-lcp-detector-css', $lcp_highlight_css);
        }
    }    /**
     * Handles incoming LCP data from REST API requests
     * 
     * Processes LCP (Largest Contentful Paint) data submitted by the client-side
     * detection script. Validates the data, manages storage, and implements
     * intelligent deduplication and data retention policies.
     * 
     * Data processing features:
     * - CORS headers for cross-origin requests
     * - Multiple data input methods (JSON params, POST body, $_POST)
     * - Data validation and sanitization
     * - Intelligent deduplication (keeps largest LCP per URL/device)
     * - Automatic data retention (max 100 records, 30-day expiry)
     * - Timestamp-based cleanup of old records
     * 
     * @since 1.0.0
     * @param \WP_REST_Request $request The REST API request object containing LCP data
     * @return \WP_REST_Response JSON response with success/error status and message
     * 
     * @example
     * POST /wp-json/zizi-cache/v1/lcp
     * {
     *   "lcpInfo": {
     *     "url": "https://example.com/page",
     *     "deviceType": "desktop",
     *     "size": 1024,
     *     "timestamp": 1609459200
     *   }
     * }
     */
    public static function handle_lcp_data(\WP_REST_Request $request): \WP_REST_Response
    {
        // Add CORS headers for each request
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, X-WP-Nonce');
        
        // If it's an OPTIONS request, return a successful response and stop processing
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            return new \WP_REST_Response(null, 200);
        }
        
        $params = $request->get_json_params();
        
        // If data is not in get_json_params(), try to get data from POST
        if (empty($params)) {
            $body = $request->get_body();
            if (!empty($body)) {
                $params = json_decode($body, true);
            }
        }
        
        // If we still don't have data, try to get it directly from $_POST
        if (empty($params) && !empty($_POST)) {
            $params = $_POST;
        }
        
        if (empty($params['lcpInfo'])) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Missing LCP data'
            ], 400);
        }

        $lcp_info = $params['lcpInfo'];
        
        // Basic data validation
        if (empty($lcp_info['url']) || empty($lcp_info['deviceType'])) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Invalid LCP data'
            ], 400);
        }

        // Get existing data
        $lcp_data = get_transient('zizicache_lcp_data');
        if ($lcp_data === false) {
            $lcp_data = [];
        }

        // Find existing record for this URL and device type
        $found = false;
        foreach ($lcp_data as $key => $item) {
            if ($item['url'] === $lcp_info['url'] && $item['deviceType'] === $lcp_info['deviceType']) {
                // Update existing record only if the new element is larger or has a better score
                if ($lcp_info['size'] > $item['size']) {
                    $lcp_data[$key] = $lcp_info;
                }
                $found = true;
                break;
            }
        }

        // If record doesn't exist, add a new one
        if (!$found) {
            $lcp_data[] = $lcp_info;
            
            // Limit the number of records to 100
            if (count($lcp_data) > 100) {
                // Sort by time to remove the oldest
                usort($lcp_data, function($a, $b) {
                    return $a['timestamp'] <=> $b['timestamp'];
                });
                
                // Remove oldest records
                $lcp_data = array_slice($lcp_data, -100);
            }
        }

        // Save data to transients (valid for 30 days)
        set_transient('zizicache_lcp_data', $lcp_data, 30 * DAY_IN_SECONDS);

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'LCP data has been successfully saved'
        ], 200);
    }

    /**
     * AJAX handler for LCP data submission
     * 
     * Provides a fallback method for LCP data submission when REST API
     * is not available or accessible. Uses WordPress AJAX hooks and
     * implements the same data processing logic as the REST handler.
     * 
     * Features:
     * - Fallback for REST API issues
     * - JSON string decoding support
     * - Same validation and storage logic as REST handler
     * - WordPress AJAX response format
     * - Support for both logged-in and non-logged-in users
     * 
     * @since 1.0.0
     * @return void Outputs JSON response and exits
     * 
     * @example
     * POST /wp-admin/admin-ajax.php
     * action=zizi_lcp_data&lcpInfo={"url":"https://example.com","deviceType":"mobile"}
     */
    public static function ajax_handle_lcp_data()
    {
        // Get raw data from POST
        $lcp_info_raw = isset($_POST['lcpInfo']) ? $_POST['lcpInfo'] : null;

        // If data came as a JSON string, decode it
        if (is_string($lcp_info_raw)) {
            // Use stripslashes as WordPress often adds them to POST data
            $lcp_info_raw = json_decode(stripslashes($lcp_info_raw), true);
        }

        // Basic structure validation
        if (empty($lcp_info_raw) || !is_array($lcp_info_raw) || empty($lcp_info_raw['url']) || empty($lcp_info_raw['deviceType'])) {
            wp_send_json_error(['message' => 'Invalid LCP data structure'], 400);
            exit;
        }

        // Whitelist of expected keys and their sanitizers
        $allowed_keys = [
            'url'        => 'esc_url_raw',
            'deviceType' => 'sanitize_text_field',
            'size'       => 'intval',
            'timestamp'  => 'intval',
            'src'        => 'esc_url_raw',
            'selector'   => 'sanitize_text_field',
        ];

        $lcp_info = [];
        foreach ($allowed_keys as $key => $sanitizer) {
            // Only process keys that are present in the raw data
            if (isset($lcp_info_raw[$key])) {
                $lcp_info[$key] = call_user_func($sanitizer, $lcp_info_raw[$key]);
            }
        }

        // Re-check essential fields after sanitization
        if (empty($lcp_info['url']) || empty($lcp_info['deviceType'])) {
            wp_send_json_error(['message' => 'Invalid LCP data after sanitization'], 400);
            exit;
        }

        // Get existing data
        $lcp_data = get_transient('zizicache_lcp_data');
        if ($lcp_data === false || !is_array($lcp_data)) {
            $lcp_data = [];
        }

        // Find existing record for this URL and device type
        $found = false;
        foreach ($lcp_data as $key => $item) {
            // Ensure the item from transient has the required keys before comparing
            if (isset($item['url'], $item['deviceType']) && $item['url'] === $lcp_info['url'] && $item['deviceType'] === $lcp_info['deviceType']) {
                // Update existing record only if the new element is larger
                if (isset($lcp_info['size'], $item['size']) && $lcp_info['size'] > $item['size']) {
                    // Merge to preserve other potential data, but overwrite with new sanitized data
                    $lcp_data[$key] = array_merge($item, $lcp_info);
                }
                $found = true;
                break;
            }
        }

        // If record doesn't exist, add a new one
        if (!$found) {
            $lcp_data[] = $lcp_info;
        }

        // Limit the number of records to 100
        if (count($lcp_data) > 100) {
            // Sort by timestamp to remove the oldest records (newest first)
            usort($lcp_data, function ($a, $b) {
                $ts_a = $a['timestamp'] ?? 0;
                $ts_b = $b['timestamp'] ?? 0;
                return $ts_b <=> $ts_a; // Sort descending
            });
            // Keep only the newest 100 records
            $lcp_data = array_slice($lcp_data, 0, 100);
        }

        // Save data to transient (valid for 30 days)
        set_transient('zizicache_lcp_data', $lcp_data, 30 * DAY_IN_SECONDS);

        wp_send_json_success(['message' => 'LCP data has been successfully saved']);
        exit;
    }

    /**
     * Clears all stored LCP data from WordPress transients
     * 
     * Removes all collected LCP performance data from the database.
     * This can be useful for debugging, privacy compliance, or
     * when resetting performance monitoring data.
     * 
     * @since 1.0.0
     * @return bool True if the transient was successfully deleted, false otherwise
     * 
     * @example
     * if (LcpHandler::clear_lcp_data()) {
     *     echo "LCP data cleared successfully";
     * } else {
     *     echo "Failed to clear LCP data or no data found";
     * }
     */
    public static function clear_lcp_data(): bool
    {
        return delete_transient('zizicache_lcp_data');
    }
}
