<?php
declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\CacheSys;
use ZiziCache\SysConfig;
use ZiziCache\SysTool;
use MatthiasMullie\Minify;

/**
 * JavaScript Optimization and Management
 * 
 * This class provides comprehensive JavaScript optimization features including:
 * - Minification of JavaScript files
 * - Defer and async script loading
 * - Module script handling
 * - Script combination and optimization
 * - Third-party script optimization
 * - JavaScript execution timing control
 * 
 * @package ZiziCache
 */
class JavaScript
{
  /**
   * Initialize JavaScript optimization features
   * 
   * Sets up necessary hooks for JavaScript optimization including:
   * - Preload library injection
   * - Configuration updates
   * - Cleanup on deactivation
   */
  public static function init(): void
  {
    // Make sure we have fresh configurations
    if (!defined('REST_REQUEST') || !REST_REQUEST) {
      add_action('wp_enqueue_scripts', [__CLASS__, 'inject_preload_lib']);
    }
    
    // Add action to refresh config on certain admin actions (saving settings)
    add_action('zizi_cache_update_config:after', function($config) {
      // Force a config refresh when settings are updated
      SysConfig::$config = $config;
    });
    
    // Clean up third-party files on plugin deactivation
    register_deactivation_hook(ZIZI_CACHE_FILE_NAME, [__CLASS__, 'cleanup_on_deactivation']);
  }
  
  /**
   * Cleanup third-party files when plugin is deactivated
   */
  public static function cleanup_on_deactivation(): void
  {
    if (class_exists('\\ZiziCache\\SysTool')) {
      SysTool::cleanup_third_party_files(true); // Force cleanup on deactivation
    }
  }

  /**
   * Minify JavaScript files in the HTML content
   * 
   * Processes all script tags with src attributes, minifies the JavaScript,
   * and serves the minified version from the cache directory.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with minified JavaScript files
   */
  public static function minify(string $html): string
  {
    if (!SysConfig::$config['js_minify']) {
      return $html;
    }

    // get all the scripts with src attribute
    preg_match_all('/<script[^>]*src=[\'"][^\'"]+[\'"][^>]*><\/script>/i', $html, $scripts);

    // Get excluded keywords from filter
    $exclude_keywords = apply_filters('zizi_cache_exclude_from_minify:js', []);

    try {
      // loop through all the scripts
      foreach ($scripts[0] as $script_tag) {
        // skip if script is excluded
        if (SysTool::any_keywords_match_string($exclude_keywords, $script_tag)) {
          continue;
        }

        $script = new HTML($script_tag);
        $src = $script->src;
        if (empty($src)) {
            continue;
        }
        $file_path = CacheSys::get_file_path_from_url($src);

        // Skip if file doesn't exist or empty
        if (empty($file_path) || !is_file($file_path) || !filesize($file_path)) {
          continue;
        }

        // Generate hash
        $hash = substr(hash_file('md5', $file_path), 0, 12);

        // If already minified, add hash to the query string and skip minification
        if (preg_match('/\.min\.js/', $src)) {
          $html = str_replace($src, strtok($src, '?') . "?ver=$hash", $html);
          continue;
        }

        // Generate minified file path and URL
        $file_name = $hash . '.' . basename($file_path);
        $minified_path = ZIZI_CACHE_CACHE_DIR . $file_name;
        $minified_url = ZIZI_CACHE_CACHE_URL . $file_name;

        // Create minified version if it doesn't exist
        if (!is_file($minified_path)) {
          $minifier = new Minify\JS($file_path);
          $minifier->minify($minified_path);
          
          // Create gzipped version for OLS server compatibility
          $gzipped_path = $minified_path . '.gz';
          if (function_exists('gzencode')) {
              $minified_js = file_get_contents($minified_path);
              $gzipped_js = gzencode($minified_js, 9);
              file_put_contents($gzipped_path, $gzipped_js);
          }
        }

        // Check if minified version is smaller than original
        $original_file_size = filesize($file_path);
        $minified_file_size = filesize($minified_path);
        
        if ($minified_file_size === 0) { // If minified file is empty, use original with hash
            $minified_url = strtok($src, '?') . "?ver=$hash";
        } else {
            $wasted_bytes = $original_file_size - $minified_file_size;
            $wasted_percent = ($original_file_size > 0) ? (($wasted_bytes / $original_file_size) * 100) : 0;
    
            if ($wasted_bytes < 2048 || $wasted_percent < 10) {
              $minified_url = strtok($src, '?') . "?ver=$hash";
            }
        }

        $html = str_replace($src, $minified_url, $html);
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('JS Minify Error: ' . $e->getMessage(), 'Security');
    } finally {
      return $html;
    }
  }

  /**
   * Move module scripts to the end of the document
   * 
   * Relocates module scripts (type="module") to the end of the document
   * to improve page load performance.
   * 
   * @param string $html The HTML content to process
   * @return string The modified HTML with relocated module scripts
   */
  public static function move_module_scripts(string $html): string
  {
    // Ensure js_defer is properly checked with strict comparison and type validation
    if (empty(SysConfig::$config['js_defer']) || SysConfig::$config['js_defer'] !== true) {
      return $html;
    }

    // Regex to capture both self-closing and standard script tags with type="module"
    $module_scripts_regex = '/<script[^>]*type=["\']module["\'][^>]*?(?:><\/script>|\/>)/i';
    preg_match_all($module_scripts_regex, $html, $matches);

    // If no module scripts found, return the original HTML
    if (empty($matches[0])) {
      return $html;
    }

    // Extract the found module scripts
    $module_scripts = implode("\n", $matches[0]);

    // Remove the module scripts from the original HTML
    $html = preg_replace($module_scripts_regex, '', $html);

    // Place the module scripts before the closing body tag
    $html = preg_replace('/<\/body>/i', $module_scripts . "\n</body>", $html);

    return $html;
  }

  /**
   * Add defer attribute to external JavaScript files
   * 
   * Processes external scripts and adds the defer attribute to improve
   * page load performance while maintaining execution order.
   * 
   * @param string $original_html_input The HTML content to process
   * @return string The processed HTML with deferred external scripts
   */
  public static function defer_external(string $original_html_input): string
  {
    // Ensure js_defer is properly checked with strict comparison and type validation
    if (empty(SysConfig::$config['js_defer']) || SysConfig::$config['js_defer'] !== true) {
      // --- ZIZI DEBUG START ---
      \ZiziCache\CacheSys::writeLog('ZiziCache Defer External SKIPPED: js_defer is not true. Value: ' . var_export(SysConfig::$config['js_defer'] ?? 'NOT SET', true) . ' Type: ' . gettype(SysConfig::$config['js_defer'] ?? null));
      // --- ZIZI DEBUG END ---
      return $original_html_input;
    }

    // get excluded keywords from filter and config
    $exclude_keywords = apply_filters(
      'zizi_cache_exclude_from_defer:js',
      is_array(SysConfig::$config['js_defer_excludes']) ? SysConfig::$config['js_defer_excludes'] : []
    );

    // --- ZIZI DEBUG START ---
    $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
    if ($debug_level === 'verbose') {
      $initial_debug_log = [
        'function' => __METHOD__,
        'js_defer_active' => (empty(SysConfig::$config['js_defer']) || SysConfig::$config['js_defer'] !== true) ? 'NO' : 'YES',
        'excludes_count' => count($exclude_keywords),
      ];
      \ZiziCache\CacheSys::writeLog('[DEBUG] Defer External START: ' . json_encode($initial_debug_log));
    } elseif ($debug_level === 'basic') {
      \ZiziCache\CacheSys::writeLog('[INFO] Defer External: Processing scripts with ' . count($exclude_keywords) . ' exclusion rules');
    }
    // --- ZIZI DEBUG END ---

    $processed_html = preg_replace_callback(
        '/(<script\s+[^>]*src\s*=\s*(["\'])(?:(?!\2).)*\2[^>]*>)([\s\S]*?)<\/script>/i',
        function ($matches) use ($exclude_keywords) {
            $original_script_element_string = $matches[0];
            $opening_tag_string = $matches[1];
            // $quote_char = $matches[2]; // Available if needed
            $inner_content = $matches[3];

            $is_excluded = SysTool::any_keywords_match_string($exclude_keywords, $original_script_element_string);
            // --- ZIZI DEBUG START (Callback) ---
            $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
            if ($debug_level === 'verbose') {
                // Extract script identifier (id or src filename)
                $script_id = '';
                if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                    $script_id = $id_match[1];
                } elseif (preg_match('/src=["\'][^"\']*\/([^\/\?"\']+)/', $original_script_element_string, $src_match)) {
                    $script_id = $src_match[1];
                } else {
                    $script_id = 'unknown';
                }
                \ZiziCache\CacheSys::writeLog('[DEBUG] Defer External Script: ' . $script_id . ' | Excluded: ' . ($is_excluded ? 'YES' : 'NO'));
            }
            // --- ZIZI DEBUG END ---

            if ($is_excluded) {
                $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
                if ($debug_level === 'verbose') {
                    // Extract script identifier for concise logging
                    $script_id = '';
                    if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                        $script_id = $id_match[1];
                    } elseif (preg_match('/src=["\'][^"\']*\/([^\/\?"\']+)/', $original_script_element_string, $src_match)) {
                        $script_id = $src_match[1];
                    } else {
                        $script_id = 'unknown';
                    }
                    \ZiziCache\CacheSys::writeLog('[DEBUG] Defer External SKIPPED: ' . $script_id . ' | Reason: Excluded by rules');
                }
                return $original_script_element_string;
            }

            try {
                // Direct approach to ensure proper formatting
                // Remove async attribute if present
                $modified_opening_tag = preg_replace('/\sasync(?=[\s>])/', '', $opening_tag_string);
                
                // Add defer attribute right before the closing bracket of the opening tag
                if (strpos($modified_opening_tag, ' defer') === false) {
                    $modified_opening_tag = rtrim($modified_opening_tag, '>') . ' defer>';
                }
                
                // Combine the modified opening tag with the inner content and closing tag
                $modified_script_element_string = $modified_opening_tag . $inner_content . '</script>';

                // --- ZIZI DEBUG START (Callback) ---
                $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
                if ($debug_level === 'verbose') {
                    // Extract script identifier for concise logging
                    $script_id = '';
                    if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                        $script_id = $id_match[1];
                    } elseif (preg_match('/src=["\'][^"\']*\/([^\/\?"\']+)/', $original_script_element_string, $src_match)) {
                        $script_id = $src_match[1];
                    } else {
                        $script_id = 'inline-script';
                    }
                    \ZiziCache\CacheSys::writeLog('[DEBUG] Defer External Applied: ' . $script_id . ' | Success: ' . ($original_script_element_string !== $modified_script_element_string ? 'YES' : 'NO'));
                } elseif ($debug_level === 'basic') {
                    // Only log significant events in basic mode
                    if ($original_script_element_string === $modified_script_element_string) {
                        \ZiziCache\CacheSys::writeLog('[WARNING] Defer External: Script unchanged - check configuration');
                    }
                }
                
                // Keep critical warnings regardless of debug level
                if (strpos($modified_script_element_string, 'defer') === false) {
                    \ZiziCache\CacheSys::writeLog('[WARNING] Defer External: "defer" attribute not found in processed script');
                }
                // --- ZIZI DEBUG END ---
                return $modified_script_element_string;
            } catch (\Exception $e) {
                \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Defer External Callback Exception for tag ' . $original_script_element_string . ': ' . $e->getMessage());
                return $original_script_element_string; // Return original on error within callback
            }
        },
        $original_html_input
    );

    if ($processed_html === null && preg_last_error() !== PREG_NO_ERROR) {
        \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Defer External: preg_replace_callback error: ' . preg_last_error_msg() . '. Returning original HTML.');
        return $original_html_input; // Return original input on catastrophic preg error
    }
    
    return $processed_html;
  }

  /**
   * Convert inline scripts to external scripts with defer attribute
   * 
   * Converts inline JavaScript to external files loaded with defer
   * to improve page load performance.
   * 
   * @param string $original_html_input The HTML content to process
   * @return string The processed HTML with deferred inline scripts
   */
  public static function defer_inline(string $original_html_input): string
  {
    // Ensure js_defer is properly checked with strict comparison and type validation
    if (empty(SysConfig::$config['js_defer']) || SysConfig::$config['js_defer'] !== true) {
      // --- ZIZI DEBUG START ---
      \ZiziCache\CacheSys::writeLog('ZiziCache Defer Inline SKIPPED: js_defer is not true. Value: ' . var_export(SysConfig::$config['js_defer'] ?? 'NOT SET', true) . ' Type: ' . gettype(SysConfig::$config['js_defer'] ?? null));
      // --- ZIZI DEBUG END ---
      return $original_html_input;
    }

    // get excluded keywords from filter and config
    $exclude_keywords = apply_filters(
      'zizi_cache_exclude_from_defer:js',
      is_array(SysConfig::$config['js_defer_excludes']) ? SysConfig::$config['js_defer_excludes'] : []
    );

    // --- ZIZI DEBUG START ---
    $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
    if ($debug_level === 'verbose') {
      $initial_debug_log_inline = [
        'function' => __METHOD__,
        'js_defer_active' => (empty(SysConfig::$config['js_defer']) || SysConfig::$config['js_defer'] !== true) ? 'NO' : 'YES',
        'excludes_count' => count($exclude_keywords),
      ];
      \ZiziCache\CacheSys::writeLog('[DEBUG] Defer Inline START: ' . json_encode($initial_debug_log_inline));
    } elseif ($debug_level === 'basic') {
      \ZiziCache\CacheSys::writeLog('[INFO] Defer Inline: Processing inline scripts with ' . count($exclude_keywords) . ' exclusion rules');
    }
    // --- ZIZI DEBUG END ---

    $processed_html = preg_replace_callback(
        '/<script(?![^>]*src)[^>]*>([\s\S]*?)<\/script>/i', // Matches full inline script tag
        function ($matches) use ($exclude_keywords) {
            $original_script_element_string = $matches[0]; // Full <script>...</script>
            // $inner_content_from_regex = $matches[1]; // Content from regex capture group 1

            $is_excluded = SysTool::any_keywords_match_string($exclude_keywords, $original_script_element_string);
            // --- ZIZI DEBUG START (Callback) ---
            $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
            if ($debug_level === 'verbose') {
                // Extract script identifier for concise logging
                $script_id = '';
                if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                    $script_id = $id_match[1];
                } else {
                    // For inline scripts, use first 30 chars of content
                    $content_preview = trim(preg_replace('/\s+/', ' ', substr(strip_tags($original_script_element_string), 0, 30)));
                    $script_id = 'inline:' . $content_preview . '...';
                }
                \ZiziCache\CacheSys::writeLog('[DEBUG] Defer Inline Script: ' . $script_id . ' | Excluded: ' . ($is_excluded ? 'YES' : 'NO'));
            }
            // --- ZIZI DEBUG END ---

            if ($is_excluded) {
                $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
                if ($debug_level === 'verbose') {
                    // Extract script identifier for concise logging
                    $script_id = '';
                    if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                        $script_id = $id_match[1];
                    } else {
                        $content_preview = trim(preg_replace('/\s+/', ' ', substr(strip_tags($original_script_element_string), 0, 30)));
                        $script_id = 'inline:' . $content_preview . '...';
                    }
                    \ZiziCache\CacheSys::writeLog('[DEBUG] Defer Inline SKIPPED: ' . $script_id . ' | Reason: Excluded by rules');
                }
                return $original_script_element_string;
            }

            try {
                $script_object = new HTML($original_script_element_string);

                if ($script_object->type && $script_object->type !== 'text/javascript' && $script_object->type !== 'application/javascript') {
                    $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
                    if ($debug_level === 'verbose') {
                        // Extract script identifier for concise logging
                        $script_id = '';
                        if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                            $script_id = $id_match[1];
                        } else {
                            $content_preview = trim(preg_replace('/\s+/', ' ', substr(strip_tags($original_script_element_string), 0, 30)));
                            $script_id = 'inline:' . $content_preview . '...';
                        }
                        \ZiziCache\CacheSys::writeLog('[DEBUG] Defer Inline SKIPPED: ' . $script_id . ' | Reason: Non-standard type (' . $script_object->type . ')');
                    } elseif ($debug_level === 'basic') {
                        \ZiziCache\CacheSys::writeLog('[INFO] Defer Inline: Skipped non-standard script type: ' . $script_object->type);
                    }
                    return $original_script_element_string;
                }

                $content_to_encode = $script_object->getContent();
                if(empty($content_to_encode)) {
                    $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
                    if ($debug_level === 'verbose') {
                        $script_id = '';
                        if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                            $script_id = $id_match[1];
                        } else {
                            $script_id = 'inline:empty';
                        }
                        \ZiziCache\CacheSys::writeLog('[DEBUG] Defer Inline SKIPPED: ' . $script_id . ' | Reason: Empty content');
                    }
                    return $original_script_element_string;
                }

                $data_uri_src = 'data:text/javascript,' . rawurlencode($content_to_encode);

                // Extract existing attributes from the opening tag
                preg_match('/<script([^>]*)>/', $original_script_element_string, $tag_parts);
                $attributes = $tag_parts[1];
                
                // Remove any type attribute if present (we'll add our own)
                $attributes = preg_replace('/\stype=(["\']).*?\1/', '', $attributes);
                
                // Remove any async attribute if present
                $attributes = preg_replace('/\sasync(?=[\s>])/', '', $attributes);
                
                // Build new script tag with src, defer, and appropriate type
                $modified_script_element_string = "<script{$attributes} src=\"{$data_uri_src}\" defer></script>";

                // --- ZIZI DEBUG START (Callback) ---
                $debug_level = SysConfig::$config['js_debug_level'] ?? 'basic';
                if ($debug_level === 'verbose') {
                    // Extract script identifier for concise logging
                    $script_id = '';
                    if (preg_match('/id=["\']([^"\']+)["\']/', $original_script_element_string, $id_match)) {
                        $script_id = $id_match[1];
                    } else {
                        $content_preview = trim(preg_replace('/\s+/', ' ', substr(strip_tags($original_script_element_string), 0, 30)));
                        $script_id = 'inline:' . $content_preview . '...';
                    }
                    \ZiziCache\CacheSys::writeLog('[DEBUG] Defer Inline Applied: ' . $script_id . ' | Externalized: YES');
                } elseif ($debug_level === 'basic') {
                    // Only log significant events in basic mode
                    if ($original_script_element_string === $modified_script_element_string) {
                        \ZiziCache\CacheSys::writeLog('[WARNING] Defer Inline: Script unchanged - check configuration');
                    }
                }
                
                // Keep critical warnings regardless of debug level
                if (strpos($modified_script_element_string, 'defer') === false) {
                    \ZiziCache\CacheSys::writeLog('[WARNING] Defer Inline: "defer" attribute not found in processed script');
                }
                // --- ZIZI DEBUG END ---
                return $modified_script_element_string;
            } catch (\Exception $e) {
                \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Defer Inline Callback Exception for tag ' . $original_script_element_string . ': ' . $e->getMessage());
                return $original_script_element_string; // Return original on error
            }
        },
        $original_html_input
    );

    if ($processed_html === null && preg_last_error() !== PREG_NO_ERROR) {
        \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Defer Inline: preg_replace_callback error: ' . preg_last_error_msg() . '. Returning original HTML.');
        return $original_html_input; // Return original input on catastrophic preg error
    }

    return $processed_html;
  }

  /**
   * Delay loading of non-critical JavaScript
   * 
   * Delays loading of selected scripts until user interaction
   * or when they're about to enter the viewport.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with delayed script loading
   */
  public static function delay_scripts(string $html): string
  {
    $config = SysConfig::$config;

    // Check if js_delay is enabled in configuration
    if (empty($config['js_delay']) || $config['js_delay'] !== true) {
      \ZiziCache\CacheSys::writeLog('ZiziCache Delay Scripts SKIPPED: js_delay is not enabled. Value: ' . var_export($config['js_delay'] ?? 'NOT SET', true) . ' Type: ' . gettype($config['js_delay'] ?? null));
      return $html;
    }
    
    // CRITICAL: Prevent conflict with User Interaction Script Delay function
    if (!empty($config['js_user_interaction_delay'])) {
      \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts SKIPPED: Advanced User Interaction Script Delay is already enabled. Only one delay method can be active at a time.');
      return $html;
    }

    // Get all the scripts with more precise pattern handling different formats
    preg_match_all('/<script[^>]*(?:src=[\'"][^\'"]+[\'"][^>]*|>([\s\S]*?)<\/script>)/i', $html, $scripts);
    
    // Log how many scripts we found
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: Found ' . count($scripts[0]) . ' script tags to process');

    // Get delay method from configuration
    $delay_method = $config['js_delay_method'] ?? 'selected'; // selected or all
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: Using delay method: ' . $delay_method);

    // Get keywords based on delay method
    $keywords = [];
    if ($delay_method === 'selected') {
      $keywords = is_array($config['js_delay_selected']) ? $config['js_delay_selected'] : [];
      \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: Using js_delay_selected keywords: ' . print_r($keywords, true));
    } else { // 'all' method
      $keywords = is_array($config['js_delay_all_excludes']) ? $config['js_delay_all_excludes'] : [];
      \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: Using js_delay_all_excludes keywords: ' . print_r($keywords, true));
    }
        
    // Get excluded keywords with filter allowing additions
    $exclude_keywords = apply_filters(
      'zizi_cache_exclude_from_delay:js',
      is_array($config['js_delay_excludes']) ? $config['js_delay_excludes'] : []
    );
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: Using exclude keywords: ' . print_r($exclude_keywords, true));

    try {
      $delayedScriptCount = 0;
      $excludedScriptCount = 0;
      $skippedMethodScriptCount = 0;
      $skippedForOtherReasons = 0;
      
      // Loop through all the scripts
      foreach ($scripts[0] as $script_tag) {
        // Verify we have a complete script tag
        if (strpos($script_tag, '</script>') === false && !preg_match('/<script[^>]*\/>/', $script_tag)) {
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: Skipping incomplete script tag: ' . substr($script_tag, 0, 100) . '...');
          $skippedForOtherReasons++;
          continue;
        }

        // First check if script is excluded
        if (SysTool::any_keywords_match_string($exclude_keywords, $script_tag)) {
          $excludedScriptCount++;
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: EXCLUDED script: ' . substr($script_tag, 0, 100) . '...');
          continue;
        }
        
        // Skip scripts that are already delayed
        if (strpos($script_tag, 'data-src=') !== false) {
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: SKIPPED already delayed script: ' . substr($script_tag, 0, 100) . '...');
          $skippedForOtherReasons++;
          continue;
        }
        
        // Skip the core library script
        if (strpos($script_tag, 'id="zizi-core-lib"') !== false) {
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: SKIPPED core lib script: ' . substr($script_tag, 0, 100) . '...');
          $skippedForOtherReasons++;
          continue;
        }
        
        // Then check delay method
        $is_keyword_matched = SysTool::any_keywords_match_string($keywords, $script_tag);
        if (
          ($delay_method === 'selected' && !$is_keyword_matched) ||
          ($delay_method === 'all' && $is_keyword_matched)
        ) {
          $skippedMethodScriptCount++;
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: SKIPPED by method rule script: ' . substr($script_tag, 0, 100) . 
            '... [Method: ' . $delay_method . ', Keyword matched: ' . ($is_keyword_matched ? 'YES' : 'NO') . ']');
          continue;
        }

        // Create HTML object for script tag manipulation
        $script = new HTML($script_tag);

        // Skip Rest API script injected by ZiziCache
        if ($script->id === 'zizi-cache-rest') {
          $skippedForOtherReasons++;
          continue;
        }

        // Skip scripts with non-standard types
        if ($script->type && $script->type !== 'text/javascript' && $script->type !== 'application/javascript') {
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: SKIPPED non-standard type script: ' . substr($script_tag, 0, 100) . '... Type: ' . $script->type);
          $skippedForOtherReasons++;
          continue;
        }

        // Get script source - either from src attribute or convert inline content to data URI
        $script_content = $script->getContent();
        $src = $script->src ?? '';
        
        if (empty($src) && !empty($script_content)) {
          $src = 'data:text/javascript,' . rawurlencode($script_content);
          \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: Created data URI for inline script, length: ' . strlen($src));
        }

        // Skip if no valid source could be determined
        if (empty($src)) {
          \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts: SKIPPED (no valid src) script: ' . substr($script_tag, 0, 100) . '...');
          $skippedForOtherReasons++;
          continue;
        }

        // Construct delayed script - important order: clear content first!
        $script->setContent('');
        
        // Remove src attribute properly before adding data-src
        $original_src = $script->src ?? '';
        if (!empty($original_src)) {
            // Store the old src attribute as data-src
            $script->{'data-src'} = $original_src;
            // Use unset to properly remove the src attribute
            unset($script->src);
        } else if (!empty($script_content)) {
            // For inline scripts with content
            $script->{'data-src'} = $src; // This is the data URI we created earlier
        }
        
        // Verify the data-src attribute is present and src is removed
        $modified_tag = (string)$script;
        
        if (strpos($modified_tag, 'data-src=') === false) {
            \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Delay Scripts ERROR: Failed to add data-src attribute: ' . substr($modified_tag, 0, 100));
            $skippedForOtherReasons++;
            continue;
        }
        
        if (strpos($modified_tag, ' src=') !== false) {
            \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Delay Scripts ERROR: Failed to remove src attribute: ' . substr($modified_tag, 0, 100));
            // Try manual string replacement as a fallback
            $modified_tag = preg_replace('/\ssrc=([\'"])[^\'"]+\\1/i', '', $modified_tag);
        }

        // Replace original tag with delayed version
        $html = str_replace($script_tag, $modified_tag, $html);
        $delayedScriptCount++;
        \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts: DELAYED script: ' . substr($script_tag, 0, 100) . '... -> ' . substr($modified_tag, 0, 100) . '...');
        
        // Verify the replacement was done
        if (strpos($html, $modified_tag) === false) {
            \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache Delay Scripts WARNING: Script tag replacement may have failed. Check HTML output.');
        }
      }
      
      // Log final statistics
      \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Delay Scripts Summary: Total scripts: ' . count($scripts[0]) . 
        ', Delayed: ' . $delayedScriptCount . 
        ', Excluded: ' . $excludedScriptCount . 
        ', Skipped by method rules: ' . $skippedMethodScriptCount . 
        ', Skipped for other reasons: ' . $skippedForOtherReasons);
        
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache delay_scripts error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
      \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache delay_scripts stack trace: ' . $e->getTraceAsString());
    }
    
    return $html;
  }

  /**
   * Inject core JavaScript library
   * 
   * Adds the core JavaScript library required for various
   * optimization features to function properly.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with injected core library
   */
  public static function inject_core_lib(string $html): string
  {
    // Check if js_delay is enabled in configuration - only inject core lib if needed
    $should_inject = false;
    
    // 1. Check if js_delay is enabled
    if (!empty(SysConfig::$config['js_delay']) && SysConfig::$config['js_delay'] === true) {
        $should_inject = true;
        \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Core Lib: Injection needed because js_delay is enabled');
    }
    
    // 2. Check if there are any delayed scripts in the HTML already
    if (strpos($html, 'data-src=') !== false) {
        $should_inject = true;
        \ZiziCache\CacheSys::writeLog('ZiziCache Core Lib: Injection needed because delayed scripts (data-src) found in HTML');
    }
    
    // 3. Check if there are any DOMContentLoaded CSS elements in the HTML
    if (strpos($html, 'data-dcl-href=') !== false || strpos($html, 'zizi-dcl-css') !== false) {
        $should_inject = true;
        \ZiziCache\CacheSys::writeLog('ZiziCache Core Lib: Injection needed because DOMContentLoaded CSS (data-dcl-href) found in HTML');
    }
    
    // Skip injection if not needed
    if (!$should_inject) {
        \ZiziCache\CacheSys::writeLog('ZiziCache Core Lib: Skipping injection - not needed (js_delay not active and no delayed scripts found)');
        return $html;
    }
    
    // Proceed with core lib injection
    $core_js_path = ZIZI_CACHE_PLUGIN_DIR . 'assets/js/core.min.js';
    if (!is_file($core_js_path) || !is_readable($core_js_path)) {
        \ZiziCache\CacheSys::writeLog('ZiziCache Core Lib ERROR: core.min.js not found or not readable at ' . $core_js_path);
        return $html;
    }
    
    // Security: Check file size to prevent memory exhaustion attacks
    $file_size = filesize($core_js_path);
    if ($file_size === false || $file_size > 2 * 1024 * 1024) { // 2MB limit
        \ZiziCache\CacheSys::writeLog("ZiziCache Core Lib ERROR: JS file too large or unreadable: {$core_js_path} ({$file_size} bytes)");
        return $html;
    }
    
    $js_code = file_get_contents($core_js_path);
    if ($js_code === false) {
        \ZiziCache\CacheSys::writeLog('ZiziCache Core Lib ERROR: Failed to read core.min.js');
        return $html;
    }
    
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache Core Lib: Successfully loaded core.min.js (' . strlen($js_code) . ' bytes)');

    // Get timeout from filter and convert it to milliseconds (default to 10s)
    $timeout = apply_filters('zizi_cache_js_delay_timeout', 10); // Default timeout in seconds
    $timeout_ms = (int) ($timeout * 1000); // Convert to milliseconds
    
    // Verify the timeout value is reasonable
    if ($timeout_ms <= 0 || $timeout_ms > 60000) { // Between 0 and 60 seconds
        $timeout_ms = 10000; // Default to 10 seconds if value is unreasonable
        \ZiziCache\CacheSys::writeWarning('Core Lib WARNING: Invalid timeout value, using default 10 seconds. Received: ' . $timeout, 'INFO');
    }

    // Log the timeout being used
    \ZiziCache\CacheSys::writeLog('Core Lib: Setting interaction timeout to ' . $timeout_ms . 'ms', 'INFO');

    // Replace timeout placeholder with actual timeout
    // Ensure we're converting to string and the replacement is actually happening
    $js_code = str_replace('INTERACTION_TIMEOUT', (string)$timeout_ms, $js_code);
    
    // Verify the replacement was done
    if (strpos($js_code, 'INTERACTION_TIMEOUT') !== false) {
        \ZiziCache\CacheSys::writeError('Core Lib ERROR: Failed to replace INTERACTION_TIMEOUT placeholder', 'Security');
    } else {
        \ZiziCache\CacheSys::writeLog('Core Lib: Successfully replaced INTERACTION_TIMEOUT with ' . $timeout_ms . 'ms', 'INFO');
    }

    // create script tag and add append it to the body tag
    // Ensure the script is not processed by other optimizers by adding a specific attribute
    $script_tag = PHP_EOL . "<script id=\"zizi-core-lib\" data-zizi-ignore=\"true\">$js_code</script>" . PHP_EOL;
    
    // Check if the script is already injected
    if (strpos($html, 'id="zizi-core-lib"') !== false) {
        \ZiziCache\CacheSys::writeLog('Core Lib: Script already injected, skipping duplicate injection', 'INFO');
        return $html;
    }
    
    // More robust way to inject before </body>, handles variations in whitespace/case
    $body_end_pattern = '/(<\/body\s*>)/i';
    if (preg_match($body_end_pattern, $html)) {
        $html = preg_replace($body_end_pattern, $script_tag . "$1", $html, 1);
        \ZiziCache\CacheSys::writeLog('Core Lib: Successfully injected before </body> tag', 'INFO');
    } else {
        // Only inject if content appears to be HTML (has HTML tags)
        if (preg_match('/<[^>]+>/', $html)) {
            $html .= $script_tag;
            \ZiziCache\CacheSys::writeWarning('Core Lib WARNING: </body> tag not found for injecting core.min.js. Appended to end of HTML.', 'INFO');
        }
    }
    return $html;
  }

  /**
   * Inject preload library for resource hints
   * 
   * Enqueues the preload library which handles resource hints
   * for optimized loading of critical resources.
   */
  public static function inject_preload_lib(): void
  {
    if (empty(SysConfig::$config['js_preload_links'])) { // Ensure js_preload_links is checked properly
      return;
    }

    // Skip if logged in
    if (is_user_logged_in()) {
      return;
    }

    $preload_js_url = ZIZI_CACHE_PLUGIN_URL . 'assets/js/preload.min.js';

    // Enqueue preload script
    wp_enqueue_script(
      'zizi_cache_preload',
      $preload_js_url,
      [],
      ZIZI_CACHE_VERSION,
      [
        'strategy' => 'defer',
        'in_footer' => true,
      ]
    );
  }

  /**
   * Check if optimization should be skipped
   * 
   * Determines if the current request context should skip optimization
   * (AJAX, admin, REST API, etc.)
   * 
   * @return bool True if optimization should be skipped
   */
  private static function should_skip_optimization(): bool
  {
    // Skip for AJAX requests
    if (wp_doing_ajax()) {
      return true;
    }
    
    // Skip for admin pages
    if (is_admin()) {
      return true;
    }
    
    // Skip for REST API requests
    if (defined('REST_REQUEST') && REST_REQUEST) {
      return true;
    }
    
    // Skip if URL contains wp-json (REST API)
    if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
      return true;
    }
    
    // Skip if Content-Type is JSON
    $headers = headers_list();
    foreach ($headers as $header) {
      if (stripos($header, 'Content-Type:') === 0 && stripos($header, 'application/json') !== false) {
        return true;
      }
    }
    
    return false;
  }

  /**
   * Inject enhanced lazy render library
   * 
   * Adds the enhanced lazy render library which handles hybrid CSS + JavaScript
   * deferred rendering of non-critical elements.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with injected enhanced lazy render library
   */
  public static function inject_lazy_render_lib(string $html): string
  {
    // Skip optimization for AJAX, admin, REST API and non-HTML requests
    if (self::should_skip_optimization()) {
      return $html;
    }
    
    // Check if the content has lazy render element or if lazy render is generally enabled
    $lazy_render_enabled = !empty(SysConfig::$config['js_lazy_render']);
    $has_lazy_elements = strpos($html, 'data-zizi-cache-lazy') !== false || strpos($html, 'zizi-lazy-hybrid') !== false;

    if ($lazy_render_enabled || $has_lazy_elements) {
      $method = SysConfig::$config['js_lazy_render_method'] ?? 'hybrid';
      $root_margin = SysConfig::$config['js_lazy_render_root_margin'] ?? '200px 0px';
      $height_cache = SysConfig::$config['js_lazy_render_height_cache'] ?? true;
      $debug = defined('WP_DEBUG') && WP_DEBUG;
      
      // Use enhanced library for hybrid/css methods, fallback to original for javascript-only
      $library_file = ($method === 'javascript') ? 'lazyrender.min.js' : 'lazyrender-enhanced.min.js';
      $lazy_render_js_url = ZIZI_CACHE_PLUGIN_URL . 'assets/js/' . $library_file;
      
      // Create configuration script
      $config_script = '<script id="zizi-lazy-render-config">
        window.ziziCacheLazyConfig = {
          rootMargin: "' . esc_js($root_margin) . '",
          threshold: 0.01,
          debounceDelay: 50,
          debug: ' . ($debug ? 'true' : 'false') . ',
          disabled: false,
          method: "' . esc_js($method) . '",
          heightCache: ' . ($height_cache ? 'true' : 'false') . ',
          fallbackDelay: 100
        };
      </script>';
      
      // Create a new script tag for the library
      $script_tag_html = new HTML('<script></script>');
      $script_tag_html->id = 'zizi-lazy-render-lib';
      $script_tag_html->src = $lazy_render_js_url;
      $script_tag_html->defer = true;
      $script_tag_html->{'data-zizi-ignore'} = 'true'; // Mark to be ignored by other optimizers

      // Append both config and library scripts to the body tag
      $body_end_pattern = '/(<\/body\s*>)/i';
      if (preg_match($body_end_pattern, $html)) {
          $html = preg_replace($body_end_pattern, $config_script . (string)$script_tag_html . "$1", $html, 1);
      } else {
          // Only inject if content appears to be HTML (has HTML tags)
          if (preg_match('/<[^>]+>/', $html)) {
              $html .= $config_script . (string)$script_tag_html;
              \ZiziCache\CacheSys::writeWarning('</body> tag not found for injecting enhanced lazyrender library. Appended to end of HTML.', 'INFO');
          }
      }
    }
    return $html;
  }

  /**
   * Mark content for lazy rendering
   * 
   * Wraps content in markers for deferred rendering.
   * 
   * @param string $html_content The content to mark for lazy rendering
   * @return string The wrapped content with lazy render markers
   */
  public static function lazy_render(string $html_content): string
  {
    if (empty(SysConfig::$config['js_lazy_render'])) { // Ensure js_lazy_render is checked properly
      return $html_content;
    }

    if (empty($html_content) || !is_string($html_content)) {
      return $html_content;
    }

    return "<!-- begin-zizi-cache-lazy-render -->" . $html_content . "<!-- end-zizi-cache-lazy-render -->";
  }

  /**
   * Apply lazy rendering to elements matching selectors
   * 
   * Processes elements matching configured selectors and applies
   * lazy rendering to improve initial page load performance.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with lazy rendering applied to selectors
   */
  public static function lazy_render_selectors(string $html): string
  {
    // get all lazy render selectors
    $selectors = SysConfig::$config['js_lazy_render_selectors'] ?? [];

    // add lazy render class to all elements
    $selectors[] = '.lazy-render';

    // remove empty selectors
    $selectors = array_filter($selectors);
    if(empty($selectors)){
        return $html;
    }

    try {
      $html_obj = new HTML($html); // Create HTML object once
      foreach ($selectors as $selector) {
        if(empty($selector) || !is_string($selector)) continue;
        // find all elements with the selector
        $elements = $html_obj->getElementsBySelector($selector);

        foreach ($elements as $element) {
          if ($element instanceof \DOMNode) { // Ensure $element is a DOMNode before processing
             $element_html = $html_obj->saveHTML($element); // Get outer HTML of the element
             // add lazy render comment to the element
             $lazy_render_element = self::lazy_render($element_html);
             // Replace the original element HTML with the modified one
             // This replacement needs to be done carefully to avoid issues if $html_obj is modified in loop
             // It might be safer to collect changes and apply them once, or use a method that replaces nodes directly.
             // For now, we assume str_replace is sufficient if elements are unique enough or processed carefully.
             $html = str_replace($element_html, $lazy_render_element, $html);
          } else if (is_string($element)) { // Fallback for string elements, though getElementsBySelector should return DOMNodes
             $lazy_render_element = self::lazy_render($element);
             $html = str_replace($element, $lazy_render_element, $html);
          }
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Enhanced Lazy Load Error: ' . $e->getMessage(), 'Security');
    } finally {
      return is_string($html) ? $html : (string) $html_obj; // Ensure string is returned
    }
  }

  /**
   * Replace lazy render markers with actual markup
   * 
   * Converts lazy render markers into the appropriate HTML structure
   * for client-side processing.
   * 
   * @param string $html The HTML content with lazy render markers
   * @return string The HTML with processed lazy render markup
   */
  public static function replace_lazy_render_markers(string $html): string
  {
    return str_replace(
      ['<!-- begin-zizi-cache-lazy-render -->', '<!-- end-zizi-cache-lazy-render -->'],
      [
        '<div data-zizi-cache-lazy-render style="height:1000px; width:100%;"><noscript>',
        '</noscript></div>',
      ],
      $html
    );
  }

  /**
   * Self-host third-party JavaScript files
   * 
   * Downloads and serves third-party scripts from the local server
   * to improve performance and reliability.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with self-hosted third-party scripts
   */
  public static function self_host_third_party_js(string $html): string
  {
    if (empty(SysConfig::$config['self_host_third_party_css_js'])) { // Ensure self_host_third_party_css_js is checked
      return $html;
    }

    try {
      // Find all the script tags with src attribute
      preg_match_all('/<script[^>]*src=[\'"][^\'"]+[\'"][^>]*><\/script>/i', $html, $scripts);

      foreach ($scripts[0] as $script_tag) {
        $script = new HTML($script_tag);
        $original_src = $script->src;

        if(empty($original_src) || !SysTool::is_external_url($original_src, home_url())) {
            continue; // Skip local or empty srcs
        }

        // Download the external file if allowed
        $url = SysTool::download_external_file($original_src, 'js');

        if (!$url) {
          continue;
        }

        // Remove resource hints
        $html = SysTool::remove_resource_hints($original_src, $html);

        // Save the original src
        $script->{'data-origin-src'} = $original_src;

        // Set the locally hosted file as the new src
        $script->src = $url;

        // Remove integrity and crossorigin attributes if exist
        unset($script->integrity);
        unset($script->crossorigin);

        // Replace the source with the new file
        $html = str_replace($script_tag, (string)$script, $html);
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Self-Host JS Error: ' . $e->getMessage(), 'Security');
    } finally {
      return $html;
    }
  }

  /**
   * Debug method to validate configuration settings
   * Can be called via a hook or admin action to verify settings
   */
  public static function debug_config(): array
  {
    $config = SysConfig::$config;
    
    return [
      'js_defer' => [
        'value' => $config['js_defer'] ?? null,
        'type' => gettype($config['js_defer'] ?? null),
        'active' => !empty($config['js_defer']) && $config['js_defer'] === true,
      ],
      'js_defer_excludes' => [
        'value' => $config['js_defer_excludes'] ?? [],
        'type' => gettype($config['js_defer_excludes'] ?? null),
        'is_array' => is_array($config['js_defer_excludes'] ?? null),
      ],
      'js_delay' => [
        'value' => $config['js_delay'] ?? null,
        'type' => gettype($config['js_delay'] ?? null),
        'active' => !empty($config['js_delay']) && $config['js_delay'] === true,
      ],
      'cache_timestamp' => current_time('timestamp'),
    ];
  }

  /**
   * User Interaction Script Delay - Advanced JavaScript optimization
   * 
   * Delays JavaScript execution until user interaction with network awareness
   * and intelligent fallback strategies. Superior to Flying Press implementation.
   * 
   * @param string $content HTML content
   * @return string Modified HTML content with delayed scripts
   */
  public static function user_interaction_delay(string $content): string
  {
    $config = SysConfig::$config;
    
    // Check if feature is enabled
    if (empty($config['js_user_interaction_delay'])) {
      return $content;
    }
    
    // CRITICAL: Prevent conflict with existing delay_scripts function
    if (!empty($config['js_delay'])) {
      \ZiziCache\CacheSys::writeLog('[WARNING] ZiziCache User Interaction Delay SKIPPED: Basic Delay JavaScript execution is already enabled. Only one delay method can be active at a time.');
      return $content;
    }
    
    // Extract script tags for processing
    preg_match_all('/<script([^>]*)>(.*?)<\/script>/is', $content, $matches, PREG_SET_ORDER);
    
    if (empty($matches)) {
      return $content;
    }
    
    $delayed_scripts = [];
    $method = $config['js_user_interaction_method'] ?? 'smart';
    $selected_scripts = $config['js_user_interaction_selected'] ?? [];
    $exclude_scripts = $config['js_user_interaction_excludes'] ?? [];
    
    foreach ($matches as $match) {
      $full_tag = $match[0];
      $attributes = $match[1] ?? '';
      $script_content = $match[2] ?? '';
      
      // Skip if already has delay attributes or is module
      if (strpos($attributes, 'data-zizi-delay') !== false || 
          strpos($attributes, 'type="module"') !== false ||
          strpos($attributes, 'type=\'module\'') !== false ||
          strpos($attributes, 'data-no-defer') !== false) {
        continue;
      }
      
      // Check excludes - add critical hardcoded excludes for security
      $critical_excludes = ['jquery', 'wp-includes', 'wp-admin', 'core.min.js'];
      $all_excludes = array_merge($critical_excludes, $exclude_scripts);
      
      $should_exclude = false;
      foreach ($all_excludes as $exclude) {
        if (!empty($exclude) && (strpos($full_tag, $exclude) !== false || strpos($script_content, $exclude) !== false)) {
          $should_exclude = true;
          break;
        }
      }
      
      if ($should_exclude) {
        continue;
      }
      
      // Determine if script should be delayed based on method
      $should_delay = false;
      
      switch ($method) {
        case 'all':
          $should_delay = true;
          break;
          
        case 'selected':
          foreach ($selected_scripts as $selected) {
            if (!empty($selected) && (strpos($full_tag, $selected) !== false || strpos($script_content, $selected) !== false)) {
              $should_delay = true;
              break;
            }
          }
          break;
          
        case 'smart':
        default:
          // Smart detection - delay scripts that are likely non-critical
          $smart_patterns = [
            // Analytics & Tracking (Enhanced 2025)
            'google-analytics', 'gtag', 'ga.js', 'analytics.js',
            'google-tag-manager', 'googletagmanager', 'googleoptimize',
            'segment.com', 'mixpanel.com', 'amplitude.com',
            'plausible.io', 'usefathom.com', 'clarity.ms',
            'matomo.js', 'piwik.js',
            
            // Social Media (Enhanced)
            'facebook.net', 'fbevents.js', 'connect.facebook', 'xfbml',
            'twitter.com', 'platform.twitter',
            'instagram.com', 'pinterest.com',
            'linkedin.com/px', 'snap.licdn.com',
            'snapchat.com/p13n', 'reddit.com/gtm',
            'quora.com/qevents',
            
            // Video & Media (Enhanced)
            'youtube.com', 'embed', 'player', 'iframe_api',
            'vimeo.com', 'dailymotion.com', 'player.vimeo',
            
            // Live Chat & Support (Enhanced)
            'disqus', 'livechat', 'intercom', 'livechatinc',
            'zendesk', 'tawk.to', 'crisp.chat', 'client.crisp',
            'olark.com', 'freshchat.com', 'zopim.com',
            'code.tidio.co', 'js.driftt.com', 'widget.manychat',
            
            // Heat Mapping & User Analysis (Enhanced)
            'hotjar', 'mouseflow', 'crazyegg',
            'fullstory', 'logrocket',
            'bugsnag.com', 'rollbar.com', 'sentry.io',
            
            // Social Sharing (Enhanced)
            'addthis', 'sharethis', 'sharebuttons',
            'addthis_widget.js',
            
            // Comments
            'comments', 'comment', 'disqus',
            'disqus.com/embed.js',
            
            // Advertising (Enhanced)
            'ads', 'advertisement', 'doubleclick',
            'googlesyndication', 'googleadservices',
            'adsbygoogle.js', 'adsct.microsoft.com',
            'media.net', 'amazon-adsystem.com',
            'outbrain.com', 'bat.js',
            
            // Marketing & Lead Generation
            'mailchimp.com', 'leadpages.com', 'calendly.com',
            'typeform.com', 'constantcontact.com',
            'trustpilot.com', 'embed.acimage.com',
            
            // Cookie & GDPR Tools
            'cookiebot.com', 'cdn.cookielaw.org',
            'cookie-law-info', 'metomic.io',
            
            // Performance Monitoring
            'cdn.onesignal.com', 'hs-scripts.com',
            'cloudflareinsights.com', 'perimeterx.net',
            
            // User Testing & Analytics Extended
            'luckyorange.com', 'visualwebsiteoptimizer.com',
            'usertesting.com',
            
            // Cookie & GDPR Tools Extended
            'cookieyes.com', 'iubenda.com',
            
            // CDN & Fonts (non-critical)
            'use.fontawesome.com', 'cdnjs.cloudflare.com',
            'unpkg.com', 'cdn.jsdelivr.net',
            
            // General Tracking
            'tracking', 'pixel', 'beacon',
            'grecaptcha.execute',
            
            // WordPress Specific (non-critical)
            'wp-embed', 'jetpack', 'gravatar',
            'woocommerce-analytics', 'wp-statistics',
            
            // E-commerce & Payments (2025)
            'shopify.com', 'stripe.js', 'paypal.com',
            'klarna.com', 'checkout.com', 'square.com',
            'amazon-pay.com', 'apple-pay.js',
            
            // Modern CRM & Customer Support
            'salesforce.com', 'hubspot.com', 'drift.com',
            'helpscout.net', 'pipedrive.com',
            
            // A/B Testing & Personalization
            'optimizely.com', 'google-optimize.com',
            'unbounce.com', 'landingi.com', 'instapage.com',
            
            // Security & Monitoring
            'recaptcha', 'cloudflare.com', 'pingdom.com',
            'datadoghq.com', 'newrelic.com',
            
            // Modern Analytics (2025)
            'posthog.com', 'june.so', 'heap.js',
            'splitbee.io', 'simple-analytics.com'
          ];
          
          foreach ($smart_patterns as $pattern) {
            if (strpos(strtolower($full_tag), $pattern) !== false || 
                strpos(strtolower($script_content), $pattern) !== false) {
              $should_delay = true;
              break;
            }
          }
          break;
      }
      
      if ($should_delay) {
        // Create delayed script version
        $delayed_script = self::create_delayed_script($full_tag, $attributes, $script_content);
        $delayed_scripts[] = [
          'original' => $full_tag,
          'delayed' => $delayed_script
        ];
      }
    }
    
    // Replace original scripts with delayed versions
    foreach ($delayed_scripts as $script) {
      $content = str_replace($script['original'], $script['delayed'], $content);
    }
    
    // Inject interaction handler if we have delayed scripts
    if (!empty($delayed_scripts)) {
      $content = self::inject_user_interaction_handler($content);
    }
    
    return $content;
  }
  
  /**
   * Create delayed script version with security validation
   * 
   * @param string $full_tag Original script tag
   * @param string $attributes Script attributes
   * @param string $script_content Script content
   * @return string Modified script tag
   */
  private static function create_delayed_script(string $full_tag, string $attributes, string $script_content): string
  {
    // Extract src attribute if present
    $src = '';
    if (preg_match('/src=["\']([^"\']+)["\']/', $attributes, $src_match)) {
      $src = $src_match[1];
    }
    
    if (!empty($src)) {
      // External script - validate URL for security
      if (!filter_var($src, FILTER_VALIDATE_URL) && !preg_match('/^\//', $src)) {
        // Invalid URL format, return original tag
        return $full_tag;
      }
      
      $new_attributes = preg_replace('/src=["\'][^"\']*["\']/', '', $attributes);
      $new_attributes .= ' data-zizi-delay-src="' . esc_attr($src) . '"';
      $new_attributes .= ' data-zizi-delay="true"';
      
      return '<script' . $new_attributes . '></script>';
    } else {
      // Inline script - validate content length for security
      $inline_limit = intval(SysConfig::$config['js_user_interaction_inline_limit'] ?? 50000);
      
      // If limit is 0, disable size checking
      if ($inline_limit > 0 && strlen($script_content) > $inline_limit) {
        return $full_tag; // Return original if too large
      }
      
      $encoded_content = base64_encode($script_content);
      // Additional security: check base64 encoding success
      if (base64_decode($encoded_content, true) === false) {
        return $full_tag; // Return original if encoding failed
      }
      
      $new_attributes = $attributes . ' data-zizi-delay-inline="' . esc_attr($encoded_content) . '"';
      $new_attributes .= ' data-zizi-delay="true"';
      
      return '<script' . $new_attributes . '></script>';
    }
  }
  
  /**
   * Inject user interaction handler script
   * 
   * Loads an external JavaScript file that manages delayed script execution 
   * based on user interaction. The script is loaded conditionally only when 
   * user interaction delay is enabled and there are delayed scripts present.
   * 
   * @param string $content HTML content to modify
   * @return string Modified HTML content with injected script
   */
  private static function inject_user_interaction_handler(string $content): string
  {
    $config = SysConfig::$config;
    
    // Check if user interaction delay is enabled
    if (empty($config['js_user_interaction_delay']) || $config['js_user_interaction_delay'] !== true) {
      return $content;
    }
    
    $timeout = max(1000, min(10000, intval($config['js_user_interaction_timeout'] ?? 3000)));
    $debug = defined('WP_DEBUG') && WP_DEBUG;
    
    // Get plugin URL for assets
    $plugin_url = plugin_dir_url(ZIZI_CACHE_FILE);
    
    // Choose minified or development version based on debug settings
    $script_file = $debug ? 'user-interaction-handler.js' : 'user-interaction-handler.min.js';
    $script_url = $plugin_url . 'assets/js/' . $script_file;
    
    // Generate nonce for CSP compatibility
    $nonce = wp_create_nonce('zizi_user_interaction_' . get_current_blog_id());
    
    // Configuration script - inject config before loading the handler
    $config_script = '<script data-no-defer="1" data-zizi-core="config" nonce="' . esc_attr($nonce) . '">
window.ziziUserInteractionConfig = {
  timeout: ' . intval($timeout) . ',
  debug: ' . ($debug ? 'true' : 'false') . '
};
</script>';
    
    // Main handler script - load external file
    $handler_script = '<script data-no-defer="1" data-zizi-core="interaction-handler" src="' . esc_url($script_url) . '" nonce="' . esc_attr($nonce) . '"></script>';
    
    $full_script = $config_script . $handler_script;
    
    // Insert before </head> or at the beginning of <body>
    if (strpos($content, '</head>') !== false) {
      $content = str_replace('</head>', $full_script . '</head>', $content);
    } elseif (strpos($content, '<body') !== false) {
      $content = preg_replace('/(<body[^>]*>)/', '$1' . $full_script, $content, 1);
    }
    
    return $content;
  }

  /**
   * Get intelligent auto-detection selectors for common WordPress elements
   * 
   * Returns CSS selectors for elements that are typically below-the-fold
   * and benefit from lazy rendering optimization.
   * 
   * @return array List of CSS selectors for auto-detection
   */
  public static function get_auto_detect_selectors(): array
  {
    return [
      // WordPress Core Elements
      '#comments',                    // Comments section
      '.comments-area',              // Comment area wrapper
      '.comment-list',               // Individual comments
      '.comment-respond',            // Comment form
      '.wp-block-comments',          // Gutenberg comments block
      
      // Sidebar & Widget Areas
      '.sidebar',                    // Main sidebar
      '.widget-area',               // Widget area containers
      '.footer-sidebar',            // Footer sidebars
      'aside.widget',               // Individual widgets
      '.wp-block-widget-area',      // Gutenberg widget areas
      
      // Footer Content
      'footer',                     // Main footer
      '.site-footer',              // Site footer
      '.footer-content',           // Footer content areas
      '.footer-widgets',           // Footer widget areas
      
      // E-commerce (WooCommerce)
      '.related.products',         // Related products
      '.cross-sells',              // Cross-sell products  
      '.up-sells',                 // Up-sell products
      '.woocommerce-tabs',         // Product tabs
      '.reviews_tab',              // Product reviews
      '.single-product-summary .woocommerce-product-details__short-description', // Product descriptions
      
      // Content Heavy Areas
      '.post-content blockquote',   // Long quotes
      '.wp-block-quote.is-style-large', // Large quote blocks
      '.wp-block-pullquote',        // Pull quotes
      '.author-bio',                // Author biography sections
      '.related-posts',             // Related content
      '.post-navigation',           // Post navigation
      
      // Media & Embeds
      '.wp-block-embed',            // Embedded content
      '.wp-block-media-text',       // Media text blocks (heavy)
      '.wp-block-gallery',          // Image galleries
      '.gallery',                   // Classic galleries
      
      // Third-party Plugins
      '.mailchimp-form',            // Newsletter forms
      '.contact-form',              // Contact forms  
      '.social-share',              // Social sharing widgets
      '.recent-posts-widget',       // Recent posts widgets
      '.tag-cloud',                 // Tag clouds
      
      // Performance Heavy Elements
      '.elementor-widget-container', // Elementor widgets (can be heavy)
      '.vc_row',                    // Visual Composer rows
      '.fusion-builder-row',        // Fusion Builder content
    ];
  }

  /**
   * Enhanced lazy render with hybrid CSS + JavaScript approach
   * 
   * Implements intelligent lazy rendering using modern CSS content-visibility
   * with JavaScript fallback for maximum compatibility and performance.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with enhanced lazy rendering applied
   */
  public static function enhanced_lazy_render(string $html): string
  {
    if (empty(SysConfig::$config['js_lazy_render'])) {
      return $html;
    }

    $method = SysConfig::$config['js_lazy_render_method'] ?? 'hybrid';
    $auto_detect = SysConfig::$config['js_lazy_render_auto_detect'] ?? true;
    $custom_selectors = SysConfig::$config['js_lazy_render_selectors'] ?? [];
    $height_cache = SysConfig::$config['js_lazy_render_height_cache'] ?? true;
    $root_margin = SysConfig::$config['js_lazy_render_root_margin'] ?? '200px 0px';

    // Build complete selector list
    $selectors = [];
    
    if ($auto_detect) {
      $selectors = array_merge($selectors, self::get_auto_detect_selectors());
    }
    
    if (!empty($custom_selectors)) {
      $selectors = array_merge($selectors, $custom_selectors);
    }

    // Always include manual lazy render class
    $selectors[] = '.lazy-render';

    // Remove duplicates and empty selectors
    $selectors = array_unique(array_filter($selectors));
    
    if (empty($selectors)) {
      return $html;
    }

    try {
      $html_obj = new HTML($html);
      $processed_elements = [];

      foreach ($selectors as $selector) {
        if (empty($selector) || !is_string($selector)) continue;
        
        $elements = $html_obj->getElementsBySelector($selector);
        
        foreach ($elements as $element) {
          if (!($element instanceof \DOMNode)) continue;
          
          // Skip if already processed
          $element_id = spl_object_id($element);
          if (isset($processed_elements[$element_id])) continue;
          
          $element_html = $html_obj->saveHTML($element);
          
          // Apply appropriate lazy render method
          switch ($method) {
            case 'css':
              $processed_html = self::apply_css_lazy_render($element, $element_html, $height_cache);
              break;
              
            case 'javascript':
              $processed_html = self::apply_javascript_lazy_render($element_html, $root_margin);
              break;
              
            case 'hybrid':
            default:
              $processed_html = self::apply_hybrid_lazy_render($element, $element_html, $height_cache, $root_margin);
              break;
          }
          
          if ($processed_html !== $element_html) {
            $html = str_replace($element_html, $processed_html, $html);
            $processed_elements[$element_id] = true;
          }
        }
      }
      
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Enhanced Lazy Render Error: ' . $e->getMessage(), 'Security');
    }

    return $html;
  }

  /**
   * Apply CSS-only lazy rendering using content-visibility
   * 
   * Uses modern CSS content-visibility and contain-intrinsic-size
   * for native browser optimization (modern browsers only).
   * 
   * @param \DOMNode $element The DOM element to process
   * @param string $element_html The element's HTML content
   * @param bool $height_cache Whether to cache element heights
   * @return string The processed HTML with CSS lazy rendering
   */
  public static function apply_css_lazy_render(\DOMNode $element, string $element_html, bool $height_cache = true): string
  {
    // Extract or estimate element height for contain-intrinsic-size
    $height = self::estimate_element_height($element, $height_cache);
    
    // Add CSS content-visibility attributes
    $element_obj = new HTML($element_html);
    
    if ($element_obj->tag) {
      // Generate unique ID for height caching if needed
      $element_id = $element_obj->id ?: 'zizi-lazy-' . uniqid();
      if (!$element_obj->id) {
        $element_obj->id = $element_id;
      }
      
      // Add CSS properties for lazy rendering
      $style = $element_obj->style ?? '';
      $style .= '; content-visibility: auto';
      $style .= '; contain-intrinsic-size: auto ' . $height . 'px';
      $element_obj->style = ltrim($style, '; ');
      
      // Add data attribute for height caching
      if ($height_cache) {
        $element_obj->{'data-zizi-height'} = $height;
      }
      
      return (string) $element_obj;
    }
    
    return $element_html;
  }

  /**
   * Apply JavaScript-only lazy rendering using IntersectionObserver
   * 
   * Traditional approach using JavaScript for universal browser compatibility.
   * 
   * @param string $element_html The element's HTML content
   * @param string $root_margin Viewport offset for triggering
   * @return string The processed HTML with JavaScript lazy rendering markers
   */
  public static function apply_javascript_lazy_render(string $element_html, string $root_margin = '200px 0px'): string
  {
    // Use existing lazy render system with enhanced configuration
    $lazy_content = "<!-- begin-zizi-cache-lazy-render -->" . $element_html . "<!-- end-zizi-cache-lazy-render -->";
    
    // Convert to noscript wrapper format
    return str_replace(
      ['<!-- begin-zizi-cache-lazy-render -->', '<!-- end-zizi-cache-lazy-render -->'],
      [
        '<div data-zizi-cache-lazy-render data-root-margin="' . esc_attr($root_margin) . '"><noscript>',
        '</noscript></div>',
      ],
      $lazy_content
    );
  }

  /**
   * Apply hybrid lazy rendering (CSS + JavaScript fallback)
   * 
   * Uses CSS content-visibility for modern browsers with JavaScript fallback
   * for older browsers. Best of both worlds approach.
   * 
   * @param \DOMNode $element The DOM element to process  
   * @param string $element_html The element's HTML content
   * @param bool $height_cache Whether to cache element heights
   * @param string $root_margin Viewport offset for JavaScript fallback
   * @return string The processed HTML with hybrid lazy rendering
   */
  public static function apply_hybrid_lazy_render(\DOMNode $element, string $element_html, bool $height_cache = true, string $root_margin = '200px 0px'): string
  {
    // First apply CSS content-visibility for modern browsers
    $css_processed = self::apply_css_lazy_render($element, $element_html, $height_cache);
    
    // Add JavaScript fallback attributes for older browsers
    $hybrid_obj = new HTML($css_processed);
    
    if ($hybrid_obj->tag) {
      // Add JavaScript fallback attributes
      $hybrid_obj->{'data-zizi-lazy-fallback'} = 'true';
      $hybrid_obj->{'data-root-margin'} = $root_margin;
      
      // Add CSS feature detection class
      $classes = $hybrid_obj->class ?? '';
      $classes .= ' zizi-lazy-hybrid';
      $hybrid_obj->class = trim($classes);
      
      return (string) $hybrid_obj;
    }
    
    return $css_processed;
  }

  /**
   * Estimate element height for contain-intrinsic-size
   * 
   * Provides height estimation for elements to maintain layout stability
   * during lazy rendering with CSS content-visibility.
   * 
   * @param \DOMNode $element The DOM element to analyze
   * @param bool $use_cache Whether to use cached height values
   * @return int Estimated height in pixels
   */
  public static function estimate_element_height(\DOMNode $element, bool $use_cache = true): int
  {
    // Default heights based on element type and common patterns
    $height_estimates = [
      'header' => 80,
      'nav' => 60,
      'article' => 400,
      'section' => 300,
      'aside' => 250,
      'footer' => 200,
      'div' => 150,
      'blockquote' => 100,
      'figure' => 200,
      'table' => 300,
      'form' => 200,
      'iframe' => 315, // Common video aspect ratio
    ];
    
    $tag_name = strtolower($element->nodeName ?? 'div');
    $base_height = $height_estimates[$tag_name] ?? 150;
    
    // Check for cached height in data attribute if using cache
    if ($use_cache && $element instanceof \DOMElement) {
      $cached_height = $element->getAttribute('data-zizi-height');
      if (!empty($cached_height) && is_numeric($cached_height)) {
        return (int) $cached_height;
      }
    }
    
    // Adjust height based on element characteristics
    if ($element instanceof \DOMElement) {
      $class_list = $element->getAttribute('class') ?? '';
      $id = $element->getAttribute('id') ?? '';
      
      // Specific adjustments based on common patterns
      if (strpos($class_list, 'comments') !== false || strpos($id, 'comments') !== false) {
        $base_height = 400; // Comments sections tend to be tall
      } elseif (strpos($class_list, 'footer') !== false || strpos($id, 'footer') !== false) {
        $base_height = 250; // Footer sections
      } elseif (strpos($class_list, 'sidebar') !== false || strpos($class_list, 'widget') !== false) {
        $base_height = 300; // Sidebar/widget areas
      } elseif (strpos($class_list, 'gallery') !== false) {
        $base_height = 400; // Image galleries
      } elseif (strpos($class_list, 'embed') !== false) {
        $base_height = 315; // Embedded content (videos)
      }
      
      // Count child elements for rough size estimation
      if ($element->hasChildNodes()) {
        $child_count = $element->childNodes->length;
        $base_height += min($child_count * 10, 200); // Add up to 200px based on children
      }
    }
    
    return max($base_height, 50); // Minimum 50px height
  }
}