<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Image Thumbnail Manager - Smart thumbnail management for ZiziCache
 * 
 * Provides intelligent thumbnail size management with prevention and cleanup
 * capabilities. Integrates seamlessly with existing ImageConverter functionality.
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.4
 */
class ImageThumbnailManager
{
    /**
     * WordPress default thumbnail sizes
     */
    private const WP_DEFAULT_SIZES = [
        'thumbnail' => 'Thumbnail (150x150)',
        'medium' => 'Medium (300x300)', 
        'medium_large' => 'Medium Large (768x0)',
        'large' => 'Large (1024x1024)',
        '1536x1536' => 'Extra Large (1536x1536)',
        '2048x2048' => 'XXL (2048x2048)'
    ];

    /**
     * Initialize Thumbnail Manager
     */
    public static function init(): void
    {
        // Register WordPress filters for thumbnail prevention with high priority
        add_filter('intermediate_image_sizes_advanced', [__CLASS__, 'filter_thumbnail_sizes'], 9999);
        
        // Also hook into earlier filter to catch edge cases
        add_filter('intermediate_image_sizes', [__CLASS__, 'filter_basic_sizes'], 9999);
        
        // Register additional WordPress filters based on settings
        self::register_additional_filters();
        
        // Register settings hooks
        add_action('init', [__CLASS__, 'register_settings']);
        
        // Log initialization
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('ImageThumbnailManager initialized with high priority filters', 'ThumbnailManager');
        }
    }

    /**
     * Register additional WordPress filters based on settings
     */
    private static function register_additional_filters(): void
    {
        $config = SysConfig::$config;
        
        // Disable big image size threshold if enabled
        if ($config['thumbnail_disable_big_image_threshold'] ?? false) {
            add_filter('big_image_size_threshold', '__return_false');
        }
        
        // Disable EXIF auto-rotation if enabled  
        if ($config['thumbnail_disable_exif_rotate'] ?? false) {
            add_filter('wp_image_maybe_exif_rotate', '__return_false');
        }
    }

    /**
     * Get ALL registered thumbnail sizes (WordPress + theme + plugins)
     * 
     * @return array Complete list of thumbnail sizes with metadata
     */
    public static function get_all_thumbnail_sizes(): array
    {
        global $_wp_additional_image_sizes;
        
        $sizes = [];
        
        // WordPress default sizes
        foreach (self::WP_DEFAULT_SIZES as $size => $label) {
            $width = get_option($size . '_size_w', 0);
            $height = get_option($size . '_size_h', 0);
            $crop = get_option($size . '_crop', 0);
            
            $sizes[$size] = [
                'label' => $label,
                'width' => (int)$width,
                'height' => (int)$height,
                'crop' => (bool)$crop,
                'source' => 'wordpress',
                'active' => true
            ];
        }
        
        // Theme and plugin registered sizes
        if (!empty($_wp_additional_image_sizes)) {
            foreach ($_wp_additional_image_sizes as $size => $data) {
                $source = self::detect_size_source($size);
                
                $sizes[$size] = [
                    'label' => self::generate_size_label($size, $data),
                    'width' => (int)$data['width'],
                    'height' => (int)$data['height'], 
                    'crop' => (bool)$data['crop'],
                    'source' => $source,
                    'active' => true
                ];
            }
        }
        
        // Add disabled status from settings
        $disabled_sizes = self::get_disabled_sizes();
        foreach ($disabled_sizes as $disabled_size) {
            if (isset($sizes[$disabled_size])) {
                $sizes[$disabled_size]['active'] = false;
            }
        }
        
        return $sizes;
    }

    /**
     * Detect source of thumbnail size (theme/plugin name)
     * 
     * @param string $size_name Thumbnail size name
     * @return string Source identification
     */
    private static function detect_size_source(string $size_name): string
    {
        // Common theme prefixes
        $theme_patterns = [
            'twentytwentyfive' => 'Twenty Twenty-Five Theme',
            'twentytwentyfour' => 'Twenty Twenty-Four Theme', 
            'astra' => 'Astra Theme',
            'oceanwp' => 'OceanWP Theme',
            'generatepress' => 'GeneratePress Theme',
            'divi' => 'Divi Theme',
            'avada' => 'Avada Theme'
        ];
        
        // Common plugin prefixes
        $plugin_patterns = [
            'woocommerce' => 'WooCommerce Plugin',
            'elementor' => 'Elementor Plugin',
            'wp_portfolio' => 'Portfolio Plugin',
            'gallery' => 'Gallery Plugin',
            'slider' => 'Slider Plugin'
        ];
        
        // Check theme patterns
        foreach ($theme_patterns as $pattern => $name) {
            if (strpos($size_name, $pattern) !== false) {
                return $name;
            }
        }
        
        // Check plugin patterns
        foreach ($plugin_patterns as $pattern => $name) {
            if (strpos($size_name, $pattern) !== false) {
                return $name;
            }
        }
        
        // Try to detect from active theme
        $current_theme = wp_get_theme();
        $theme_name = strtolower($current_theme->get('Name'));
        if (strpos($size_name, $theme_name) !== false) {
            return $current_theme->get('Name') . ' Theme';
        }
        
        return 'Unknown Plugin/Theme';
    }

    /**
     * Generate user-friendly label for thumbnail size
     * 
     * @param string $size_name Size name
     * @param array $data Size data
     * @return string Human-readable label
     */
    private static function generate_size_label(string $size_name, array $data): string
    {
        $width = $data['width'] ?? 0;
        $height = $data['height'] ?? 0;
        $crop = $data['crop'] ?? false;
        
        // Clean up size name for display
        $clean_name = ucwords(str_replace(['_', '-'], ' ', $size_name));
        
        // Add dimensions
        $dimensions = $crop ? "{$width}x{$height} (cropped)" : "{$width}x{$height}";
        
        return "{$clean_name} ({$dimensions})";
    }

    /**
     * Get currently disabled thumbnail sizes
     * 
     * @return array List of disabled size names
     */
    public static function get_disabled_sizes(): array
    {
        $config = SysConfig::$config;
        return $config['thumbnail_disabled_sizes'] ?? [];
    }

    /**
     * Update disabled thumbnail sizes
     * 
     * @param array $disabled_sizes List of size names to disable
     * @return bool Success status
     */
    public static function update_disabled_sizes(array $disabled_sizes): bool
    {
        try {
            $config = SysConfig::$config;
            $config['thumbnail_disabled_sizes'] = array_values(array_unique($disabled_sizes));
            
            $result = SysConfig::updateConfig($config);
            
            if ($result && class_exists('\\ZiziCache\\CacheSys')) {
                $count = count($disabled_sizes);
                \ZiziCache\CacheSys::writeLog("Updated disabled thumbnail sizes: {$count} sizes disabled", 'ThumbnailManager');
            }
            
            return $result;
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Failed to update disabled sizes: " . $e->getMessage(), 'ThumbnailManager');
            }
            return false;
        }
    }

    /**
     * WordPress filter to remove disabled thumbnail sizes during generation
     * 
     * @param array $sizes Image sizes array from WordPress
     * @return array Filtered sizes array
     */
    public static function filter_thumbnail_sizes(array $sizes): array
    {
        // Check if thumbnail prevention is enabled
        $config = SysConfig::$config;
        if (!($config['thumbnail_prevention_enabled'] ?? true)) {
            return $sizes; // Prevention disabled, allow all sizes
        }
        
        $disabled_sizes = self::get_disabled_sizes();
        
        foreach ($disabled_sizes as $disabled_size) {
            unset($sizes[$disabled_size]);
        }

        // Debug logging if enabled
        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
            $removed_count = count($disabled_sizes);
            $remaining_count = count($sizes);
            \ZiziCache\CacheSys::writeLog(
                "Thumbnail filter applied: removed {$removed_count} disabled sizes, {$remaining_count} remaining",
                'ThumbnailManager'
            );
        }

        return $sizes;
    }

    /**
     * Filter basic intermediate image sizes (earlier hook)
     * 
     * @param array $sizes Basic image size names
     * @return array Filtered sizes array
     */
    public static function filter_basic_sizes(array $sizes): array
    {
        // Check if thumbnail prevention is enabled
        $config = SysConfig::$config;
        if (!($config['thumbnail_prevention_enabled'] ?? true)) {
            return $sizes; // Prevention disabled, allow all sizes
        }
        
        $disabled_sizes = self::get_disabled_sizes();
        
        // Remove disabled sizes from basic sizes array
        $filtered_sizes = array_diff($sizes, $disabled_sizes);
        
        // Debug logging if enabled
        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
            $removed_count = count($sizes) - count($filtered_sizes);
            \ZiziCache\CacheSys::writeLog(
                "Basic thumbnail filter applied: removed {$removed_count} disabled sizes from basic sizes",
                'ThumbnailManager'
            );
        }

        return array_values($filtered_sizes);
    }

    /**
     * Analyze existing thumbnails usage across site content
     * 
     * @param int $batch_size Number of attachments to process per batch
     * @return array Usage analysis results
     */
    public static function analyze_thumbnail_usage(int $batch_size = 50, array $options = []): array
    {
        $results = [
            'total_attachments' => 0,
            'total_thumbnails' => 0,
            'used_thumbnails' => 0,      // Active thumbnail sizes (enabled by user)
            'unused_thumbnails' => 0,    // Inactive thumbnail sizes (disabled by user)
            'size_usage' => [],
            'potential_savings' => 0,    // Savings from deleting inactive size thumbnails
            'recommended_disabled' => []
        ];
        
        // Get user's thumbnail size settings - which sizes are active/inactive
        $all_sizes = self::get_all_thumbnail_sizes();
        $inactive_sizes = array_keys(array_filter($all_sizes, function($size) {
            return !$size['active'];  // Sizes disabled by user in Available Thumbnail Sizes
        }));
        
        // Get all image attachments
        $attachments = get_posts([
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'numberposts' => $batch_size,
            'post_status' => 'inherit'
        ]);
        
        $results['total_attachments'] = count($attachments);
        
        foreach ($attachments as $attachment) {
            $attachment_id = $attachment->ID;
            $metadata = wp_get_attachment_metadata($attachment_id);
            
            if (!$metadata || empty($metadata['sizes'])) {
                continue;
            }
            
            foreach ($metadata['sizes'] as $size_name => $size_data) {
                $results['total_thumbnails']++;
                
                // NEW LOGIC: Classification based on user's Available Thumbnail Sizes settings
                $is_inactive_size = in_array($size_name, $inactive_sizes);
                
                if ($is_inactive_size) {
                    // This thumbnail size is DISABLED by user - count as unused
                    $results['unused_thumbnails']++;
                    
                    // Calculate potential savings from deleting inactive size thumbnails
                    $file_path = self::get_thumbnail_file_path($attachment_id, $size_name);
                    if ($file_path && file_exists($file_path)) {
                        $results['potential_savings'] += filesize($file_path);
                    }
                } else {
                    // This thumbnail size is ENABLED by user - count as used
                    $results['used_thumbnails']++;
                }
                
                // Track usage by size with activation status
                if (!isset($results['size_usage'][$size_name])) {
                    $results['size_usage'][$size_name] = [
                        'used' => 0, 
                        'unused' => 0,
                        'is_active' => !$is_inactive_size,
                        'total_files' => 0,
                        'total_size' => 0
                    ];
                }
                
                // Count files and calculate size for this thumbnail size
                $file_path = self::get_thumbnail_file_path($attachment_id, $size_name);
                if ($file_path && file_exists($file_path)) {
                    $results['size_usage'][$size_name]['total_files']++;
                    $results['size_usage'][$size_name]['total_size'] += filesize($file_path);
                }
                
                // Increment counters based on user settings
                $results['size_usage'][$size_name][$is_inactive_size ? 'unused' : 'used']++;
            }
        }
        
        // Add size information for each thumbnail size
        foreach ($results['size_usage'] as $size_name => &$usage) {
            $usage['size_info'] = $all_sizes[$size_name] ?? null;
            $usage['unused_size'] = $usage['is_active'] ? 0 : $usage['total_size'];
        }
        
        // Recommended disabled are already inactive sizes with files
        $results['recommended_disabled'] = array_keys(array_filter($results['size_usage'], function($usage) {
            return !$usage['is_active'] && $usage['total_files'] > 0;
        }));
        
        return $results;
    }

    /**
     * Scan all site content for image URL references
     * 
     * @return array List of image URLs found in content
     */
    private static function scan_content_for_used_images(): array
    {
        $used_images = [];
        
        // Scan post content
        $posts = get_posts([
            'numberposts' => -1,
            'post_type' => ['post', 'page'],
            'post_status' => ['publish', 'draft']
        ]);
        
        foreach ($posts as $post) {
            $content = $post->post_content;
            $found_images = self::extract_image_urls_from_content($content);
            $used_images = array_merge($used_images, $found_images);
        }
        
        // Scan post meta (page builders, widgets, etc.)
        $meta_images = self::scan_post_meta_for_images();
        $used_images = array_merge($used_images, $meta_images);
        
        // Scan theme customizer settings
        $theme_images = self::scan_theme_settings_for_images();
        $used_images = array_merge($used_images, $theme_images);
        
        return array_unique($used_images);
    }

    /**
     * Extract image URLs from content using regex patterns
     * 
     * @param string $content Content to scan
     * @return array Found image URLs
     */
    private static function extract_image_urls_from_content(string $content): array
    {
        $images = [];
        
        // Pattern for img src attributes
        if (preg_match_all('/src=["\']([^"\']*wp-content\/uploads[^"\']*)["\']/', $content, $matches)) {
            $images = array_merge($images, $matches[1]);
        }
        
        // Pattern for CSS background-image URLs
        if (preg_match_all('/background-image:\s*url\(["\']?([^"\']*wp-content\/uploads[^"\']*)["\']?\)/', $content, $matches)) {
            $images = array_merge($images, $matches[1]);
        }
        
        // Pattern for inline CSS url() functions
        if (preg_match_all('/url\(["\']?([^"\']*wp-content\/uploads[^"\']*)["\']?\)/', $content, $matches)) {
            $images = array_merge($images, $matches[1]);
        }
        
        return $images;
    }

    /**
     * Scan post meta fields for image references
     * 
     * @return array Found image URLs
     */
    private static function scan_post_meta_for_images(): array
    {
        $images = [];
        
        // This will catch page builder content stored in meta fields
        global $wpdb;
        $meta_values = $wpdb->get_col(
            "SELECT meta_value FROM {$wpdb->postmeta} 
             WHERE meta_value LIKE '%wp-content/uploads%'"
        );
        
        foreach ($meta_values as $meta_value) {
            $found_images = self::extract_image_urls_from_content($meta_value);
            $images = array_merge($images, $found_images);
        }
        
        return $images;
    }

    /**
     * Scan theme customizer and theme mod settings for image references
     * 
     * @return array Found image URLs
     */
    private static function scan_theme_settings_for_images(): array
    {
        $images = [];
        
        // Scan theme mods
        $theme_mods = get_theme_mods();
        if ($theme_mods) {
            $theme_mods_json = json_encode($theme_mods);
            $found_images = self::extract_image_urls_from_content($theme_mods_json);
            $images = array_merge($images, $found_images);
        }
        
        // Scan customizer settings
        $customizer_settings = get_option('theme_mods_' . get_option('stylesheet'));
        if ($customizer_settings) {
            $settings_json = json_encode($customizer_settings);
            $found_images = self::extract_image_urls_from_content($settings_json);
            $images = array_merge($images, $found_images);
        }
        
        return $images;
    }

    /**
     * Check if specific image URL is used in content
     * 
     * @param string $image_url Image URL to check
     * @param array $used_images List of used image URLs
     * @return bool True if image is used
     */
    private static function is_image_used_in_content(string $image_url, array $used_images): bool
    {
        $image_basename = basename($image_url);
        
        foreach ($used_images as $used_url) {
            if (strpos($used_url, $image_basename) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Get file path for specific thumbnail size
     * 
     * @param int $attachment_id Attachment ID
     * @param string $size_name Thumbnail size name
     * @return string|null File path or null if not found
     */
    private static function get_thumbnail_file_path(int $attachment_id, string $size_name): ?string
    {
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        if (!$metadata || empty($metadata['sizes'][$size_name])) {
            return null;
        }
        
        $upload_dir = wp_upload_dir();
        $file_dir = dirname($metadata['file']);
        $filename = $metadata['sizes'][$size_name]['file'];
        
        return $upload_dir['basedir'] . '/' . $file_dir . '/' . $filename;
    }

    /**
     * Register settings for thumbnail management
     */
    public static function register_settings(): void
    {
        // Settings are handled through SysConfig, no additional registration needed
        // This method is kept for future extensibility
    }

    /**
     * Get statistics for thumbnail management dashboard
     * 
     * @return array Statistics data
     */
    public static function get_statistics(): array
    {
        $all_sizes = self::get_all_thumbnail_sizes();
        $disabled_count = count(array_filter($all_sizes, function($size) {
            return !$size['active'];
        }));
        
        return [
            'total_sizes' => count($all_sizes),
            'active_sizes' => count($all_sizes) - $disabled_count,
            'disabled_sizes' => $disabled_count,
            'wordpress_sizes' => count(array_filter($all_sizes, function($size) {
                return $size['source'] === 'wordpress';
            })),
            'theme_plugin_sizes' => count($all_sizes) - count(array_filter($all_sizes, function($size) {
                return $size['source'] === 'wordpress';
            }))
        ];
    }

    /**
     * Get thumbnail settings for API/frontend
     * 
     * @return array Thumbnail settings
     */
    public static function get_thumbnail_settings(): array
    {
        $config = SysConfig::$config;
        
        return [
            'prevention_enabled' => $config['thumbnail_prevention_enabled'] ?? true,
            'disabled_sizes' => $config['thumbnail_disabled_sizes'] ?? [],
            'disable_big_image_threshold' => $config['thumbnail_disable_big_image_threshold'] ?? false,
            'disable_exif_rotate' => $config['thumbnail_disable_exif_rotate'] ?? false
        ];
    }

    /**
     * Update thumbnail settings from API/frontend
     * 
     * @param array $settings New settings
     * @return bool Success status
     */
    public static function update_thumbnail_settings(array $settings): bool
    {
        try {
            $config = SysConfig::$config;
            
            // Update individual settings
            if (isset($settings['prevention_enabled'])) {
                $config['thumbnail_prevention_enabled'] = (bool)$settings['prevention_enabled'];
            }
            
            if (isset($settings['disabled_sizes'])) {
                $config['thumbnail_disabled_sizes'] = array_values(array_unique((array)$settings['disabled_sizes']));
            }
            
            if (isset($settings['disable_big_image_threshold'])) {
                $config['thumbnail_disable_big_image_threshold'] = (bool)$settings['disable_big_image_threshold'];
            }
            
            if (isset($settings['disable_exif_rotate'])) {
                $config['thumbnail_disable_exif_rotate'] = (bool)$settings['disable_exif_rotate'];
            }
            
            $result = SysConfig::updateConfig($config);
            
            if ($result && class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog("Updated thumbnail settings", 'ThumbnailManager');
                
                // Re-register filters with new settings
                self::unregister_additional_filters();
                self::register_additional_filters();
            }
            
            return $result;
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Failed to update thumbnail settings: " . $e->getMessage(), 'ThumbnailManager');
            }
            return false;
        }
    }

    /**
     * Unregister additional filters (for re-registration with new settings)
     */
    private static function unregister_additional_filters(): void
    {
        remove_filter('big_image_size_threshold', '__return_false');
        remove_filter('wp_image_maybe_exif_rotate', '__return_false');
    }
}
