<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Image Thumbnail API - REST API endpoints for thumbnail management
 * 
 * Provides RESTful interface for thumbnail size management, analysis,
 * and cleanup operations. Integrates with ImageThumbnailManager.
 * 
 * @package ZiziCache  
 * @since 0.4.8-beta.4
 */
class ImageThumbnailAPI
{
    /**
     * Initialize API endpoints
     */
    public static function init(): void
    {
        add_action('rest_api_init', [__CLASS__, 'register_endpoints']);
    }

    /**
     * Register REST API endpoints
     */
    public static function register_endpoints(): void
    {
        // Get all thumbnail sizes
        register_rest_route('zizi-cache/v1', '/thumbnail/sizes', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_thumbnail_sizes'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        // Update disabled sizes
        register_rest_route('zizi-cache/v1', '/thumbnail/sizes', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'update_disabled_sizes'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'disabled_sizes' => [
                    'required' => true,
                    'type' => 'array',
                    'items' => ['type' => 'string']
                ]
            ]
        ]);

        // Get thumbnail settings (prevention_enabled, etc.)
        register_rest_route('zizi-cache/v1', '/thumbnail/settings', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_thumbnail_settings'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        // Update thumbnail settings (prevention_enabled, etc.)
        register_rest_route('zizi-cache/v1', '/thumbnail/settings', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'update_thumbnail_settings'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'prevention_enabled' => [
                    'type' => 'boolean'
                ],
                'disabled_sizes' => [
                    'type' => 'array',
                    'items' => ['type' => 'string']
                ],
                'disable_big_image_threshold' => [
                    'type' => 'boolean'
                ],
                'disable_exif_rotate' => [
                    'type' => 'boolean'
                ]
            ]
        ]);

        // Analyze thumbnail usage
        register_rest_route('zizi-cache/v1', '/thumbnail/analyze', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'analyze_usage'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'batch_size' => [
                    'type' => 'integer',
                    'default' => 50,
                    'minimum' => 10,
                    'maximum' => 100
                ]
            ]
        ]);

        // Get thumbnail statistics
        register_rest_route('zizi-cache/v1', '/thumbnail/stats', [
            'methods' => 'GET', 
            'callback' => [__CLASS__, 'get_statistics'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        // Cleanup unused thumbnails (dry run)
        register_rest_route('zizi-cache/v1', '/thumbnail/cleanup/preview', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'preview_cleanup'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'sizes' => [
                    'type' => 'array',
                    'items' => ['type' => 'string']
                ]
            ]
        ]);

        // Execute thumbnail cleanup
        register_rest_route('zizi-cache/v1', '/thumbnail/cleanup/execute', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'execute_cleanup'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'sizes' => [
                    'required' => true,
                    'type' => 'array',
                    'items' => ['type' => 'string']
                ],
                'confirm' => [
                    'required' => true,
                    'type' => 'boolean'
                ]
            ]
        ]);

        // WooCommerce specific endpoints
        register_rest_route('zizi-cache/v1', '/thumbnail/woocommerce/analyze', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'analyze_woocommerce_usage'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'batch_size' => [
                    'type' => 'integer',
                    'default' => 100
                ]
            ]
        ]);

        register_rest_route('zizi-cache/v1', '/thumbnail/woocommerce/orphaned', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_orphaned_wc_images'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        register_rest_route('zizi-cache/v1', '/thumbnail/woocommerce/cleanup', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'cleanup_orphaned_wc_images'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'confirm' => [
                    'required' => true,
                    'type' => 'boolean'
                ]
            ]
        ]);

        // Page Builder endpoints
        register_rest_route('zizi-cache/v1', '/thumbnail/builders/detect', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'detect_page_builders'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);

        register_rest_route('zizi-cache/v1', '/thumbnail/builders/analyze', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'analyze_builder_usage'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'batch_size' => [
                    'type' => 'integer',
                    'default' => 50
                ]
            ]
        ]);

        // Advanced analysis endpoint
        register_rest_route('zizi-cache/v1', '/thumbnail/advanced/analyze', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'advanced_usage_analysis'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'include_content' => [
                    'type' => 'boolean',
                    'default' => true
                ],
                'include_meta' => [
                    'type' => 'boolean',
                    'default' => true
                ],
                'include_builders' => [
                    'type' => 'boolean',
                    'default' => true
                ],
                'include_woocommerce' => [
                    'type' => 'boolean',
                    'default' => true
                ],
                'batch_size' => [
                    'type' => 'integer',
                    'default' => 100
                ]
            ]
        ]);

        // Quick stats for dashboard
        register_rest_route('zizi-cache/v1', '/thumbnail/quick-stats', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_quick_stats'],
            'permission_callback' => [__CLASS__, 'check_permissions']
        ]);
    }

    /**
     * Check user permissions for API access
     * 
     * @return bool Permission status
     */
    public static function check_permissions(): bool
    {
        return current_user_can('manage_options');
    }

    /**
     * Get all thumbnail sizes with metadata
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function get_thumbnail_sizes(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $sizes = ImageThumbnailManager::get_all_thumbnail_sizes();
            $statistics = ImageThumbnailManager::get_statistics();
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => [
                    'sizes' => $sizes,
                    'statistics' => $statistics,
                    'total_count' => count($sizes)
                ]
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to retrieve thumbnail sizes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update disabled thumbnail sizes
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function update_disabled_sizes(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $disabled_sizes = $request->get_param('disabled_sizes');
            
            if (!is_array($disabled_sizes)) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Invalid disabled_sizes parameter'
                ], 400);
            }
            
            $result = ImageThumbnailManager::update_disabled_sizes($disabled_sizes);
            
            if (!$result) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Failed to update disabled sizes'
                ], 500);
            }
            
            // Get updated sizes for response
            $updated_sizes = ImageThumbnailManager::get_all_thumbnail_sizes();
            $statistics = ImageThumbnailManager::get_statistics();
            
            return new \WP_REST_Response([
                'success' => true,
                'message' => 'Thumbnail size settings updated successfully',
                'data' => [
                    'sizes' => $updated_sizes,
                    'statistics' => $statistics,
                    'disabled_count' => count($disabled_sizes)
                ]
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to update settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get thumbnail settings (prevention_enabled, etc.)
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function get_thumbnail_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $settings = ImageThumbnailManager::get_thumbnail_settings();
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $settings
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to retrieve thumbnail settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update thumbnail settings (prevention_enabled, etc.)
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function update_thumbnail_settings(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $settings = $request->get_params();
            
            $result = ImageThumbnailManager::update_thumbnail_settings($settings);
            
            if (!$result) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Failed to update thumbnail settings'
                ], 500);
            }
            
            return new \WP_REST_Response([
                'success' => true,
                'message' => 'Thumbnail settings updated successfully'
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to update thumbnail settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Analyze thumbnail usage across site content
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function analyze_usage(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $batch_size = $request->get_param('batch_size') ?? 50;
            $active_sizes = $request->get_param('active_sizes') ?? [];
            $inactive_sizes = $request->get_param('inactive_sizes') ?? [];
            $respect_size_settings = $request->get_param('respect_size_settings') ?? false;
            
            // If size settings provided, use them to inform the analysis
            $analysis_options = ['batch_size' => $batch_size];
            if ($respect_size_settings && !empty($inactive_sizes)) {
                $analysis_options['focus_on_sizes'] = $inactive_sizes;
            }
            
            $analysis = ImageThumbnailManager::analyze_thumbnail_usage($batch_size, $analysis_options);
            
            // If we have size settings, filter and highlight results
            if ($respect_size_settings && !empty($inactive_sizes)) {
                // Calculate potential savings focusing on inactive sizes
                $inactive_savings = 0;
                $inactive_count = 0;
                
                foreach ($analysis['size_usage'] as $size_name => $usage) {
                    if (in_array($size_name, $inactive_sizes)) {
                        $inactive_savings += $usage['unused_size'] ?? 0;
                        $inactive_count += $usage['unused'] ?? 0;
                    }
                }
                
                $analysis['inactive_sizes_savings'] = $inactive_savings;
                $analysis['inactive_sizes_count'] = $inactive_count;
                $analysis['inactive_sizes_savings_formatted'] = size_format($inactive_savings);
                $analysis['recommended_disabled'] = $inactive_sizes;
            }
            
            // Format potential savings for display
            $analysis['potential_savings_formatted'] = size_format($analysis['potential_savings'] ?? 0);
            
            // Calculate usage percentages
            if (isset($analysis['size_usage'])) {
                foreach ($analysis['size_usage'] as $size_name => &$usage) {
                    $total = ($usage['used'] ?? 0) + ($usage['unused'] ?? 0);
                    $usage['usage_percentage'] = $total > 0 ? round((($usage['used'] ?? 0) / $total) * 100, 1) : 0;
                    $usage['total'] = $total;
                    $usage['is_active'] = !in_array($size_name, $inactive_sizes);
                }
            }
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $analysis
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to analyze thumbnail usage: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get thumbnail management statistics
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function get_statistics(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $statistics = ImageThumbnailManager::get_statistics();
            
            // Add environment info
            $statistics['environment'] = [
                'php_version' => PHP_VERSION,
                'wordpress_version' => get_bloginfo('version'),
                'theme' => wp_get_theme()->get('Name'),
                'active_plugins' => count(get_option('active_plugins', []))
            ];
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $statistics
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get statistics: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Preview thumbnail cleanup (dry run)
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function preview_cleanup(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            // Support both old and new parameter names
            $sizes_to_cleanup = $request->get_param('sizes') ?? $request->get_param('inactive_sizes') ?? [];
            $respect_size_settings = $request->get_param('respect_size_settings') ?? false;
            
            if (empty($sizes_to_cleanup)) {
                // If no specific sizes requested, find all unused thumbnails
                $analysis = ImageThumbnailManager::analyze_thumbnail_usage(100);
                $sizes_to_cleanup = $analysis['recommended_disabled'] ?? [];
            }
            
            if (empty($sizes_to_cleanup)) {
                return new \WP_REST_Response([
                    'success' => true,
                    'data' => [
                        'files_found' => 0,
                        'total_size' => 0,
                        'total_size_formatted' => '0 B',
                        'message' => 'No inactive thumbnail sizes found for cleanup'
                    ]
                ], 200);
            }
            
            $preview_data = self::scan_thumbnails_for_cleanup($sizes_to_cleanup, true);
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $preview_data
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to preview cleanup: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Execute thumbnail cleanup
     * 
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response API response
     */
    public static function execute_cleanup(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            // Support both old and new parameter names
            $sizes_to_cleanup = $request->get_param('sizes') ?? $request->get_param('inactive_sizes') ?? [];
            $confirm = $request->get_param('confirm') ?? $request->get_param('confirmed') ?? false;
            $respect_size_settings = $request->get_param('respect_size_settings') ?? false;
            
            if (!$confirm) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Cleanup must be confirmed with confirmed=true parameter'
                ], 400);
            }
            
            if (empty($sizes_to_cleanup)) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'No inactive thumbnail sizes specified for cleanup'
                ], 400);
            }
            
            $cleanup_results = self::scan_thumbnails_for_cleanup($sizes_to_cleanup, false);
            
            return new \WP_REST_Response([
                'success' => true,
                'message' => 'Thumbnail cleanup completed successfully',
                'data' => [
                    'files_deleted' => $cleanup_results['files_found'] ?? 0,
                    'space_freed' => $cleanup_results['total_size'] ?? 0,
                    'space_freed_formatted' => $cleanup_results['total_size_formatted'] ?? '0 B'
                ]
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to execute cleanup: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Scan and optionally cleanup thumbnail files
     * 
     * @param array $sizes_to_cleanup Thumbnail sizes to process
     * @param bool $dry_run If true, only preview without actual deletion
     * @return array Cleanup results
     */
    private static function scan_thumbnails_for_cleanup(array $sizes_to_cleanup, bool $dry_run = true): array
    {
        $results = [
            'files_found' => 0,
            'files_processed' => 0,
            'total_size_bytes' => 0,
            'total_size_formatted' => '',
            'sizes_processed' => [],
            'errors' => [],
            'dry_run' => $dry_run
        ];
        
        // Get all image attachments
        $attachments = get_posts([
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'numberposts' => -1,
            'post_status' => 'inherit'
        ]);
        
        foreach ($attachments as $attachment) {
            $attachment_id = $attachment->ID;
            $metadata = wp_get_attachment_metadata($attachment_id);
            
            if (!$metadata || empty($metadata['sizes'])) {
                continue;
            }
            
            foreach ($sizes_to_cleanup as $size_name) {
                if (!isset($metadata['sizes'][$size_name])) {
                    continue;
                }
                
                $file_path = self::get_thumbnail_file_path($attachment_id, $size_name, $metadata);
                
                if (!$file_path || !file_exists($file_path)) {
                    continue;
                }
                
                $file_size = filesize($file_path);
                $results['files_found']++;
                $results['total_size_bytes'] += $file_size;
                
                if (!isset($results['sizes_processed'][$size_name])) {
                    $results['sizes_processed'][$size_name] = [
                        'count' => 0,
                        'size_bytes' => 0
                    ];
                }
                
                $results['sizes_processed'][$size_name]['count']++;
                $results['sizes_processed'][$size_name]['size_bytes'] += $file_size;
                
                // Actually delete file if not dry run
                if (!$dry_run) {
                    if (@unlink($file_path)) {
                        $results['files_processed']++;
                        
                        // Update attachment metadata to remove deleted size
                        unset($metadata['sizes'][$size_name]);
                        wp_update_attachment_metadata($attachment_id, $metadata);
                        
                    } else {
                        $results['errors'][] = "Failed to delete: {$file_path}";
                    }
                } else {
                    $results['files_processed']++;
                }
            }
        }
        
        $results['total_size_formatted'] = size_format($results['total_size_bytes']);
        
        // Format size data for each processed size
        foreach ($results['sizes_processed'] as $size_name => &$size_data) {
            $size_data['size_formatted'] = size_format($size_data['size_bytes']);
        }
        
        return $results;
    }

    /**
     * Get file path for specific thumbnail
     * 
     * @param int $attachment_id Attachment ID
     * @param string $size_name Thumbnail size name
     * @param array $metadata Attachment metadata
     * @return string|null File path or null if not found
     */
    private static function get_thumbnail_file_path(int $attachment_id, string $size_name, array $metadata): ?string
    {
        if (!isset($metadata['sizes'][$size_name])) {
            return null;
        }
        
        $upload_dir = wp_upload_dir();
        $file_dir = dirname($metadata['file']);
        $filename = $metadata['sizes'][$size_name]['file'];
        
        return $upload_dir['basedir'] . '/' . $file_dir . '/' . $filename;
    }

    /**
     * Analyze WooCommerce thumbnail usage
     */
    public static function analyze_woocommerce_usage(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            if (!class_exists('\\ZiziCache\\WooCommerceThumbnailManager')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'WooCommerce Thumbnail Manager not available'
                ], 500);
            }

            $batch_size = $request->get_param('batch_size') ?? 100;
            $analysis = WooCommerceThumbnailManager::analyze_woocommerce_usage($batch_size);
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $analysis
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to analyze WooCommerce usage: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get orphaned WooCommerce images
     */
    public static function get_orphaned_wc_images(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            if (!class_exists('\\ZiziCache\\WooCommerceThumbnailManager')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'WooCommerce not available'
                ], 400);
            }

            // This would need to be implemented in WooCommerceThumbnailManager
            $orphaned_data = [
                'count' => 0,
                'total_size' => 0,
                'images' => []
            ];
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $orphaned_data
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get orphaned images: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cleanup orphaned WooCommerce images
     */
    public static function cleanup_orphaned_wc_images(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $confirm = $request->get_param('confirm');
            
            if (!$confirm) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Cleanup must be confirmed'
                ], 400);
            }

            // Implementation would go here
            $cleanup_results = [
                'images_deleted' => 0,
                'space_freed' => 0
            ];
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $cleanup_results
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to cleanup orphaned images: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Detect active page builders
     */
    public static function detect_page_builders(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            if (!class_exists('\\ZiziCache\\PageBuilderThumbnailDetector')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Page Builder Detector not available'
                ], 500);
            }

            $active_builders = PageBuilderThumbnailDetector::detect_active_builders();
            $statistics = PageBuilderThumbnailDetector::get_builder_statistics();
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => [
                    'active_builders' => $active_builders,
                    'statistics' => $statistics
                ]
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to detect page builders: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Analyze page builder usage
     */
    public static function analyze_builder_usage(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            if (!class_exists('\\ZiziCache\\PageBuilderThumbnailDetector')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Page Builder Detector not available'
                ], 500);
            }

            $batch_size = $request->get_param('batch_size') ?? 50;
            $analysis = PageBuilderThumbnailDetector::analyze_builder_efficiency();
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $analysis
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to analyze page builder usage: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Advanced usage analysis
     */
    public static function advanced_usage_analysis(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            if (!class_exists('\\ZiziCache\\AdvancedUsageAnalyzer')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Advanced Usage Analyzer not available'
                ], 500);
            }

            $options = [
                'include_content' => $request->get_param('include_content') ?? true,
                'include_meta' => $request->get_param('include_meta') ?? true,
                'include_builders' => $request->get_param('include_builders') ?? true,
                'include_woocommerce' => $request->get_param('include_woocommerce') ?? true,
                'batch_size' => $request->get_param('batch_size') ?? 100
            ];

            $analysis = AdvancedUsageAnalyzer::analyze_comprehensive_usage($options);
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $analysis
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to perform advanced analysis: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get quick statistics for dashboard
     */
    public static function get_quick_stats(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $stats = [];

            // Basic stats
            if (class_exists('\\ZiziCache\\AdvancedUsageAnalyzer')) {
                $stats['basic'] = AdvancedUsageAnalyzer::get_quick_statistics();
            }

            // WooCommerce stats
            if (class_exists('\\ZiziCache\\WooCommerceThumbnailManager')) {
                $stats['woocommerce'] = WooCommerceThumbnailManager::get_wc_statistics();
            }

            // Page builder stats
            if (class_exists('\\ZiziCache\\PageBuilderThumbnailDetector')) {
                $stats['builders'] = PageBuilderThumbnailDetector::get_builder_statistics();
            }

            return new \WP_REST_Response([
                'success' => true,
                'data' => $stats
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get quick stats: ' . $e->getMessage()
            ], 500);
        }
    }

    // ==============================================
    // HANDLE METHODS FOR REST API CALLBACKS
    // ==============================================

    /**
     * Handle quick stats endpoint
     */
    public function handle_quick_stats(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->get_quick_stats($request);
    }

    /**
     * Handle sizes endpoint
     */
    public function handle_sizes(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->get_all_sizes($request);
    }

    /**
     * Handle analyze endpoint
     */
    public function handle_analyze(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->analyze_usage($request);
    }

    /**
     * Handle WooCommerce status endpoint
     */
    public function handle_woocommerce_status(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->get_woocommerce_status($request);
    }

    /**
     * Handle WooCommerce analyze endpoint
     */
    public function handle_woocommerce_analyze(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->analyze_woocommerce($request);
    }

    /**
     * Handle WooCommerce cleanup endpoint
     */
    public function handle_woocommerce_cleanup(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->cleanup_woocommerce($request);
    }

    /**
     * Handle builders detect endpoint
     */
    public function handle_builders_detect(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->detect_page_builders($request);
    }

    /**
     * Handle builders analyze endpoint
     */
    public function handle_builders_analyze(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->analyze_page_builders($request);
    }

    /**
     * Handle advanced analyze endpoint
     */
    public function handle_advanced_analyze(\WP_REST_Request $request): \WP_REST_Response
    {
        return $this->analyze_advanced($request);
    }

    /**
     * Handle advanced status endpoint
     */
    public function handle_advanced_status(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $status = [
                'analysis_available' => class_exists('\\ZiziCache\\AdvancedUsageAnalyzer'),
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
                'estimated_images' => wp_count_posts('attachment')->inherit ?? 0,
                'recommended_batch_size' => $this->calculate_optimal_batch_size()
            ];

            return new \WP_REST_Response([
                'success' => true,
                'data' => $status
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to get advanced status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle advanced optimize endpoint
     */
    public function handle_advanced_optimize(\WP_REST_Request $request): \WP_REST_Response
    {
        try {
            $confirm = $request->get_param('confirm');
            if (!$confirm) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Confirmation required for optimization operations'
                ], 400);
            }

            if (!class_exists('\\ZiziCache\\AdvancedUsageAnalyzer')) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Advanced analyzer not available'
                ], 500);
            }

            $options = [
                'remove_unused' => $request->get_param('remove_unused') ?? false,
                'batch_size' => $request->get_param('batch_size') ?? 100,
                'dry_run' => $request->get_param('dry_run') ?? true
            ];

            $result = AdvancedUsageAnalyzer::optimize_thumbnails($options);
            
            return new \WP_REST_Response([
                'success' => true,
                'data' => $result
            ], 200);

        } catch (\Exception $e) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Failed to optimize thumbnails: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Calculate optimal batch size based on system resources
     */
    private function calculate_optimal_batch_size(): int
    {
        $memory_limit = $this->parse_memory_limit(ini_get('memory_limit'));
        $available_memory = $memory_limit - memory_get_usage();
        
        // Estimate ~2MB per image processing
        $estimated_per_image = 2 * 1024 * 1024;
        $max_by_memory = intval($available_memory / $estimated_per_image * 0.5); // 50% safety margin
        
        // Clamp between 50 and 500
        return max(50, min(500, $max_by_memory));
    }

    /**
     * Parse memory limit string to bytes
     */
    private function parse_memory_limit(string $limit): int
    {
        $unit = strtolower(substr($limit, -1));
        $value = intval($limit);
        
        switch ($unit) {
            case 'g':
                return $value * 1024 * 1024 * 1024;
            case 'm':
                return $value * 1024 * 1024;
            case 'k':
                return $value * 1024;
            default:
                return $value;
        }
    }
}
