<?php

namespace ZiziCache;

/**
 * Enhanced Image Converter Statistics
 * Provides accurate and performant statistics for image conversion
 * 
 * Key improvements:
 * - Precise original vs converted file detection
 * - Optimized file system scanning with batching
 * - Smart caching with proper invalidation
 * - User settings-based calculations
 * - Detailed space savings analysis
 */
class ImageConverterStatsEnhanced extends ImageConverterStats
{
    // Supported original image formats
    const ORIGINAL_FORMATS = [
        'image/jpeg' => ['jpg', 'jpeg'],
        'image/png' => ['png'],
        'image/gif' => ['gif'],
        'image/bmp' => ['bmp'],
        'image/tiff' => ['tiff', 'tif']
    ];

    // Converted formats
    const CONVERTED_FORMATS = [
        'image/webp' => ['webp'],
        'image/avif' => ['avif']
    ];

    // Enhanced cache keys
    const CACHE_KEY_ENHANCED_STATS = 'zizi_image_converter_enhanced_stats_v1';
    const CACHE_KEY_SCAN = 'zizi_image_converter_enhanced_scan_v1';
    const CACHE_TTL = 300; // 5 minutes

    /**
     * Get comprehensive enhanced image converter statistics
     * 
     * @param bool $force_refresh Force refresh of cached data
     * @return array Complete enhanced statistics
     */
    public static function getEnhancedStats(bool $force_refresh = false): array
    {
        $start_time = microtime(true);

        // Try cache first unless forcing refresh
        if (!$force_refresh) {
            $cached_stats = get_transient(self::CACHE_KEY_ENHANCED_STATS);
            if ($cached_stats !== false && is_array($cached_stats)) {
                $cached_stats['cached'] = true;
                $cached_stats['cache_age'] = time() - ($cached_stats['generated_at'] ?? time());
                return $cached_stats;
            }
        }

        try {
            // Get user settings for enabled formats
            $config = SysConfig::$config ?? [];
            $enabled_formats = $config['image_converter_formats'] ?? ['webp'];
            
            // Perform comprehensive scan
            $scan_results = self::performComprehensiveScan();
            
            // Calculate enhanced statistics based on user settings
            $stats = self::calculateEnhancedStatistics($scan_results, $enabled_formats);
            
            // Add metadata
            $stats['generated_at'] = time();
            $stats['processing_time'] = round(microtime(true) - $start_time, 3);
            $stats['enabled_formats'] = $enabled_formats;
            $stats['cached'] = false;
            $stats['cache_age'] = 0;

            // Cache results
            set_transient(self::CACHE_KEY_ENHANCED_STATS, $stats, self::CACHE_TTL);

            return $stats;

        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("ImageConverterStatsEnhanced: Error generating stats: " . $e->getMessage(), 'ImageConverter');
            }

            // Return safe fallback with enhanced structure
            return self::getEmptyEnhancedStats();
        }
    }

    /**
     * Perform comprehensive file system scan
     * Optimized for performance with batching
     * 
     * @return array Scan results with detailed file analysis
     */
    private static function performComprehensiveScan(): array
    {
        $uploads_dir = wp_upload_dir()['basedir'];
        if (!is_dir($uploads_dir)) {
            throw new \Exception('Uploads directory not accessible');
        }

        // Check for cached scan results
        $cache_key = self::CACHE_KEY_SCAN . '_' . md5($uploads_dir);
        $cached_scan = get_transient($cache_key);
        
        // Use cached scan if recent and valid
        if ($cached_scan !== false && is_array($cached_scan)) {
            $cache_age = time() - ($cached_scan['scan_time'] ?? 0);
            if ($cache_age < 600) { // 10 minutes cache for scan
                return $cached_scan;
            }
        }

        $results = [
            'original_files' => [],
            'converted_files' => [
                'webp' => [],
                'avif' => []
            ],
            'file_sizes' => [],
            'scan_time' => time(),
            'total_scanned' => 0
        ];

        // Use optimized iterator with progress tracking
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($uploads_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::LEAVES_ONLY
        );

        $batch_count = 0;
        $batch_size = 1000; // Process in batches to avoid memory issues

        foreach ($iterator as $file) {
            if (!$file->isFile()) continue;

            $batch_count++;
            $results['total_scanned']++;

            $file_path = $file->getRealPath();
            $file_size = $file->getSize();
            $extension = strtolower($file->getExtension());
            
            // Store file size for calculations
            $results['file_sizes'][$file_path] = $file_size;

            // Categorize file by type
            if (self::isOriginalImageFile($extension)) {
                $results['original_files'][] = [
                    'path' => $file_path,
                    'size' => $file_size,
                    'extension' => $extension
                ];
            } elseif ($extension === 'webp') {
                $results['converted_files']['webp'][] = [
                    'path' => $file_path,
                    'size' => $file_size
                ];
            } elseif ($extension === 'avif') {
                $results['converted_files']['avif'][] = [
                    'path' => $file_path,
                    'size' => $file_size
                ];
            }

            // Garbage collection every batch
            if ($batch_count >= $batch_size) {
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
                $batch_count = 0;
            }
        }

        // Cache scan results
        set_transient($cache_key, $results, 600); // 10 minutes

        return $results;
    }

    /**
     * Calculate comprehensive statistics from scan results
     * 
     * @param array $scan_results Results from file system scan
     * @param array $enabled_formats User-enabled conversion formats
     * @return array Calculated enhanced statistics
     */
    private static function calculateEnhancedStatistics(array $scan_results, array $enabled_formats): array
    {
        $stats = [
            // Basic counts
            'total_original_images' => count($scan_results['original_files']),
            'total_conversions' => 0,
            'remaining_images' => 0,
            'webp_count' => count($scan_results['converted_files']['webp']),
            'avif_count' => count($scan_results['converted_files']['avif']),
            
            // Format distribution with detailed analytics
            'format_distribution' => [
                'webp' => ['count' => 0, 'size' => 0, 'savings' => 0],
                'avif' => ['count' => 0, 'size' => 0, 'savings' => 0]
            ],
            
            // Size analytics
            'total_original_size' => 0,
            'total_converted_size' => 0,
            'total_savings' => 0,
            'savings_percent' => 0,
            'potential_savings' => 0,
            
            // Optimization status
            'fully_optimized_count' => 0,
            'partially_optimized_count' => 0,
            'unoptimized_count' => 0,
            
            // WordPress integration
            'wordpress_registered' => 0,
            'orphaned_files' => 0,
            
            // Scan metadata
            'scan_metadata' => [
                'total_scanned' => $scan_results['total_scanned'],
                'scan_time' => $scan_results['scan_time']
            ]
        ];

        // Analyze each original image
        foreach ($scan_results['original_files'] as $original) {
            $original_path = $original['path'];
            $original_size = $original['size'];
            $stats['total_original_size'] += $original_size;

            $conversion_analysis = self::analyzeImageConversions(
                $original_path, 
                $scan_results['converted_files'], 
                $enabled_formats
            );

            // Update optimization counts
            if ($conversion_analysis['fully_optimized']) {
                $stats['fully_optimized_count']++;
            } elseif ($conversion_analysis['partially_optimized']) {
                $stats['partially_optimized_count']++;
            } else {
                $stats['unoptimized_count']++;
                $stats['remaining_images']++;
                $stats['potential_savings'] += self::estimatePotentialSavings($original_size, $enabled_formats);
            }

            // Update format distribution
            foreach ($conversion_analysis['converted_formats'] as $format) {
                if (isset($stats['format_distribution'][$format])) {
                    $converted_path = $original_path . '.' . $format;
                    if (isset($scan_results['file_sizes'][$converted_path])) {
                        $converted_size = $scan_results['file_sizes'][$converted_path];
                        $savings = max(0, $original_size - $converted_size);
                        
                        $stats['format_distribution'][$format]['count']++;
                        $stats['format_distribution'][$format]['size'] += $converted_size;
                        $stats['format_distribution'][$format]['savings'] += $savings;
                        $stats['total_converted_size'] += $converted_size;
                    }
                }
            }
        }

        // Calculate total conversions (unique originals with at least one conversion)
        $converted_originals = [];
        foreach (['webp', 'avif'] as $format) {
            foreach ($scan_results['converted_files'][$format] as $converted_file) {
                $original_path = substr($converted_file['path'], 0, -(strlen($format) + 1));
                $converted_originals[$original_path] = true;
            }
        }
        $stats['total_conversions'] = count($converted_originals);

        // Calculate overall savings
        $stats['total_savings'] = $stats['total_original_size'] - $stats['total_converted_size'];
        if ($stats['total_original_size'] > 0) {
            $stats['savings_percent'] = round(($stats['total_savings'] / $stats['total_original_size']) * 100, 1);
        }

        // Add WordPress integration stats
        $stats = array_merge($stats, self::getWordPressIntegrationStats($scan_results));

        return $stats;
    }

    /**
     * Analyze conversion status for a single image
     * 
     * @param string $original_path Path to original image
     * @param array $converted_files Array of converted files by format
     * @param array $enabled_formats User-enabled formats
     * @return array Analysis results
     */
    private static function analyzeImageConversions(
        string $original_path, 
        array $converted_files, 
        array $enabled_formats
    ): array {
        $result = [
            'fully_optimized' => false,
            'partially_optimized' => false,
            'converted_formats' => []
        ];

        $original_basename = pathinfo($original_path, PATHINFO_DIRNAME) . '/' . pathinfo($original_path, PATHINFO_FILENAME);

        foreach ($enabled_formats as $format) {
            $expected_converted_path = $original_path . '.' . $format;
            
            // Check if converted file exists in our scan results
            foreach ($converted_files[$format] as $converted_file) {
                if ($converted_file['path'] === $expected_converted_path) {
                    $result['converted_formats'][] = $format;
                    break;
                }
            }
        }

        // Determine optimization status
        $total_enabled = count($enabled_formats);
        $total_converted = count($result['converted_formats']);

        if ($total_converted === $total_enabled && $total_enabled > 0) {
            $result['fully_optimized'] = true;
        } elseif ($total_converted > 0) {
            $result['partially_optimized'] = true;
        }

        return $result;
    }

    /**
     * Estimate potential savings for an image
     * 
     * @param int $original_size Original file size
     * @param array $enabled_formats Enabled conversion formats
     * @return int Estimated bytes saved
     */
    private static function estimatePotentialSavings(int $original_size, array $enabled_formats): int
    {
        $savings_rates = [
            'webp' => 0.25, // 25% savings on average
            'avif' => 0.35  // 35% savings on average
        ];

        $best_savings = 0;
        foreach ($enabled_formats as $format) {
            $rate = $savings_rates[$format] ?? 0.20;
            $estimated_savings = $original_size * $rate;
            $best_savings = max($best_savings, $estimated_savings);
        }

        return (int)$best_savings;
    }

    /**
     * Get WordPress integration statistics
     * 
     * @param array $scan_results File system scan results
     * @return array WordPress-specific statistics
     */
    private static function getWordPressIntegrationStats(array $scan_results): array
    {
        global $wpdb;

        $stats = [
            'wordpress_registered' => 0,
            'orphaned_files' => 0,
            'registered_vs_filesystem' => [
                'wp_only' => 0,
                'fs_only' => 0,
                'both' => 0
            ]
        ];

        try {
            // Get all image attachments from WordPress
            $wp_images = $wpdb->get_results("
                SELECT p.ID, pm.meta_value as file_path 
                FROM {$wpdb->posts} p 
                INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
                WHERE p.post_type = 'attachment' 
                AND p.post_mime_type LIKE 'image/%' 
                AND pm.meta_key = '_wp_attached_file'
            ", ARRAY_A);

            $wp_file_paths = [];
            foreach ($wp_images as $wp_image) {
                $uploads_dir = wp_upload_dir()['basedir'];
                $full_path = $uploads_dir . '/' . $wp_image['file_path'];
                $wp_file_paths[wp_normalize_path($full_path)] = $wp_image['ID'];
            }

            $fs_file_paths = [];
            foreach ($scan_results['original_files'] as $original_file) {
                $fs_file_paths[wp_normalize_path($original_file['path'])] = true;
            }

            // Compare WordPress vs filesystem
            foreach ($wp_file_paths as $wp_path => $attachment_id) {
                if (isset($fs_file_paths[$wp_path])) {
                    $stats['registered_vs_filesystem']['both']++;
                    $stats['wordpress_registered']++;
                } else {
                    $stats['registered_vs_filesystem']['wp_only']++;
                }
            }

            foreach ($fs_file_paths as $fs_path => $ignored) {
                if (!isset($wp_file_paths[$fs_path])) {
                    $stats['registered_vs_filesystem']['fs_only']++;
                    $stats['orphaned_files']++;
                }
            }

        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("ImageConverterStatsEnhanced: WordPress integration error: " . $e->getMessage(), 'ImageConverter');
            }
        }

        return $stats;
    }

    /**
     * Check if file extension represents an original image format
     * 
     * @param string $extension File extension
     * @return bool True if original format
     */
    private static function isOriginalImageFile(string $extension): bool
    {
        foreach (self::ORIGINAL_FORMATS as $mime => $extensions) {
            if (in_array($extension, $extensions)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get empty enhanced statistics structure
     * 
     * @return array Empty enhanced stats
     */
    private static function getEmptyEnhancedStats(): array
    {
        return [
            'total_original_images' => 0,
            'total_conversions' => 0,
            'remaining_images' => 0,
            'webp_count' => 0,
            'avif_count' => 0,
            'format_distribution' => [
                'webp' => ['count' => 0, 'size' => 0, 'savings' => 0],
                'avif' => ['count' => 0, 'size' => 0, 'savings' => 0]
            ],
            'total_original_size' => 0,
            'total_converted_size' => 0,
            'total_savings' => 0,
            'savings_percent' => 0,
            'potential_savings' => 0,
            'fully_optimized_count' => 0,
            'partially_optimized_count' => 0,
            'unoptimized_count' => 0,
            'wordpress_registered' => 0,
            'orphaned_files' => 0,
            'registered_vs_filesystem' => [
                'wp_only' => 0,
                'fs_only' => 0,
                'both' => 0
            ],
            'scan_metadata' => [
                'total_scanned' => 0,
                'scan_time' => time()
            ],
            'generated_at' => time(),
            'processing_time' => 0,
            'enabled_formats' => [],
            'cached' => false,
            'cache_age' => 0
        ];
    }

    /**
     * Clear enhanced statistics cache
     */
    public static function clearEnhancedCache(): void
    {
        delete_transient(self::CACHE_KEY_ENHANCED_STATS);
        
        // Clear scan cache for all possible directories
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare("
                DELETE FROM {$wpdb->options} 
                WHERE option_name LIKE %s
            ", '_transient_' . self::CACHE_KEY_SCAN . '%')
        );

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog("ImageConverterStatsEnhanced: Enhanced cache cleared", 'ImageConverter');
        }
    }

    /**
     * Initialize enhanced statistics tracking
     */
    public static function init(): void
    {
        // Clear cache when images are converted
        add_action('zizi_image_converted', [__CLASS__, 'clearEnhancedCache']);
        
        // Clear cache daily
        add_action('wp_scheduled_delete', [__CLASS__, 'clearEnhancedCache']);
        
        // Clear cache when new images are uploaded
        add_action('add_attachment', [__CLASS__, 'clearEnhancedCache']);
        
        // Clear cache when attachments are deleted
        add_action('delete_attachment', [__CLASS__, 'clearEnhancedCache']);
    }
}