<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Image Converter Statistics
 * 
 * Lightweight statistics for Image Converter functionality.
 * Provides quick overview of converted images without performance impact.
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.4
 */
class ImageConverterStats
{
    /**
     * Create database table for statistics
     * 
     * @return bool Success status
     */
    public static function createTable(): bool
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'zizi_image_converter_stats';

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            attachment_id bigint(20) unsigned NOT NULL,
            original_format varchar(20) NOT NULL,
            output_format varchar(20) NOT NULL,
            original_size bigint(20) unsigned NOT NULL,
            optimized_size bigint(20) unsigned NOT NULL,
            size_savings bigint(20) unsigned NOT NULL,
            processing_time float DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY attachment_id (attachment_id),
            KEY output_format (output_format),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        $result = dbDelta($sql);

        return !empty($result);
    }
    /**
     * Get image conversion statistics
     * 
     * @return array Statistics data
     */
    public static function getStats(): array
    {
        global $wpdb;
        
        // Cache key for stats - v4 includes total media library count
        $cache_key = 'zizi_image_converter_stats_v4';
        
        // Try object cache first
        $cached_stats = wp_cache_get($cache_key);
        if ($cached_stats !== false) {
            return $cached_stats;
        }
        
        // Try transient cache as fallback
        $cached_stats = get_transient($cache_key);
        if ($cached_stats !== false) {
            // Also store in object cache for faster access
            wp_cache_set($cache_key, $cached_stats, '', 120);
            return $cached_stats;
        }
        
        $table_name = $wpdb->prefix . 'zizi_image_converter_stats';
        
        $stats = [
            'total_conversions' => 0,
            'total_images' => 0,
            'remaining_images' => 0,
            'webp_count' => 0,
            'avif_count' => 0,
            'total_savings' => 0,
            'savings_percent' => 0,
            'avg_processing_time' => 0,
            'success_rate' => 100,
            'last_conversion' => null,
            // Enhanced statistics for Vue component
            'upload_directory_stats' => [
                'total_supported_images' => 0,
                'webp_percentage' => 0,
                'avif_percentage' => 0,
                'original_images_count' => 0
            ]
        ];
        
        try {
            // Get total number of images in media library
            $total_images_data = self::getTotalImagesCount();
            $stats['total_images'] = $total_images_data['total_images'];
            $stats['remaining_images'] = $total_images_data['remaining_images'];
            
            // Always perform real-time file system verification for accurate counts
            try {
                $verified_stats = self::performFileSystemScan();
            } catch (\Throwable $t) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError('ImageConverterStats: performFileSystemScan failed: ' . $t->getMessage(), 'ImageConverter');
                }
                $verified_stats = [
                    'total_conversions' => 0,
                    'webp_count' => 0,
                    'avif_count' => 0,
                    'total_savings' => 0,
                    'savings_percent' => 0
                ];
                $stats['error'] = $t->getMessage();
            }
            
            // Get processing times from database if available
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
            if ($table_exists) {
                // Clean up stale database records by verifying file existence
                self::cleanupStaleRecords();
                
                // Get average processing time from valid records
                $avg_time = $wpdb->get_var("SELECT AVG(processing_time) FROM $table_name WHERE processing_time > 0");
                $stats['avg_processing_time'] = $avg_time ? round((float)$avg_time, 2) : 0;
                
                // Get last conversion timestamp
                $last_conversion = $wpdb->get_var("SELECT created_at FROM $table_name ORDER BY created_at DESC LIMIT 1");
                $stats['last_conversion'] = $last_conversion;
            }
            
            // Use file system results as authoritative source for counts
            $stats['total_conversions'] = $verified_stats['total_conversions'];
            $stats['webp_count'] = $verified_stats['webp_count'];
            $stats['avif_count'] = $verified_stats['avif_count'];
            $stats['total_savings'] = $verified_stats['total_savings'];
            $stats['savings_percent'] = $verified_stats['savings_percent'];
            
            // Include enhanced upload directory statistics
            if (isset($verified_stats['upload_directory_stats'])) {
                $stats['upload_directory_stats'] = $verified_stats['upload_directory_stats'];
            }
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("ImageConverterStats error: " . $e->getMessage(), 'ImageConverter');
            }
        }
        
        // Cache for 2 minutes (both object cache and transient)
        wp_cache_set($cache_key, $stats, '', 120);
        set_transient($cache_key, $stats, 120);
        
        return $stats;
    }
    
    /**
     * Get total images count and remaining images to convert
     * Includes both WordPress Media Library and zizi-cache directory
     * 
     * @return array Array with total_images and remaining_images
     */
    private static function getTotalImagesCount(): array
    {
        // Get current config to check allowed types
        $config = \ZiziCache\SysConfig::$config ?? [];
        $allowed_types = $config['image_converter_allowed_types'] ?? ['image/jpeg', 'image/png', 'image/gif'];
        
        // Ensure we only include original formats, not already converted ones
        $original_formats_only = array_filter($allowed_types, function($mime_type) {
            return !self::isAlreadyConvertedFormat($mime_type);
        });
        
        // Debug logging
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "ImageConverterStats: Allowed types from config: " . implode(', ', $allowed_types) . 
                " | Original formats only: " . implode(', ', $original_formats_only), 
                'ImageConverter'
            );
        }
        
        // If no valid original formats, return empty counts
        if (empty($original_formats_only)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog("ImageConverterStats: No valid original formats found", 'ImageConverter');
            }
            return [
                'total_images' => 0,
                'remaining_images' => 0
            ];
        }

        // Get current enabled formats from settings
        $enabled_formats = $config['image_converter_formats'] ?? ['webp'];

        // 1. Get WordPress Media Library images
        $query_args = [
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => -1,
            'post_mime_type' => $original_formats_only, // Only original formats, not webp/avif
            'fields' => 'ids'
        ];
        
        $query = new \WP_Query($query_args);
        $wordpress_total = $query->found_posts;
        
        // 2. Get zizi-cache directory images
        $zizi_cache_total = 0;
        $zizi_cache_remaining = 0;
        $uploads_dir = wp_upload_dir()['basedir'];
        $zizi_cache_dir = $uploads_dir . '/zizi-cache';
        
        if (is_dir($zizi_cache_dir)) {
            try {
                $supported_extensions = [];
                
                // Use ImageConverter's supported extensions for consistency
                if (class_exists('\\ZiziCache\\ImageConverter') && method_exists('\\ZiziCache\\ImageConverter', 'get_supported_extensions')) {
                    $supported_extensions = \ZiziCache\ImageConverter::get_supported_extensions();
                } else {
                    // Fallback to common extensions
                    $supported_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'tiff', 'tif'];
                }
                
                $iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator($zizi_cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                    \RecursiveIteratorIterator::LEAVES_ONLY
                );
                
                foreach ($iterator as $file) {
                    if (!$file->isFile()) continue;
                    
                    $extension = strtolower(pathinfo($file->getPathname(), PATHINFO_EXTENSION));
                    
                    // Only count supported original formats
                    if (in_array($extension, $supported_extensions, true)) {
                        $zizi_cache_total++;
                        
                        // Check if this image needs optimization
                        $file_path = $file->getPathname();
                        $needs_optimization = false;
                        
                        foreach ($enabled_formats as $format) {
                            $converted_file = $file_path . '.' . $format;
                            if (!file_exists($converted_file)) {
                                $needs_optimization = true;
                                break;
                            }
                        }
                        
                        if ($needs_optimization) {
                            $zizi_cache_remaining++;
                        }
                    }
                }
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "ImageConverterStats: zizi-cache scan found {$zizi_cache_total} images, {$zizi_cache_remaining} need optimization",
                        'ImageConverter'
                    );
                }
                
            } catch (Exception $e) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "ImageConverterStats: Error scanning zizi-cache directory: " . $e->getMessage(),
                        'ImageConverter'
                    );
                }
            }
        }
        
        $total_images = $wordpress_total + $zizi_cache_total;
        
        // Debug logging
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "ImageConverterStats: WordPress: {$wordpress_total}, zizi-cache: {$zizi_cache_total}, total: {$total_images}. Enabled formats: " . implode(', ', $enabled_formats), 
                'ImageConverter'
            );
        }
        
        // If Image Converter is disabled, all images are "remaining"
        if (!($config['image_converter_enabled'] ?? false)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog("ImageConverterStats: Image Converter disabled, all images are remaining", 'ImageConverter');
            }
            return [
                'total_images' => $total_images,
                'remaining_images' => $total_images
            ];
        }

        // Count WordPress images that need optimization
        $wordpress_remaining = 0;
        $skipped_converted = 0;
        $skipped_optimized = 0;
        
        if ($wordpress_total > 0) {
            foreach ($query->posts as $attachment_id) {
                // Triple-check that this is a supported original format
                $mime_type = get_post_mime_type($attachment_id);
                
                if (!in_array($mime_type, $original_formats_only)) {
                    continue; // Skip if not in original formats
                }
                
                if (self::isAlreadyConvertedFormat($mime_type)) {
                    $skipped_converted++;
                    continue; // Skip already converted formats
                }
                
                if (self::imageNeedsOptimization($attachment_id, $enabled_formats)) {
                    $wordpress_remaining++;
                } else {
                    $skipped_optimized++;
                }
            }
        }
        
        // Total remaining includes both WordPress and zizi-cache images
        $total_remaining = $wordpress_remaining + $zizi_cache_remaining;
        
        // Debug logging
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "ImageConverterStats: Processing complete - Total: {$total_images}, WordPress remaining: {$wordpress_remaining}, zizi-cache remaining: {$zizi_cache_remaining}, Total remaining: {$total_remaining}, Skipped converted: {$skipped_converted}, Skipped optimized: {$skipped_optimized}", 
                'ImageConverter'
            );
        }
        
        return [
            'total_images' => $total_images,
            'remaining_images' => $total_remaining
        ];
    }
    
    /**
     * Check if an image needs optimization based on enabled formats
     * 
     * @param int $attachment_id WordPress attachment ID
     * @param array $enabled_formats Enabled image formats (avif, webp)
     * @return bool True if image needs optimization
     */
    private static function imageNeedsOptimization(int $attachment_id, array $enabled_formats): bool
    {
        $original_file = get_attached_file($attachment_id);
        if (!$original_file || !file_exists($original_file)) {
            return false;
        }
        
        // Check if any enabled format is missing
        foreach ($enabled_formats as $format) {
            $converted_file = $original_file . '.' . $format;
            if (!file_exists($converted_file)) {
                return true; // This format is missing, needs optimization
            }
        }
        
        // Check metadata consistency
        $existing_meta = get_post_meta($attachment_id, '_zizi_image_optimized', true);
        if (!is_array($existing_meta)) {
            return true; // No metadata, needs optimization
        }
        
        // Check if metadata matches enabled formats
        $meta_formats = $existing_meta['formats'] ?? [];
        foreach ($enabled_formats as $format) {
            if (!in_array($format, $meta_formats)) {
                return true; // Format not in metadata, needs optimization
            }
        }
        
        return false; // All formats exist and are in metadata
    }
    
    /**
     * Check if the MIME type represents an already converted format
     * 
     * @param string $mime_type MIME type to check
     * @return bool True if it's an already converted format
     */
    private static function isAlreadyConvertedFormat(string $mime_type): bool
    {
        $converted_formats = [
            'image/webp',
            'image/avif',
            // Additional modern formats that are considered "already optimized"
            'image/heic',
            'image/heif'
        ];
        
        return in_array(strtolower($mime_type), $converted_formats);
    }
    
    /**
     * Verify existing conversions by checking file system
     * Used when database is empty but files might exist
     * 
     * @return array Statistics based on file system scan
     */
    private static function verifyExistingConversions(): array
    {
        // This method now delegates to the more comprehensive file system scan
        return self::performFileSystemScan();
    }
    
    /**
     * Perform file system scan for accurate statistics
     * This is the authoritative source for counts since files are what matter
     * 
     * @return array File system based statistics with enhanced upload directory analysis
     */
    private static function performFileSystemScan(): array
    {
        $stats = [
            'total_conversions' => 0,
            'webp_count' => 0,
            'avif_count' => 0,
            'total_savings' => 0,
            'savings_percent' => 0,
            // Enhanced statistics for upload directory analysis
            'upload_directory_stats' => [
                'total_supported_images' => 0,
                'webp_percentage' => 0,
                'avif_percentage' => 0,
                'original_images_count' => 0
            ]
        ];
        
        // Scan uploads directory for all files
        $uploads_dir = wp_upload_dir()['basedir'];
        
        if (!is_dir($uploads_dir)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("ImageConverterStats: Uploads directory not found: $uploads_dir", 'ImageConverter');
            }
            return $stats;
        }
        
        $webp_files = [];
        $avif_files = [];
        $original_files = [];
        $total_original_size = 0;
        $total_optimized_size = 0;
        $debug_scan_count = 0;

        // Get supported original formats from ImageConverter class for consistency
        $supported_extensions = [];
        
        // Use ImageConverter's supported formats if available
        if (class_exists('\\ZiziCache\\ImageConverter') && method_exists('\\ZiziCache\\ImageConverter', 'get_supported_extensions')) {
            $supported_extensions = \ZiziCache\ImageConverter::get_supported_extensions();
        } else {
            // Fallback to config-based approach
            $config = \ZiziCache\SysConfig::$config ?? [];
            $allowed_types = $config['image_converter_allowed_types'] ?? ['image/jpeg', 'image/png', 'image/gif'];
            
            // Convert MIME types to extensions for file system scanning
            foreach ($allowed_types as $mime_type) {
                switch ($mime_type) {
                    case 'image/jpeg':
                        $supported_extensions = array_merge($supported_extensions, ['jpg', 'jpeg']);
                        break;
                    case 'image/png':
                        $supported_extensions[] = 'png';
                        break;
                    case 'image/gif':
                        $supported_extensions[] = 'gif';
                        break;
                    case 'image/webp':
                        // Skip WebP as it's a converted format, not original
                        break;
                    case 'image/avif':
                        // Skip AVIF as it's a converted format, not original
                        break;
                }
            }
        }

        // Determine whether verbose per-file logging should be enabled.
        $debug_log_enabled = false;
        $log_sample_limit = 5;
        $webp_sample_count = 0;
        $avif_sample_count = 0;

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog("ImageConverterStats: Starting enhanced file system scan in directory: $uploads_dir", 'ImageConverter');
            $debug_log_enabled = (\ZiziCache\SysConfig::$config['image_converter_debug_logging'] ?? false) === true;
            if (isset(\ZiziCache\SysConfig::$config['image_converter_log_sample_limit'])) {
                $log_sample_limit = max(1, intval(\ZiziCache\SysConfig::$config['image_converter_log_sample_limit']));
            }
        }
        
        try {
            // Recursively scan uploads directory for all image files
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($uploads_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );
            
            foreach ($iterator as $file) {
                if (!$file->isFile()) continue;
                
                $debug_scan_count++;
                $file_path = $file->getPathname();
                $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
                
                // Count original supported image files
                if (in_array($extension, $supported_extensions)) {
                    $original_files[] = $file_path;
                    $stats['upload_directory_stats']['original_images_count']++;
                }
                
                // Count WebP files
                if ($extension === 'webp') {
                    $webp_files[] = $file_path;
                    $stats['webp_count']++;
                    
                    // Try to find original file for size calculation
                    $original_file = substr($file_path, 0, -5); // Remove .webp (5 characters)
                    if (file_exists($original_file)) {
                        $original_size = filesize($original_file);
                        $optimized_size = filesize($file_path);
                        $total_original_size += $original_size;
                        $total_optimized_size += $optimized_size;

                        if ($debug_log_enabled && $webp_sample_count < $log_sample_limit) {
                            \ZiziCache\CacheSys::writeLog("ImageConverterStats: WebP file size calc - Original: $original_size, Optimized: $optimized_size for $file_path", 'ImageConverter');
                            $webp_sample_count++;
                        }
                    }
                }
                
                // Count AVIF files  
                elseif ($extension === 'avif') {
                    $avif_files[] = $file_path;
                    $stats['avif_count']++;
                    
                    // Try to find original file for size calculation
                    $original_file = substr($file_path, 0, -5); // Remove .avif (5 characters)
                    if (file_exists($original_file)) {
                        $original_size = filesize($original_file);
                        $optimized_size = filesize($file_path);
                        $total_original_size += $original_size;
                        $total_optimized_size += $optimized_size;

                        if ($debug_log_enabled && $avif_sample_count < $log_sample_limit) {
                            \ZiziCache\CacheSys::writeLog("ImageConverterStats: AVIF file size calc - Original: $original_size, Optimized: $optimized_size for $file_path", 'ImageConverter');
                            $avif_sample_count++;
                        }
                    }
                }
            }
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("ImageConverterStats: File scan exception: " . $e->getMessage(), 'ImageConverter');
            }
        }
        
        // Calculate enhanced upload directory statistics
        $stats['upload_directory_stats']['total_supported_images'] = 
            $stats['upload_directory_stats']['original_images_count'] + 
            $stats['webp_count'] + 
            $stats['avif_count'];
            
        if ($stats['upload_directory_stats']['total_supported_images'] > 0) {
            $stats['upload_directory_stats']['webp_percentage'] = round(
                ($stats['webp_count'] / $stats['upload_directory_stats']['total_supported_images']) * 100, 1
            );
            $stats['upload_directory_stats']['avif_percentage'] = round(
                ($stats['avif_count'] / $stats['upload_directory_stats']['total_supported_images']) * 100, 1
            );
        }
        
        // Count unique conversions (some files may have both WebP and AVIF)
        $converted_originals = [];
        $debug_unique_count = 0;
        
        foreach (array_merge($webp_files, $avif_files) as $converted_file) {
            // For .webp files, remove last 5 characters (.webp)
            // For .avif files, remove last 5 characters (.avif)
            if (str_ends_with($converted_file, '.webp')) {
                $original_file = substr($converted_file, 0, -5);
            } elseif (str_ends_with($converted_file, '.avif')) {
                $original_file = substr($converted_file, 0, -5);
            } else {
                continue; // Skip if not webp or avif
            }
            
            if (!isset($converted_originals[$original_file])) {
                $converted_originals[$original_file] = true;
                $debug_unique_count++;
            }
        }
        $stats['total_conversions'] = count($converted_originals);
        
        // Calculate savings
        $stats['total_savings'] = max(0, $total_original_size - $total_optimized_size);
        
        // Comprehensive debug logging
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "ImageConverterStats enhanced scan COMPLETE: " .
                "Scanned $debug_scan_count files, " .
                "Original={$stats['upload_directory_stats']['original_images_count']}, " .
                "WebP={$stats['webp_count']} ({$stats['upload_directory_stats']['webp_percentage']}%), " .
                "AVIF={$stats['avif_count']} ({$stats['upload_directory_stats']['avif_percentage']}%), " .
                "Total supported images={$stats['upload_directory_stats']['total_supported_images']}, " .
                "Unique conversions={$debug_unique_count}", 
                'ImageConverter'
            );
        }
        
        if ($total_original_size > 0) {
            $stats['savings_percent'] = round(($stats['total_savings'] / $total_original_size) * 100, 1);
        }
        
        return $stats;
    }
    
    /**
     * Clean up stale records from database where files no longer exist
     * 
     * @return int Number of records cleaned up
     */
    private static function cleanupStaleRecords(): int
    {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'zizi_image_converter_stats';
        $cleaned_count = 0;
        
        // Get all records
        $records = $wpdb->get_results("
            SELECT id, attachment_id, output_format 
            FROM $table_name
        ", ARRAY_A);
        
        foreach ($records as $record) {
            $attachment_id = (int) $record['attachment_id'];
            $format = $record['output_format'];
            
            // Check if original file exists
            $original_file = get_attached_file($attachment_id);
            if (!$original_file) {
                // Original attachment deleted
                $wpdb->delete($table_name, ['id' => $record['id']]);
                $cleaned_count++;
                continue;
            }
            
            // Check if converted file exists
            $converted_file = $original_file . '.' . $format;
            if (!file_exists($converted_file)) {
                // Converted file no longer exists
                $wpdb->delete($table_name, ['id' => $record['id']]);
                $cleaned_count++;
            }
        }
        
        return $cleaned_count;
    }
    
    /**
     * Get recent conversion activity
     * 
     * @param int $limit Number of recent conversions to return
     * @return array Recent conversions
     */
    public static function getRecentActivity(int $limit = 10): array
    {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'zizi_image_converter_stats';
        $cache_key = "zizi_image_converter_recent_{$limit}";
        $cached_activity = wp_cache_get($cache_key);
        
        if ($cached_activity !== false) {
            return $cached_activity;
        }
        
        try {
            // Check if table exists
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
            if (!$table_exists) {
                return [];
            }
            
            $results = $wpdb->get_results($wpdb->prepare(
                "SELECT attachment_id, output_format, size_savings, created_at 
                 FROM $table_name 
                 ORDER BY created_at DESC 
                 LIMIT %d",
                $limit
            ), ARRAY_A);
            
            // Cache for 2 minutes
            wp_cache_set($cache_key, $results, '', 120);
            
            return $results;
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("ImageConverterStats recent activity error: " . $e->getMessage(), 'ImageConverter');
            }
            return [];
        }
    }
    
    /**
     * Clear statistics cache
     */
    public static function clearCache(): void
    {
        // Clear WordPress object cache
        wp_cache_delete('zizi_image_converter_stats_v4');
        wp_cache_delete('zizi_image_converter_stats_v3'); // Keep for backward compatibility
        
        // Clear transients (persistent cache)
        delete_transient('zizi_image_converter_stats_v4');
        delete_transient('zizi_image_converter_stats_v3');
        delete_transient('zizi_image_converter_stats');
        
        // Clear recent activity cache (multiple limits)
        foreach ([5, 10, 20] as $limit) {
            wp_cache_delete("zizi_image_converter_recent_{$limit}");
            delete_transient("zizi_image_converter_recent_{$limit}");
        }
        
        // Force garbage collection for good measure
        if (function_exists('wp_cache_flush_group')) {
            wp_cache_flush_group('zizi_image_converter');
        }
        
        // Debug logging
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog("ImageConverterStats: Cache cleared for refresh", 'ImageConverter');
        }
    }

    /**
     * Force full cache clear and refresh statistics
     * Use this method when troubleshooting statistics issues
     */
    public static function forceRefresh(): array
    {
        // Clear all caches aggressively
        self::clearCache();
        
        // Clear any WordPress object cache
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
        
        // Force clear opcache if available
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog("ImageConverterStats: Force refresh initiated - all caches cleared", 'ImageConverter');
        }
        
        // Get fresh statistics with debug logging
        $fresh_stats = self::getStats();
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "ImageConverterStats: Force refresh complete - " .
                "WebP={$fresh_stats['webp_count']}, " .
                "AVIF={$fresh_stats['avif_count']}, " .
                "Total={$fresh_stats['total_conversions']}", 
                'ImageConverter'
            );
        }
        
        return $fresh_stats;
    }

    /**
     * Initialize stats tracking
     */
    public static function init(): void
    {
        // Clear cache when images are converted
        add_action('zizi_image_converted', [__CLASS__, 'clearCache']);
        
        // Clear cache daily
        add_action('wp_scheduled_delete', [__CLASS__, 'clearCache']);
    }
}
