<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Image Converter Frontend Serving Manager
 * 
 * Handles transparent serving of optimized images on the frontend
 * with browser compatibility detection and intelligent fallback.
 * Complements existing Image optimization features without conflicts.
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.4
 */
class ImageConverterFrontend
{
    /**
     * Browser support cache
     * 
     * @var array|null
     */
    private static $browser_support = null;

    /**
     * Initialize frontend hooks
     */
    public static function init(): void
    {
        // Only enable if frontend serving is enabled
        if (!self::is_frontend_serving_enabled()) {
            return;
        }

        // Content filtering hooks
        add_filter('wp_get_attachment_image_src', [__CLASS__, 'filter_attachment_image_src'], 10, 4);
        add_filter('wp_get_attachment_image_attributes', [__CLASS__, 'filter_attachment_image_attributes'], 10, 3);
        add_filter('the_content', [__CLASS__, 'filter_content_images'], 999);
        add_filter('post_thumbnail_html', [__CLASS__, 'filter_post_thumbnail'], 999);
        
        // Responsive images support
        add_filter('wp_calculate_image_srcset', [__CLASS__, 'filter_responsive_images'], 10, 5);
        
        // REST API image URLs
        add_filter('rest_prepare_attachment', [__CLASS__, 'filter_rest_attachment'], 10, 2);
    }

    /**
     * Filter attachment image source to serve optimized versions
     * 
     * @param array|false $image Image data array or false
     * @param int $attachment_id Attachment ID
     * @param string|int[] $size Image size
     * @param bool $icon Whether to use icon
     * @return array|false Modified image data
     */
    public static function filter_attachment_image_src($image, int $attachment_id, $size, bool $icon)
    {
        // Never modify images in admin area - always return originals
        if (is_admin()) {
            // Enhanced debugging for admin thumbnail issues
            if (WP_DEBUG && class_exists('\\ZiziCache\\CacheSys') && $image && !$icon) {
                $debug_key = "admin_thumb_debug_{$attachment_id}";
                if (!get_transient($debug_key)) {
                    $image_url = is_array($image) ? $image[0] : 'not-array';
                    $image_width = is_array($image) && isset($image[1]) ? $image[1] : 'unknown';
                    $image_height = is_array($image) && isset($image[2]) ? $image[2] : 'unknown';
                    \ZiziCache\CacheSys::writeLog(
                        "Admin thumbnail request for attachment {$attachment_id}: URL={$image_url}, Size={$image_width}x{$image_height}, RequestedSize=" . (is_array($size) ? implode('x', $size) : $size),
                        'ImageConverter'
                    );
                    set_transient($debug_key, true, MINUTE_IN_SECONDS * 5);
                }
            }
            return $image;
        }
        
        if (!$image || $icon || !wp_attachment_is_image($attachment_id)) {
            return $image;
        }

        $optimized_url = self::get_optimized_image_url($image[0], $attachment_id);
        if ($optimized_url !== $image[0]) {
            $image[0] = $optimized_url;
        }

        return $image;
    }

    /**
     * Filter attachment image attributes for optimized serving
     * 
     * @param array $attr Image attributes
     * @param \WP_Post $attachment Attachment post object
     * @param string|int[] $size Image size
     * @return array Modified attributes
     */
    public static function filter_attachment_image_attributes(array $attr, \WP_Post $attachment, $size): array
    {
        // Never modify images in admin area - always return originals
        if (is_admin()) {
            return $attr;
        }
        
        if (!wp_attachment_is_image($attachment->ID)) {
            return $attr;
        }

        // Update src if present
        if (!empty($attr['src'])) {
            $optimized_url = self::get_optimized_image_url($attr['src'], $attachment->ID);
            if ($optimized_url !== $attr['src']) {
                $attr['src'] = $optimized_url;
            }
        }

        return $attr;
    }

    /**
     * Filter content images to serve optimized versions
     * 
     * @param string $content Post content
     * @return string Modified content
     */
    public static function filter_content_images(string $content): string
    {
        if (empty($content) || is_admin() || is_feed()) {
            return $content;
        }

        // Find all img tags
        $pattern = '/<img([^>]*?)src=["\']([^"\']*?)["\']([^>]*?)>/i';
        
        return preg_replace_callback($pattern, function($matches) {
            $before_src = $matches[1];
            $src_url = $matches[2];
            $after_src = $matches[3];
            
            // Try to get attachment ID from URL
            $attachment_id = self::get_attachment_id_from_url($src_url);
            
            if ($attachment_id) {
                $optimized_url = self::get_optimized_image_url($src_url, $attachment_id);
                if ($optimized_url !== $src_url) {
                    return '<img' . $before_src . 'src="' . $optimized_url . '"' . $after_src . '>';
                }
            }
            
            return $matches[0]; // Return original if no optimization available
        }, $content);
    }

    /**
     * Filter post thumbnail HTML
     * 
     * @param string $html Thumbnail HTML
     * @return string Modified HTML
     */
    public static function filter_post_thumbnail(string $html): string
    {
        if (empty($html) || is_admin() || is_feed()) {
            return $html;
        }

        return self::filter_content_images($html);
    }

    /**
     * Filter responsive images srcset
     * 
     * @param array $sources Array of image sources
     * @param array $size_array Image size array
     * @param string $image_src Image source URL
     * @param array $image_meta Image metadata
     * @param int $attachment_id Attachment ID
     * @return array Modified sources array
     */
    public static function filter_responsive_images(array $sources, array $size_array, string $image_src, array $image_meta, int $attachment_id): array
    {
        // Never modify images in admin area - always return originals
        if (is_admin()) {
            return $sources;
        }
        
        if (empty($sources) || !wp_attachment_is_image($attachment_id)) {
            return $sources;
        }

        foreach ($sources as $width => &$source) {
            $optimized_url = self::get_optimized_image_url($source['url'], $attachment_id);
            if ($optimized_url !== $source['url']) {
                $source['url'] = $optimized_url;
            }
        }

        return $sources;
    }

    /**
     * Filter REST API attachment data
     * 
     * @param \WP_REST_Response $response Response object
     * @param \WP_Post $post Post object
     * @return \WP_REST_Response Modified response
     */
    public static function filter_rest_attachment(\WP_REST_Response $response, \WP_Post $post): \WP_REST_Response
    {
        // Never modify images in admin area - always return originals
        if (is_admin()) {
            return $response;
        }
        
        if (!wp_attachment_is_image($post->ID)) {
            return $response;
        }

        $data = $response->get_data();
        
        // Update media_details URLs
        if (!empty($data['media_details']['sizes'])) {
            foreach ($data['media_details']['sizes'] as $size => &$size_data) {
                if (!empty($size_data['source_url'])) {
                    $optimized_url = self::get_optimized_image_url($size_data['source_url'], $post->ID);
                    if ($optimized_url !== $size_data['source_url']) {
                        $size_data['source_url'] = $optimized_url;
                    }
                }
            }
        }

        // Update main source_url
        if (!empty($data['source_url'])) {
            $optimized_url = self::get_optimized_image_url($data['source_url'], $post->ID);
            if ($optimized_url !== $data['source_url']) {
                $data['source_url'] = $optimized_url;
            }
        }

        $response->set_data($data);
        return $response;
    }

    /**
     * Get optimized image URL based on browser support
     * 
     * @param string $original_url Original image URL
     * @param int $attachment_id Attachment ID
     * @return string Optimized URL or original if not available
     */
    public static function get_optimized_image_url(string $original_url, int $attachment_id): string
    {
        // IMPORTANT: Never serve optimized images in admin area to prevent 404 errors with thumbnails
        // Admin area should always show original images or native WordPress thumbnails
        if (is_admin()) {
            // Enhanced debugging for admin area optimization skipping
            if (WP_DEBUG && class_exists('\\ZiziCache\\CacheSys')) {
                $log_key = "admin_skip_logged_{$attachment_id}";
                if (!get_transient($log_key)) {
                    $parsed_url = parse_url($original_url);
                    $filename = basename($parsed_url['path'] ?? '');
                    \ZiziCache\CacheSys::writeLog(
                        "Skipping optimization in admin area for attachment {$attachment_id}: {$filename} (Full URL: {$original_url})",
                        'ImageConverter'
                    );
                    set_transient($log_key, true, MINUTE_IN_SECONDS * 5);
                }
            }
            return $original_url;
        }
        
        // Skip if disabled for frontend
        if (!self::is_frontend_serving_enabled()) {
            return $original_url;
        }
        
        // Check if image is optimized
        $optimization_meta = get_post_meta($attachment_id, '_zizi_image_optimized', true);
        if (empty($optimization_meta) || empty($optimization_meta['formats'])) {
            // Throttled logging for debugging - only log once per attachment per session to prevent log spam
            if (WP_DEBUG && class_exists('\\ZiziCache\\CacheSys')) {
                $log_key = "no_meta_logged_{$attachment_id}";
                if (!get_transient($log_key)) {
                    \ZiziCache\CacheSys::writeLog("No optimization metadata for attachment {$attachment_id}, URL: {$original_url}", 'ImageConverter');
                    // Set transient for 1 hour to prevent repeated logging of same attachment
                    set_transient($log_key, true, HOUR_IN_SECONDS);
                }
            }
            return $original_url;
        }

        // Get browser support with enhanced detection
        $browser_support = self::get_browser_support();
        
        // Determine best format based on browser support and availability
        $preferred_format = null;
        $available_formats = $optimization_meta['formats'];
        
        // Priority: AVIF > WebP > Original
        if ($browser_support['avif'] && in_array('avif', $available_formats)) {
            $preferred_format = 'avif';
        } elseif ($browser_support['webp'] && in_array('webp', $available_formats)) {
            $preferred_format = 'webp';
        }

        if (!$preferred_format) {
            // Log browser support info for debugging
            if (WP_DEBUG && class_exists('\\ZiziCache\\CacheSys')) {
                $debug_info = sprintf(
                    "No supported format for attachment %d. Browser support: AVIF=%s, WebP=%s, Available: %s",
                    $attachment_id,
                    $browser_support['avif'] ? 'yes' : 'no',
                    $browser_support['webp'] ? 'yes' : 'no',
                    implode(', ', $available_formats)
                );
                \ZiziCache\CacheSys::writeLog($debug_info, 'ImageConverter');
            }
            return $original_url; // No supported optimized format
        }

        // Build optimized URL - check if it already has extension
        if (strpos($original_url, '.' . $preferred_format) !== false) {
            return $original_url; // Already optimized URL
        }
        
        $optimized_url = $original_url . '.' . $preferred_format;
        
        // Verify file exists before serving
        if (self::optimized_file_exists($optimized_url, $attachment_id, $preferred_format)) {
            // Update serving statistics
            self::update_serving_statistics($preferred_format);

            // Log successful optimization for debugging / monitoring.
            // Rate-limit to avoid log flooding on high-traffic sites. Logging occurs when:
            //  - explicit debug logging is enabled via config 'image_converter_debug_logging'
            //  - OR WP_DEBUG is enabled but the per-attachment transient hasn't been set yet (sampled log)
            if (class_exists('\ZiziCache\\CacheSys')) {
                $debug_log_enabled = (\ZiziCache\SysConfig::$config['image_converter_debug_logging'] ?? false) === true;
                $serve_log_ttl = max(0, (int)(\ZiziCache\SysConfig::$config['image_converter_serve_log_ttl'] ?? HOUR_IN_SECONDS));
                $transient_key = "zizi_img_serve_log_{$attachment_id}_{$preferred_format}";

                if ($debug_log_enabled) {
                    \ZiziCache\CacheSys::writeLog("Serving {$preferred_format} for attachment {$attachment_id}: {$optimized_url}", 'ImageConverter');
                } elseif (WP_DEBUG && $serve_log_ttl > 0 && !get_transient($transient_key)) {
                    // Sampled log while in WP_DEBUG to help debugging without flooding logs
                    \ZiziCache\CacheSys::writeLog("Serving {$preferred_format} for attachment {$attachment_id}: {$optimized_url}", 'ImageConverter');
                    set_transient($transient_key, 1, $serve_log_ttl);
                }
            }
            
            return $optimized_url;
        }

        // File doesn't exist, update fallback statistics and return original
        self::update_serving_statistics('fallback');

        // Log fallback with the same rate-limiting rules as successful serves
        if (class_exists('\ZiziCache\\CacheSys')) {
            $debug_log_enabled = (\ZiziCache\SysConfig::$config['image_converter_debug_logging'] ?? false) === true;
            $serve_log_ttl = max(0, (int)(\ZiziCache\SysConfig::$config['image_converter_serve_log_ttl'] ?? HOUR_IN_SECONDS));
            $transient_key = "zizi_img_serve_log_{$attachment_id}_fallback";

            if ($debug_log_enabled) {
                \ZiziCache\CacheSys::writeLog("Fallback to original for attachment {$attachment_id}: optimized file not found ({$preferred_format})", 'ImageConverter');
            } elseif (WP_DEBUG && $serve_log_ttl > 0 && !get_transient($transient_key)) {
                \ZiziCache\CacheSys::writeLog("Fallback to original for attachment {$attachment_id}: optimized file not found ({$preferred_format})", 'ImageConverter');
                set_transient($transient_key, 1, $serve_log_ttl);
            }
        }

        return $original_url;
    }

    /**
     * Get browser support for modern image formats
     * 
     * Uses multiple detection methods for better reliability:
     * - HTTP Accept header (server-side)
     * - Query parameter override (for testing)
     * - User agent string analysis (fallback)
     * 
     * @return array Browser support information
     */
    public static function get_browser_support(): array
    {
        if (self::$browser_support !== null) {
            return self::$browser_support;
        }

        $accept_header = $_SERVER['HTTP_ACCEPT'] ?? '';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Allow query parameter override for testing
        $force_format = $_GET['zizi_force_format'] ?? '';
        if (in_array($force_format, ['avif', 'webp', 'original'])) {
            self::$browser_support = [
                'avif' => $force_format === 'avif',
                'webp' => in_array($force_format, ['avif', 'webp']),
                'debug_mode' => true,
                'forced_format' => $force_format
            ];
            return self::$browser_support;
        }
        
        // Primary detection via Accept header
        $avif_support = strpos($accept_header, 'image/avif') !== false;
        $webp_support = strpos($accept_header, 'image/webp') !== false;
        
        // Fallback detection via User Agent for better compatibility
        if (!$webp_support && $user_agent) {
            // Chrome 23+, Edge 18+, Firefox 65+, Safari 14+ support WebP
            $webp_support = preg_match('/Chrome\/([2-9]\d|[1-9]\d{2,})|Edge\/([1-9]\d|[2-9]\d{2,})|Firefox\/([6-9]\d|[1-9]\d{2,})|Version\/1[4-9].*Safari/', $user_agent);
        }
        
        if (!$avif_support && $user_agent) {
            // Chrome 85+, Edge 121+, Firefox 93+ support AVIF
            $avif_support = preg_match('/Chrome\/(8[5-9]|9\d|[1-9]\d{2,})|Edge\/(12[1-9]|1[3-9]\d|[2-9]\d{2,})|Firefox\/(9[3-9]|[1-9]\d{2,})/', $user_agent);
        }

        self::$browser_support = [
            'avif' => $avif_support,
            'webp' => $webp_support,
            'accept_header' => $accept_header,
            'user_agent' => substr($user_agent, 0, 100), // Limited for privacy
            'detection_method' => $accept_header ? 'accept_header' : 'user_agent',
            'debug_mode' => false
        ];

        return self::$browser_support;
    }

    /**
     * Check if optimized file exists
     * 
     * Uses multiple methods to verify file existence:
     * 1. Direct file path check (fastest)
     * 2. URL to path conversion
     * 3. WordPress attachment metadata check
     * 
     * @param string $optimized_url Optimized image URL
     * @param int $attachment_id Attachment ID
     * @param string $format Image format (avif, webp)
     * @return bool Whether file exists
     */
    private static function optimized_file_exists(string $optimized_url, int $attachment_id, string $format): bool
    {
        // Method 1: Direct path check using attachment metadata
        if ($attachment_id > 0) {
            $original_file = get_attached_file($attachment_id);
            if ($original_file) {
                $optimized_file = $original_file . '.' . $format;
                if (file_exists($optimized_file) && is_readable($optimized_file)) {
                    return true;
                }
            }
        }
        
        // Method 2: URL to path conversion
        $upload_dir = wp_upload_dir();
        if (!empty($upload_dir['baseurl']) && !empty($upload_dir['basedir'])) {
            // Handle potential URL encoding and normalize
            $optimized_url = urldecode($optimized_url);
            $file_path = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $optimized_url);
            
            // Normalize path separators for Windows compatibility
            $file_path = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $file_path);
            
            if (file_exists($file_path) && is_readable($file_path)) {
                return true;
            }
        }
        
        // Method 3: Alternative URL parsing (for edge cases)
        $parsed_url = parse_url($optimized_url);
        if (!empty($parsed_url['path'])) {
            $relative_path = ltrim($parsed_url['path'], '/');
            $full_path = ABSPATH . $relative_path;
            
            if (file_exists($full_path) && is_readable($full_path)) {
                return true;
            }
        }

        // Log for debugging if file not found
        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
            \ZiziCache\CacheSys::writeLog("Optimized file not found: {$optimized_url} (format: {$format}, attachment: {$attachment_id})", 'ImageConverter');
        }

        return false;
    }

    /**
     * Get attachment ID from image URL
     * 
     * @param string $url Image URL
     * @return int|null Attachment ID or null if not found
     */
    private static function get_attachment_id_from_url(string $url): ?int
    {
        // Remove query strings and fragments
        $url = strtok($url, '?');
        $url = strtok($url, '#');
        
        // Try WordPress function first
        $attachment_id = attachment_url_to_postid($url);
        if ($attachment_id) {
            return $attachment_id;
        }

        // Fallback: Try without size suffix
        $url_without_size = preg_replace('/-\d+x\d+(?=\.[a-z]{3,4}$)/i', '', $url);
        if ($url_without_size !== $url) {
            $attachment_id = attachment_url_to_postid($url_without_size);
            if ($attachment_id) {
                return $attachment_id;
            }
        }

        return null;
    }

    /**
     * Check if frontend serving is enabled
     * 
     * @return bool Whether frontend serving is enabled
     */
    private static function is_frontend_serving_enabled(): bool
    {
        if (!ImageConverter::is_enabled()) {
            return false;
        }

        $config = SysConfig::$config;
        return !empty($config['image_converter_frontend_serve']);
    }

    /**
     * Get optimization statistics for monitoring
     * 
     * @return array Statistics data
     */
    public static function get_serving_statistics(): array
    {
        $stats = get_transient('zizi_image_converter_serving_stats') ?: [
            'total_requests' => 0,
            'optimized_served' => 0,
            'avif_served' => 0,
            'webp_served' => 0,
            'fallback_served' => 0,
            'last_reset' => current_time('mysql')
        ];

        return $stats;
    }

    /**
     * Update serving statistics
     * 
     * @param string $type Type of serving: 'avif', 'webp', 'fallback'
     */
    public static function update_serving_statistics(string $type): void
    {
        $stats = self::get_serving_statistics();
        
        $stats['total_requests']++;
        
        switch ($type) {
            case 'avif':
                $stats['optimized_served']++;
                $stats['avif_served']++;
                break;
            case 'webp':
                $stats['optimized_served']++;
                $stats['webp_served']++;
                break;
            case 'fallback':
                $stats['fallback_served']++;
                break;
        }

        set_transient('zizi_image_converter_serving_stats', $stats, 24 * HOUR_IN_SECONDS);
    }

    /**
     * Reset serving statistics
     */
    public static function reset_serving_statistics(): void
    {
        delete_transient('zizi_image_converter_serving_stats');
    }

    /**
     * Get diagnostic information for troubleshooting
     * 
     * @return array Comprehensive diagnostic data
     */
    public static function get_diagnostic_info(): array
    {
        $config = SysConfig::$config;
        $browser_support = self::get_browser_support();
        $stats = self::get_serving_statistics();
        
        return [
            'image_converter_enabled' => ImageConverter::is_enabled(),
            'frontend_serving_enabled' => self::is_frontend_serving_enabled(),
            'htaccess_rules_enabled' => !empty($config['image_converter_htaccess_rules']),
            'enabled_formats' => $config['image_converter_formats'] ?? [],
            'browser_support' => $browser_support,
            'serving_statistics' => $stats,
            'server_info' => [
                'http_accept' => $_SERVER['HTTP_ACCEPT'] ?? 'not_set',
                'user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? 'not_set', 0, 100),
                'request_uri' => $_SERVER['REQUEST_URI'] ?? 'not_set',
                'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'not_set'
            ],
            'environment' => ImageConverter::get_environment_info(),
            'wp_debug' => WP_DEBUG,
            'is_admin' => is_admin(),
            'upload_dir' => wp_upload_dir()
        ];
    }

    /**
     * Test image conversion for a specific attachment
     * 
     * @param int $attachment_id Attachment ID to test
     * @return array Test results
     */
    public static function test_attachment_conversion(int $attachment_id): array
    {
        if (!wp_attachment_is_image($attachment_id)) {
            return ['error' => 'Not a valid image attachment'];
        }
        
        $original_url = wp_get_attachment_url($attachment_id);
        $optimization_meta = get_post_meta($attachment_id, '_zizi_image_optimized', true);
        $browser_support = self::get_browser_support();
        
        $results = [
            'attachment_id' => $attachment_id,
            'original_url' => $original_url,
            'optimization_meta' => $optimization_meta,
            'browser_support' => $browser_support,
            'tests' => []
        ];
        
        // Test each available format
        if (!empty($optimization_meta['formats'])) {
            foreach ($optimization_meta['formats'] as $format) {
                $optimized_url = $original_url . '.' . $format;
                $file_exists = self::optimized_file_exists($optimized_url, $attachment_id, $format);
                
                $results['tests'][$format] = [
                    'optimized_url' => $optimized_url,
                    'file_exists' => $file_exists,
                    'browser_supported' => $browser_support[$format] ?? false,
                    'would_serve' => $file_exists && ($browser_support[$format] ?? false)
                ];
            }
        }
        
        // Test what would actually be served
        $served_url = self::get_optimized_image_url($original_url, $attachment_id);
        $results['final_served_url'] = $served_url;
        $results['optimization_applied'] = $served_url !== $original_url;
        
        return $results;
    }
}
