<?php

namespace ZiziCache;

/**
 * Enhanced Image Converter Configuration
 * Configuration flags for enhanced image converter features
 */
class ImageConverterEnhancedConfig
{
    /**
     * Check if enhanced mode is enabled
     * Can be controlled via wp-config.php or admin setting
     */
    public static function isEnhancedModeEnabled(): bool
    {
        // Check wp-config.php constant first (override for developers)
        if (defined('ZIZI_IMAGE_CONVERTER_ENHANCED_MODE')) {
            return (bool) ZIZI_IMAGE_CONVERTER_ENHANCED_MODE;
        }

        // Check admin setting
        $config = SysConfig::$config ?? [];
        return $config['image_converter_enhanced_mode'] ?? false;
    }

    /**
     * Check if enhanced UI is enabled
     */
    public static function isEnhancedUIEnabled(): bool
    {
        // Enhanced UI only works if enhanced mode is enabled
        if (!self::isEnhancedModeEnabled()) {
            return false;
        }

        // Check specific UI setting
        $config = SysConfig::$config ?? [];
        return $config['image_converter_enhanced_ui'] ?? false;
    }

    /**
     * Check if debug logging is enabled for enhanced features
     */
    public static function isDebugLoggingEnabled(): bool
    {
        $config = SysConfig::$config ?? [];
        return $config['image_converter_enhanced_debug'] ?? WP_DEBUG;
    }

    /**
     * Get feature flags for frontend
     */
    public static function getFeatureFlags(): array
    {
        return [
            'enhanced_mode' => self::isEnhancedModeEnabled(),
            'enhanced_ui' => self::isEnhancedUIEnabled(),
            'debug_logging' => self::isDebugLoggingEnabled()
        ];
    }

    /**
     * Log enhanced feature usage
     */
    public static function logFeatureUsage(string $feature, array $data = []): void
    {
        if (!self::isDebugLoggingEnabled()) {
            return;
        }

        if (class_exists('\\ZiziCache\\CacheSys')) {
            $message = "Enhanced feature '$feature' used";
            if (!empty($data)) {
                $message .= ' with data: ' . wp_json_encode($data);
            }
            \ZiziCache\CacheSys::writeLog($message, 'ImageConverterEnhanced');
        }
    }
}