<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Image Converter Bulk Processing Manager
 * 
 * Handles bulk image optimization using Action Scheduler for robust
 * background processing with progress tracking, error handling, and
 * load balancing capabilities.
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.4
 */
class ImageConverterBulk
{
    /**
     * Action Scheduler hook name for single image processing
     */
    private const SINGLE_ACTION_HOOK = 'zizi_image_converter_process_single';

    /**
     * Action Scheduler group name
     */
    private const ACTION_GROUP = 'zizi-image-converter';

    /**
     * Transient key for bulk optimization status
     */
    private const STATUS_TRANSIENT = 'zizi_bulk_optimization_running';

    /**
     * Transient key for bulk statistics
     */
    private const STATS_TRANSIENT = 'zizi_bulk_optimization_stats';

    /**
     * Default maximum batch size (fallback)
     */
    private const DEFAULT_MAX_BATCH_SIZE = 25;

    /**
     * Maximum retry attempts for failed conversions
     */
    private const MAX_RETRY_ATTEMPTS = 3;
    
    /**
     * Default time interval between batches (fallback)
     */
    private const DEFAULT_BATCH_DELAY = 120;

    /**
     * Get optimal batch size based on server performance and total images
     * 
     * @param int $total_images Total number of images to process
     * @return int Optimal batch size
     */
    private static function get_optimal_batch_size(int $total_images): int
    {
        $config = SysConfig::$config ?? [];
        
        // Check for manual override
        $manual_batch_size = $config['bulk_max_batch_size'] ?? 0;
        if ($manual_batch_size > 0) {
            return $manual_batch_size;
        }
        
        // Use performance profiler for auto-detection
        if (class_exists('\\ZiziCache\\PerformanceProfiler')) {
            $hosting_type = $config['bulk_performance_mode'] ?? 'auto';
            if ($hosting_type === 'auto') {
                $hosting_type = null; // Let profiler auto-detect
            }
            
            return \ZiziCache\PerformanceProfiler::get_optimal_batch_size($total_images, $hosting_type);
        }
        
        // Fallback to legacy logic
        return $total_images > 1000 ? min(self::DEFAULT_MAX_BATCH_SIZE, 15) : self::DEFAULT_MAX_BATCH_SIZE;
    }
    
    /**
     * Get optimal batch delay based on server performance
     * 
     * @return int Optimal delay in seconds
     */
    private static function get_optimal_batch_delay(): int
    {
        $config = SysConfig::$config ?? [];
        
        // Check for manual override
        $manual_delay = $config['bulk_batch_delay'] ?? 0;
        if ($manual_delay > 0) {
            return $manual_delay;
        }
        
        // Use performance profiler for auto-detection
        if (class_exists('\\ZiziCache\\PerformanceProfiler')) {
            $hosting_type = $config['bulk_performance_mode'] ?? 'auto';
            if ($hosting_type === 'auto') {
                $hosting_type = null; // Let profiler auto-detect
            }
            
            return \ZiziCache\PerformanceProfiler::get_optimal_batch_delay($hosting_type);
        }
        
        // Fallback to default
        return self::DEFAULT_BATCH_DELAY;
    }
    
    /**
     * Check if server can handle additional load
     * 
     * @return bool True if can handle more load
     */
    private static function can_handle_more_load(): bool
    {
        $config = SysConfig::$config ?? [];
        
        // Use performance profiler if available
        if (class_exists('\\ZiziCache\\PerformanceProfiler')) {
            $hosting_type = $config['bulk_performance_mode'] ?? 'auto';
            if ($hosting_type === 'auto') {
                $hosting_type = null; // Let profiler auto-detect
            }
            
            return \ZiziCache\PerformanceProfiler::can_handle_more_load($hosting_type);
        }
        
        // Fallback to basic memory check
        $memory_threshold = $config['bulk_memory_threshold'] ?? 80;
        $memory_limit = ini_get('memory_limit');
        if ($memory_limit === '-1') {
            return true; // Unlimited memory
        }
        
        $memory_limit_bytes = self::convert_to_bytes($memory_limit);
        $memory_usage = memory_get_usage(true);
        $memory_usage_percent = ($memory_usage / $memory_limit_bytes) * 100;
        
        return $memory_usage_percent < $memory_threshold;
    }

    /**
     * Initialize Action Scheduler hooks
     */
    public static function init(): void
    {
        // Register Action Scheduler hooks
        add_action(self::SINGLE_ACTION_HOOK, [__CLASS__, 'process_single_image']);
        add_action('action_scheduler_failed_action', [__CLASS__, 'handle_failed_action'], 10, 2);
        add_action('action_scheduler_completed_action', [__CLASS__, 'handle_completed_action'], 10, 2);
        
        // Cleanup hooks
        add_action('wp_head', [__CLASS__, 'cleanup_completed_optimization']);
        add_action('admin_init', [__CLASS__, 'cleanup_stale_optimizations']);
    }

    /**
     * Start bulk image optimization process
     * 
     * @param array $options Optimization options
     * @return array Operation result
     */
    public static function start_bulk_optimization(array $options = []): array
    {
        try {
            // Check if already running
            if (self::is_optimization_running()) {
                return [
                    'success' => false,
                    'message' => 'Bulk optimization is already running',
                    'code' => 'ALREADY_RUNNING'
                ];
            }

            // Check if Image Converter is enabled
            if (!ImageConverter::is_enabled()) {
                return [
                    'success' => false,
                    'message' => 'Image Converter is not enabled',
                    'code' => 'NOT_ENABLED'
                ];
            }

            // Get images that need optimization (either no metadata or missing files)
            // Include both standard WordPress media library and zizi-cache directory
            $query_args = [
                'post_type' => 'attachment',
                'post_status' => 'inherit',
                'posts_per_page' => -1,
                'post_mime_type' => ['image/jpeg', 'image/png', 'image/gif']
            ];

            $query = new \WP_Query($query_args);
            $attachments = $query->posts;
            
            // Also scan zizi-cache directory for thumbnail images
            $uploads_dir = wp_upload_dir()['basedir'];
            $zizi_cache_dir = $uploads_dir . '/zizi-cache';
            $zizi_cache_attachments = [];
            
            if (is_dir($zizi_cache_dir)) {
                $iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator($zizi_cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                    \RecursiveIteratorIterator::LEAVES_ONLY
                );
                
                foreach ($iterator as $file) {
                    if ($file->isFile()) {
                        $extension = strtolower(pathinfo($file->getPathname(), PATHINFO_EXTENSION));
                        // Only include original formats, not already converted
                        if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
                            // Create pseudo attachment object for zizi-cache images
                            $zizi_cache_attachments[] = (object)[
                                'ID' => 'zizi_cache_' . md5($file->getPathname()),
                                'file_path' => $file->getPathname(),
                                'is_zizi_cache' => true
                            ];
                        }
                    }
                }
            }
            
            // Also scan uploads directory for orphan images (not in Media Library)
            $orphan_attachments = [];
            if (is_dir($uploads_dir)) {
                $uploads_iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator($uploads_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                    \RecursiveIteratorIterator::LEAVES_ONLY
                );
                
                // Get list of existing attachment file paths for comparison
                $existing_attachment_paths = [];
                foreach ($attachments as $attachment) {
                    $file_path = get_attached_file($attachment->ID);
                    if ($file_path) {
                        $existing_attachment_paths[] = $file_path;
                    }
                }
                
                foreach ($uploads_iterator as $file) {
                    if ($file->isFile()) {
                        $file_path = $file->getPathname();
                        $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
                        
                        // Skip already processed zizi-cache files
                        if (strpos($file_path, '/zizi-cache/') !== false) {
                            continue;
                        }
                        
                        // Only include original image formats
                        if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
                            // Check if this file is NOT already in WordPress attachments
                            if (!in_array($file_path, $existing_attachment_paths)) {
                                // This is an orphan file - not registered in WP
                                $orphan_attachments[] = (object)[
                                    'ID' => 'orphan_' . md5($file_path),
                                    'file_path' => $file_path,
                                    'is_orphan' => true
                                ];
                            }
                        }
                    }
                }
            }
            
            // Combine WordPress attachments, zizi-cache images, and orphan files
            $all_attachments = array_merge($attachments, $zizi_cache_attachments, $orphan_attachments);

            if (empty($all_attachments)) {
                return [
                    'success' => false,
                    'message' => 'No images found for optimization',
                    'code' => 'NO_IMAGES'
                ];
            }

            // Filter supported images and check if they need optimization
            $supported_images = [];
            
            // Ensure SysConfig is initialized before accessing configuration
            if (empty(SysConfig::$config)) {
                SysConfig::init();
            }
            
            $config = SysConfig::$config;
            $enabled_formats = $options['enabled_formats'] ?? ($config['image_converter_formats'] ?? ['webp']);
            
            // Validate enabled_formats - ensure it's not empty
            if (empty($enabled_formats) || !is_array($enabled_formats)) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        'Bulk optimization warning: enabled_formats is empty or invalid. Using WebP as fallback.',
                        'ImageConverterBulk'
                    );
                }
                $enabled_formats = ['webp']; // Safe fallback
            }
            
            foreach ($all_attachments as $attachment) {
                // Handle zizi-cache images differently
                if (isset($attachment->is_zizi_cache)) {
                    $file_path = $attachment->file_path;
                    if (file_exists($file_path)) {
                        // Check if this zizi-cache image needs optimization
                        if (self::zizi_cache_image_needs_optimization($file_path, $enabled_formats)) {
                            $supported_images[] = $attachment->ID;
                        }
                    }
                    continue;
                }
                
                // Handle orphan images (not in Media Library)
                if (isset($attachment->is_orphan)) {
                    $file_path = $attachment->file_path;
                    if (file_exists($file_path)) {
                        // Check if this orphan image needs optimization
                        if (self::zizi_cache_image_needs_optimization($file_path, $enabled_formats)) {
                            $supported_images[] = $attachment->ID;
                        }
                    }
                    continue;
                }
                
                // Handle regular WordPress attachments
                if (wp_attachment_is_image($attachment->ID)) {
                    $mime_type = get_post_mime_type($attachment->ID);
                    
                    // Additional check for real file MIME type to catch database inconsistencies
                    $file_path = get_attached_file($attachment->ID);
                    if ($file_path && file_exists($file_path)) {
                        $file_extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
                        
                        // If file has .webp or .avif extension, treat as already converted
                        if (in_array($file_extension, ['webp', 'avif'])) {
                            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                                \ZiziCache\CacheSys::writeLog(
                                    "Skipping attachment {$attachment->ID} - file extension '{$file_extension}' indicates already converted format, despite DB MIME type '{$mime_type}'",
                                    'ImageConverterBulk'
                                );
                            }
                            continue; // Skip already converted files
                        }
                    }
                    
                    // Only process original image formats, not already converted ones
                    if (ImageConverter::is_supported_format($mime_type) && !self::is_already_converted_format($mime_type)) {
                        // Check if this image needs optimization
                        if (self::image_needs_optimization($attachment->ID, $enabled_formats)) {
                            $supported_images[] = $attachment->ID;
                        }
                    }
                }
            }

            if (empty($supported_images)) {
                return [
                    'success' => false,
                    'message' => 'No images found that need optimization',
                    'code' => 'NO_IMAGES_NEED_OPTIMIZATION'
                ];
            }

            // Schedule actions in batches for load balancing
            $scheduled_count = self::schedule_bulk_actions($supported_images, $enabled_formats);

            // Set running status
            $status_data = [
                'started_at' => current_time('mysql'),
                'total_images' => count($supported_images),
                'scheduled_images' => $scheduled_count,
                'options' => $options,
                'status' => 'running'
            ];

            set_transient(self::STATUS_TRANSIENT, $status_data, 24 * HOUR_IN_SECONDS);

            // Initialize statistics
            self::init_bulk_statistics($status_data);

            // Log start
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Bulk optimization started: {$scheduled_count} images scheduled from {$status_data['total_images']} total",
                    'ImageConverterBulk'
                );
            }

            return [
                'success' => true,
                'message' => 'Bulk optimization started successfully',
                'total_images' => $status_data['total_images'],
                'scheduled_images' => $scheduled_count
            ];

        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    'Failed to start bulk optimization: ' . $e->getMessage(),
                    'ImageConverterBulk'
                );
            }

            return [
                'success' => false,
                'message' => 'Failed to start bulk optimization: ' . $e->getMessage(),
                'code' => 'START_FAILED'
            ];
        }
    }

    /**
     * Schedule bulk actions with load balancing
     * 
     * @param array $image_ids Array of attachment IDs
     * @param array $enabled_formats Array of enabled image formats
     * @return int Number of scheduled actions
     */
    private static function schedule_bulk_actions(array $image_ids, array $enabled_formats = []): int
    {
        // Check if Action Scheduler exists
        if (!function_exists('as_schedule_single_action')) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Bulk scheduling failed: Action Scheduler not available",
                    'ImageConverterBulk'
                );
            }
            return 0;
        }
        
        $scheduled_count = 0;
        $batch_delay = 0;

        // Get number of images for better planning
        $total_images = count($image_ids);
        
        // Get optimal batch size based on server performance
        $effective_batch_size = self::get_optimal_batch_size($total_images);
        
        // Get optimal delay between batches
        $batch_delay_increment = self::get_optimal_batch_delay();
            
        // Process in batches to avoid server overload
        $batches = array_chunk($image_ids, $effective_batch_size);
        $batch_count = count($batches);
        
        // Log planning information
        if (class_exists('\\ZiziCache\\CacheSys')) {
            $config = SysConfig::$config ?? [];
            $performance_logging = $config['bulk_performance_logging'] ?? false;
            
            \ZiziCache\CacheSys::writeLog(
                "Planning bulk optimization: {$total_images} images in {$batch_count} batches (batch_size: {$effective_batch_size}, delay: {$batch_delay_increment}s)",
                'ImageConverterBulk'
            );
            
            // Log performance details if enabled
            if ($performance_logging && class_exists('\\ZiziCache\\PerformanceProfiler')) {
                $server_analysis = \ZiziCache\PerformanceProfiler::get_server_analysis();
                \ZiziCache\CacheSys::writeLog(
                    "Performance analysis: " . json_encode($server_analysis),
                    'ImageConverterBulk'
                );
            }
        }

        foreach ($batches as $batch_index => $batch) {
            foreach ($batch as $image_index => $attachment_id) {
                // Adaptive delay calculation for load distribution
                $base_delay = $batch_index * $batch_delay_increment; // Use dynamic delay between batches
                
                // Larger random delay spread for bigger batches
                $max_random_delay = min(45, 15 + ($total_images / 1000));
                $random_delay = rand(1, (int)$max_random_delay); 
                $total_delay = $base_delay + $random_delay;

                // Schedule single action
                $scheduled = as_schedule_single_action(
                    time() + $total_delay,
                    self::SINGLE_ACTION_HOOK,
                    [$attachment_id, $enabled_formats],
                    self::ACTION_GROUP
                );

                if ($scheduled) {
                    $scheduled_count++;
                }
            }
        }

        return $scheduled_count;
    }

    /**
     * Process single image in Action Scheduler
     * 
     * @param int $attachment_id Attachment ID
     * @param array $enabled_formats Array of enabled image formats
     */
    public static function process_single_image($attachment_id, array $enabled_formats = []): void
    {
    // Determine whether verbose per-item logging should be enabled for bulk runs.
    // Controlled via SysConfig::$config['image_converter_debug_logging'] (bool)
    // and sampling limit via SysConfig::$config['image_converter_log_sample_limit'] (int).
    $debug_log_enabled = (\ZiziCache\SysConfig::$config['image_converter_debug_logging'] ?? false) === true;
    $log_sample_limit = max(1, (int)(\ZiziCache\SysConfig::$config['image_converter_log_sample_limit'] ?? 10));
    $log_samples = get_transient('zizi_bulk_log_samples') ?: ['success' => 0, 'skipped' => 0, 'failed' => 0, 'reschedule' => 0];

        // Check server load before processing using performance profiler
        if (!self::can_handle_more_load()) {
            // Check how many times this image has been rescheduled to prevent infinite loops
            $reschedule_count = get_transient("zizi_reschedule_count_{$attachment_id}") ?: 0;
            
            if ($reschedule_count >= 3) {
                // Too many reschedules, mark as failed to prevent infinite loop
                self::update_bulk_statistics('failed');
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Image {$attachment_id} processing failed: Too many reschedules due to server overload",
                        'ImageConverterBulk'
                    );
                }
                return;
            }
            
            // Increment reschedule counter
            set_transient("zizi_reschedule_count_{$attachment_id}", $reschedule_count + 1, HOUR_IN_SECONDS);
            
            // Reschedule the job with a small delay to allow server recovery
            as_schedule_single_action(
                time() + rand(300, 600), // 5-10 minute delay
                self::SINGLE_ACTION_HOOK,
                [$attachment_id, $enabled_formats],
                self::ACTION_GROUP
            );
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                // Only write reschedule logs when debug is enabled or sample limit not reached
                if ($debug_log_enabled || ($log_samples['reschedule'] < $log_sample_limit)) {
                    \ZiziCache\CacheSys::writeLog(
                        "Server is overloaded, processing of image {$attachment_id} postponed (attempt " . ($reschedule_count + 2) . "/4)",
                        'ImageConverterBulk'
                    );
                    if (!$debug_log_enabled) {
                        $log_samples['reschedule']++;
                        set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                    }
                }
            }
            return;
        }
        
        // Clear reschedule counter if processing starts normally
        delete_transient("zizi_reschedule_count_{$attachment_id}");
        
        try {
            // Measure processing time for statistics
            $start_time = microtime(true);
            
            // Check if this is a zizi-cache image, orphan image, or regular WordPress attachment
            $is_zizi_cache = is_string($attachment_id) && strpos($attachment_id, 'zizi_cache_') === 0;
            $is_orphan = is_string($attachment_id) && strpos($attachment_id, 'orphan_') === 0;
            
            if ($is_zizi_cache) {
                // Handle zizi-cache images
                // Find the actual file path for this zizi-cache hash
                $uploads_dir = wp_upload_dir()['basedir'];
                $zizi_cache_dir = $uploads_dir . '/zizi-cache';
                $file_path = null;
                
                if (is_dir($zizi_cache_dir)) {
                    $iterator = new \RecursiveIteratorIterator(
                        new \RecursiveDirectoryIterator($zizi_cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                        \RecursiveIteratorIterator::LEAVES_ONLY
                    );
                    
                    $target_hash = str_replace('zizi_cache_', '', $attachment_id);
                    
                    foreach ($iterator as $file) {
                        if ($file->isFile()) {
                            $file_hash = md5($file->getPathname());
                            if ($file_hash === $target_hash) {
                                $file_path = $file->getPathname();
                                break;
                            }
                        }
                    }
                }
                
                if (!$file_path || !file_exists($file_path)) {
                    self::update_bulk_statistics('skipped');
                    throw new \Exception("ZiziCache image file not found for hash: {$attachment_id}");
                }
                
            } elseif ($is_orphan) {
                // Handle orphan images (not in Media Library)
                // Find the actual file path for this orphan hash
                $uploads_dir = wp_upload_dir()['basedir'];
                $file_path = null;
                
                if (is_dir($uploads_dir)) {
                    $iterator = new \RecursiveIteratorIterator(
                        new \RecursiveDirectoryIterator($uploads_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                        \RecursiveIteratorIterator::LEAVES_ONLY
                    );
                    
                    $target_hash = str_replace('orphan_', '', $attachment_id);
                    
                    foreach ($iterator as $file) {
                        if ($file->isFile()) {
                            $file_hash = md5($file->getPathname());
                            if ($file_hash === $target_hash) {
                                $file_path = $file->getPathname();
                                break;
                            }
                        }
                    }
                }
                
                if (!$file_path || !file_exists($file_path)) {
                    self::update_bulk_statistics('skipped');
                    throw new \Exception("Orphan image file not found for hash: {$attachment_id}");
                }
                
            } else {
                // Handle regular WordPress attachments
                $attachment_id = (int)$attachment_id; // Ensure integer type
                
                // Verify image still exists and is valid
                if (!wp_attachment_is_image($attachment_id)) {
                    self::update_bulk_statistics('skipped');
                    throw new \Exception("Invalid image attachment: {$attachment_id}");
                }

                $file_path = get_attached_file($attachment_id);
                if (!$file_path || !file_exists($file_path)) {
                    self::update_bulk_statistics('skipped');
                    throw new \Exception("Image file not found for attachment {$attachment_id}: {$file_path}");
                }
            }

            // Additional file accessibility check
            if (!is_readable($file_path)) {
                self::update_bulk_statistics('failed');
                throw new \Exception("Image file not readable for attachment {$attachment_id}: {$file_path}");
            }

            // Check if already optimized for current formats
            // Only check metadata for WordPress attachments, not zizi-cache or orphan images
            if (!$is_zizi_cache && !$is_orphan) {
                $existing_meta = get_post_meta($attachment_id, '_zizi_image_optimized', true);
                if (!empty($existing_meta)) {
                    // Use enabled formats passed from bulk start
                    $current_formats = !empty($enabled_formats) ? $enabled_formats : (get_option('zizi_cache_settings', [])['image_converter_formats'] ?? ['webp']);
                    
                    // Check if optimized formats match current settings and files actually exist
                    $existing_formats = $existing_meta['formats'] ?? [];
                    $formats_match = !empty(array_intersect($current_formats, $existing_formats));
                    
                    if ($formats_match) {
                        // Verify that the optimized files actually exist
                        $upload_dir = wp_upload_dir();
                        $relative_path = str_replace($upload_dir['basedir'], '', $file_path);
                        $files_exist = true;
                        
                        foreach ($current_formats as $format) {
                            $converted_path = $upload_dir['basedir'] . $relative_path . '.' . $format;
                            if (!file_exists($converted_path)) {
                                $files_exist = false;
                                break;
                            }
                        }
                        
                        if ($files_exist) {
                            self::update_bulk_statistics('skipped');
                            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                                if ($debug_log_enabled || ($log_samples['skipped'] < $log_sample_limit)) {
                                    \ZiziCache\CacheSys::writeLog(
                                        "Skipping attachment {$attachment_id} - already optimized for current formats",
                                        'ImageConverterBulk'
                                    );
                                    if (!$debug_log_enabled) {
                                        $log_samples['skipped']++;
                                        set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                                    }
                                }
                            }
                            return;
                        } else {
                            // Files don't exist, clear metadata and proceed with optimization
                            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                                if ($debug_log_enabled || ($log_samples['skipped'] < $log_sample_limit)) {
                                    \ZiziCache\CacheSys::writeLog(
                                        "Clearing stale metadata for attachment {$attachment_id} - optimized files don't exist",
                                        'ImageConverterBulk'
                                    );
                                    if (!$debug_log_enabled) {
                                        $log_samples['skipped']++;
                                        set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                                    }
                                }
                            }
                            delete_post_meta($attachment_id, '_zizi_image_optimized');
                        }
                    } else {
                        // Format settings changed, clear metadata and proceed
                        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                            if ($debug_log_enabled || ($log_samples['skipped'] < $log_sample_limit)) {
                                \ZiziCache\CacheSys::writeLog(
                                    "Clearing metadata for attachment {$attachment_id} - format settings changed",
                                    'ImageConverterBulk'
                                );
                                if (!$debug_log_enabled) {
                                    $log_samples['skipped']++;
                                    set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                                }
                            }
                        }
                        delete_post_meta($attachment_id, '_zizi_image_optimized');
                    }
                }
            } else {
                // For zizi-cache and orphan images, check if optimized files already exist
                $current_formats = !empty($enabled_formats) ? $enabled_formats : (get_option('zizi_cache_settings', [])['image_converter_formats'] ?? ['webp']);
                $files_exist = true;
                
                foreach ($current_formats as $format) {
                    $converted_path = $file_path . '.' . $format;
                    if (!file_exists($converted_path)) {
                        $files_exist = false;
                        break;
                    }
                }
                
                if ($files_exist) {
                    self::update_bulk_statistics('skipped');
                    if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                        if ($debug_log_enabled || ($log_samples['skipped'] < $log_sample_limit)) {
                            \ZiziCache\CacheSys::writeLog(
                                "Skipping {$attachment_id} - converted files already exist",
                                'ImageConverterBulk'
                            );
                            if (!$debug_log_enabled) {
                                $log_samples['skipped']++;
                                set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                            }
                        }
                    }
                    return;
                }
            }

            // Check file format support (only for WordPress attachments)
            if (!$is_zizi_cache && !$is_orphan) {
                $mime_type = get_post_mime_type($attachment_id);
                
                // Additional check: Skip if file extension indicates already converted format
                $file_path_check = get_attached_file($attachment_id);
                if ($file_path_check) {
                    $file_extension = strtolower(pathinfo($file_path_check, PATHINFO_EXTENSION));
                    if (in_array($file_extension, ['webp', 'avif'])) {
                        self::update_bulk_statistics('skipped');
                        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                            if ($debug_log_enabled || ($log_samples['skipped'] < $log_sample_limit)) {
                                \ZiziCache\CacheSys::writeLog(
                                    "Skipping attachment {$attachment_id} - already converted format by extension: .{$file_extension}",
                                    'ImageConverterBulk'
                                );
                                if (!$debug_log_enabled) {
                                    $log_samples['skipped']++;
                                    set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                                }
                            }
                        }
                        return;
                    }
                }
                
                if (!ImageConverter::is_supported_format($mime_type) || self::is_already_converted_format($mime_type)) {
                    self::update_bulk_statistics('skipped');
                    if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                        $skip_reason = self::is_already_converted_format($mime_type) 
                            ? "already converted format" 
                            : "unsupported format";
                        \ZiziCache\CacheSys::writeLog(
                            "Skipping attachment {$attachment_id} - {$skip_reason}: {$mime_type}",
                            'ImageConverterBulk'
                        );
                    }
                    return;
                }
            } else {
                // For zizi-cache images, assume they're already in supported format since they passed initial filtering
                if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                    \ZiziCache\CacheSys::writeLog(
                        "Processing zizi-cache image: {$attachment_id} -> {$file_path}",
                        'ImageConverterBulk'
                    );
                }
            }

            // Process the image
            // Set attachment_id in request for identification in conversion process
            $_REQUEST['action'] = 'zizi_image_converter_process_single';
            $_REQUEST['attachment_id'] = $attachment_id;
            
            try {
                if ($is_zizi_cache || $is_orphan) {
                    // For zizi-cache and orphan images, use convert_single_format for each enabled format
                    $formats_to_convert = !empty($enabled_formats) ? $enabled_formats : ['webp'];
                    $result = true;
                    
                    foreach ($formats_to_convert as $format) {
                        $conversion_result = ImageConverter::convert_single_format($file_path, $format, 0);
                        if (!$conversion_result['success']) {
                            $result = false;
                            break;
                        }
                    }
                } else {
                    // For WordPress attachments, use the standard method
                    $result = ImageConverter::convert_image_formats($file_path, (int)$attachment_id);
                }
            } catch (\Error $e) {
                // Catch fatal errors (like missing function)
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Fatal error in conversion for attachment {$attachment_id}: " . $e->getMessage(),
                        'ImageConverterBulk'
                    );
                }
                $result = false;
            } catch (\Exception $e) {
                // Catch regular exceptions
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Exception in conversion for attachment {$attachment_id}: " . $e->getMessage(),
                        'ImageConverterBulk'
                    );
                }
                $result = false;
            }
            
            // Clean up request after conversion
            unset($_REQUEST['action'], $_REQUEST['attachment_id']);
            $process_time = microtime(true) - $start_time;

            if ($result) {
                // Clear reschedule counter on successful conversion
                delete_transient("zizi_reschedule_count_{$attachment_id}");
                
                // Get optimization metadata for better statistics
                if ($is_zizi_cache || $is_orphan) {
                    // For zizi-cache and orphan images, get metadata differently since they don't have post meta
                    $metadata = [];
                    if ($result && is_array($result) && isset($result['metadata'])) {
                        $metadata = $result['metadata'];
                    } else {
                        // Fallback: detect formats from enabled_formats array
                        $metadata['formats'] = $enabled_formats;
                        // Try to get file sizes if possible
                        if (file_exists($file_path)) {
                            $metadata['sizes']['original'] = filesize($file_path);
                        }
                    }
                } else {
                    // For WordPress attachments, get from post meta
                    $metadata = get_post_meta($attachment_id, '_zizi_image_optimized', true) ?: [];
                }
                
                // Get filename for better logging
                $filename = basename($file_path);
                
                // Formats that were created
                $formats = $metadata['formats'] ?? [];
                $format_info = !empty($formats) ? ' (' . implode(', ', $formats) . ')' : '';
                
                // Space savings
                $original_size = $metadata['sizes']['original'] ?? 0;
                $total_savings_percent = $metadata['total_savings_percent'] ?? 0;
                $savings_info = $total_savings_percent > 0 ? ", savings: {$total_savings_percent}%" : '';
                
                self::update_bulk_statistics('completed', $process_time, $metadata);
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    // Log per-image optimization success only when debug enabled or sample quota remains
                    if ($debug_log_enabled || ($log_samples['success'] < $log_sample_limit)) {
                        \ZiziCache\CacheSys::writeLog(
                            "Image {$attachment_id} ({$filename}) optimized in " . round($process_time, 2) . 
                            "s{$format_info}{$savings_info}",
                            'ImageConverterBulk'
                        );
                        if (!$debug_log_enabled) {
                            $log_samples['success']++;
                            set_transient('zizi_bulk_log_samples', $log_samples, HOUR_IN_SECONDS);
                        }
                    }
                }
            } else {
                self::update_bulk_statistics('failed');
                
                // Try to get more specific error information
                $error_details = "Conversion returned false";
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    // Check if it's an environment issue
                    $env_info = ImageConverter::get_environment_info();
                    if (!$env_info['gd_available'] && !$env_info['imagick_available']) {
                        $error_details = "No image processing extensions available (GD/Imagick)";
                    } elseif (!function_exists('imageavif') && !$env_info['imagick_avif_support']) {
                        $error_details = "AVIF support not available";
                    }
                }
                
                throw new \Exception("Conversion failed for image {$attachment_id}: {$error_details}");
            }

        } catch (\Exception $e) {
            // Only increment failed counter if we haven't already counted this attempt
            $error_message = $e->getMessage();
            if (strpos($error_message, 'already optimized') === false && 
                strpos($error_message, 'unsupported format') === false &&
                strpos($error_message, 'Invalid image attachment') === false &&
                strpos($error_message, 'Image file not found') === false) {
                
                // This is a real processing failure, not a skip condition
                if (strpos($error_message, 'Conversion failed for image') === false) {
                    self::update_bulk_statistics('failed');
                }
            }
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                $log_method = (strpos($error_message, 'Skipping') === 0) ? 'writeLog' : 'writeError';
                \ZiziCache\CacheSys::$log_method(
                    "Bulk processing error for image {$attachment_id}: " . $error_message,
                    'ImageConverterBulk'
                );
            }

            throw $e; // Re-throw for Action Scheduler retry mechanism
        }
    }

    /**
     * Stop bulk optimization process
     * 
     * @return array Operation result
     */
    public static function stop_bulk_optimization(): array
    {
        try {
            // Get pending actions
            $pending_actions = as_get_scheduled_actions([
                'hook' => self::SINGLE_ACTION_HOOK,
                'status' => \ActionScheduler_Store::STATUS_PENDING,
                'per_page' => -1
            ]);

            $cancelled_count = 0;

            // Cancel all pending actions
            foreach ($pending_actions as $action) {
                try {
                    as_unschedule_action(self::SINGLE_ACTION_HOOK, $action->get_args(), self::ACTION_GROUP);
                    $cancelled_count++;
                } catch (\Exception $e) {
                    // Continue cancelling other actions
                    continue;
                }
            }

            // Update status
            $status_data = get_transient(self::STATUS_TRANSIENT);
            if ($status_data) {
                $status_data['status'] = 'stopped';
                $status_data['stopped_at'] = current_time('mysql');
                set_transient(self::STATUS_TRANSIENT, $status_data, HOUR_IN_SECONDS);
            }

            // Log stop
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Bulk optimization stopped: {$cancelled_count} actions cancelled",
                    'ImageConverterBulk'
                );
            }

            return [
                'success' => true,
                'message' => 'Bulk optimization stopped successfully',
                'cancelled_actions' => $cancelled_count
            ];

        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    'Failed to stop bulk optimization: ' . $e->getMessage(),
                    'ImageConverterBulk'
                );
            }

            return [
                'success' => false,
                'message' => 'Failed to stop bulk optimization: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get bulk optimization status (optimized for large datasets)
     * 
     * @return array Status information
     */
    public static function get_bulk_status(): array
    {
        // Performance monitoring - measure first, optimize second
        $start_time = microtime(true);
        $start_memory = memory_get_usage(true);
        
        try {
            // Check if any optimization is configured or running
            $status_data = get_transient(self::STATUS_TRANSIENT);
            
            if (!$status_data) {
                return [
                    'running' => false,
                    'message' => 'No bulk optimization is currently running',
                    'performance' => [
                        'execution_time_ms' => round((microtime(true) - $start_time) * 1000, 2),
                        'memory_used_mb' => round((memory_get_usage(true) - $start_memory) / 1024 / 1024, 2)
                    ]
                ];
            }

            // Use cached status for frequently accessed data (30-second cache for production)
            $cache_key = 'zizi_bulk_status_cache';
            $cached_status = get_transient($cache_key);
            
            if ($cached_status && is_array($cached_status)) {
                // Add freshness indicator to cached data
                $cached_status['cached'] = true;
                $cached_status['cache_age_seconds'] = time() - ($cached_status['generated_at'] ?? time());
                return $cached_status;
            }

            // Memory protection - check if we have enough memory for processing
            $memory_limit = self::convert_to_bytes(ini_get('memory_limit'));
            $current_memory = memory_get_usage(true);
            $memory_threshold = 0.8; // 80% threshold
            
            if ($memory_limit > 0 && $current_memory > ($memory_limit * $memory_threshold)) {
                // Graceful degradation - return basic status
                return [
                    'running' => ($status_data['status'] ?? '') === 'running',
                    'status' => $status_data['status'] ?? 'unknown',
                    'message' => 'Limited status due to memory constraints',
                    'total_images' => $status_data['total_images'] ?? 0,
                    'error' => 'insufficient_memory',
                    'performance' => [
                        'execution_time_ms' => round((microtime(true) - $start_time) * 1000, 2),
                        'memory_used_mb' => round((memory_get_usage(true) - $start_memory) / 1024 / 1024, 2),
                        'memory_limit_reached' => true
                    ]
                ];
            }

            // Optimize for common case - efficient pending count via SQL instead of loading all data
            $pending_count = self::get_pending_actions_count_optimized();
            
            // If SQL optimization failed, fall back to Action Scheduler API with limit
            if ($pending_count === false) {
                $pending_actions = as_get_scheduled_actions([
                    'hook' => self::SINGLE_ACTION_HOOK,
                    'status' => \ActionScheduler_Store::STATUS_PENDING,
                    'per_page' => 1000 // Reasonable limit instead of -1
                ]);
                $pending_count = count($pending_actions);
                
                // Log performance issue for monitoring
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Bulk status: SQL optimization failed, using AS API with limit. Pending: {$pending_count}",
                        'ImageConverterBulk'
                    );
                }
            }

            // Get statistics with timeout protection
            $stats = get_transient(self::STATS_TRANSIENT) ?: [];
            $total_images = $status_data['total_images'] ?? 0;
            $completed_count = $stats['completed'] ?? 0;
            $failed_count = $stats['failed'] ?? 0;
            $skipped_count = $stats['skipped'] ?? 0;
            
            // Progress calculation with bounds checking
            $processed_count = $completed_count + $failed_count + $skipped_count;
            $progress = $total_images > 0 
                ? round(($processed_count / $total_images) * 100, 1) 
                : 0;
            $progress = max(0, min(100, $progress)); // Ensure bounds [0,100]
                
            // Format information
            $format_stats = $stats['formats'] ?? ['avif' => 0, 'webp' => 0];
            
            // Space savings calculation with validation
            $original_size = max(0, $stats['original_size'] ?? 0);
            $optimized_size = max(0, $stats['optimized_size'] ?? 0);
            $savings_bytes = $original_size > $optimized_size ? $original_size - $optimized_size : 0;
            $savings_percent = $original_size > 0 ? round(($savings_bytes / $original_size) * 100, 1) : 0;

            // Build result with performance data
            $result = [
                'running' => $status_data['status'] === 'running' && $pending_count > 0,
                'status' => $status_data['status'] ?? 'unknown',
                'started_at' => $status_data['started_at'] ?? '',
                'stopped_at' => $status_data['stopped_at'] ?? null,
                'total_images' => $total_images,
                'pending_count' => $pending_count,
                'completed_count' => $completed_count,
                'failed_count' => $failed_count,
                'skipped_count' => $skipped_count,
                'progress' => $progress,
                'estimated_time_remaining' => self::calculate_eta($stats, $pending_count),
                'average_processing_time' => $stats['avg_processing_time'] ?? 0,
                'formats' => $format_stats,
                'original_size' => $original_size,
                'optimized_size' => $optimized_size,
                'savings_bytes' => $savings_bytes,
                'savings_percent' => $savings_percent,
                'server_load' => function_exists('sys_getloadavg') ? sys_getloadavg()[0] : null,
                'memory_usage' => memory_get_usage(true),
                'generated_at' => time(),
                'cached' => false,
                'performance' => [
                    'execution_time_ms' => round((microtime(true) - $start_time) * 1000, 2),
                    'memory_used_mb' => round((memory_get_usage(true) - $start_memory) / 1024 / 1024, 2),
                    'sql_optimized' => $pending_count !== false,
                    'total_images' => $total_images
                ]
            ];

            // Cache result for 30 seconds to reduce load (minimize resource usage)
            if ($total_images > 1000) { // Only cache for large datasets
                set_transient($cache_key, $result, 30);
            }

            return $result;

        } catch (\Exception $e) {
            // Comprehensive error handling with graceful degradation
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    'Bulk status error: ' . $e->getMessage() . ' | Memory: ' . 
                    round(memory_get_usage(true) / 1024 / 1024, 2) . 'MB | Time: ' . 
                    round((microtime(true) - $start_time) * 1000, 2) . 'ms',
                    'ImageConverterBulk'
                );
            }
            
            // Return minimal status on error
            $status_data = get_transient(self::STATUS_TRANSIENT) ?: [];
            return [
                'running' => ($status_data['status'] ?? '') === 'running',
                'status' => $status_data['status'] ?? 'error',
                'message' => 'Status retrieval failed: ' . $e->getMessage(),
                'error' => true,
                'total_images' => $status_data['total_images'] ?? 0,
                'performance' => [
                    'execution_time_ms' => round((microtime(true) - $start_time) * 1000, 2),
                    'memory_used_mb' => round((memory_get_usage(true) - $start_memory) / 1024 / 1024, 2),
                    'error_occurred' => true
                ]
            ];
        }
    }

    /**
     * Get pending actions count optimized for large datasets
     * Uses direct SQL query instead of loading all actions into memory
     * 
     * @return int|false Pending actions count or false on error
     */
    private static function get_pending_actions_count_optimized()
    {
        global $wpdb;
        
        try {
            // Use direct SQL query for better performance with large datasets
            $table_name = $wpdb->prefix . 'actionscheduler_actions';
            
            // Check if table exists first
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name;
            if (!$table_exists) {
                return false; // Fall back to AS API
            }
            
            // Efficient COUNT query with proper indexing
            $count = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*) 
                FROM {$table_name} 
                WHERE hook = %s 
                AND status = %s
            ", self::SINGLE_ACTION_HOOK, \ActionScheduler_Store::STATUS_PENDING));
            
            return (int)$count;
            
        } catch (\Exception $e) {
            // Log SQL error but don't fail - fall back to AS API
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    'SQL optimization failed for pending count: ' . $e->getMessage(),
                    'ImageConverterBulk'
                );
            }
            return false;
        }
    }

    /**
     * Check if bulk optimization is currently running
     * 
     * @return bool
     */
    public static function is_optimization_running(): bool
    {
        $status_data = get_transient(self::STATUS_TRANSIENT);
        return !empty($status_data) && ($status_data['status'] ?? '') === 'running';
    }

    /**
     * Initialize bulk statistics
     * 
     * @param array $status_data Initial status data
     */
    private static function init_bulk_statistics(array $status_data): void
    {
        $stats = [
            'started_at' => $status_data['started_at'],
            'completed' => 0,
            'failed' => 0,
            'skipped' => 0,
            'total_processing_time' => 0,
            'avg_processing_time' => 0,
            'formats' => ['avif' => 0, 'webp' => 0],
            'original_size' => 0,
            'optimized_size' => 0,
            'last_update' => current_time('mysql')
        ];

        set_transient(self::STATS_TRANSIENT, $stats, 24 * HOUR_IN_SECONDS);
    }

    /**
     * Update bulk statistics
     * 
     * @param string $status Status type: 'completed', 'failed', 'skipped'
     * @param float $processing_time Processing time in seconds
     * @param array $metadata Metadata about processed image
     */
    private static function update_bulk_statistics(string $status, float $processing_time = 0, array $metadata = []): void
    {
        $stats = get_transient(self::STATS_TRANSIENT) ?: [];
        
        $stats[$status] = ($stats[$status] ?? 0) + 1;
        $stats['last_update'] = current_time('mysql');
        
        if ($processing_time > 0) {
            $stats['total_processing_time'] = ($stats['total_processing_time'] ?? 0) + $processing_time;
            $completed_count = $stats['completed'] ?? 1;
            $stats['avg_processing_time'] = $stats['total_processing_time'] / $completed_count;
        }
        
        // Update statistics about formats and file sizes
        if (!empty($metadata) && $status === 'completed') {
            // Format counters
            if (!empty($metadata['formats'])) {
                $stats['formats'] = $stats['formats'] ?? ['avif' => 0, 'webp' => 0];
                foreach ($metadata['formats'] as $format) {
                    if (isset($stats['formats'][$format])) {
                        $stats['formats'][$format]++;
                    }
                }
            }
            
            // File sizes
            if (isset($metadata['sizes']['original']) && $metadata['sizes']['original'] > 0) {
                $stats['original_size'] = ($stats['original_size'] ?? 0) + $metadata['sizes']['original'];
                
                // Sum optimized format sizes
                $optimized_size = 0;
                foreach (['webp', 'avif'] as $format) {
                    if (isset($metadata['sizes'][$format]) && $metadata['sizes'][$format] > 0) {
                        $optimized_size += $metadata['sizes'][$format];
                    }
                }
                
                if ($optimized_size > 0) {
                    $stats['optimized_size'] = ($stats['optimized_size'] ?? 0) + $optimized_size;
                }
            }
        }

        set_transient(self::STATS_TRANSIENT, $stats, 24 * HOUR_IN_SECONDS);
    }

    /**
     * Calculate estimated time remaining
     * 
     * @param array $stats Current statistics
     * @param int $pending_count Number of pending images
     * @return int Estimated seconds remaining
     */
    private static function calculate_eta(array $stats, int $pending_count): int
    {
        $avg_time = $stats['avg_processing_time'] ?? 2; // Default 2 seconds per image
        return (int)($pending_count * $avg_time);
    }

    /**
     * Handle failed Action Scheduler action
     * 
     * @param int $action_id Action ID
     * @param \Exception $exception Exception that caused failure
     */
    public static function handle_failed_action(int $action_id, \Exception $exception): void
    {
        // Log failure for monitoring
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeError(
                "Action {$action_id} failed: " . $exception->getMessage(),
                'ImageConverterBulk'
            );
        }
    }

    /**
     * Server load check
     * 
     * Determines if the server is too overloaded and should temporarily pause processing
     * 
     * @return bool True if the server is too overloaded
     */
    private static function is_server_overloaded(): bool
    {
        // Check CPU usage (Linux)
        if (function_exists('sys_getloadavg')) {
            $load = sys_getloadavg();
            if ($load[0] > 0.8) { // If server load is higher than 80%
                return true;
            }
        }
        
        // Check available memory
        $memory_limit = ini_get('memory_limit');
        $memory_limit_bytes = self::convert_to_bytes($memory_limit);
        $memory_usage = memory_get_usage(true);
        
        if ($memory_limit_bytes > 0 && $memory_usage > ($memory_limit_bytes * 0.9)) {
            // More than 90% of memory is used
            return true;
        }
        
        return false;
    }
    
    /**
     * Convert textual memory representation (e.g. "128M") to bytes
     *
     * @param string $value
     * @return int
     */
    private static function convert_to_bytes(string $value): int
    {
        $value = trim($value);
        $last = strtolower($value[strlen($value) - 1]);
        $value = (int)$value;
        
        switch ($last) {
            case 'g':
                $value *= 1024;
                // no break
            case 'm':
                $value *= 1024;
                // no break
            case 'k':
                $value *= 1024;
        }
        
        return $value;
    }
    
    /**
     * Handle completed Action Scheduler action
     * 
     * @param int $action_id Action ID
     */
    public static function handle_completed_action(int $action_id): void
    {
        $pending_actions = as_get_scheduled_actions([
            'hook' => self::SINGLE_ACTION_HOOK,
            'status' => \ActionScheduler_Store::STATUS_PENDING,
            'per_page' => 1
        ]);

        // If no pending actions, mark optimization as completed
        if (empty($pending_actions)) {
            $status_data = get_transient(self::STATUS_TRANSIENT);
            if ($status_data && ($status_data['status'] ?? '') === 'running') {
                $status_data['status'] = 'completed';
                $status_data['completed_at'] = current_time('mysql');
                set_transient(self::STATUS_TRANSIENT, $status_data, HOUR_IN_SECONDS);
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('Bulk optimization completed', 'ImageConverterBulk');
                }
                
                // Clean up reschedule counters for completed optimization
                self::cleanup_reschedule_counters();
            }
        }
    }
    
    /**
     * Clean up reschedule counters for completed bulk optimization
     */
    private static function cleanup_reschedule_counters(): void
    {
        global $wpdb;
        
        // Clean up reschedule transients for this optimization batch
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_zizi_reschedule_count_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_zizi_reschedule_count_%'");
    }

    /**
     * Cleanup completed optimization data
     */
    public static function cleanup_completed_optimization(): void
    {
        if (!is_admin()) {
            return;
        }

        $status_data = get_transient(self::STATUS_TRANSIENT);
        
        if ($status_data && 
            in_array($status_data['status'] ?? '', ['completed', 'stopped']) &&
            isset($status_data['completed_at']) &&
            strtotime($status_data['completed_at']) < (time() - 3600)) { // 1 hour old
            
            delete_transient(self::STATUS_TRANSIENT);
            delete_transient(self::STATS_TRANSIENT);
        }
    }

    /**
     * Check if an image needs optimization based on enabled formats
     * 
     * @param int $attachment_id WordPress attachment ID
     * @param array $enabled_formats Enabled image formats (avif, webp)
     * @return bool True if image needs optimization
     */
    private static function image_needs_optimization(int $attachment_id, array $enabled_formats): bool
    {
        $original_file = get_attached_file($attachment_id);
        if (!$original_file || !file_exists($original_file)) {
            return false;
        }
        
        // Check if any enabled format is missing
        foreach ($enabled_formats as $format) {
            $converted_file = $original_file . '.' . $format;
            if (!file_exists($converted_file)) {
                return true; // This format is missing, needs optimization
            }
        }
        
        // Check metadata consistency
        $existing_meta = get_post_meta($attachment_id, '_zizi_image_optimized', true);
        if (!is_array($existing_meta)) {
            return true; // No metadata, needs optimization
        }
        
        // Check if metadata matches enabled formats
        $meta_formats = $existing_meta['formats'] ?? [];
        foreach ($enabled_formats as $format) {
            if (!in_array($format, $meta_formats)) {
                return true; // Format not in metadata, needs optimization
            }
        }
        
        return false; // All formats exist and are in metadata
    }

    /**
     * Check if the MIME type represents an already converted format
     * 
     * @param string $mime_type MIME type to check
     * @return bool True if it's an already converted format
     */
    private static function is_already_converted_format(string $mime_type): bool
    {
        $converted_formats = [
            'image/webp',
            'image/avif'
        ];
        
        return in_array($mime_type, $converted_formats);
    }

    /**
     * Cleanup stale optimization data
     */
    public static function cleanup_stale_optimizations(): void
    {
        $status_data = get_transient(self::STATUS_TRANSIENT);
        
        if ($status_data && 
            ($status_data['status'] ?? '') === 'running' &&
            isset($status_data['started_at']) &&
            strtotime($status_data['started_at']) < (time() - 24 * 3600)) { // 24 hours old
            // Mark as stale and stop
            self::stop_bulk_optimization();
        }
    }

    /**
     * Check if zizi-cache image needs optimization
     * 
     * @param string $file_path Full path to the image file
     * @param array $enabled_formats Enabled conversion formats
     * @return bool True if optimization is needed
     */
    public static function zizi_cache_image_needs_optimization(string $file_path, array $enabled_formats): bool
    {
        if (!file_exists($file_path)) {
            return false;
        }
        
        // First check if this is a supported image format for conversion
        $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        $supported_extensions = [];
        
        // Use ImageConverter's supported extensions for consistency
        if (class_exists('\\ZiziCache\\ImageConverter') && method_exists('\\ZiziCache\\ImageConverter', 'get_supported_extensions')) {
            $supported_extensions = \ZiziCache\ImageConverter::get_supported_extensions();
        } else {
            // Fallback to common formats
            $supported_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'tiff', 'tif'];
        }
        
        // Skip if not a supported format
        if (!in_array($extension, $supported_extensions, true)) {
            return false;
        }
        
        // Check if any enabled format is missing
        foreach ($enabled_formats as $format) {
            $converted_file = $file_path . '.' . $format;
            if (!file_exists($converted_file)) {
                return true; // This format is missing, needs optimization
            }
        }
        
        return false; // All formats exist
    }
}
