<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Image Converter - Core functionality for AVIF/WebP image format conversion
 * 
 * Provides comprehensive image format conversion with AVIF-first approach,
 * WebP fallback, and intelligent quality optimization. Integrates seamlessly 
 * with existing ZiziCache Image optimization features.
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.4
 */
class ImageConverter
{
    /**
     * Supported input image formats for conversion
     */
    private const SUPPORTED_INPUT_FORMATS = [
        'image/jpeg',
        'image/png', 
        'image/gif',
        'image/bmp',
        'image/tiff'
    ];

    /**
     * Output format priority (AVIF-first approach)
     */
    private const FORMAT_PRIORITY = ['avif', 'webp'];

    /**
     * Initialize Image Converter functionality
     * Hooks into WordPress upload process
     */
    public static function init(): void
    {
        // Hook into metadata generation for stable conversion process
        add_filter('wp_generate_attachment_metadata', [__CLASS__, 'handle_attachment_metadata'], 10, 2);
        
        // NOTE: Action Scheduler hook for bulk processing now handled in ImageConverterBulk
        // Old hook removed to avoid conflicts with new zizi-cache support
        
        // Hook into attachment deletion to cleanup converted files
        add_action('delete_attachment', [__CLASS__, 'cleanup_converted_files'], 10, 1);
        
        // Hook for async original cleanup
        add_action('zizi_cleanup_original_after_conversion', [__CLASS__, 'cleanup_original_after_conversion']);
        
        // Log initialization
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('ImageConverter module initialized with cleanup hooks', 'ImageConverter');
        }
    }

    /**
     * Handle automatic image conversion on upload
     * 
     * @param array $upload Upload data from wp_handle_upload
     * @param string $context Upload context
     * @return array Modified upload data
     */
    /**
     * Handle attachment metadata generation for automatic conversion
     * More stable than wp_handle_upload hook
     * 
     * @param array $metadata Attachment metadata
     * @param int $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public static function handle_attachment_metadata(array $metadata, int $attachment_id): array
    {
        if (!self::is_enabled() || !wp_attachment_is_image($attachment_id)) {
            return $metadata;
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }

        $mime_type = get_post_mime_type($attachment_id);
        if (!self::is_supported_format($mime_type)) {
            return $metadata;
        }

        try {
            // Perform conversion based on user settings
            $conversion_result = self::perform_image_conversion($attachment_id, $metadata);
            
            if ($conversion_result['success']) {
                // Update metadata if conversion was successful
                $metadata = $conversion_result['metadata'];
                
                // Schedule original cleanup if conversion successful and setting is enabled
                if (!self::should_keep_originals()) {
                    self::schedule_original_cleanup($attachment_id);
                }
            }
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Image conversion completed for attachment {$attachment_id}: " . 
                    ($conversion_result['success'] ? 'SUCCESS' : 'FAILED'),
                    'ImageConverter'
                );
            }
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    "Metadata conversion failed for attachment {$attachment_id}: " . $e->getMessage(),
                    'ImageConverter'
                );
            }
        }

        return $metadata;
    }

    /**
     * Perform image conversion based on user settings and enabled formats
     * Follows CompressX pattern but with our own implementation
     * 
     * @param int $attachment_id Attachment ID
     * @param array $metadata Attachment metadata
     * @return array Result with success status and updated metadata
     */
    public static function perform_image_conversion(int $attachment_id, array $metadata): array
    {
        $result = [
            'success' => false,
            'metadata' => $metadata,
            'formats_converted' => [],
            'errors' => []
        ];

        $config = SysConfig::$config;
        $enabled_formats = $config['image_converter_formats'] ?? [];
        
        // If no formats enabled, skip conversion
        if (empty($enabled_formats)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog('No conversion formats enabled, skipping conversion', 'ImageConverter');
            }
            $result['success'] = true; // Not an error, just disabled
            return $result;
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            $result['errors'][] = 'Original file not found: ' . $file_path;
            return $result;
        }

        $original_size = filesize($file_path);
        $has_any_success = false;

        // Convert original + all thumbnails to each enabled format
        foreach ($enabled_formats as $format) {
            if (!in_array($format, ['webp', 'avif'])) {
                continue; // Skip unsupported formats
            }

            try {
                // Convert original file
                $conversion_result = self::convert_single_format($file_path, $format, $attachment_id);
                
                if ($conversion_result['success']) {
                    $has_any_success = true;
                    $result['formats_converted'][] = $format;
                    
                    // Register converted file in WordPress
                    self::register_converted_file($attachment_id, $conversion_result['converted_file'], $format);
                    
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeLog(
                            "Successfully converted original to {$format}: " . $conversion_result['converted_file'],
                            'ImageConverter'
                        );
                    }
                }

                // Convert all thumbnail sizes
                if (!empty($metadata['sizes'])) {
                    $upload_dir = wp_upload_dir();
                    $base_dir = dirname($file_path);
                    $converted_thumbnails = [];
                    
                    foreach ($metadata['sizes'] as $size_name => $size_data) {
                        $thumbnail_path = $base_dir . '/' . $size_data['file'];
                        
                        if (file_exists($thumbnail_path)) {
                            $thumb_result = self::convert_single_format($thumbnail_path, $format, $attachment_id);
                            
                            if ($thumb_result['success']) {
                                $converted_thumbnails[$size_name] = basename($thumb_result['converted_file']);
                                
                                if (class_exists('\\ZiziCache\\CacheSys')) {
                                    \ZiziCache\CacheSys::writeLog(
                                        "Successfully converted thumbnail {$size_name} to {$format}: " . $thumb_result['converted_file'],
                                        'ImageConverter'
                                    );
                                }
                            } else {
                                if (class_exists('\\ZiziCache\\CacheSys')) {
                                    \ZiziCache\CacheSys::writeError(
                                        "Failed to convert thumbnail {$size_name} to {$format}: " . $thumb_result['error'],
                                        'ImageConverter'
                                    );
                                }
                            }
                        }
                    }
                    
                    // Store converted thumbnail info
                    if (!empty($converted_thumbnails)) {
                        update_post_meta($attachment_id, "_zizi_converted_thumbnails_{$format}", $converted_thumbnails);
                    }
                }
                
                if (!$conversion_result['success']) {
                    $result['errors'][] = "Failed to convert to {$format}: " . $conversion_result['error'];
                }
                
            } catch (\Exception $e) {
                $result['errors'][] = "Exception converting to {$format}: " . $e->getMessage();
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Exception during {$format} conversion for attachment {$attachment_id}: " . $e->getMessage(),
                        'ImageConverter'
                    );
                }
            }
        }

        $result['success'] = $has_any_success;
        
        // Log summary
        if (class_exists('\\ZiziCache\\CacheSys')) {
            $formats_str = implode(', ', $result['formats_converted']);
            $status = $has_any_success ? 'SUCCESS' : 'FAILED';
            \ZiziCache\CacheSys::writeLog(
                "Conversion summary for attachment {$attachment_id}: {$status} - Formats: {$formats_str}",
                'ImageConverter'
            );
        }

        return $result;
    }

    /**
     * Convert image to a single format (WebP or AVIF)
     * 
     * @param string $file_path Source file path
     * @param string $format Target format (webp|avif)
     * @param int $attachment_id Attachment ID for logging
     * @return array Result with success status and converted file path
     */
    public static function convert_single_format(string $file_path, string $format, int $attachment_id = 0): array
    {
        $result = [
            'success' => false,
            'converted_file' => '',
            'error' => '',
            'original_size' => 0,
            'converted_size' => 0,
            'savings_percent' => 0
        ];

        if (!file_exists($file_path) || !is_readable($file_path)) {
            $result['error'] = 'Source file not accessible';
            return $result;
        }

        $result['original_size'] = filesize($file_path);
        
        // Generate output path
        $output_path = self::get_converted_file_path($file_path, $format);
        
        // Ensure output directory exists
        $output_dir = dirname($output_path);
        if (!wp_mkdir_p($output_dir)) {
            $result['error'] = 'Cannot create output directory: ' . $output_dir;
            return $result;
        }

        // Check if conversion is needed (file doesn't exist or is older)
        if (file_exists($output_path) && filemtime($output_path) >= filemtime($file_path)) {
            $result['success'] = true;
            $result['converted_file'] = $output_path;
            $result['converted_size'] = filesize($output_path);
            $result['savings_percent'] = self::calculate_savings($result['original_size'], $result['converted_size']);
            
            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                \ZiziCache\CacheSys::writeLog(
                    "Using existing {$format} file: " . basename($output_path),
                    'ImageConverter'
                );
            }
            return $result;
        }

        // Perform actual conversion
        $config = SysConfig::$config;
        $quality = ($format === 'avif') 
            ? ($config['image_converter_avif_quality'] ?? 75)
            : ($config['image_converter_quality'] ?? 80);

        if ($format === 'webp') {
            $convert_result = self::convert_to_webp_internal($file_path, $output_path, $quality);
        } elseif ($format === 'avif') {
            $convert_result = self::convert_to_avif_internal($file_path, $output_path, $quality);
        } else {
            $result['error'] = 'Unsupported format: ' . $format;
            return $result;
        }

        if ($convert_result && file_exists($output_path)) {
            $result['success'] = true;
            $result['converted_file'] = $output_path;
            $result['converted_size'] = filesize($output_path);
            $result['savings_percent'] = self::calculate_savings($result['original_size'], $result['converted_size']);
            
            // Check if converted file is actually smaller
            if ($result['converted_size'] >= $result['original_size']) {
                $config = SysConfig::$config;
                if ($config['image_converter_remove_larger'] ?? true) {
                    @unlink($output_path);
                    $result['error'] = 'Converted file is larger than original, removed';
                    $result['success'] = false;
                    
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeLog(
                            "Removed larger {$format} file for attachment {$attachment_id}",
                            'ImageConverter'
                        );
                    }
                }
            }
        } else {
            $result['error'] = 'Conversion failed or output file not created';
        }

        return $result;
    }

    /**
     * Convert image using Imagick library
     * 
     * @param string $source_file Source image path
     * @param string $output_file Output image path
     * @param string $format Target format
     * @param int $quality Image quality (0-100)
     * @return bool Success status
     */
    private static function convert_with_imagick(string $source_file, string $output_file, string $format, int $quality): bool
    {
        try {
            if (!class_exists('Imagick')) {
                return false;
            }

            // Check AVIF support for Imagick
            if ($format === 'avif' && !self::imagick_supports_avif()) {
                return false;
            }

            $imagick = new \Imagick($source_file);
            
            // Preserve transparency for PNG/GIF
            $has_transparency = self::has_transparency($imagick);
            if ($has_transparency) {
                $imagick->setImageBackgroundColor(new \ImagickPixel('transparent'));
                $imagick->setImageAlphaChannel(\Imagick::ALPHACHANNEL_ACTIVATE);
            }
            
            // Set format
            $imagick->setImageFormat($format);
            
            // Set quality
            $imagick->setImageCompressionQuality($quality);
            
            // AVIF-specific optimizations
            if ($format === 'avif') {
                $imagick->setOption('heic:speed', '4'); // Balance speed vs quality
                if ($has_transparency) {
                    $imagick->setOption('heic:preserve-transparency', 'true');
                }
            }
            
            // WebP-specific optimizations
            if ($format === 'webp') {
                $config = SysConfig::$config;
                $method = $config['image_converter_method'] ?? 6;
                $imagick->setOption('webp:method', (string)$method);
                $imagick->setOption('webp:auto-filter', 'true');
                if ($has_transparency) {
                    $imagick->setOption('webp:alpha-quality', '100');
                }
            }

            // Strip metadata for smaller files (but preserve transparency info)
            $imagick->stripImage();
            
            // Write converted image
            $result = $imagick->writeImage($output_file);
            
            // Cleanup
            $imagick->clear();
            $imagick->destroy();
            
            return $result && file_exists($output_file);
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Imagick conversion failed: " . $e->getMessage(), 'ImageConverter');
            }
            return false;
        }
    }

    /**
     * Convert image using GD library (WebP only)
     * 
     * @param string $source_file Source image path
     * @param string $output_file Output image path  
     * @param int $quality Image quality (0-100)
     * @return bool Success status
     */
    private static function convert_with_gd(string $source_file, string $output_file, int $quality): bool
    {
        try {
            if (!function_exists('imagewebp')) {
                return false;
            }

            $image_editor = wp_get_image_editor($source_file);
            if (is_wp_error($image_editor)) {
                return false;
            }

            // Preserve transparency for PNG images
            $mime_type = wp_check_filetype($source_file)['type'] ?? '';
            $preserve_transparency = in_array($mime_type, ['image/png', 'image/gif']);

            $result = $image_editor->save($output_file, 'image/webp', [
                'quality' => $quality,
                'alpha_quality' => $preserve_transparency ? 100 : 80
            ]);
            
            return !is_wp_error($result) && file_exists($output_file);
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("GD conversion failed: " . $e->getMessage(), 'ImageConverter');
            }
            return false;
        }
    }

    /**
     * Convert image to AVIF using GD library
     * 
     * @param string $source_file Source image path  
     * @param string $output_file Output AVIF image path  
     * @param int $quality Image quality (0-100)
     * @return bool Success status
     */
    private static function convert_with_gd_avif(string $source_file, string $output_file, int $quality): bool
    {
        try {
            if (!function_exists('imageavif')) {
                if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                    \ZiziCache\CacheSys::writeLog("AVIF conversion failed: imageavif() function not available", 'ImageConverter');
                }
                return false;
            }

            // Verify source file exists and is readable
            if (!file_exists($source_file) || !is_readable($source_file)) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError("AVIF conversion failed: Source file not accessible: {$source_file}", 'ImageConverter');
                }
                return false;
            }

            // Load source image
            $source_image = null;
            $image_info = @getimagesize($source_file);
            if (!$image_info) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError("AVIF conversion failed: Cannot get image info from: {$source_file}", 'ImageConverter');
                }
                return false;
            }

            $mime_type = $image_info['mime'];
            
            // Log conversion attempt for debugging
            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                \ZiziCache\CacheSys::writeLog(
                    "Converting to AVIF: {$source_file} (MIME: {$mime_type}) -> {$output_file} (Quality: {$quality})", 
                    'ImageConverter'
                );
            }
            
            switch ($mime_type) {
                case 'image/jpeg':
                    $source_image = @imagecreatefromjpeg($source_file);
                    break;
                case 'image/png':
                    $source_image = @imagecreatefrompng($source_file);
                    if ($source_image) {
                        // Preserve transparency for PNG
                        imagealphablending($source_image, false);
                        imagesavealpha($source_image, true);
                    }
                    break;
                case 'image/gif':
                    $source_image = @imagecreatefromgif($source_file);
                    break;
                case 'image/webp':
                    $source_image = @imagecreatefromwebp($source_file);
                    break;
                default:
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeError("AVIF conversion failed: Unsupported MIME type: {$mime_type}", 'ImageConverter');
                    }
                    return false;
            }

            if (!$source_image) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError("AVIF conversion failed: Cannot create image resource from: {$source_file}", 'ImageConverter');
                }
                return false;
            }

            // Convert to AVIF
            $result = @imageavif($source_image, $output_file, $quality);
            
            // Clean up memory
            imagedestroy($source_image);
            
            if (!$result) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError("AVIF conversion failed: imageavif() returned false for: {$source_file}", 'ImageConverter');
                }
                return false;
            }
            
            // Verify output file was created
            if (!file_exists($output_file) || filesize($output_file) === 0) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError("AVIF conversion failed: Output file not created or empty: {$output_file}", 'ImageConverter');
                }
                return false;
            }
            
            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                $output_size = filesize($output_file);
                \ZiziCache\CacheSys::writeLog("AVIF conversion successful: {$output_file} ({$output_size} bytes)", 'ImageConverter');
            }
            
            return true;
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("GD AVIF conversion failed: " . $e->getMessage(), 'ImageConverter');
            }
            return false;
        }
    }

    /**
     * Check if image has transparency
     * 
     * @param \Imagick $imagick Imagick object
     * @return bool True if image has transparency
     */
    private static function has_transparency(\Imagick $imagick): bool
    {
        try {
            // Check if image has alpha channel
            return $imagick->getImageAlphaChannel() !== \Imagick::ALPHACHANNEL_UNDEFINED;
        } catch (\Exception $e) {
            // Fallback: check by format
            $format = strtolower($imagick->getImageFormat());
            return in_array($format, ['png', 'gif', 'webp']);
        }
    }

    /**
     * Resize image if it exceeds configured dimensions
     * 
     * @param string $file_path Original image path
     * @return string|null Resized image path or null if no resize needed
     */
    private static function resize_if_needed(string $file_path): ?string
    {
        $config = SysConfig::$config;
        $max_width = $config['image_converter_max_width'] ?? 1920;
        $max_height = $config['image_converter_max_height'] ?? 1080;

        if ($max_width <= 0 && $max_height <= 0) {
            return null;
        }

        $image_editor = wp_get_image_editor($file_path);
        if (is_wp_error($image_editor)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Failed to get image editor for resize: " . $image_editor->get_error_message(), 'ImageConverter');
            }
            return null;
        }

        $size = $image_editor->get_size();
        if (!$size || ($size['width'] <= $max_width && $size['height'] <= $max_height)) {
            return null;
        }

        // Create temporary resized file with proper extension
        $original_extension = pathinfo($file_path, PATHINFO_EXTENSION);
        
        // Vytvoříme dočasný soubor v uploads adresáři s příponou
        $uploads_dir = wp_upload_dir();
        $temp_filename = wp_unique_filename($uploads_dir['path'], uniqid() . '.' . $original_extension);
        $temp_file = $uploads_dir['path'] . '/' . $temp_filename;
        
        // Log resize attempt for debugging
        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
            \ZiziCache\CacheSys::writeLog(
                "Resizing image from {$size['width']}x{$size['height']} to max {$max_width}x{$max_height}, temp file: " . basename($temp_file), 
                'ImageConverter'
            );
        }
        
        $resize_result = $image_editor->resize($max_width, $max_height);
        
        if (is_wp_error($resize_result)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Failed to resize image: " . $resize_result->get_error_message(), 'ImageConverter');
            }
            return null;
        }

        $save_result = $image_editor->save($temp_file);
        if (is_wp_error($save_result)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Failed to save resized image: " . $save_result->get_error_message(), 'ImageConverter');
            }
            return null;
        }

        // Verify the temporary file was created and is readable
        if (!file_exists($temp_file) || !is_readable($temp_file)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError("Temporary resized file not accessible: {$temp_file}", 'ImageConverter');
            }
            return null;
        }

        return $temp_file;
    }

    /**
     * Check if Imagick supports AVIF format
     * 
     * @return bool AVIF support status
     */
    public static function imagick_supports_avif(): bool
    {
        if (!class_exists('Imagick')) {
            return false;
        }

        try {
            $imagick = new \Imagick();
            $formats = $imagick->queryFormats('AVIF');
            $imagick->destroy();
            return !empty($formats);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Convert image to enabled formats (Main conversion function for bulk processing)
     * 
     * @param string $file_path Source file path
     * @param int $attachment_id WordPress attachment ID
     * @return bool Success status
     */
    public static function convert_image_formats(string $file_path, int $attachment_id): bool
    {
        try {
            // Check if Image Converter is enabled
            if (!self::is_enabled()) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Image Converter is disabled, skipping conversion for attachment {$attachment_id}",
                        'ImageConverter'
                    );
                }
                return false;
            }

            // Verify file exists and is readable
            if (!file_exists($file_path) || !is_readable($file_path)) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Source file not accessible for attachment {$attachment_id}: {$file_path}",
                        'ImageConverter'
                    );
                }
                return false;
            }

            // Check MIME type support
            $mime_type = wp_get_image_mime($file_path);
            if (!self::is_supported_format($mime_type)) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Unsupported format for attachment {$attachment_id}: {$mime_type}",
                        'ImageConverter'
                    );
                }
                return false;
            }

            // Get enabled formats from configuration
            $config = SysConfig::$config;
            $enabled_formats = $config['image_converter_formats'] ?? [];
            
            if (empty($enabled_formats)) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "No conversion formats enabled for attachment {$attachment_id}",
                        'ImageConverter'
                    );
                }
                return false;
            }

            // Perform conversions
            $conversion_success = false;
            $results = [];
            $original_size = filesize($file_path);

            foreach ($enabled_formats as $format) {
                if (!in_array($format, ['webp', 'avif'])) {
                    continue; // Skip unsupported formats
                }

                try {
                    $conversion_result = self::convert_single_format($file_path, $format, $attachment_id);
                    
                    if ($conversion_result['success']) {
                        $conversion_success = true;
                        $results[$format] = [
                            'file' => $conversion_result['converted_file'],
                            'size' => $conversion_result['converted_size'],
                            'savings' => $conversion_result['savings_percent']
                        ];
                        
                        if (class_exists('\\ZiziCache\\CacheSys')) {
                            \ZiziCache\CacheSys::writeLog(
                                "Successfully converted attachment {$attachment_id} to {$format}: " . 
                                size_format($conversion_result['converted_size']) . 
                                " ({$conversion_result['savings_percent']}% savings)",
                                'ImageConverter'
                            );
                        }
                    } else {
                        if (class_exists('\\ZiziCache\\CacheSys')) {
                            \ZiziCache\CacheSys::writeError(
                                "Failed to convert attachment {$attachment_id} to {$format}: " . 
                                $conversion_result['error'],
                                'ImageConverter'
                            );
                        }
                    }

                } catch (\Exception $e) {
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeError(
                            "Exception converting attachment {$attachment_id} to {$format}: " . $e->getMessage(),
                            'ImageConverter'
                        );
                    }
                }
            }

            // Update metadata if any conversion was successful
            if ($conversion_success && !empty($results)) {
                $metadata = [
                    'attachment_id' => $attachment_id,
                    'formats' => array_keys($results),
                    'sizes' => [
                        'original' => $original_size
                    ],
                    'savings' => [],
                    'optimized_at' => current_time('mysql'),
                    'total_savings_bytes' => 0,
                    'total_savings_percent' => 0
                ];

                $total_optimized_size = 0;
                $total_savings_bytes = 0;

                foreach ($results as $format => $data) {
                    $metadata['sizes'][$format] = $data['size'];
                    $metadata['savings'][$format] = $data['savings'];
                    $total_optimized_size += $data['size'];
                    $total_savings_bytes += ($original_size - $data['size']);
                }

                if ($original_size > 0) {
                    $metadata['total_savings_bytes'] = $total_savings_bytes;
                    $metadata['total_savings_percent'] = round(($total_savings_bytes / $original_size) * 100, 1);
                }

                // Save metadata
                update_post_meta($attachment_id, '_zizi_image_optimized', $metadata);

                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Saved optimization metadata for attachment {$attachment_id}: " . 
                        implode(', ', array_keys($results)) . 
                        " (Total savings: {$metadata['total_savings_percent']}%)",
                        'ImageConverter'
                    );
                }
            }

            return $conversion_success;

        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    "Conversion failed for attachment {$attachment_id}: " . $e->getMessage(),
                    'ImageConverter'
                );
            }
            return false;
        }
    }

    /**
     * Get optimal quality setting for specific format
     * 
     * @param string $format Image format ('avif' or 'webp')
     * @return int Quality value (0-100)
     */
    private static function get_quality_for_format(string $format): int
    {
        $config = SysConfig::$config;
        
        switch ($format) {
            case 'avif':
                return (int)($config['image_converter_avif_quality'] ?? 75);
            case 'webp':
                return (int)($config['image_converter_quality'] ?? 80);
            default:
                return 80;
        }
    }

    /**
     * Calculate file size savings percentage
     * 
     * @param int $original_size Original file size in bytes
     * @param int $converted_size Converted file size in bytes
     * @return float Savings percentage
     */
    private static function calculate_savings(int $original_size, int $converted_size): float
    {
        if ($original_size <= 0) {
            return 0.0;
        }
        
        return round((($original_size - $converted_size) / $original_size) * 100, 1);
    }

    /**
     * Save conversion metadata to WordPress post_meta
     * 
     * @param int $attachment_id WordPress attachment ID
     * @param array $results Conversion results data
     * @param int $original_size Original file size in bytes
     */
    private static function save_conversion_metadata(int $attachment_id, array $results, int $original_size = 0): void
    {
        $metadata = [
            'formats' => array_keys($results),
            'sizes' => ['original' => $original_size],
            'savings' => [],
            'optimized_at' => current_time('mysql'),
            'quality' => [],
            'total_savings_bytes' => 0,
            'total_savings_percent' => 0
        ];

        $total_optimized_size = 0;

        foreach ($results as $format => $data) {
            $metadata['sizes'][$format] = $data['size'];
            $metadata['savings'][$format] = $data['savings'];
            $metadata['quality'][$format] = self::get_quality_for_format($format);
            $total_optimized_size += $data['size'];
            
            // Record statistics if ImageConverterStats is available
            if (class_exists('\\ZiziCache\\ImageConverterStats') && $original_size > 0) {
                $original_format = wp_check_filetype(get_attached_file($attachment_id))['type'] ?? 'unknown';
                
                \ZiziCache\ImageConverterStats::recordConversion(
                    $attachment_id,
                    $original_format,
                    $format,
                    $original_size,
                    $data['size'],
                    0.0 // Processing time not tracked here
                );
            }
        }

        // Calculate total savings
        if ($original_size > 0 && $total_optimized_size > 0) {
            $metadata['total_savings_bytes'] = $original_size - $total_optimized_size;
            $metadata['total_savings_percent'] = round(($metadata['total_savings_bytes'] / $original_size) * 100, 1);
        }

        update_post_meta($attachment_id, '_zizi_image_optimized', $metadata);
        
        // Auto-generate alt text if enabled and missing
        $config = SysConfig::$config;
        if (!empty($config['image_converter_auto_alt'])) {
            self::generate_alt_text_if_missing($attachment_id);
        }
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            $formats_str = implode(', ', array_keys($results));
            $savings_info = $metadata['total_savings_percent'] > 0 
                ? " (Total savings: {$metadata['total_savings_percent']}%)" 
                : "";
            \ZiziCache\CacheSys::writeLog("Image {$attachment_id} converted to: {$formats_str}{$savings_info}", 'ImageConverter');
        }
    }

    /**
     * Generate alt text for image if missing
     * 
     * @param int $attachment_id WordPress attachment ID
     */
    private static function generate_alt_text_if_missing(int $attachment_id): void
    {
        $current_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
        
        // Skip if alt text already exists
        if (!empty($current_alt)) {
            return;
        }
        
        $attachment = get_post($attachment_id);
        if (!$attachment) {
            return;
        }
        
        // Generate alt text from filename and title
        $alt_text = '';
        
        // Try post title first
        if (!empty($attachment->post_title)) {
            $alt_text = $attachment->post_title;
        } else {
            // Fallback to filename
            $filename = basename(get_attached_file($attachment_id));
            $alt_text = pathinfo($filename, PATHINFO_FILENAME);
        }
        
        // Clean up the alt text
        $alt_text = self::clean_alt_text($alt_text);
        
        if (!empty($alt_text)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
            
            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                \ZiziCache\CacheSys::writeLog("Auto-generated alt text for image {$attachment_id}: {$alt_text}", 'ImageConverter');
            }
        }
    }

    /**
     * Clean and improve alt text
     * 
     * @param string $text Raw text to clean
     * @return string Cleaned alt text
     */
    private static function clean_alt_text(string $text): string
    {
        // Remove file extensions
        $text = preg_replace('/\.(jpe?g|png|gif|webp|avif|bmp|tiff?)$/i', '', $text);
        
        // Replace common separators with spaces
        $text = preg_replace('/[-_\.]+/', ' ', $text);
        
        // Remove numbers that look like timestamps or IDs
        $text = preg_replace('/\b\d{10,}\b/', '', $text);
        
        // Clean up multiple spaces
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Capitalize first letter of each word
        $text = ucwords(strtolower(trim($text)));
        
        // Remove very short or generic terms
        if (strlen($text) < 3 || in_array(strtolower($text), ['img', 'image', 'pic', 'photo'])) {
            return '';
        }
        
        return $text;
    }

    /**
     * Process single attachment for bulk optimization
     * Action Scheduler callback function
     * 
     * @param int $attachment_id WordPress attachment ID
     */
    public static function process_single_attachment(int $attachment_id): void
    {
        if (!wp_attachment_is_image($attachment_id)) {
            return;
        }

        // Skip if already optimized
        if (get_post_meta($attachment_id, '_zizi_image_optimized', true)) {
            return;
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return;
        }

        $mime_type = get_post_mime_type($attachment_id);
        if (!self::is_supported_format($mime_type)) {
            return;
        }

        self::convert_image_formats($file_path, $attachment_id);
    }

    /**
     * Check if image format is supported for conversion
     * 
     * @param string $mime_type MIME type to check
     * @return bool Support status
     */
    public static function is_supported_format(string $mime_type): bool
    {
        return in_array($mime_type, self::SUPPORTED_INPUT_FORMATS, true);
    }

    /**
     * Get supported input formats for image conversion
     * 
     * @return array Array of supported MIME types
     */
    public static function get_supported_input_formats(): array
    {
        return self::SUPPORTED_INPUT_FORMATS;
    }

    /**
     * Get supported file extensions for image conversion
     * 
     * @return array Array of supported file extensions
     */
    public static function get_supported_extensions(): array
    {
        $extensions = [];
        foreach (self::SUPPORTED_INPUT_FORMATS as $mime_type) {
            switch ($mime_type) {
                case 'image/jpeg':
                    $extensions = array_merge($extensions, ['jpg', 'jpeg']);
                    break;
                case 'image/png':
                    $extensions[] = 'png';
                    break;
                case 'image/gif':
                    $extensions[] = 'gif';
                    break;
                case 'image/bmp':
                    $extensions[] = 'bmp';
                    break;
                case 'image/tiff':
                    $extensions = array_merge($extensions, ['tiff', 'tif']);
                    break;
            }
        }
        return $extensions;
    }

    /**
     * Test image conversion manually for debugging
     * 
     * @param int $attachment_id WordPress attachment ID to test
     * @return array Detailed test results
     */
    public static function test_conversion_for_attachment(int $attachment_id): array
    {
        $start_time = microtime(true);
        
        $result = [
            'success' => false,
            'attachment_id' => $attachment_id,
            'errors' => [],
            'warnings' => [],
            'info' => [],
            'conversions' => [],
            'process_time' => 0
        ];

        try {
            // Basic checks
            if (!wp_attachment_is_image($attachment_id)) {
                $result['errors'][] = 'Attachment is not an image';
                return $result;
            }

            $file_path = get_attached_file($attachment_id);
            if (!$file_path) {
                $result['errors'][] = 'Cannot get attached file path';
                return $result;
            }

            if (!file_exists($file_path)) {
                $result['errors'][] = "File does not exist: {$file_path}";
                return $result;
            }

            if (!is_readable($file_path)) {
                $result['errors'][] = "File is not readable: {$file_path}";
                return $result;
            }

            $result['info'][] = "File path: {$file_path}";
            $result['info'][] = "File size: " . size_format(filesize($file_path));

            // Check MIME type
            $mime_type = wp_get_image_mime($file_path);
            $result['info'][] = "MIME type: {$mime_type}";

            if (!self::is_supported_format($mime_type)) {
                $result['errors'][] = "Unsupported MIME type: {$mime_type}";
                return $result;
            }

            // Check if Image Converter is enabled
            if (!self::is_enabled()) {
                $result['errors'][] = 'Image Converter is disabled';
                return $result;
            }

            // Get environment info
            $env = self::get_environment_info();
            $result['info'][] = "GD available: " . ($env['gd_available'] ? 'YES' : 'NO');
            $result['info'][] = "GD WebP: " . ($env['gd_webp'] ? 'YES' : 'NO');
            $result['info'][] = "GD AVIF: " . ($env['gd_avif'] ? 'YES' : 'NO');
            $result['info'][] = "Imagick available: " . ($env['imagick_available'] ? 'YES' : 'NO');
            $result['info'][] = "Imagick WebP: " . ($env['imagick_webp'] ? 'YES' : 'NO');
            $result['info'][] = "Imagick AVIF: " . ($env['imagick_avif'] ? 'YES' : 'NO');

            // Get configuration
            $config = SysConfig::$config;
            $enabled_formats = $config['image_converter_formats'] ?? [];
            $result['info'][] = "Enabled formats: " . (empty($enabled_formats) ? 'NONE' : implode(', ', $enabled_formats));

            if (empty($enabled_formats)) {
                $result['errors'][] = 'No conversion formats enabled in configuration';
                return $result;
            }

            // Test each format
            foreach ($enabled_formats as $format) {
                if (!in_array($format, ['webp', 'avif'])) {
                    $result['warnings'][] = "Skipping unsupported format: {$format}";
                    continue;
                }

                $result['info'][] = "Testing {$format} conversion...";

                // Check format support
                $support_key = $format === 'avif' ? 'avif_support' : 'webp_support';
                if (!$env[$support_key]) {
                    $result['errors'][] = "No {$format} support available on this server";
                    continue;
                }

                try {
                    $conversion_result = self::convert_single_format($file_path, $format, $attachment_id);
                    
                    if ($conversion_result['success']) {
                        $result['conversions'][$format] = [
                            'success' => true,
                            'file' => $conversion_result['converted_file'],
                            'original_size' => $conversion_result['original_size'],
                            'converted_size' => $conversion_result['converted_size'],
                            'savings_percent' => $conversion_result['savings_percent'],
                            'file_exists' => file_exists($conversion_result['converted_file']),
                            'file_readable' => file_exists($conversion_result['converted_file']) ? is_readable($conversion_result['converted_file']) : false
                        ];
                        $result['info'][] = "✅ {$format} conversion successful: " . size_format($conversion_result['converted_size']) . " ({$conversion_result['savings_percent']}% savings)";
                    } else {
                        $result['conversions'][$format] = [
                            'success' => false,
                            'error' => $conversion_result['error']
                        ];
                        $result['errors'][] = "❌ {$format} conversion failed: " . $conversion_result['error'];
                    }

                } catch (\Exception $e) {
                    $result['conversions'][$format] = [
                        'success' => false,
                        'error' => $e->getMessage()
                    ];
                    $result['errors'][] = "❌ {$format} conversion exception: " . $e->getMessage();
                }
            }

            $successful_conversions = array_filter($result['conversions'], function($conv) {
                return $conv['success'] ?? false;
            });

            $result['success'] = !empty($successful_conversions);
            $result['info'][] = "Successful conversions: " . count($successful_conversions) . "/" . count($enabled_formats);

        } catch (\Exception $e) {
            $result['errors'][] = "Test failed with exception: " . $e->getMessage();
        }

        $result['process_time'] = round(microtime(true) - $start_time, 3);
        $result['info'][] = "Total test time: {$result['process_time']}s";

        return $result;
    }

    /**
     * Check if Image Converter is enabled
     * 
     * @return bool Enabled status
     */
    public static function is_enabled(): bool
    {
        $config = SysConfig::$config;
        return !empty($config['image_converter_enabled']);
    }

    /**
     * Get server environment capabilities
     * 
     * @return array Environment data
     */
    public static function get_environment_info(): array
    {
        $gd_available = extension_loaded('gd');
        $imagick_available = extension_loaded('imagick');
        
        // Check GD capabilities
        $gd_webp = $gd_available && function_exists('imagewebp');
        $gd_avif = $gd_available && function_exists('imageavif');
        
        // Check Imagick capabilities
        $imagick_webp = false;
        $imagick_avif = false;
        
        if ($imagick_available) {
            try {
                $imagick = new \Imagick();
                $formats = $imagick->queryFormats();
                $imagick_webp = in_array('WEBP', $formats);
                $imagick_avif = in_array('AVIF', $formats) || in_array('HEIC', $formats);
                $imagick->destroy();
            } catch (\Exception $e) {
                // Imagick available but can't query formats
            }
        }

        return [
            'gd_available' => $gd_available,
            'gd_webp' => $gd_webp,
            'gd_avif' => $gd_avif,
            'imagick_available' => $imagick_available,
            'imagick_webp' => $imagick_webp,
            'imagick_avif_support' => $imagick_avif,
            'imagick_avif' => $imagick_avif, // Compatibility with frontend
            'webp_support' => $gd_webp || $imagick_webp,
            'avif_support' => $gd_avif || $imagick_avif,
            'supported_formats' => self::SUPPORTED_INPUT_FORMATS,
            'output_formats' => self::FORMAT_PRIORITY,
            'php_version' => PHP_VERSION,
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size'),
            'temp_dir' => sys_get_temp_dir(),
            'temp_dir_writable' => is_writable(sys_get_temp_dir()),
            'wp_temp_dir' => get_temp_dir(),
            'wp_temp_dir_writable' => is_writable(get_temp_dir())
        ];
    }

    /**
     * Convert a single image by ID (used for testing and API)
     *
     * @param int $attachment_id Attachment ID
     * @return array Conversion result data
     */
    public static function convert_image(int $attachment_id): array
    {
        // Kontrola, zda jde o obrázek
        if (!wp_attachment_is_image($attachment_id)) {
            throw new \Exception("ID {$attachment_id} není platný obrázek.");
        }
        
        // Získání cesty k souboru
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            throw new \Exception("Soubor obrázku s ID {$attachment_id} nebyl nalezen.");
        }
        
        // Kontrola formátu
        $mime_type = get_post_mime_type($attachment_id);
        if (!self::is_supported_format($mime_type)) {
            throw new \Exception("Formát obrázku ({$mime_type}) není podporován pro konverzi.");
        }
        
        // Příprava výsledků
        $start_time = microtime(true);
        $original_size = filesize($file_path);
        
        // Diagnostické informace před konverzí
        $diagnostics = [
            'attachment_id' => $attachment_id,
            'file_path' => $file_path,
            'mime_type' => $mime_type,
            'original_size' => $original_size,
            'file_readable' => is_readable($file_path),
            'directory_writable' => is_writable(dirname($file_path)),
            'environment' => self::get_environment_info(),
            'existing_formats' => []
        ];
        
        // Kontrola existujících formátů před konverzí
        foreach (['avif', 'webp'] as $format) {
            $format_path = $file_path . '.' . $format;
            if (file_exists($format_path)) {
                $diagnostics['existing_formats'][$format] = [
                    'path' => $format_path,
                    'size' => filesize($format_path)
                ];
            }
        }
        
        // Provést konverzi
        $result = self::convert_image_formats($file_path, $attachment_id);
        $process_time = microtime(true) - $start_time;
        
        // Získat aktualizovaná data
        $metadata = get_post_meta($attachment_id, '_zizi_image_optimized', true) ?: [];
        
        // Kontrola výsledků konverze po konverzi
        $converted_formats = [];
        foreach (['avif', 'webp'] as $format) {
            $format_path = $file_path . '.' . $format;
            if (file_exists($format_path)) {
                $converted_formats[$format] = [
                    'path' => $format_path,
                    'size' => filesize($format_path),
                    'readable' => is_readable($format_path),
                    'url' => wp_get_attachment_url($attachment_id) . '.' . $format
                ];
            }
        }
        
        // Vrátit podrobné informace o konverzi
        return [
            'success' => $result,
            'attachment_id' => $attachment_id,
            'original_file' => [
                'path' => $file_path,
                'size' => $original_size,
                'url' => wp_get_attachment_url($attachment_id)
            ],
            'converted_formats' => $converted_formats,
            'metadata' => $metadata,
            'diagnostics' => $diagnostics,
            'process_time' => round($process_time, 2),
            'message' => $result 
                ? 'Konverze úspěšně dokončena' 
                : 'Konverze selhala, zkontrolujte diagnostické informace'
        ];
    }

    /**
     * Get list of optimized attachments
     * 
     * @param int $limit Number of results to return
     * @param int $offset Results offset
     * @return array Optimized attachments data
     */
    public static function get_optimized_attachments(int $limit = 10, int $offset = 0): array
    {
        $query = new \WP_Query([
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => $limit,
            'offset' => $offset,
            'meta_query' => [['key' => '_zizi_image_optimized', 'compare' => 'EXISTS']]
        ]);

        $results = [];
        foreach ($query->posts as $post) {
            $metadata = get_post_meta($post->ID, '_zizi_image_optimized', true);
            $results[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => wp_get_attachment_url($post->ID),
                'metadata' => $metadata
            ];
        }

        return $results;
    }

    /**
     * Cleanup converted files when original attachment is deleted
     * 
     * @param int $attachment_id WordPress attachment ID being deleted
     */
    public static function cleanup_converted_files(int $attachment_id): void
    {
        // Only process image attachments
        if (!wp_attachment_is_image($attachment_id)) {
            return;
        }

        // Get original file path
        $original_file = get_attached_file($attachment_id);
        if (!$original_file || !file_exists($original_file)) {
            return;
        }

        // Get all possible converted formats
        $formats_to_cleanup = ['avif', 'webp'];
        $deleted_files = [];
        $total_size_saved = 0;

        // 1. Clean up main file converted versions
        foreach ($formats_to_cleanup as $format) {
            $converted_file = $original_file . '.' . $format;
            
            if (file_exists($converted_file)) {
                $file_size = filesize($converted_file);
                
                if (@unlink($converted_file)) {
                    $deleted_files[] = basename($converted_file);
                    $total_size_saved += $file_size;
                    
                    if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                        \ZiziCache\CacheSys::writeLog(
                            "Deleted converted {$format} file: {$converted_file} (" . size_format($file_size) . ")",
                            'ImageConverter'
                        );
                    }
                } else {
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeError(
                            "Failed to delete converted {$format} file: {$converted_file}",
                            'ImageConverter'
                        );
                    }
                }
            }
        }

        // 2. Clean up thumbnail converted versions
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!empty($metadata['sizes']) && is_array($metadata['sizes'])) {
            $upload_dir = wp_get_upload_dir();
            $base_dir = dirname($original_file);
            
            foreach ($metadata['sizes'] as $size_name => $size_data) {
                if (empty($size_data['file'])) {
                    continue;
                }
                
                // Build thumbnail path
                $thumbnail_path = $base_dir . '/' . $size_data['file'];
                
                // Clean up converted versions of this thumbnail
                foreach ($formats_to_cleanup as $format) {
                    $converted_thumbnail = $thumbnail_path . '.' . $format;
                    
                    if (file_exists($converted_thumbnail)) {
                        $file_size = filesize($converted_thumbnail);
                        
                        if (@unlink($converted_thumbnail)) {
                            $deleted_files[] = basename($converted_thumbnail);
                            $total_size_saved += $file_size;
                            
                            if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                                \ZiziCache\CacheSys::writeLog(
                                    "Deleted converted {$format} thumbnail: {$converted_thumbnail} (" . size_format($file_size) . ")",
                                    'ImageConverter'
                                );
                            }
                        } else {
                            if (class_exists('\\ZiziCache\\CacheSys')) {
                                \ZiziCache\CacheSys::writeError(
                                    "Failed to delete converted {$format} thumbnail: {$converted_thumbnail}",
                                    'ImageConverter'
                                );
                            }
                        }
                    }
                }
            }
        }

        // Clean up metadata
        if (!empty($deleted_files)) {
            delete_post_meta($attachment_id, '_zizi_image_optimized');
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                $files_count = count($deleted_files);
                $files_preview = implode(', ', array_slice($deleted_files, 0, 5));
                $more_files = $files_count > 5 ? " ... +" . ($files_count - 5) . " more" : "";
                \ZiziCache\CacheSys::writeLog(
                    "Cleanup completed for attachment {$attachment_id}: deleted {$files_count} converted files, saved " . size_format($total_size_saved) . " (Files: {$files_preview}{$more_files})",
                    'ImageConverter'
                );
            }
        }
    }

    /**
     * Cleanup original file after successful conversion
     * Uses attachment metadata redirection instead of physical deletion
     * to avoid orphaned database records
     * 
     * @param int $attachment_id Attachment ID
     */
    /**
     * Clean up original file after successful conversion (async)
     * Based on CompressX pattern but with safety checks
     * 
     * @param int $attachment_id Attachment ID
     */
    public static function cleanup_original_after_conversion(int $attachment_id): void
    {
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Starting cleanup_original_after_conversion for attachment {$attachment_id}",
                'ImageConverter'
            );
        }

        // Double-check that cleanup is enabled
        if (self::should_keep_originals()) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Original cleanup disabled in settings for attachment {$attachment_id}",
                    'ImageConverter'
                );
            }
            return;
        }

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Keep originals is FALSE - proceeding with cleanup for attachment {$attachment_id}",
                'ImageConverter'
            );
        }

        // CRITICAL SAFETY CHECK: Verify converted files exist before deleting original
        if (!self::converted_files_exist($attachment_id)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "SAFETY ABORT: No converted files exist for attachment {$attachment_id}, skipping cleanup to prevent data loss",
                    'ImageConverter'
                );
            }
            return;
        }

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Converted files verified to exist for attachment {$attachment_id}, proceeding with cleanup",
                'ImageConverter'
            );
        }

        // Safety checks before deletion
        if (!self::can_safely_delete_original($attachment_id)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Safety check failed for original cleanup of attachment {$attachment_id}",
                    'ImageConverter'
                );
            }
            return;
        }

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Safety checks passed for attachment {$attachment_id}",
                'ImageConverter'
            );
        }

        // Verify converted files exist and are accessible
        $converted_files = get_post_meta($attachment_id, '_zizi_converted_files', true);
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Found converted files metadata for attachment {$attachment_id}: " . 
                (empty($converted_files) ? 'NONE' : json_encode($converted_files)),
                'ImageConverter'
            );
        }
        
        if (empty($converted_files)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "No converted files found for attachment {$attachment_id}, skipping cleanup",
                    'ImageConverter'
                );
            }
            return;
        }

        // Verify at least one converted file exists
        $has_valid_conversion = false;
        $valid_files = [];
        foreach ($converted_files as $format => $converted_path) {
            // Normalize path separators for Windows compatibility
            $normalized_path = str_replace('/', DIRECTORY_SEPARATOR, $converted_path);
            $exists = file_exists($normalized_path);
            $size = $exists ? filesize($normalized_path) : 0;
            $valid_files[] = "{$format}: {$normalized_path} (exists: " . ($exists ? 'YES' : 'NO') . ", size: {$size})";
            
            if ($exists && $size > 0) {
                $has_valid_conversion = true;
            }
            
            // Update the array with normalized path for further processing
            $converted_files[$format] = $normalized_path;
        }

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Converted files validation for attachment {$attachment_id}: " . implode(', ', $valid_files),
                'ImageConverter'
            );
        }

        if (!$has_valid_conversion) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "No valid converted files found for attachment {$attachment_id}, skipping cleanup",
                    'ImageConverter'
                );
            }
            return;
        }

        $original_file = get_attached_file($attachment_id);
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Original file path for attachment {$attachment_id}: {$original_file} (exists: " . 
                (file_exists($original_file) ? 'YES' : 'NO') . ")",
                'ImageConverter'
            );
        }
        
        if (!$original_file || !file_exists($original_file)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Original file not found for attachment {$attachment_id}: {$original_file}",
                    'ImageConverter'
                );
            }
            return;
        }

        // Perform cleanup by deleting original
        $backup_made = false;
        $config = SysConfig::$config;
        
        // Optionally create backup before deletion
        if ($config['image_converter_backup_before_delete'] ?? false) {
            $backup_path = $original_file . '.backup';
            if (copy($original_file, $backup_path)) {
                $backup_made = true;
            }
        }

        try {
            $deleted_files = [];
            $total_size_saved = 0;

            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog(
                    "Starting file deletion process for attachment {$attachment_id}",
                    'ImageConverter'
                );
            }

            // Delete original file
            if (file_exists($original_file)) {
                $original_size = filesize($original_file);
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Attempting to delete original file: {$original_file} (size: {$original_size})",
                        'ImageConverter'
                    );
                }
                
                if (wp_delete_file($original_file)) {
                    $deleted_files[] = 'original';
                    $total_size_saved += $original_size;
                    
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeLog(
                            "Successfully deleted original file: {$original_file}",
                            'ImageConverter'
                        );
                    }
                } else {
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeError(
                            "Failed to delete original file: {$original_file}",
                            'ImageConverter'
                        );
                    }
                }
            }

            // Delete original thumbnails
            $metadata = wp_get_attachment_metadata($attachment_id);
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                $sizes_count = !empty($metadata['sizes']) ? count($metadata['sizes']) : 0;
                \ZiziCache\CacheSys::writeLog(
                    "Found {$sizes_count} thumbnail sizes in metadata for attachment {$attachment_id}",
                    'ImageConverter'
                );
            }
            
            if (!empty($metadata['sizes'])) {
                $base_dir = dirname($original_file);
                
                foreach ($metadata['sizes'] as $size_name => $size_data) {
                    $thumbnail_path = $base_dir . '/' . $size_data['file'];
                    
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeLog(
                            "Checking thumbnail {$size_name}: {$thumbnail_path} (exists: " . 
                            (file_exists($thumbnail_path) ? 'YES' : 'NO') . ")",
                            'ImageConverter'
                        );
                    }
                    
                    if (file_exists($thumbnail_path)) {
                        $thumb_size = filesize($thumbnail_path);
                        if (wp_delete_file($thumbnail_path)) {
                            $deleted_files[] = "thumbnail-{$size_name}";
                            $total_size_saved += $thumb_size;
                            
                            if (class_exists('\\ZiziCache\\CacheSys')) {
                                \ZiziCache\CacheSys::writeLog(
                                    "Successfully deleted thumbnail {$size_name}: {$thumbnail_path}",
                                    'ImageConverter'
                                );
                            }
                        } else {
                            if (class_exists('\\ZiziCache\\CacheSys')) {
                                \ZiziCache\CacheSys::writeError(
                                    "Failed to delete thumbnail {$size_name}: {$thumbnail_path}",
                                    'ImageConverter'
                                );
                            }
                        }
                    }
                }
            }

            if (!empty($deleted_files)) {
                // Update attachment metadata to mark as converted-only
                update_post_meta($attachment_id, '_zizi_original_deleted', current_time('mysql'));
                
                // Update attachment file path to point to converted version
                $primary_format = array_keys($converted_files)[0] ?? 'webp';
                $converted_primary = $converted_files[$primary_format] ?? '';
                
                if ($converted_primary && file_exists($converted_primary)) {
                    update_attached_file($attachment_id, $converted_primary);
                }
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    $files_str = implode(', ', $deleted_files);
                    \ZiziCache\CacheSys::writeLog(
                        "Successfully deleted original files for attachment {$attachment_id}: {$files_str}, saved " . size_format($total_size_saved) .
                        ($backup_made ? " (backup created)" : ""),
                        'ImageConverter'
                    );
                }
                
                // Update WordPress metadata to point to converted files
                $metadata_updated = self::update_wordpress_metadata_after_cleanup($attachment_id, $converted_files);
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "WordPress metadata update " . ($metadata_updated ? 'successful' : 'failed') . " for attachment {$attachment_id}",
                        'ImageConverter'
                    );
                }
                
                // Clean WordPress file cache
                wp_cache_delete($attachment_id, 'posts');
                clean_attachment_cache($attachment_id);
                
            } else {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Failed to delete any original files for attachment {$attachment_id}",
                        'ImageConverter'
                    );
                }
            }
            
        } catch (\Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    "Exception during original cleanup for attachment {$attachment_id}: " . $e->getMessage(),
                    'ImageConverter'
                );
            }
        }
    }

    /**
     * Check if original file can be safely deleted
     *
                    \ZiziCache\CacheSys::writeLog(
                        "Original cleanup completed: Redirected attachment {$attachment_id} to {$best_format}, deleted {$original_file}",
                        'ImageConverter'
                    );
                    
                    // Mark as processed
                    update_post_meta($attachment_id, '_zizi_original_cleaned', time());
                    update_post_meta($attachment_id, '_zizi_redirected_to', $best_format);
                    
                } elseif (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeError(
                        "Failed to delete original file after redirect: {$original_file} (attachment {$attachment_id})",
                        'ImageConverter'
                    );
                }
                
                return;
            }
        }
        
        // Fallback: If redirection fails, log but don't break anything
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeError(
                "Could not redirect attachment {$attachment_id} to converted format - keeping original",
                'ImageConverter'
            );
        }
    }

    /**
     * Get the best converted format for cleanup redirection
     * 
     * @param int $attachment_id Attachment ID
     * @param array $available_formats Available converted formats
     * @return string|null Best format or null if none suitable
     */
    private static function get_best_converted_format(int $attachment_id, array $available_formats): ?string
    {
        // Priority: AVIF > WebP
        $format_priority = ['avif', 'webp'];
        
        foreach ($format_priority as $format) {
            if (in_array($format, $available_formats)) {
                return $format;
            }
        }
        
        return null;
    }

    /**
     * Check if original can be safely deleted
     * 
     * @param int $attachment_id WordPress attachment ID
     * @return bool True if safe to delete
     */
    private static function can_safely_delete_original(int $attachment_id): bool
    {
        // Check WooCommerce usage (leverage existing infrastructure)
        if (class_exists('\\ZiziCache\\WooCommerceThumbnailManager')) {
            if (\ZiziCache\WooCommerceThumbnailManager::is_image_used_elsewhere($attachment_id, 0)) {
                return false;
            }
        }

        // Check featured image usage
        if (self::is_featured_image($attachment_id)) {
            return false;
        }

        // Check content usage
        if (self::is_used_in_content($attachment_id)) {
            return false;
        }

        return true;
    }

    /**
     * Check if image is used as featured image
     * 
     * @param int $attachment_id WordPress attachment ID
     * @return bool True if used as featured image
     */
    private static function is_featured_image(int $attachment_id): bool
    {
        global $wpdb;

        $usage = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE p.post_status IN ('publish', 'private', 'draft')
             AND pm.meta_key = '_thumbnail_id'
             AND pm.meta_value = %d",
            $attachment_id
        ));

        return $usage > 0;
    }

    /**
     * Check if image is used in post content
     * 
     * @param int $attachment_id WordPress attachment ID
     * @return bool True if used in content
     */
    private static function is_used_in_content(int $attachment_id): bool
    {
        global $wpdb;

        $usage = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts}
             WHERE post_status IN ('publish', 'private')
             AND post_content LIKE %s",
            '%wp-image-' . $attachment_id . '%'
        ));

        return $usage > 0;
    }

    /**
     * Check if converted files exist for attachment
     * 
     * @param int $attachment_id WordPress attachment ID
     * @return bool True if converted files exist
     */
    private static function converted_files_exist(int $attachment_id): bool
    {
        $original_file = get_attached_file($attachment_id);
        if (!$original_file) {
            return false;
        }

        $formats = ['avif', 'webp'];
        foreach ($formats as $format) {
            $converted_file = $original_file . '.' . $format;
            if (file_exists($converted_file)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Start bulk cleanup of existing original files
     * 
     * @return array Result with success status and message
     */
    public static function start_bulk_original_cleanup(): array
    {
        if (SysConfig::get('image_converter_keep_originals', true)) {
            return [
                'success' => false,
                'message' => 'Original cleanup is disabled in settings'
            ];
        }

        // Get converted images
        $candidates = self::get_cleanup_candidates();

        if (empty($candidates)) {
            return [
                'success' => false,
                'message' => 'No original files can be safely deleted'
            ];
        }

        // Schedule cleanup actions with intervals
        $scheduled = 0;
        foreach ($candidates as $i => $attachment_id) {
            $delay = $i * 10; // 10 second intervals

            if (wp_schedule_single_event(
                time() + $delay,
                'zizi_cleanup_original_after_conversion',
                [$attachment_id]
            )) {
                $scheduled++;
            }
        }

        return [
            'success' => true,
            'message' => "Scheduled cleanup for {$scheduled} original files",
            'scheduled_count' => $scheduled,
            'total_candidates' => count($candidates)
        ];
    }

    /**
     * Get candidates for original cleanup
     * 
     * @return array Array of attachment IDs
     */
    private static function get_cleanup_candidates(): array
    {
        $query = new \WP_Query([
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => 100, // Limit for safety
            'meta_query' => [
                [
                    'key' => '_zizi_image_optimized',
                    'compare' => 'EXISTS'
                ]
            ]
        ]);

        $candidates = [];
        foreach ($query->posts as $post) {
            if (self::can_safely_delete_original($post->ID) && self::converted_files_exist($post->ID)) {
                $candidates[] = $post->ID;
            }
        }

        return $candidates;
    }

    /**
     * Get converted file path for specific format
     * 
     * @param string $original_path Original file path
     * @param string $format Target format
     * @return string Converted file path
     */
    public static function get_converted_file_path(string $original_path, string $format): string
    {
        return $original_path . '.' . $format;
    }

    /**
     * Register converted file in WordPress metadata
     * 
     * @param int $attachment_id Attachment ID
     * @param string $converted_file Converted file path
     * @param string $format Format (webp|avif)
     */
    public static function register_converted_file(int $attachment_id, string $converted_file, string $format): void
    {
        $existing_meta = get_post_meta($attachment_id, '_zizi_converted_files', true) ?: [];
        
        // Normalize path separators for Windows compatibility
        $normalized_path = str_replace('\\', '/', $converted_file);
        $existing_meta[$format] = $normalized_path;
        
        update_post_meta($attachment_id, '_zizi_converted_files', $existing_meta);
        
        // Also store conversion timestamp
        update_post_meta($attachment_id, '_zizi_conversion_date', current_time('mysql'));
        
        // Debug logging
        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
            \ZiziCache\CacheSys::writeLog(
                "Registered converted file for attachment {$attachment_id}: {$format} => {$normalized_path}",
                'ImageConverter'
            );
        }
    }

    /**
     * Update WordPress attachment metadata after cleanup to point to converted files
     * 
     * @param int $attachment_id Attachment ID
     * @param array $converted_files Array of converted files [format => path]
     * @return bool Success status
     */
    public static function update_wordpress_metadata_after_cleanup(int $attachment_id, array $converted_files): bool
    {
        if (empty($converted_files)) {
            return false;
        }

        // Get preferred format (AVIF > WebP)
        $preferred_format = 'avif';
        if (!isset($converted_files['avif']) && isset($converted_files['webp'])) {
            $preferred_format = 'webp';
        }

        if (!isset($converted_files[$preferred_format])) {
            return false;
        }

        $converted_main_path = $converted_files[$preferred_format];
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Updating WordPress metadata for attachment {$attachment_id} to use {$preferred_format} files",
                'ImageConverter'
            );
        }

        // Update the main attached file path
        $upload_dir = wp_upload_dir();
        $relative_path = str_replace($upload_dir['basedir'] . '/', '', $converted_main_path);
        $relative_path = str_replace('\\', '/', $relative_path); // Normalize slashes
        
        update_post_meta($attachment_id, '_wp_attached_file', $relative_path);

        // Update post MIME type
        $mime_type = $preferred_format === 'avif' ? 'image/avif' : 'image/webp';
        wp_update_post([
            'ID' => $attachment_id,
            'post_mime_type' => $mime_type
        ]);

        // Get current attachment metadata
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!$metadata || !is_array($metadata)) {
            $metadata = [];
        }

        // Update main file reference in metadata
        $metadata['file'] = $relative_path;

        // Update thumbnail sizes to point to converted versions
        if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
            $original_dir = dirname(get_attached_file($attachment_id));
            
            foreach ($metadata['sizes'] as $size_name => &$size_data) {
                if (isset($size_data['file'])) {
                    // Construct path to original thumbnail
                    $original_thumb = $original_dir . '/' . $size_data['file'];
                    $converted_thumb = $original_thumb . '.' . $preferred_format;
                    
                    // Check if converted thumbnail exists
                    $normalized_thumb_path = str_replace('/', DIRECTORY_SEPARATOR, $converted_thumb);
                    if (file_exists($normalized_thumb_path)) {
                        // Update filename to include format extension
                        $size_data['file'] = $size_data['file'] . '.' . $preferred_format;
                        $size_data['mime-type'] = $mime_type;
                        
                        if (class_exists('\\ZiziCache\\CacheSys') && WP_DEBUG) {
                            \ZiziCache\CacheSys::writeLog(
                                "Updated thumbnail {$size_name} metadata to: {$size_data['file']}",
                                'ImageConverter'
                            );
                        }
                    } else {
                        // Remove size if converted version doesn't exist
                        if (class_exists('\\ZiziCache\\CacheSys')) {
                            \ZiziCache\CacheSys::writeLog(
                                "Removing thumbnail {$size_name} metadata - converted version not found: {$normalized_thumb_path}",
                                'ImageConverter'
                            );
                        }
                        unset($metadata['sizes'][$size_name]);
                    }
                }
            }
        }

        // Save updated metadata
        $result = wp_update_attachment_metadata($attachment_id, $metadata);

        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "WordPress metadata update " . ($result ? 'successful' : 'failed') . " for attachment {$attachment_id}",
                'ImageConverter'
            );
        }

        return (bool) $result;
    }

    /**
     * Check if originals should be kept based on settings
     * 
     * @return bool True if originals should be kept
     */
    public static function should_keep_originals(): bool
    {
        $config = SysConfig::$config;
        return $config['image_converter_keep_originals'] ?? true;
    }

    /**
     * Schedule original file cleanup asynchronously
     * 
     * @param int $attachment_id Attachment ID
     */
    public static function schedule_original_cleanup(int $attachment_id): void
    {
        if (!function_exists('as_schedule_single_action')) {
            return; // Action Scheduler not available
        }

        // Schedule cleanup in 30 seconds to ensure conversion is complete
        as_schedule_single_action(
            time() + 30,
            'zizi_cleanup_original_after_conversion',
            [$attachment_id],
            'zizi-image-converter'
        );
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog(
                "Scheduled original cleanup for attachment {$attachment_id}",
                'ImageConverter'
            );
        }
    }

    /**
     * Convert image to WebP format using available methods
     * 
     * @param string $source_path Source image path
     * @param string $output_path Output WebP path
     * @param int $quality WebP quality (0-100)
     * @return bool Success status
     */
    public static function convert_to_webp_internal(string $source_path, string $output_path, int $quality = 80): bool
    {
        // Try Imagick first if available
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($source_path);
                $imagick->setImageFormat('WEBP');
                $imagick->setImageCompressionQuality($quality);
                $imagick->stripImage(); // Remove metadata
                
                $result = $imagick->writeImage($output_path);
                $imagick->destroy();
                
                if ($result) {
                    return true;
                }
            } catch (\Exception $e) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Imagick WebP conversion failed: " . $e->getMessage(),
                        'ImageConverter'
                    );
                }
            }
        }

        // Fallback to GD
        if (extension_loaded('gd') && function_exists('imagewebp')) {
            try {
                $image = self::create_image_from_file($source_path);
                if ($image) {
                    $result = imagewebp($image, $output_path, $quality);
                    imagedestroy($image);
                    return $result;
                }
            } catch (\Exception $e) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "GD WebP conversion failed: " . $e->getMessage(),
                        'ImageConverter'
                    );
                }
            }
        }

        return false;
    }

    /**
     * Convert image to AVIF format using available methods
     * 
     * @param string $source_path Source image path
     * @param string $output_path Output AVIF path
     * @param int $quality AVIF quality (0-100)
     * @return bool Success status
     */
    public static function convert_to_avif_internal(string $source_path, string $output_path, int $quality = 75): bool
    {
        // Try Imagick first if available and supports AVIF
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($source_path);
                
                // Check if AVIF is supported
                $formats = $imagick->queryFormats('AVIF');
                if (!empty($formats)) {
                    $imagick->setImageFormat('AVIF');
                    $imagick->setImageCompressionQuality($quality);
                    $imagick->stripImage(); // Remove metadata
                    
                    $result = $imagick->writeImage($output_path);
                    $imagick->destroy();
                    
                    if ($result) {
                        return true;
                    }
                }
            } catch (\Exception $e) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Imagick AVIF conversion failed: " . $e->getMessage(),
                        'ImageConverter'
                    );
                }
            }
        }

        // Try GD if available and supports AVIF (PHP 8.1+)
        if (extension_loaded('gd') && function_exists('imageavif')) {
            try {
                $image = self::create_image_from_file($source_path);
                if ($image) {
                    $result = imageavif($image, $output_path, $quality);
                    imagedestroy($image);
                    return $result;
                }
            } catch (\Exception $e) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "GD AVIF conversion failed: " . $e->getMessage(),
                        'ImageConverter'
                    );
                }
            }
        }

        return false;
    }

    /**
     * Create GD image resource from file
     * 
     * @param string $file_path Image file path
     * @return resource|false GD image resource or false on failure
     */
    private static function create_image_from_file(string $file_path)
    {
        $image_info = getimagesize($file_path);
        if (!$image_info) {
            return false;
        }

        $image = false;
        
        switch ($image_info[2]) {
            case IMAGETYPE_JPEG:
                $image = imagecreatefromjpeg($file_path);
                break;
            case IMAGETYPE_PNG:
                $image = imagecreatefrompng($file_path);
                // Convert palette images to truecolor for WebP compatibility
                if ($image && !imageistruecolor($image)) {
                    $truecolor_image = imagecreatetruecolor(imagesx($image), imagesy($image));
                    
                    // Preserve transparency for palette images
                    imagealphablending($truecolor_image, false);
                    imagesavealpha($truecolor_image, true);
                    $transparent = imagecolorallocatealpha($truecolor_image, 0, 0, 0, 127);
                    imagefill($truecolor_image, 0, 0, $transparent);
                    
                    // Copy palette image to truecolor
                    imagecopy($truecolor_image, $image, 0, 0, 0, 0, imagesx($image), imagesy($image));
                    imagedestroy($image);
                    $image = $truecolor_image;
                }
                break;
            case IMAGETYPE_GIF:
                $image = imagecreatefromgif($file_path);
                // Convert GIF to truecolor for WebP compatibility
                if ($image && !imageistruecolor($image)) {
                    $truecolor_image = imagecreatetruecolor(imagesx($image), imagesy($image));
                    
                    // Preserve transparency
                    imagealphablending($truecolor_image, false);
                    imagesavealpha($truecolor_image, true);
                    $transparent = imagecolorallocatealpha($truecolor_image, 0, 0, 0, 127);
                    imagefill($truecolor_image, 0, 0, 0, $transparent);
                    
                    imagecopy($truecolor_image, $image, 0, 0, 0, 0, imagesx($image), imagesy($image));
                    imagedestroy($image);
                    $image = $truecolor_image;
                }
                break;
            case IMAGETYPE_BMP:
                $image = function_exists('imagecreatefrombmp') ? imagecreatefrombmp($file_path) : false;
                break;
            case IMAGETYPE_WEBP:
                $image = function_exists('imagecreatefromwebp') ? imagecreatefromwebp($file_path) : false;
                break;
            default:
                return false;
        }
        
        return $image;
    }
}
