<?php
declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\Caching;
use ZiziCache\SysTool;
use ZiziCache\SysConfig;

/**
 * Image Optimization and Management
 * 
 * This class provides comprehensive image optimization features including:
 * - Lazy loading of images
 * - Responsive image handling with srcset
 * - Image dimension management
 * - WebP conversion
 * - Image compression and optimization
 * - Above-the-fold image prioritization
 * 
 * @package ZiziCache
 */
class Image
{
  private static array $images = [];

  public static function parse_images(string $html): void
  {
    // Remove tags that are not needed for image parsing
    $html_without_scripts = preg_replace(
      '/<script.*?<\/script>|<noscript.*?<\/noscript>|<template.*?<\/template>/is',
      '',
      $html
    );

    if ($html_without_scripts === null) {
        // Handle preg_replace error, perhaps log it or return early
        \ZiziCache\CacheSys::writeError("preg_replace failed in parse_images");
        self::$images = [];
        return;
    }

    // Find all images with src attribute
    preg_match_all('/<img[^>]+src=[\"\'][^>]+>/i', $html_without_scripts, $image_tags);
    $image_tags = $image_tags[0] ?? [];

    // Filter out base64 images
    $image_tags = array_filter($image_tags, function (string $image_tag): bool {
      return strpos($image_tag, 'data:image') === false;
    });

    // Parse image using HTML class
    $parsed_images = [];
    foreach($image_tags as $image_tag){
        try {
            $parsed_images[] = new HTML($image_tag);
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeError("Failed to parse image tag: {$image_tag} - " . $e->getMessage());
        }
    }

    // Store images in the static variable
    self::$images = $parsed_images;
  }

  public static function add_width_height(string $html): string // HTML parameter is not used, consider removing or using it.
  {
    if (empty(SysConfig::$config['img_width_height'])) {
      return $html;
    }

    try {
      foreach (self::$images as $image) {
        if (!$image instanceof HTML) continue; 
        // get src attribute
        $src = $image->src;
        if (empty($src) || !is_string($src)) {
            continue;
        }

        // Skip if both width and height are already set
        $current_width = $image->width;
        $current_height = $image->height;

        if (is_numeric($current_width) && is_numeric($current_height)) {
          continue;
        }

        // Get width and height
        $dimensions = self::get_dimensions($src);

        // Skip if no dimensions found
        if ($dimensions === false) {
          continue;
        }

        // Add missing width and height attributes
        $ratio = ($dimensions['height'] > 0) ? ($dimensions['width'] / $dimensions['height']) : 1;

        if (!is_numeric($current_width) && !is_numeric($current_height)) {
          $image->width = (string)$dimensions['width'];
          $image->height = (string)$dimensions['height'];
        } elseif (!is_numeric($current_width) && is_numeric($current_height)) {
          $image->width = (string)round(((float)$current_height * $ratio));
        } elseif (is_numeric($current_width) && !is_numeric($current_height)) {
          $image->height = ($ratio > 0) ? (string)round(((float)$current_width / $ratio)) : (string)$dimensions['height'];
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError("Error in add_width_height - " . $e->getMessage());
    }
    return $html; // Returning original $html as modifications are on self::$images
  }

  public static function exclude_above_fold(string $html): string // HTML parameter is not used
  {
    if (empty(SysConfig::$config['img_lazyload']) || empty(SysConfig::$config['img_lazyload_exclude_count'])) {
      return $html;
    }

    $count = (int)SysConfig::$config['img_lazyload_exclude_count'];

    foreach (self::$images as $key => $image) {
      if (!$image instanceof HTML) continue;
      if ($key < $count) {
        $image->loading = 'eager';
      }
    }
    return $html;
  }

  public static function lazy_load(string $html): string // HTML parameter is not used
  {
    if (empty(SysConfig::$config['img_lazyload'])) {
      return $html;
    }

    $default_exclude_keywords = ['eager', 'skip-lazy'];
    $user_exclude_keywords = SysConfig::$config['img_lazyload_excludes'] ?? [];

    // Merge default and user excluded keywords
    $exclude_keywords = array_merge($default_exclude_keywords, $user_exclude_keywords);

    try {
      foreach (self::$images as $image) {
        if (!$image instanceof HTML) continue;
        // Image is excluded from lazy loading
        if (SysTool::any_keywords_match_string($exclude_keywords, (string)$image)) {
          $image->loading = 'eager';
          $image->fetchpriority = 'high';
          $image->decoding = 'async';
        } else {
          $image->loading = 'lazy';
          $image->fetchpriority = 'low';
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError("Error in lazy_load (images) - " . $e->getMessage());
    }
    return $html;
  }

  public static function responsive_images(string $html): string // HTML parameter is not used
  {
    if (empty(SysConfig::$config['img_responsive'])) {
      return $html;
    }

    $site_url = site_url();
    // Get all images from the page
    $images_on_site = array_filter(self::$images, function ($image) use ($site_url): bool {
      return ($image instanceof HTML && !empty($image->src) && is_string($image->src) && strpos($image->src, $site_url) !== false);
    });

    try {
      foreach ($images_on_site as $image) {
        if (!$image instanceof HTML) continue;
        // Skip images with loading="eager" attribute
        if ($image->loading === 'eager') {
          continue;
        }

        // Skip SVG images
        if (!empty($image->src) && is_string($image->src) && strpos($image->src, '.svg') !== false) {
          continue;
        }

        // Skip if width and height are not set or not numeric
        $img_width = $image->width;
        $img_height = $image->height;
        if (!is_numeric($img_width) || !is_numeric($img_height) || (int)$img_width <= 0 || (int)$img_height <= 0) {
          continue;
        }

        $srcset = self::generate_srcset($image);
        if ($srcset) {
          $image->srcset = $srcset;
          // Set sizes="auto" if srcset is present or successfully generated
          $image->sizes = 'auto';
        } else {
          // Remove srcset and sizes if srcset is unavailable
          unset($image->srcset, $image->sizes);
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError("Error in responsive_images - " . $e->getMessage());
    }

    return $html;
  }

  /**
   * Generate srcset attribute for an image using WordPress functions
   * 
   * @param HTML $image The image HTML element
   * @return string|null The generated srcset attribute or null on failure
   */
  private static function generate_srcset(HTML $image): ?string
  {
    if (empty($image->src) || !is_string($image->src)) return null;
    // Extract the attachment ID from the image URL
    $attachment_id = attachment_url_to_postid(preg_replace('/-\d+x\d+/', '', $image->src)); 
    if($attachment_id === 0) return null; // attachment_url_to_postid returns 0 on failure

    // Use wp_get_attachment_image_srcset to generate the srcset
    $img_width = $image->width;
    $img_height = $image->height;
    if (!is_numeric($img_width) || !is_numeric($img_height)) return null;

    $srcset = wp_get_attachment_image_srcset($attachment_id, [(int)$img_width, (int)$img_height]);
    return $srcset === false ? null : $srcset;
  }

  /**
   * Replace Gravatar URLs with locally hosted versions
   * 
   * @param string $html The HTML content
   * @return string The modified HTML with local Gravatars
   */
  public static function localhost_gravatars(string $html): string
  {
    if (empty(SysConfig::$config['img_localhost_gravatar'])) {
      return $html;
    }

    try {
      foreach (self::$images as $image) {
        if (!$image instanceof HTML || empty($image->src) || !is_string($image->src) || strpos($image->src, 'gravatar.com/avatar/') === false) {
          continue;
        }

        // Get the self-hosted Gravatar URL for src
        $self_hosted_url_src = self::get_self_hosted_gravatar($image->src);
        if($self_hosted_url_src === null) continue;

        // Change src to self hosted url
        $image->src = $self_hosted_url_src;

        // Skip if image does not have srcset
        $current_srcset = $image->srcset;
        if (empty($current_srcset) || !is_string($current_srcset)) {
          continue;
        }
        
        $new_srcset_parts = [];
        foreach (explode(',', $current_srcset) as $descriptor) {
          $trimmed_descriptor = trim($descriptor);
          // Extract the URL before the first space
          // Use the entire descriptor if no space is found.
          $url_part = strtok($trimmed_descriptor, ' ');
          $size_part = strtok(' ');

          if($url_part === false) continue;

          // Get the self-hosted Gravatar URL for srcset
          $self_hosted_url_srcset = self::get_self_hosted_gravatar($url_part);
          if($self_hosted_url_srcset === null) {
            $new_srcset_parts[] = $trimmed_descriptor; // Keep original if self-hosting fails
            continue;
          }
          $new_srcset_parts[] = $self_hosted_url_srcset . ($size_part ? ' ' . $size_part : '');
        }
        $image->srcset = implode(', ', $new_srcset_parts);
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError("Error in localhost_gravatars - " . $e->getMessage());
    }

    return $html;
  }

  /**
   * Download and cache a Gravatar image locally
   * 
   * @param string $url The Gravatar URL to cache
   * @return string|null The local URL of the cached Gravatar or null on failure
   */
  private static function get_self_hosted_gravatar(string $url): ?string
  {
    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        return null;
    }
    
    // Security: SSRF protection - validate URL and check for private IP ranges
    $parsed_url = parse_url($url);
    if (isset($parsed_url['host'])) {
        $ip = gethostbyname($parsed_url['host']);
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
            // Block private/reserved IP ranges
            \ZiziCache\CacheSys::writeWarning("Blocked Gravatar request to private IP range: {$url}", 'Security');
            return null;
        }
    }
    
    $file_name = 'gravatar-' . substr(md5($url), 0, 12) . '.png';
    $cache_file_path = ZIZI_CACHE_CACHE_DIR . $file_name;

    if (!file_exists($cache_file_path) || filesize($cache_file_path) === 0) {
      $gravatar_request = wp_remote_get($url, [
          'timeout' => 10, // Security: Increased timeout for better reliability 
          'sslverify' => true, // Security: Enable SSL verification
          'user-agent' => 'ZiziCache/1.0 (+https://zizicache.com)',
          'headers' => [
              'Accept' => 'image/*'
          ]
      ]);
      if (is_wp_error($gravatar_request)) {
          \ZiziCache\CacheSys::writeError("Failed to fetch Gravatar from {$url}: " . $gravatar_request->get_error_message());
          return null;
      }
      $gravatar_body = wp_remote_retrieve_body($gravatar_request);
      if (empty($gravatar_body)) {
          error_log("ZiziCache: Empty response body for Gravatar from {$url}");
          return null;
      }
      $put_contents_result = file_put_contents($cache_file_path, $gravatar_body);
      if($put_contents_result === false){
          error_log("ZiziCache: Failed to write Gravatar to {$cache_file_path}");
          return null;
      }
    }

    return ZIZI_CACHE_CACHE_URL . $file_name;
  }

  /**
   * Write modified images back to the HTML
   * 
   * @param string $html The original HTML content
   * @return string The modified HTML with updated image tags
   */
  public static function write_images(string $html): string
  {
    foreach (self::$images as $image) {
      if (!$image instanceof HTML || empty($image->original_tag) || !is_string($image->original_tag)) continue;
      $html = str_replace($image->original_tag, (string)$image, $html);
    }
    self::$images = []; // Clear images after writing to prevent issues on subsequent calls or different pages
    return $html;
  }

  /**
   * Lazy load background images defined in style attributes
   * 
   * @param string $html The HTML content to process
   * @return string The modified HTML with lazy-loaded background images
   */
  public static function lazy_load_bg_style(string $html): string
  {
    if (empty(SysConfig::$config['img_lazyload'])) {
      return $html;
    }

    // Get excluded keywords
    $exclude_keywords = SysConfig::$config['img_lazyload_excludes'] ?? [];

    // Get all the elements with url in style attribute
    // Improved regex to be more specific and avoid catastrophic backtracking
    preg_match_all('/<([^>\s]+)\s*[^>]*style=([\'"])([^\'"]*url\(([^)]+)\)[^\'"]*)\2[^>]*>/i', $html, $elements, PREG_SET_ORDER);

    try {
      // Loop through elements
      foreach ($elements as $element_match) {
        $element_tag = $element_match[0];
        $current_style = $element_match[3]; // The full style attribute content

        $element = new HTML($element_tag);

        // Continue if element is excluded
        if (SysTool::any_keywords_match_string($exclude_keywords, $element_tag)) {
          continue;
        }

        // Lazy load background images by lazy loading style attribute
        $element->{'data-lazy-style'} = $current_style;
        $element->{'data-lazy-method'} = 'viewport';
        $element->{'data-lazy-attributes'} = 'style';
        unset($element->style);

        $html = str_replace($element_tag, (string)$element, $html);
      }
    } catch (\Exception $e) {
      error_log("ZiziCache: Error in lazy_load_bg_style - " . $e->getMessage());
    } finally {
      return $html;
    }
  }

  /**
   * Lazy load background images defined by CSS classes
   * 
   * @param string $html The HTML content to process
   * @return string The modified HTML with lazy-loaded background classes
   */
  public static function lazy_load_bg_class(string $html): string
  {
    if (empty(SysConfig::$config['img_lazyload'])) {
      return $html;
    }

    // get all elements with lazy-bg class
    // Improved regex to be more specific
    preg_match_all('/<([^>\s]+)\s*[^>]*class=([\'"])([^\'"]*lazy-bg[^\'"]*)\2[^>]*>/i', $html, $elements, PREG_SET_ORDER);

    try {
      foreach ($elements as $element_match) {
        $element_tag = $element_match[0];
        $current_class = $element_match[3]; // The full class attribute content

        $element = new HTML($element_tag);

        // Lazy load class attribute
        $element->{'data-lazy-class'} = $current_class;
        $element->{'data-lazy-method'} = 'viewport';
        $element->{'data-lazy-attributes'} = 'class';
        unset($element->class);

        // Lazy load id attribute
        if (!empty($element->id) && is_string($element->id)) {
          $id = $element->id;
          $element->{'data-lazy-id'} = $id;
          $current_attributes = $element->{'data-lazy-attributes'};
          $element->{'data-lazy-attributes'} = $current_attributes . ',id';
          unset($element->id);
        }

        $html = str_replace($element_tag, (string)$element, $html);
      }
    } catch (\Exception $e) {
      error_log("ZiziCache: Error in lazy_load_bg_class - " . $e->getMessage());
    } finally {
      return $html;
    }
  }

  /**
   * Add preload links for above-the-fold images
   * 
   * @param string $html The HTML content
   * @return string The modified HTML with preload links
   */
  public static function preload(string $html): string
  {
    if (empty(SysConfig::$config['img_preload'])) {
      return $html;
    }

    // Filter self::$images to get only images with loading="eager"
    $eager_images = array_filter(self::$images, function ($image): bool {
      return ($image instanceof HTML && $image->loading === 'eager');
    });

    if(empty($eager_images)){
        return $html;
    }

    $preload_image_tags_array = [];

    try {
      foreach ($eager_images as $image) {
        if (!$image instanceof HTML) continue;
        $src = $image->src;
        if(empty($src) || !is_string($src)) continue;

        $srcset = $image->srcset ?? '';
        $sizes = $image->sizes ?? '';
        $importance = $image->fetchpriority === 'high' ? 'high' : 'low';
        
        // Ensure attributes are properly escaped
        $escaped_src = esc_url($src);
        $escaped_srcset = !empty($srcset) && is_string($srcset) ? esc_attr($srcset) : '';
        $escaped_sizes = !empty($sizes) && is_string($sizes) ? esc_attr($sizes) : '';

        // Enhanced <link> tag with LCP attributes
        $link_tag = "<link rel='preload' href='".$escaped_src."' as='image' importance='".$importance."'";
        if(!empty($escaped_srcset)) $link_tag .= " imagesrcset='".$escaped_srcset."'";
        if(!empty($escaped_sizes)) $link_tag .= " imagesizes='".$escaped_sizes."'";
        $link_tag .= " />";
        $preload_image_tags_array[] = $link_tag;
      }

      // Get unique preload tags
      $preload_image_tags_array = array_unique($preload_image_tags_array);

      // Convert array to string
      $preload_image_tags_string = implode(PHP_EOL, $preload_image_tags_array);

      // Add preload tags after head tag opening or title tag closing
      if (strpos($html, '</title>') !== false) {
        $html = SysTool::str_replace_first(
          '</title>',
          '</title>' . PHP_EOL . $preload_image_tags_string,
          $html
        );
      } elseif (strpos($html, '<head>') !== false) {
         $html = SysTool::str_replace_first(
          '<head>',
          '<head>' . PHP_EOL . $preload_image_tags_string,
          $html
        );
      }

    } catch (\Exception $e) {
      error_log("ZiziCache: Error in preload (images) - " . $e->getMessage());
    } finally {
      return $html;
    }
  }

  /**
   * Applies Priority Hints to images based on their importance
   * 
   * @param string $html The HTML content
   * @return string The modified HTML with priority hints
   */
  public static function apply_priority_hints(string $html): string
  {
    if (empty(SysConfig::$config['img_priority_hints'])) {
      return $html;
    }

    try {
      foreach (self::$images as $image) {
        if (!$image instanceof HTML) continue;

        // Automatically assign high priority to above-the-fold images
        if ($image->loading === 'eager') {
          $image->fetchpriority = 'high';
          $image->importance = 'high';
        } else {
          // Set low priority for other images if they don't already have high priority
          if (empty($image->fetchpriority) || $image->fetchpriority !== 'high') {
            $image->fetchpriority = 'low';
            $image->importance = 'auto';
          }
        }
      }
    } catch (\Exception $e) {
      error_log("ZiziCache: Error in apply_priority_hints - " . $e->getMessage());
    }
    
    return $html;
  }

  /**
   * Automatic detection of LCP (Largest Contentful Paint) images
   * based on Layout Instability API data
   * 
   * @param string $html The HTML content
   * @return string The modified HTML with LCP optimizations
   */
  public static function auto_detect_lcp(string $html): string
  {
    if (empty(SysConfig::$config['img_auto_lcp'])) {
      return $html;
    }

    try {
      // Get LCP data from the database (if it exists)
      $lcp_data = self::get_lcp_data();
      
      if (empty($lcp_data)) {
        return $html;
      }
      
      // Identify the device type
      $is_mobile = wp_is_mobile();
      
      // Get LCP information for the current device type and URL
      $current_url = $_SERVER['REQUEST_URI'] ?? '/';
      $device_type = $is_mobile ? 'mobile' : 'desktop';
      
      // Find the record for the current URL and device type
      $lcp_item = null;
      foreach ($lcp_data as $item) {
        if ($item['url'] === $current_url && $item['deviceType'] === $device_type) {
          $lcp_item = $item;
          break;
        }
      }
      
      if (empty($lcp_item) || empty($lcp_item['src']) || empty($lcp_item['selector'])) {
        return $html;
      }
      
      // Process all images to find the LCP
      foreach (self::$images as $image) {
        if (!$image instanceof HTML) continue;
        
        $src = $image->src;
        if (empty($src) || !is_string($src)) continue;
        
        // Porovnejme URL obrázku s URL LCP
        if (strpos($src, $lcp_item['src']) !== false || 
            (string)$image === $lcp_item['selector']) {
          
          // We found the LCP image - set it with the highest priority
          $image->loading = 'eager';
          $image->fetchpriority = 'high';
          $image->importance = 'high';
          $image->decoding = 'sync'; // Synchronní dekódování pro LCP
          
          // Since we've found the LCP, we can break the loop
          break;
        }
      }
    } catch (\Exception $e) {
      error_log("ZiziCache: Error in auto_detect_lcp - " . $e->getMessage());
    }
    
    return $html;
  }

  /**
   * Get stored LCP (Largest Contentful Paint) element data
   * 
   * @return array LCP data
   */
  private static function get_lcp_data(): array
  {
    try {
      $lcp_data = get_transient('zizicache_lcp_data');
      if ($lcp_data === false) {
        return [];
      }
      
      return $lcp_data;
    } catch (\Exception $e) {
      error_log("ZiziCache: Error in get_lcp_data - " . $e->getMessage());
      return [];
    }
  }

  /**
   * Get image dimensions from URL or file
   * 
   * @param string $url The image URL
   * @return array|false Array with 'width' and 'height' or false on failure
   */
  private static function get_dimensions(string $url): array|false
  {
    try {
      if (empty($url) || !is_string($url)) return false;
      // Extract width if found the the url. For example something-100x100.jpg
      if (preg_match('/(?:.+)-([0-9]+)x([0-9]+)\.(jpg|jpeg|png|gif|webp)$/i', $url, $matches)) { // Added webp, case-insensitive
        list($_, $width, $height) = array_slice($matches, 1, 2); // Ensure correct extraction
        return ['width' => (int)$width, 'height' => (int)$height];
      }

      // Get width and height for Gravatar images
      if (strpos($url, 'gravatar.com/avatar/') !== false) {
        $query_string = parse_url($url, PHP_URL_QUERY);
        parse_str($query_string ?? '', $query_vars);
        $size = isset($query_vars['s']) && is_numeric($query_vars['s']) ? (int)$query_vars['s'] : 80;
        return ['width' => $size, 'height' => $size];
      }

      $file_path = \ZiziCache\CacheSys::get_file_path_from_url($url);

      if (empty($file_path) || !is_string($file_path) || !is_file($file_path) || !is_readable($file_path) || filesize($file_path) === 0) { // Added readable check and filesize > 0
        return false;
      }

      // Get width and height from svg
      if (pathinfo($file_path, PATHINFO_EXTENSION) === 'svg') {
        $xml_content = file_get_contents($file_path);
        if($xml_content === false) return false;
        $xml = @simplexml_load_string($xml_content); // Use @ to suppress errors on invalid XML
        if ($xml === false) return false;

        $attr = $xml->attributes();
        $width_attr = isset($attr->width) ? (string)$attr->width : null;
        $height_attr = isset($attr->height) ? (string)$attr->height : null;
        $viewbox_attr = isset($attr->viewBox) ? (string)$attr->viewBox : null;

        $width = null;
        $height = null;

        if ($width_attr && preg_match('/^\d+(\.?\d+)?(px)?$/', $width_attr, $value)) {
            $width = (int)$value[1];
        }
        if ($height_attr && preg_match('/^\d+(\.?\d+)?(px)?$/', $height_attr, $value)) {
            $height = (int)$value[1];
        }

        if ($width !== null && $height !== null) return ['width' => $width, 'height' => $height];
        
        if ($viewbox_attr) {
            $viewbox = preg_split('/\s+|,/', trim($viewbox_attr));
            if (count($viewbox) === 4 && is_numeric($viewbox[2]) && is_numeric($viewbox[3])) {
                 $vb_width = (int)$viewbox[2];
                 $vb_height = (int)$viewbox[3];
                 if ($width === null && $height === null) return ['width' => $vb_width, 'height' => $vb_height];
                 if ($width === null && $height !== null && $vb_height > 0) $width = (int)($height * $vb_width / $vb_height);
                 if ($height === null && $width !== null && $vb_width > 0) $height = (int)($width * $vb_height / $vb_width);
                 if ($width !== null && $height !== null) return ['width' => $width, 'height' => $height];
            }
        }
        return false; // Could not determine dimensions from SVG
      }

      // Get image size by checking the file
      $image_size = @getimagesize($file_path); // Use @ to suppress errors on invalid image files
      if ($image_size !== false && isset($image_size[0], $image_size[1])) {
        return ['width' => $image_size[0], 'height' => $image_size[1]];
      }
    } catch (\Exception $e) {
      error_log("ZiziCache: Error in get_dimensions for URL {$url} - " . $e->getMessage());
      return false;
    }
    return false; // Default return if no dimensions found
  }

  /**
   * Integration hook for Image Converter compatibility
   * 
   * Provides integration points for the Image Converter module
   * while maintaining separation of concerns and avoiding conflicts
   * with existing Image optimization features.
   * 
   * @return array Integration status and compatibility info
   */
  public static function get_image_converter_integration(): array
  {
    $integration = [
      'compatible' => true,
      'existing_features' => [
        'lazy_loading' => !empty(SysConfig::$config['img_lazyload']),
        'responsive_images' => !empty(SysConfig::$config['img_responsive']),
        'priority_hints' => !empty(SysConfig::$config['img_priority_hints']),
        'auto_lcp' => !empty(SysConfig::$config['img_auto_lcp']),
        'preload' => !empty(SysConfig::$config['img_preload']),
        'width_height' => !empty(SysConfig::$config['img_width_height'])
      ],
      'conflicts' => [],
      'recommendations' => []
    ];

    // Check for potential conflicts (none expected with current design)
    if (class_exists('\\ZiziCache\\ImageConverter')) {
      $integration['image_converter_available'] = true;
      $integration['image_converter_enabled'] = \ZiziCache\ImageConverter::is_enabled();
    } else {
      $integration['image_converter_available'] = false;
      $integration['image_converter_enabled'] = false;
    }

    // Add recommendations for optimal configuration
    if ($integration['image_converter_enabled']) {
      $integration['recommendations'][] = 'Image Converter works best with existing lazy loading and responsive image features enabled';
      
      if (!$integration['existing_features']['lazy_loading']) {
        $integration['recommendations'][] = 'Consider enabling lazy loading for better performance with optimized images';
      }
      
      if (!$integration['existing_features']['priority_hints']) {
        $integration['recommendations'][] = 'Priority hints can improve LCP with AVIF/WebP images';
      }
    }

    return $integration;
  }

  /**
   * Hook for Image Converter metadata integration
   * 
   * Allows Image Converter to provide additional metadata
   * without modifying core Image optimization logic
   * 
   * @param string $image_url Image URL
   * @param int|null $attachment_id Attachment ID if available
   * @return array Additional metadata from Image Converter
   */
  public static function get_image_converter_metadata(string $image_url, ?int $attachment_id = null): array
  {
    $metadata = [
      'optimized' => false,
      'formats' => [],
      'savings' => [],
      'original_size' => null,
      'optimized_sizes' => []
    ];

    // Only fetch metadata if Image Converter is available and enabled
    if (!class_exists('\\ZiziCache\\ImageConverter') || !ImageConverter::is_enabled()) {
      return $metadata;
    }

    if ($attachment_id) {
      $optimization_meta = get_post_meta($attachment_id, '_zizi_image_optimized', true);
      
      if (!empty($optimization_meta)) {
        $metadata['optimized'] = true;
        $metadata['formats'] = $optimization_meta['formats'] ?? [];
        $metadata['savings'] = $optimization_meta['savings'] ?? [];
        $metadata['original_size'] = $optimization_meta['sizes']['original'] ?? null;
        $metadata['optimized_sizes'] = array_filter($optimization_meta['sizes'] ?? [], function($key) {
          return $key !== 'original';
        }, ARRAY_FILTER_USE_KEY);
      }
    }

    return $metadata;
  }
}
