<?php
declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\SysConfig;
use ZiziCache\SysTool;
use Exception;

/**
 * IFrame Optimization and Management
 * 
 * This class provides optimization features for iframes, including:
 * - Lazy loading of iframe content
 * - Placeholder generation for YouTube and Vimeo videos
 * - Performance optimization for embedded content
 * - Responsive iframe handling
 * - Resource optimization for third-party embeds
 * 
 * @package ZiziCache
 */
class IFrame
{
    /**
     * Add YouTube video placeholders for lazy loading
     * 
     * Replaces YouTube iframe elements with optimized placeholders that load the actual
     * video only when clicked, improving page load performance.
     *
     * @param string $html The HTML content to process
     * @return string The modified HTML with YouTube placeholders
     */
    public static function add_youtube_placeholder(string $html): string
    {
        if (empty(SysConfig::$config['iframe_youtube_placeholder'])) {
            return $html;
        }

        // Get all iframes with YouTube sources
        // Improved regex for robustness and to prevent potential ReDoS with overly broad patterns.
        preg_match_all(
            '/<iframe[^>]+\\bsrc=(?:\"|\')(?:https?:)?\\/\\/(?:www\\.)?(?:youtube\\.com|youtu\\.be|youtube-nocookie\\.com)\\/[^\"\']+\\b[^>]*><\\/iframe>/i',
            $html,
            $iframes,
            PREG_SET_ORDER // Ensures $iframes[0] contains the full match for each iframe
        );

        if (empty($iframes)) {
            return $html;
        }

        $resolutions = [
            'default' => ['width' => 120, 'height' => 90], // Standard Definition
            'mqdefault' => ['width' => 320, 'height' => 180], // Medium Quality
            'hqdefault' => ['width' => 480, 'height' => 360], // High Quality
            'sddefault' => ['width' => 640, 'height' => 480], // Standard Definition (larger)
            'maxresdefault' => ['width' => 1280, 'height' => 720], // Maximum Resolution
        ];

        // Get quality setting from configuration
        $quality_setting = !empty(SysConfig::$config['iframe_youtube_quality']) ? SysConfig::$config['iframe_youtube_quality'] : 'auto';
        
        // Choose appropriate resolution based on quality setting
        $default_resolution = 'hqdefault'; // Default resolution (medium quality)
        
        if ($quality_setting === 'auto') {
            // Use medium quality for mobile, high quality for desktop
            $default_resolution = wp_is_mobile() ? 'hqdefault' : 'maxresdefault';
        } elseif ($quality_setting === 'high') {
            $default_resolution = 'maxresdefault';
        } elseif ($quality_setting === 'medium') {
            $default_resolution = 'hqdefault';
        }
        
        // Allow resolution override using filter
        $resolution_key = apply_filters('zizi_cache_youtube_placeholder_resolution', $default_resolution);
        
        if (!array_key_exists($resolution_key, $resolutions)) {
            $resolution_key = 'hqdefault'; // Fallback to a default if filter provides invalid key
        }
        $selected_resolution = $resolutions[$resolution_key];

        try {
            foreach ($iframes as $iframe_match) {
                $iframe_tag = $iframe_match[0];
                $iframe = new HTML($iframe_tag);
                
                $title = $iframe->title ?? 'YouTube Video'; // Fallback title
                $src_attr = $iframe->src;

                if (empty($src_attr) || !is_string($src_attr)) {
                    continue;
                }

                // Ensure autoplay parameter is correctly appended
                $src_attr .= (strpos($src_attr, '?') === false ? '?' : '&') . 'autoplay=1';
                
                $video_id = '';
                if (preg_match('/(?:\/|=)(.{11})(?:$|&|\?)/', $src_attr, $matches)) {
                    $video_id = $matches[1];
                }

                if (empty($video_id)) {
                    continue;
                }

                $local_placeholder_url = self::get_self_hosted_placeholder($video_id, $resolution_key);
                if ($local_placeholder_url === null) {
                    continue; // Skip if placeholder can't be fetched/created
                }

                $width = (string)$selected_resolution['width'];
                $height = (string)$selected_resolution['height'];
                $escaped_src = esc_url($src_attr); // Properly escape URL for attribute
                $escaped_title = esc_attr($title); // Properly escape title for alt attribute

                // Modernized placeholder HTML with better accessibility and structure
                // Added attributes for accessibility, keyboard control, and CLS prevention
                $aspect_ratio = isset($selected_resolution['height'], $selected_resolution['width']) 
                    ? (($selected_resolution['height'] / $selected_resolution['width']) * 100) 
                    : 56.25; // Default 16:9 aspect ratio
                
                $placeholder_tag = <<<HTML
<div class='zizi-cache-youtube' data-src='{$escaped_src}' style='padding-bottom:{$aspect_ratio}%'
    onclick='load_zizi_cache_youtube_video(this)' role='button' tabindex='0' 
    aria-label='Play video: {$escaped_title}'
    onkeydown="if(event.key==='Enter'||event.key===' '){event.preventDefault();load_zizi_cache_youtube_video(this);}">
    <img src='{$local_placeholder_url}' width='{$width}' height='{$height}' 
        alt='Video preview: {$escaped_title}' loading='lazy' decoding='async' />
    <svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 68 48' width='68' height='48' 
        aria-hidden='true' focusable='false'><path fill='red' 
        d='M66.52,7.74c-0.78-2.93-2.49-5.42-5.28-6.04C55.86,0.05,34,0,34,0S12.14,0.05,6.76,1.7
        c-2.79,0.62-4.5,3.11-5.28,6.04C0.05,10.67,0,18.11,0,24s0.05,13.33,1.48,16.26c0.78,2.93,
        2.49,5.42,5.28,6.04C12.14,47.95,34,48,34,48s21.86-0.05,27.24-1.7c2.79-0.62,4.5-3.11,
        5.28-6.04C67.95,37.33,68,29.89,68,24S67.95,10.67,66.52,7.74z'/><path d='M 45,24 27,14 27,34'
        fill='#fff'/></svg>
</div>
HTML;

                $html = str_replace($iframe_tag, $placeholder_tag, $html);
            }
            // Inject CSS and JS only if any placeholders were added
            if (!empty($iframes)) { // Check if any replacements were made
                $html = self::inject_css_js($html);
            }
        } catch (Exception $e) {
            // Log detailed error message, avoid exposing raw error to user if possible
            error_log("ZiziCache IFrame Optimizer (add_youtube_placeholder): Exception - " . $e->getMessage());
        } 
        // Removed finally block as return $html is now at the end of the try or initial if
        return $html;
    }

    /**
     * Lazy load iframes to improve page load performance
     * 
     * Adds lazy loading to iframes by replacing them with placeholders that load
     * the actual content when they come into the viewport.
     *
     * @param string $html The HTML content to process
     * @return string The modified HTML with lazy-loaded iframes
     */
    public static function lazy_load(string $html): string
    {
        if (empty(SysConfig::$config['iframe_lazyload'])) {
            return $html;
        }

        // Regex to find all iframes, including those already having data-src
        preg_match_all('/<iframe\\s+[^>]*>/i', $html, $iframe_tags);
        if (empty($iframe_tags[0])) {
            return $html;
        }            try {
            foreach ($iframe_tags[0] as $iframe_tag) {
                $iframe = new HTML($iframe_tag);

                // Skip iframes that are already lazy-loaded by this plugin or have no src
                if (!empty($iframe->{'data-lazy-src'})) {
                    continue;
                }
                $src = $iframe->src ?? $iframe->{'data-src'}; // Prefer src, fallback to data-src

                if (empty($src) || !is_string($src)) {
                    continue;
                }
                
                // Skip YouTube iframes if placeholder is active, as it handles its own lazy loading logic
                if (!empty(SysConfig::$config['iframe_youtube_placeholder']) && 
                    (strpos($src, 'youtube.com/') !== false || strpos($src, 'youtu.be/') !== false)) {
                    continue;    
                }
                
                // Skip Vimeo iframes if placeholder is active, as it handles its own lazy loading logic
                if (!empty(SysConfig::$config['iframe_vimeo_placeholder']) && 
                    strpos($src, 'player.vimeo.com/') !== false) {
                    continue;    
                }
                
                // Přeskočit iframy, které jsou v seznamu vyloučených
                $exclude_lazyload = false;
                if (!empty(SysConfig::$config['iframe_lazyload_excludes']) && is_array(SysConfig::$config['iframe_lazyload_excludes'])) {
                    $iframe_html = (string)$iframe;
                    foreach (SysConfig::$config['iframe_lazyload_excludes'] as $exclude_pattern) {
                        if (empty($exclude_pattern)) {
                            continue;
                        }
                        // Kontrola, zda iframe obsahuje vyloučený vzor
                        if (strpos($iframe_html, $exclude_pattern) !== false) {
                            $exclude_lazyload = true;
                            break;
                        }
                    }
                }
                
                if ($exclude_lazyload) {
                    continue;
                }
                // Před použitím attribute-based lazy-loadingu zkontrolujeme, jestli je používán webový prohlížeč,
                // který podporuje nativní lazy-loading. Pokud ano, použijeme výhradně HTML atributy,
                // jinak použijeme kombinaci nativního a JavaScript přístupu.
                
                // Ujistit se, že iframy mají správné rozměry pro zabránění CLS
                $width = null;
                $height = null;
                
                // Extrahovat rozměry z atributů
                if (isset($iframe->width) && isset($iframe->height)) {
                    $width = str_replace('px', '', $iframe->width);
                    $height = str_replace('px', '', $iframe->height);
                }
                
                // Aplikovat optimalizaci atributů, pokud je povolena
                if (!empty(SysConfig::$config['iframe_optimize_attributes'])) {
                    // Nastavení loading='lazy' pro nativní lazy-loading
                    $iframe->loading = 'lazy';
                    
                    // Nastavit fetchpriority='low' pokud ještě není nastaveno
                    if (!isset($iframe->fetchpriority)) {
                        $iframe->fetchpriority = 'low';
                    }
                    
                    // Nastavit importance='low' pro zpětnou kompatibilitu
                    if (!isset($iframe->importance)) {
                        $iframe->importance = 'low';
                    }
                    
                    // Přidat referrerpolicy='no-referrer-when-downgrade' pro bezpečnost a soukromí
                    if (!isset($iframe->referrerpolicy)) {
                        $iframe->referrerpolicy = 'no-referrer-when-downgrade';
                    }
                    
                    // Přidat title pokud není nastaven, pro lepší přístupnost
                    if (!isset($iframe->title)) {
                        $iframe->title = 'Embedded content';
                    }
                    
                    // Zajistit, že má iframe správné rozměry pro Core Web Vitals (CLS)
                    if ($width && $height && is_numeric($width) && is_numeric($height) && (int)$width > 0 && (int)$height > 0) {
                        // Použít aspect-ratio pro moderní prohlížeče a padding-bottom jako fallback
                        $aspect_ratio = (float)$height / (float)$width;
                        $aspect_percent = $aspect_ratio * 100;
                        
                        $existing_style = isset($iframe->style) ? $iframe->style . '; ' : '';
                        $iframe->style = $existing_style . "aspect-ratio: {$width} / {$height}; height: auto;";
                        
                        // Přidat data atributy pro JS fallback
                        $iframe->{'data-aspect-ratio'} = $aspect_ratio;
                        $iframe->{'data-width'} = $width;
                        $iframe->{'data-height'} = $height;
                    }
                }
                
                // Basic data attribute for lazy-loading
                $iframe->{'data-src'} = $src;

                // If the iframe is really large and deferring its loading is effective,
                // we use JavaScript approach together with native lazy-loading
                if ((!$width || !$height || (int)$width > 300 || (int)$height > 300)) {
                    unset($iframe->src);
                    $iframe->{'data-lazy-src'} = $src;
                    $iframe->{'data-lazy-method'} = 'viewport';
                    $iframe->{'data-lazy-attributes'} = 'src';
                    
                    // Placeholder for iframe, which helps with CLS (Cumulative Layout Shift)
                    // and gives browsers an indication that something will be loaded
                    if ($width && $height) {
                        $iframe->srcdoc = "
                            <style>
                            * {padding:0;margin:0;overflow:hidden;box-sizing:border-box;}
                            body {width:100%;height:100%;display:flex;align-items:center;justify-content:center;background:#f9f9f9;color:#999;}
                            div {text-align:center;padding:20px;}
                            </style>
                            <div>Loading content...</div>
                        ";
                    }
                } else {
                    // For small iframes, there is no need to defer loading - use native lazy-loading
                    $iframe->src = $src;
                }

                $html = str_replace($iframe_tag, (string)$iframe, $html);
            }
        } catch (Exception $e) {
            // Log error for debugging, avoid breaking the site for users
            error_log("ZiziCache IFrame Optimizer (lazy_load): Exception - " . $e->getMessage());
        }

        return $html;
    }

    /**
     * Add Vimeo video placeholders for lazy loading
     *
     * Replaces Vimeo iframe elements with optimized placeholders that load the actual
     * video only when clicked, improving page load performance.
     *
     * @param string $html The HTML content to process
     * @return string The modified HTML with Vimeo placeholders
     */
    public static function add_vimeo_placeholder(string $html): string
    {
        if (empty(SysConfig::$config['iframe_vimeo_placeholder'])) {
            return $html;
        }

        // Regex to find Vimeo iframe
        preg_match_all(
            '/<iframe[^>]+\\bsrc=(?:\"|\')(?:https?:)?\\/\\/(?:www\\.)?player\.vimeo\.com\\/video\\/([0-9]+)[^\"\']*\\b[^>]*><\\/iframe>/i',
            $html,
            $iframes,
            PREG_SET_ORDER
        );

        if (empty($iframes)) {
            return $html;
        }

        $placeholders_added = false;

        try {
            foreach ($iframes as $iframe_match) {
                $iframe_tag = $iframe_match[0];
                $video_id = $iframe_match[1]; // Vimeo ID captured in regex group
                
                $iframe = new HTML($iframe_tag);
                
                $title = $iframe->title ?? 'Vimeo Video'; // Fallback title
                $src_attr = $iframe->src;

                if (empty($src_attr) || !is_string($src_attr)) {
                    continue;
                }

                // Ensure autoplay parameter is correctly appended
                if (strpos($src_attr, 'autoplay=') === false) {
                    $src_attr .= (strpos($src_attr, '?') === false ? '?' : '&') . 'autoplay=1';
                }

                // Get placeholder image
                $local_placeholder_url = self::get_self_hosted_placeholder($video_id, '', 'vimeo');
                if ($local_placeholder_url === null) {
                    continue; // Skip if placeholder can't be fetched/created
                }

                // Get width and height for aspect ratio
                $width = $iframe->width ?? '640';
                $height = $iframe->height ?? '360';
                
                // Calculate aspect ratio to prevent CLS
                $aspect_ratio = 56.25; // Default ratio 16:9 (56.25%)
                if (is_numeric($width) && is_numeric($height) && (int)$width > 0 && (int)$height > 0) {
                    $aspect_ratio = ((int)$height / (int)$width) * 100;
                }

                $escaped_src = esc_url($src_attr); // Properly escape URL for attribute
                $escaped_title = esc_attr($title); // Properly escape title for alt attribute

                // Create HTML for placeholder with aspect ratio
                $placeholder_tag = <<<HTML
<div class='zizi-cache-vimeo' data-src='{$escaped_src}' style='padding-bottom:{$aspect_ratio}%'
    data-video-id='{$video_id}' data-video-type='vimeo'
    onclick='load_zizi_cache_vimeo_video(this)' role='button' tabindex='0' 
    aria-label='Play video: {$escaped_title}'
    onkeydown="if(event.key==='Enter'||event.key===' '){event.preventDefault();load_zizi_cache_vimeo_video(this);}">
    <img src='{$local_placeholder_url}' width='{$width}' height='{$height}' 
        alt='Video thumbnail: {$escaped_title}' loading='lazy' decoding='async' />
    <svg xmlns='http://www.w3.org/2000/svg' width='56' height='56' viewBox='0 0 56 56' aria-hidden='true' focusable='false'>
        <path fill='#00adef' d='M67.6,15.2c-0.6-2.3-2.4-4.1-4.7-4.7C57.4,9,34,9,34,9s-23.4,0-28.9,1.5c-2.3,0.6-4.1,2.4-4.7,4.7 C0,21.8,0,33,0,33s0,11.2,1.5,16.8c0.6,2.3,2.4,4.1,4.7,4.7C10.6,56,34,56,34,56s23.4,0,28.9-1.5c2.3-0.6,4.1-2.4,4.7-4.7 C69,44.2,69,33,69,33S69,21.8,67.6,15.2z'/>
        <path fill='#FFF' d='M27,42l18-9L27,24V42z'/>
    </svg>
    <span class='zizi-cache-vimeo-title'>{$escaped_title}</span>
</div>
HTML;

                $html = str_replace($iframe_tag, $placeholder_tag, $html);
                $placeholders_added = true;
            }

            // Inject CSS and JS only if any placeholders were added
            if ($placeholders_added) {
                $html = self::inject_css_js($html);
            }
        } catch (Exception $e) {
            // Log detailed error message
            error_log("ZiziCache IFrame Optimizer (add_vimeo_placeholder): Exception - " . $e->getMessage());
        }
        
        return $html;
    }

    /**
     * Inject necessary CSS and JavaScript for video placeholders
     * 
     * Adds the required styles and scripts to handle the video placeholders,
     * including click handlers and animations.
     *
     * @param string $html The HTML content to process
     * @return string The HTML with injected CSS and JavaScript
     */
    private static function inject_css_js(string $html): string
    {
        // JavaScript for YouTube and Vimeo placeholders - optimized and minified, with performance monitoring
        $script = "
        function load_zizi_cache_youtube_video(t) {
            // Web Vitals monitoring - start
            const startTime = performance.now();
            let videoLoadedEvent;
            
            // Main function for loading video iframe
            const e = document.createElement('iframe');
            e.setAttribute('src', t.getAttribute('data-src'));
            e.setAttribute('frameborder', '0');
            e.setAttribute('allowfullscreen', '1');
            e.setAttribute('allow', 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture');
            e.setAttribute('loading', 'eager'); // Requires quick loading, as user clicked
            e.setAttribute('importance', 'high'); // Resource Hints Priority
            
            // Monitoring video load event for Web Vitals
            e.addEventListener('load', function() {
                const loadTime = performance.now() - startTime;
                if (window.zizi_cache_video_metrics === undefined) {
                    window.zizi_cache_video_metrics = [];
                }
                window.zizi_cache_video_metrics.push({
                    type: 'video_load',
                    duration: loadTime,
                    timestamp: new Date().toISOString(),
                    url: t.getAttribute('data-src')
                });
                
                // Notification for analytics (if exists)
                if (typeof window.ziziCacheAnalytics !== 'undefined') {
                    window.ziziCacheAnalytics('video_loaded', {
                        loadTime: loadTime,
                        videoType: t.getAttribute('data-video-type') || 'youtube',
                        videoId: t.getAttribute('data-video-id') || 'unknown'
                    });
                }
            });
            
            t.innerHTML = '';
            t.appendChild(e);
            t.classList.add('loaded');
            
            // Send interaction information to Core Web Vitals API (if available)
            if ('function' === typeof window.reportInteraction) {
                window.reportInteraction('video-play');
            }
        }
        
        function load_zizi_cache_vimeo_video(t) {
            // Similar implementation as for YouTube, but for Vimeo
            const e = document.createElement('iframe');
            e.setAttribute('src', t.getAttribute('data-src'));
            e.setAttribute('frameborder', '0');
            e.setAttribute('allowfullscreen', '1');
            e.setAttribute('allow', 'autoplay; fullscreen; picture-in-picture');
            e.setAttribute('loading', 'eager');
            e.setAttribute('importance', 'high');
            
            t.innerHTML = '';
            t.appendChild(e);
            t.classList.add('loaded');
        }";

        // CSS for YouTube/Vimeo placeholder - focused on CLS prevention and UX
        $css = "
        .zizi-cache-youtube,
        .zizi-cache-vimeo {
            position: relative;
            width: 100%;
            overflow: hidden;
            cursor: pointer;
            background-color: #000;
            transition: all 0.2s ease-out;
            margin: 0 auto;
        }
        .zizi-cache-youtube:hover,
        .zizi-cache-vimeo:hover {
            filter: brightness(1.1);
        }
        .zizi-cache-youtube:focus,
        .zizi-cache-vimeo:focus {
            outline: 3px solid #0078d4;
            filter: brightness(1.1);
        }
        .zizi-cache-youtube img,
        .zizi-cache-vimeo img {
            width: 100%;
            height: 100%;
            position: absolute;
            top: 0;
            left: 0;
            object-fit: cover;
            margin: 0;
            opacity: 0.85;
            transition: opacity 0.3s;
        }
        .zizi-cache-youtube:hover img,
        .zizi-cache-vimeo:hover img {
            opacity: 1;
        }
        .zizi-cache-youtube svg,
        .zizi-cache-vimeo svg {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%,-50%);
            width: 68px;
            height: 48px;
            transition: all 0.2s ease-out;
            filter: drop-shadow(0 1px 3px rgba(0,0,0,0.3));
        }
        .zizi-cache-youtube:hover svg,
        .zizi-cache-vimeo:hover svg {
            transform: translate(-50%,-50%) scale(1.1);
        }
        .zizi-cache-youtube.loaded iframe,
        .zizi-cache-vimeo.loaded iframe {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            border: 0;
        }
        .zizi-cache-youtube-title,
        .zizi-cache-vimeo-title {
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            background: rgba(0,0,0,0.7);
            color: #fff;
            padding: 8px;
            font-size: 14px;
            text-overflow: ellipsis;
            overflow: hidden;
            white-space: nowrap;
            opacity: 0;
            transition: opacity 0.3s;
        }
        .zizi-cache-youtube:hover .zizi-cache-youtube-title,
        .zizi-cache-vimeo:hover .zizi-cache-vimeo-title {
            opacity: 1;
        }
        
        /* CSS for ensuring correct aspect ratio and CLS prevention */
        .zizi-cache-youtube[data-aspect-ratio],
        .zizi-cache-vimeo[data-aspect-ratio] {
            height: 0;
            overflow: hidden;
        }
        
        /* Loader animation during loading */
        .zizi-cache-youtube::after,
        .zizi-cache-vimeo::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            opacity: 0;
            transform: translate(-50%, -50%);
            border-radius: 50%;
            background-color: rgba(255,255,255,0.2);
            transition: all 0.3s;
        }
        .zizi-cache-youtube.loading::after,
        .zizi-cache-vimeo.loading::after {
            width: 60px;
            height: 60px;
            opacity: 1;
            animation: zizi-pulse 1s infinite;
        }
        
        @keyframes zizi-pulse {
            0% { transform: translate(-50%, -50%) scale(0.8); opacity: 0.8; }
            50% { transform: translate(-50%, -50%) scale(1); opacity: 0.5; }
            100% { transform: translate(-50%, -50%) scale(0.8); opacity: 0.8; }
        }";

        // Inject CSS and JS into HTML
        $html = str_replace('</head>', "<style id='zizi-cache-video-styles'>{$css}</style></head>", $html);
        $html = str_replace('</body>', "<script id='zizi-cache-video-script'>{$script}</script></body>", $html);

        return $html;
    }

    /**
     * Get or generate a self-hosted thumbnail for a video
     * 
     * Downloads and caches video thumbnails from YouTube or Vimeo to improve
     * performance and reduce external dependencies.
     *
     * @param string $video_id The ID of the video
     * @param string $resolution_key The resolution key for the thumbnail (e.g., 'hqdefault', 'maxresdefault')
     * @param string $platform The video platform ('youtube' or 'vimeo')
     * @return string|null The URL of the cached thumbnail, or null on failure
     */
    private static function get_self_hosted_placeholder(string $video_id, string $resolution_key, string $platform = 'youtube'): ?string
    {
        $upload_dir = wp_upload_dir();
        
        if ($platform === 'youtube') {
            $cache_dir = $upload_dir['basedir'] . '/zizi-cache/youtube-thumbnails/';
            $cache_url = $upload_dir['baseurl'] . '/zizi-cache/youtube-thumbnails/';
            
            // Create cache directory if it doesn't exist
            if (!file_exists($cache_dir)) {
                wp_mkdir_p($cache_dir);
            }

            // Placeholder file name
            $placeholder_file = $cache_dir . $video_id . '-' . $resolution_key . '.jpg';
            $placeholder_url = $cache_url . $video_id . '-' . $resolution_key . '.jpg';

            // Check if placeholder already exists
            if (file_exists($placeholder_file) && filesize($placeholder_file) > 0) {
                return $placeholder_url;
            }

            // If it doesn't exist, download it from YouTube
            try {
                $image_url = "https://img.youtube.com/vi/{$video_id}/{$resolution_key}.jpg";
                
                // Security: SSRF protection - validate URL and check for private IP ranges
                $parsed_url = parse_url($image_url);
                if (isset($parsed_url['host'])) {
                    $ip = gethostbyname($parsed_url['host']);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
                        // Block private/reserved IP ranges
                        \ZiziCache\CacheSys::writeWarning("Blocked YouTube request to private IP range: {$image_url}", 'Security');
                        return null;
                    }
                }
                
                $response = wp_remote_get($image_url, [
                    'timeout' => 10, // Security: Increased timeout for better reliability
                    'sslverify' => true, // Security: Enable SSL verification
                    'user-agent' => 'ZiziCache/1.0 (+https://zizicache.com)'
                ]);

                if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                    // Try fallback to lower quality if we requested higher
                    if ($resolution_key === 'maxresdefault' || $resolution_key === 'sddefault') {
                        return self::get_self_hosted_placeholder($video_id, 'hqdefault');
                    }
                    return null;
                }

                $image_data = wp_remote_retrieve_body($response);
                if (!empty($image_data)) {
                    file_put_contents($placeholder_file, $image_data);
                    return $placeholder_url;
                }
            } catch (Exception $e) {
                error_log("ZiziCache Error: Failed to download YouTube thumbnail for {$video_id}: " . $e->getMessage());
            }
        }
        elseif ($platform === 'vimeo') {
            $cache_dir = $upload_dir['basedir'] . '/zizi-cache/vimeo-thumbnails/';
            $cache_url = $upload_dir['baseurl'] . '/zizi-cache/vimeo-thumbnails/';
            
            // Create cache directory if it doesn't exist
            if (!file_exists($cache_dir)) {
                wp_mkdir_p($cache_dir);
            }

            // Placeholder file name
            $placeholder_file = $cache_dir . $video_id . '.jpg';
            $placeholder_url = $cache_url . $video_id . '.jpg';

            // Check if placeholder already exists
            if (file_exists($placeholder_file) && filesize($placeholder_file) > 0) {
                return $placeholder_url;
            }

            // If it doesn't exist, download it from Vimeo API
            try {
                // Get video information from Vimeo API
                $api_url = "https://vimeo.com/api/v2/video/{$video_id}.json";
                
                // Security: SSRF protection - validate URL and check for private IP ranges  
                $parsed_url = parse_url($api_url);
                if (isset($parsed_url['host'])) {
                    $ip = gethostbyname($parsed_url['host']);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
                        // Block private/reserved IP ranges
                        \ZiziCache\CacheSys::writeWarning("Blocked Vimeo API request to private IP range: {$api_url}", 'Security');
                        return null;
                    }
                }
                
                $response = wp_remote_get($api_url, [
                    'timeout' => 10, // Security: Increased timeout for better reliability
                    'sslverify' => true, // Security: Enable SSL verification
                    'user-agent' => 'ZiziCache/1.0 (+https://zizicache.com)'
                ]);

                if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                    return null;
                }

                $video_data = json_decode(wp_remote_retrieve_body($response), true);
                
                if (empty($video_data[0]['thumbnail_large'])) {
                    return null;
                }
                
                // Download thumbnail
                $thumbnail_url = $video_data[0]['thumbnail_large'];
                
                // Security: SSRF protection for thumbnail download
                $parsed_thumbnail_url = parse_url($thumbnail_url);
                if (isset($parsed_thumbnail_url['host'])) {
                    $ip = gethostbyname($parsed_thumbnail_url['host']);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
                        // Block private/reserved IP ranges
                        \ZiziCache\CacheSys::writeWarning("Blocked Vimeo thumbnail request to private IP range: {$thumbnail_url}", 'Security');
                        return null;
                    }
                }
                
                $image_response = wp_remote_get($thumbnail_url, [
                    'timeout' => 10, // Security: Increased timeout for better reliability
                    'sslverify' => true, // Security: Enable SSL verification
                    'user-agent' => 'ZiziCache/1.0 (+https://zizicache.com)'
                ]);

                if (is_wp_error($image_response) || wp_remote_retrieve_response_code($image_response) !== 200) {
                    return null;
                }
                
                $image_data = wp_remote_retrieve_body($image_response);
                if (!empty($image_data)) {
                    file_put_contents($placeholder_file, $image_data);
                    return $placeholder_url;
                }
            } catch (Exception $e) {
                error_log("ZiziCache Error: Failed to download Vimeo thumbnail for {$video_id}: " . $e->getMessage());
            }
        }

        return null;
    }
}
