<?php
namespace ZiziCache;

/**
 * Class Htaccess
 *
 * Handles automatic management of ZiziCache .htaccess rules.
 */
class Htaccess
{
    /**
     * Constants for .htaccess markers
     * 
     * These constants define the start and end markers used to identify
     * the ZiziCache sections in the .htaccess file, along with regex patterns
     * to match the entire sections for updates or removal.
     */
    private const MARKER_BEGIN = '# BEGIN ZiziCache';
    private const MARKER_END   = '# END ZiziCache';
    private const MARKER_REGEX = '/# BEGIN ZiziCache[^\n]*\n.*?# END ZiziCache[^\n]*\n?/s';
    
    // Image Converter section markers (independent of main section)
    private const IMG_MARKER_BEGIN = '# BEGIN ZiziCache-ImageConverter';
    private const IMG_MARKER_END   = '# END ZiziCache-ImageConverter';
    private const IMG_MARKER_REGEX = '/# BEGIN ZiziCache-ImageConverter[^\n]*\n.*?# END ZiziCache-ImageConverter[^\n]*\n?/s';
    
    // Combined regex to match ALL ZiziCache sections
    private const ALL_SECTIONS_REGEX = '/# BEGIN ZiziCache(?:-ImageConverter)?[^\n]*\n.*?# END ZiziCache(?:-ImageConverter)?[^\n]*\n?/s';

    /**
     * Initializes the Htaccess management functionality.
     * 
     * Registers WordPress hooks for plugin activation, deactivation, and configuration updates
     * to automatically manage .htaccess rules. This ensures the cache rules are properly
     * maintained throughout the plugin lifecycle.
     *
     * @return void
     */
    public static function init()
    {
        // Add rules on plugin activation
        register_activation_hook(ZIZI_CACHE_FILE, [__CLASS__, 'add_htaccess_rules']);
        // Remove rules on plugin deactivation
        register_deactivation_hook(ZIZI_CACHE_FILE, [__CLASS__, 'remove_htaccess_rules']);
        // Update rules when plugin configuration changes
        add_action('zizi_cache_update_config:after', [__CLASS__, 'add_htaccess_rules']);
    }

    /**
     * Adds or updates ZiziCache rules in the .htaccess file.
     * 
     * This method handles the insertion or update of cache rules in the WordPress
     * root .htaccess file. It manages both main cache rules and Image Converter rules
     * as separate, independent sections to avoid conflicts and ensure proper removal.
     *
     * @return void
     * @throws Exception If there's an issue reading or writing the .htaccess file
     */
    public static function add_htaccess_rules()
    {
        $htaccess_file = ABSPATH . '.htaccess';
        if (!self::is_writeable($htaccess_file)) {
            // Log this for debugging
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog('Cannot write to .htaccess file: ' . $htaccess_file, 'Security');
            }
            return;
        }

        try {
            $htaccess_contents = self::read_file($htaccess_file);
            
            // Generate main cache rules (without Image Converter)
            $main_rules = self::generate_main_rules();
            
            // Validate main rules
            if (empty($main_rules) || 
                strpos($main_rules, self::MARKER_BEGIN) === false || 
                strpos($main_rules, self::MARKER_END) === false) {
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('Invalid main .htaccess rules generated, aborting write operation', 'Security');
                }
                return;
            }

            // Create backup of current content for safety
            $backup_content = $htaccess_contents;

            // Update or add main ZiziCache rules
            $htaccess_contents = self::update_section($htaccess_contents, $main_rules, self::MARKER_REGEX, 'main');
            
            // Handle Image Converter rules separately
            $config = SysConfig::$config;
            if (($config['image_converter_enabled'] ?? false) && ($config['image_converter_htaccess_rules'] ?? false)) {
                $img_rules = self::generate_image_converter_rules();
                
                if (!empty($img_rules) && 
                    strpos($img_rules, self::IMG_MARKER_BEGIN) !== false && 
                    strpos($img_rules, self::IMG_MARKER_END) !== false) {
                    
                    $htaccess_contents = self::update_section($htaccess_contents, $img_rules, self::IMG_MARKER_REGEX, 'image converter');
                } else {
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeLog('Invalid Image Converter .htaccess rules generated', 'Security');
                    }
                }
            } else {
                // Remove Image Converter rules if disabled
                $htaccess_contents = self::remove_section($htaccess_contents, self::IMG_MARKER_REGEX, 'image converter');
            }

            // Final validation: ensure we haven't corrupted the content
            if (strlen($htaccess_contents) < strlen($backup_content) * 0.5) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('Suspicious content reduction detected, aborting .htaccess write', 'Security');
                }
                return;
            }

            self::write_file($htaccess_file, $htaccess_contents);
            
        } catch (Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog('Error updating .htaccess rules: ' . $e->getMessage(), 'Security');
            }
            return;
        }
        
        // SECURITY: Ensure cache directory security is maintained
        if (class_exists('\\ZiziCache\\SysTool')) {
            \ZiziCache\SysTool::ensure_cache_security();
        }
    }

    /**
     * Removes all ZiziCache rules from the .htaccess file.
     * 
     * This method removes both main ZiziCache rules and Image Converter rules
     * during plugin deactivation to ensure complete cleanup.
     *
     * @return void
     * @throws Exception If there's an issue reading or writing the .htaccess file
     */
    public static function remove_htaccess_rules()
    {
        $htaccess_file = ABSPATH . '.htaccess';
        if (!self::is_writeable($htaccess_file)) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog('Cannot write to .htaccess file during cleanup: ' . $htaccess_file, 'Security');
            }
            return;
        }

        try {
            $htaccess = self::read_file($htaccess_file);
            $original_content = $htaccess;
            $sections_removed = 0;
            
            // Remove main ZiziCache section
            if (preg_match(self::MARKER_REGEX, $htaccess)) {
                $htaccess = preg_replace(self::MARKER_REGEX, '', $htaccess);
                $sections_removed++;
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('Main ZiziCache .htaccess section removed', 'Security');
                }
            }
            
            // Remove Image Converter section
            if (preg_match(self::IMG_MARKER_REGEX, $htaccess)) {
                $htaccess = preg_replace(self::IMG_MARKER_REGEX, '', $htaccess);
                $sections_removed++;
                
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('ZiziCache Image Converter .htaccess section removed', 'Security');
                }
            }
            
            // Check if any sections were found and removed
            if ($sections_removed === 0) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('No ZiziCache rules found in .htaccess, nothing to remove', 'Security');
                }
                return; // Nothing to remove
            }
            
            // Verify regex replacement was successful
            if ($htaccess === null) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog('Regex replacement failed during .htaccess cleanup', 'Security');
                }
                return;
            }
            
            // Additional safety: ensure we haven't removed more than expected
            $lines_removed = substr_count($original_content, "\n") - substr_count($htaccess, "\n");
            if ($lines_removed > 300) { // Increased threshold for both sections
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog("Suspicious line removal count ($lines_removed), aborting .htaccess cleanup", 'Security');
                }
                return;
            }
            
            // Clean up any multiple newlines that might result from removal
            $htaccess = preg_replace('/\n{3,}/', "\n\n", $htaccess);
            
            // Ensure content ends with single newline if it's not empty
            if (!empty(trim($htaccess))) {
                $htaccess = rtrim($htaccess) . "\n";
            }
            
            self::write_file($htaccess_file, $htaccess);
            
            // Log the successful removal for debugging
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog("All ZiziCache .htaccess rules removed successfully ($sections_removed sections, $lines_removed lines)", 'Security');
            }
            
        } catch (Exception $e) {
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog('Error removing .htaccess rules: ' . $e->getMessage(), 'Security');
            }
            return;
        }
    }

    /**
     * Verifies if a file exists and is writable.
     * 
     * @param string $file Path to the file to check.
     * @return bool True if the file exists and is writable, false otherwise.
     */
    private static function is_writeable($file)
    {
        return file_exists($file) && is_writeable($file);
    }

    /**
     * Safely reads the contents of a file.
     * 
     * @param string $file Path to the file to read.
     * @return string The file contents as a string.
     * @throws Exception If the file cannot be read.
     */
    private static function read_file($file)
    {
        return file_get_contents($file);
    }

    /**
     * Writes contents to a file with proper error handling.
     * 
     * @param string $file Path to the file to write to.
     * @param string $contents The contents to write to the file.
     * @return void
     * @throws Exception If the file cannot be written to.
     */
    private static function write_file($file, $contents)
    {
        file_put_contents($file, $contents);
    }

    /**
     * Generates the main .htaccess rules (without Image Converter rules).
     * 
     * This method loads the base rules from the template file and applies
     * server-specific modifications and dynamic replacements.
     *
     * @return string The generated main .htaccess rules as a string.
     * @throws Exception If the rules template file cannot be read.
     */
    private static function generate_main_rules()
    {
        // Load the base rules from the template file
        $rules_file = ZIZI_CACHE_PLUGIN_DIR . 'assets/templates/htaccess.txt';
        $rules = file_exists($rules_file) ? file_get_contents($rules_file) : '';
        
        if (empty($rules)) {
            throw new Exception('Could not read the .htaccess rules template.');
        }

        // Remove gzip rules for OpenLiteSpeed as it handles compression differently
        if (preg_match('/openlitespeed/i', $_SERVER['LSWS_EDITION'] ?? '')) {
            $rules = preg_replace(
                '/# GZIPed HTML, CSS, JS, Text, XML & Fonts.*# End rewrite requests to cache\n*/s',
                '',
                $rules
            );
        }

        // Replace HOSTNAME placeholder with the actual site hostname
        $hostname = parse_url(site_url(), PHP_URL_HOST);
        $rules = str_replace('HOSTNAME', $hostname, $rules);

        /**
         * Filters the generated main .htaccess rules before they are written to disk.
         *
         * @param string $rules The generated main .htaccess rules.
         * @return string The filtered rules.
         */
        $rules = apply_filters('zizi_cache_main_htaccess_rules', $rules);

        return $rules;
    }

    /**
     * Updates or adds a specific section in the .htaccess content.
     * 
     * @param string $content Current .htaccess content
     * @param string $rules New rules to insert/update
     * @param string $regex Regex pattern to match existing section
     * @param string $section_name Name of section for logging
     * @return string Updated .htaccess content
     */
    private static function update_section($content, $rules, $regex, $section_name)
    {
        // Nejprve zkontrolujeme, zda pravidla již nejsou v obsahu (i mimo značky)
        // Toto zabrání duplicitám v případě nesprávného formátu stávajících pravidel
        
        // Získáme obsah pravidel (bez BEGIN/END značek) pro detekci duplicit
        $rules_content = preg_replace('/# (BEGIN|END) ZiziCache.*?\n/s', '', $rules);
        $rules_content = trim($rules_content);
        
        if (!empty($rules_content)) {
            // Escape speciální znaky pro regex
            $escaped_rules = preg_quote($rules_content, '/');
            
            // Hledáme identický obsah pravidel
            if (preg_match('/' . $escaped_rules . '/s', $content)) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog(
                        "Identical $section_name rules already exist in .htaccess. Avoiding duplication.", 
                        'Security'
                    );
                }
                
                // Pokud sekce odpovídající našemu formátu neexistuje, přidáme formátovaná pravidla
                if (!preg_match($regex, $content)) {
                    // Nejprve smažeme existující neformátovaný obsah
                    $content = str_replace($rules_content, '', $content);
                    // Vyčistíme vícenásobné prázdné řádky
                    $content = preg_replace('/\n{3,}/', "\n\n", $content);
                    
                    // Pak přidáme správně formátovaná pravidla
                    if (strpos($content, '# BEGIN WordPress') !== false) {
                        $wp_begin_pos = strpos($content, '# BEGIN WordPress');
                        $before_wp = substr($content, 0, $wp_begin_pos);
                        $wp_section_onwards = substr($content, $wp_begin_pos);
                        $content = $before_wp . $rules . "\n\n" . $wp_section_onwards;
                    } else {
                        $content = $rules . "\n\n" . $content;
                    }
                    
                    if (class_exists('\\ZiziCache\\CacheSys')) {
                        \ZiziCache\CacheSys::writeLog(
                            "Reformatted $section_name rules in .htaccess with proper markers", 
                            'Security'
                        );
                    }
                }
                
                return $content;
            }
        }
        
        // If section already exists, replace it
        if (preg_match($regex, $content)) {
            $new_content = preg_replace($regex, $rules, $content);
            
            // Verify replacement was successful
            if ($new_content !== null && $new_content !== $content) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog("ZiziCache $section_name .htaccess rules updated", 'Security');
                }
                return $new_content;
            } else {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog("Failed to replace existing ZiziCache $section_name rules in .htaccess", 'Security');
                }
                return $content;
            }
        }
        
        // Section doesn't exist, add it
        // If WordPress rules present, insert before the entire WordPress section
        if (strpos($content, '# BEGIN WordPress') !== false) {
            // Find position of WordPress section start
            $wp_begin_pos = strpos($content, '# BEGIN WordPress');
            
            // Split content: before WordPress + WordPress section onwards
            $before_wp = substr($content, 0, $wp_begin_pos);
            $wp_section_onwards = substr($content, $wp_begin_pos);
            
            // Insert rules between the two parts
            $content = $before_wp . $rules . "\n\n" . $wp_section_onwards;
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog("ZiziCache $section_name .htaccess rules inserted before WordPress section", 'Security');
            }
        }
        // Otherwise prepend to the file
        else {
            $content = $rules . "\n\n" . $content;
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeLog("ZiziCache $section_name .htaccess rules prepended to file", 'Security');
            }
        }
        
        return $content;
    }

    /**
     * Removes a specific section from the .htaccess content.
     * 
     * @param string $content Current .htaccess content
     * @param string $regex Regex pattern to match section to remove
     * @param string $section_name Name of section for logging
     * @return string Updated .htaccess content
     */
    private static function remove_section($content, $regex, $section_name)
    {
        if (preg_match($regex, $content)) {
            $new_content = preg_replace($regex, '', $content);
            
            if ($new_content !== null) {
                if (class_exists('\\ZiziCache\\CacheSys')) {
                    \ZiziCache\CacheSys::writeLog("ZiziCache $section_name .htaccess section removed", 'Security');
                }
                return $new_content;
            }
        }
        
        return $content;
    }

    /**
     * Generates the .htaccess rules with dynamic replacements and filters.
     * 
     * @deprecated Use generate_main_rules() instead
     * This method is kept for backward compatibility but should not be used.
     *
     * @return string The generated .htaccess rules as a string.
     * @throws Exception If the rules template file cannot be read.
     */
    private static function generate_rules()
    {
        return self::generate_main_rules();
    }

    /**
     * Generates Image Converter .htaccess rules as an independent section.
     * 
     * Creates conditional rewrite rules that serve optimized image formats
     * (AVIF first, then WebP) when the browser supports them and the
     * optimized files exist on disk. This is now a separate section to avoid
     * conflicts with the main cache rules.
     *
     * @return string The Image Converter .htaccess rules as independent section.
     */
    private static function generate_image_converter_rules()
    {
        $config = SysConfig::$config;
        $formats = $config['image_converter_formats'] ?? ['avif', 'webp'];
        $rules = self::IMG_MARKER_BEGIN . "\n";
        
        // Add proper headers only for enabled formats
        $rules .= "# Set correct Content-Type headers for modern image formats\n";
        $rules .= "<IfModule mod_headers.c>\n";
        
        if (in_array('avif', $formats)) {
            $rules .= "    <FilesMatch \"\\.avif$\">\n";
            $rules .= "        Header set Content-Type \"image/avif\"\n";
            $rules .= "        Header append Vary \"Accept\"\n";
            $rules .= "        Header set x-zizi-image-format \"avif\"\n";
            $rules .= "    </FilesMatch>\n";
        }
        
        if (in_array('webp', $formats)) {
            $rules .= "    <FilesMatch \"\\.webp$\">\n";
            $rules .= "        Header set Content-Type \"image/webp\"\n";
            $rules .= "        Header append Vary \"Accept\"\n";
            $rules .= "        Header set x-zizi-image-format \"webp\"\n";
            $rules .= "    </FilesMatch>\n";
        }
        
        $rules .= "</IfModule>\n\n";
        
        // Add MIME types only for enabled formats
        $rules .= "# Ensure proper MIME types\n";
        $rules .= "<IfModule mod_mime.c>\n";
        
        if (in_array('avif', $formats)) {
            $rules .= "    AddType image/avif .avif\n";
        }
        
        if (in_array('webp', $formats)) {
            $rules .= "    AddType image/webp .webp\n";
        }
        
        $rules .= "</IfModule>\n\n";
        
        // Add rewrite rules in order of preference
        $rules .= "<IfModule mod_rewrite.c>\n";
        $rules .= "    RewriteEngine On\n\n";
        
        // AVIF rules (if enabled) - higher priority
        if (in_array('avif', $formats)) {
            $rules .= "    # Serve AVIF images if supported and available\n";
            $rules .= "    # Skip optimization for WordPress admin area\n";
            $rules .= "    RewriteCond %{REQUEST_URI} !^/wp-admin/ [NC]\n";
            $rules .= "    RewriteCond %{REQUEST_URI} !^/wp-content/plugins/.*/admin/ [NC]\n";
            $rules .= "    RewriteCond %{HTTP_REFERER} !^https?://[^/]+/wp-admin/ [NC]\n";
            $rules .= "    RewriteCond %{HTTP_ACCEPT} image/avif [NC]\n";
            $rules .= "    RewriteCond %{REQUEST_FILENAME} \\.(jpe?g|png|gif|bmp|tiff?)$ [NC]\n";
            $rules .= "    RewriteCond %{REQUEST_FILENAME}\\.avif -f\n";
            $rules .= "    RewriteRule ^(.*)\\.(jpe?g|png|gif|bmp|tiff?)$ \$1.\$2.avif [E=avif:1,L]\n\n";
        }
        
        // WebP rules (if enabled) - fallback from AVIF
        if (in_array('webp', $formats)) {
            $rules .= "    # Serve WebP images if supported and available (fallback from AVIF)\n";
            $rules .= "    # Skip optimization for WordPress admin area\n";
            $rules .= "    RewriteCond %{REQUEST_URI} !^/wp-admin/ [NC]\n";
            $rules .= "    RewriteCond %{REQUEST_URI} !^/wp-content/plugins/.*/admin/ [NC]\n";
            $rules .= "    RewriteCond %{HTTP_REFERER} !^https?://[^/]+/wp-admin/ [NC]\n";
            $rules .= "    RewriteCond %{HTTP_ACCEPT} image/webp [NC]\n";
            $rules .= "    RewriteCond %{REQUEST_FILENAME} \\.(jpe?g|png|gif|bmp|tiff?)$ [NC]\n";
            $rules .= "    RewriteCond %{REQUEST_FILENAME}\\.webp -f\n";
            $rules .= "    RewriteRule ^(.*)\\.(jpe?g|png|gif|bmp|tiff?)$ \$1.\$2.webp [E=webp:1,L]\n\n";
        }
        
        $rules .= "</IfModule>\n\n";
        
        // Debugging headers for development (only for enabled formats)
        $rules .= "# Debug headers (remove in production)\n";
        $rules .= "<IfModule mod_headers.c>\n";
        $rules .= "    Header set x-zizi-original-accept \"%{HTTP_ACCEPT}e\" env=!avif\n";
        
        if (in_array('avif', $formats)) {
            $rules .= "    Header set x-zizi-avif-served \"1\" env=avif\n";
        }
        
        if (in_array('webp', $formats)) {
            $rules .= "    Header set x-zizi-webp-served \"1\" env=webp\n";
        }
        
        $rules .= "</IfModule>\n\n";
        
        $rules .= self::IMG_MARKER_END;
        
        /**
         * Filters the generated Image Converter .htaccess rules before they are written to disk.
         *
         * @param string $rules The generated Image Converter .htaccess rules.
         * @return string The filtered rules.
         */
        $rules = apply_filters('zizi_cache_image_converter_htaccess_rules', $rules);
        
        return $rules;
    }
}