<?php

declare(strict_types=1);

namespace ZiziCache\HostProviders;

/**
 * Interface for all hosting provider implementations.
 *
 * This interface defines the contract for hosting provider integrations in the ZiziCache plugin.
 * Implementing classes must provide methods for cache purging, availability checks, and lifecycle hooks.
 *
 * @package ZiziCache\HostProviders
 */
interface HostProviderInterface
{
    /**
     * Check if this provider is available or installed in the current environment.
     *
     * Implementations should detect whether the hosting provider's environment or API is accessible.
     *
     * @return bool True if the provider is available, false otherwise.
     */
    public function isAvailable(): bool;

    /**
     * Purge specific URLs from the cache.
     *
     * Implementations should remove the provided URLs from the hosting provider's cache.
     *
     * @param string[] $urls Array of URLs to purge from cache.
     * @return bool True on success, false on failure.
     */
    public function purgeUrls(array $urls): bool;

    /**
     * Purge the entire cache for the site.
     *
     * Implementations should clear all cached content for the site from the hosting provider.
     *
     * @return bool True on success, false on failure.
     */
    public function purgeAll(): bool;

    /**
     * Run logic on plugin activation.
     *
     * Implementations can perform provider-specific setup tasks when the plugin is activated.
     *
     * @return void
     */
    public function onActivate(): void;

    /**
     * Run logic on plugin deactivation.
     *
     * Implementations can perform provider-specific cleanup tasks when the plugin is deactivated.
     *
     * @return void
     */
    public function onDeactivate(): void;
}

/**
 * Abstract base class for hosting providers.
 *
 * This class provides a template for implementing hosting provider-specific logic
 * in the ZiziCache plugin. It defines lifecycle hooks for activation and deactivation,
 * and provides a protected logging utility for subclasses. Concrete hosting provider
 * classes should extend this class and implement the HostProviderInterface.
 *
 * @package ZiziCache\HostProviders
 * @abstract
 */
abstract class AbstractHostProvider implements HostProviderInterface
{
    /**
     * Called when the plugin is activated.
     *
     * Hosting provider implementations can override this method to perform
     * provider-specific setup tasks, such as configuring environment variables,
     * writing files, or initializing resources. The default implementation does nothing.
     *
     * @return void
     */
    public function onActivate(): void
    {
        // Default implementation does nothing
    }

    /**
     * Called when the plugin is deactivated.
     *
     * Hosting provider implementations can override this method to perform
     * provider-specific cleanup tasks, such as removing files, resetting configuration,
     * or releasing resources. The default implementation does nothing.
     *
     * @return void
     */
    public function onDeactivate(): void
    {
        // Default implementation does nothing
    }

    /**
     * Log a message to the ZiziCache log file.
     *
     * This utility method writes a formatted log message to the ZiziCache log file,
     * including the log level and the calling class name. Logging only occurs if WP_DEBUG is enabled.
     *
     * @param string $message The message to log.
     * @param string $level The log level (e.g., 'info', 'warning', 'error'). Defaults to 'info'.
     *
     * @return void
     */
    protected function log(string $message, string $level = 'info'): void
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            \ZiziCache\CacheSys::writeLog(sprintf('[%s] [ZiziCache][%s] %s', strtoupper($level), static::class, $message));
        }
    }
}

/**
 * Main HostProviders class for managing hosting provider integrations.
 *
 * This class is responsible for detecting, initializing, and managing all supported hosting provider integrations
 * in the ZiziCache plugin. It provides methods to register providers, purge cache, run activation/deactivation hooks,
 * and interact with the active provider. Providers are detected dynamically based on the environment.
 *
 * @package ZiziCache\HostProviders
 */
class HostProviders
{
    /**
     * Cached array of provider instances, keyed by provider name.
     *
     * @var array<string, HostProviderInterface>|null
     */
    private static ?array $providers = null;

    /**
     * The plugin file name, used for registering activation/deactivation hooks.
     *
     * @var string
     */
    private static string $pluginFileName = '';

    /**
     * Set the plugin file name (should be called on plugin load).
     *
     * @param string $fileName The main plugin file path.
     * @return void
     */
    public static function setPluginFileName(string $fileName): void
    {
        self::$pluginFileName = $fileName;
    }

    /**
     * Get all available hosting providers.
     *
     * Returns an array of all detected hosting provider instances, keyed by provider name.
     *
     * @return array<string, HostProviderInterface> Array of provider instances.
     */
    public static function getProviders(): array
    {
        if (self::$providers === null) {
            self::$providers = self::defineProviders();
        }
        return self::$providers;
    }

    /**
     * Define all available hosting providers as instances of classes implementing HostProviderInterface.
     *
     * This method instantiates all supported hosting provider classes and returns them in an associative array.
     *
     * @return array<string, HostProviderInterface> Array of provider instances.
     */
    private static function defineProviders(): array
    {
        return [
            'cloudways' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('Breeze_CloudFlare_Helper');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable() || !\Breeze_CloudFlare_Helper::is_cloudflare_enabled()) {
                        $this->log('CloudFlare is not enabled or Cloudways provider is not available', 'warning');
                        return false;
                    }
                    try {
                        \Breeze_CloudFlare_Helper::purge_cloudflare_cache_urls($urls);
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Cloudways URLs: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable() || !\Breeze_CloudFlare_Helper::is_cloudflare_enabled()) {
                        $this->log('CloudFlare is not enabled or Cloudways provider is not available', 'warning');
                        return false;
                    }
                    try {
                        \Breeze_CloudFlare_Helper::reset_all_cache();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all Cloudways cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },
            'kinsta' => new class extends AbstractHostProvider {
                private $kinstaCache;

                public function __construct()
                {
                    global $kinsta_cache;
                    $this->kinstaCache = $kinsta_cache;
                }


                public function isAvailable(): bool
                {
                    return !empty($this->kinstaCache);
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Kinsta provider is not available', 'warning');
                        return false;
                    }
                    try {
                        $this->kinstaCache->kinsta_cache_purge->purge_complete_caches();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Kinsta cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'gridpane' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('Nginx_Cache_Purger_Admin');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('GridPane provider is not available', 'warning');
                        return false;
                    }
                    try {
                        $purger = new \Nginx_Cache_Purger_Admin();
                        $purger->register_purge();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge GridPane cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'rocketnet' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('CDN_Clear_Cache_Hooks');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Rocket.net provider is not available', 'warning');
                        return false;
                    }
                    try {
                        \CDN_Clear_Cache_Hooks::purge_cache();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Rocket.net cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'runcloud' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('RunCloud_Hub');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('RunCloud provider is not available', 'warning');
                        return false;
                    }
                    try {
                        if (is_multisite()) {
                            \RunCloud_Hub::purge_cache_all_sites();
                        } else {
                            \RunCloud_Hub::purge_cache_all();
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge RunCloud cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'siteground' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return function_exists('sg_cachepress_purge_everything') ||
                        class_exists('SiteGround_Optimizer\File_Cacher\Cache');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('SiteGround provider is not available', 'warning');
                        return false;
                    }
                    try {
                        \sg_cachepress_purge_everything();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge SiteGround cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'spinupwp' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('SpinupWp\Cache') || function_exists('spinupwp_purge_site');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('SpinupWP provider is not available', 'warning');
                        return false;
                    }
                    try {
                        \spinupwp_purge_site();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge SpinupWP cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'wpengine' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('WPeCommon');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('WP Engine provider is not available', 'warning');
                        return false;
                    }
                    try {
                        if (method_exists('WPeCommon', 'purge_memcached')) {
                            \WPeCommon::purge_memcached();
                        }
                        if (method_exists('WPeCommon', 'clear_maxcdn_cache')) {
                            \WPeCommon::clear_maxcdn_cache();
                        }
                        if (method_exists('WPeCommon', 'purge_varnish_cache')) {
                            \WPeCommon::purge_varnish_cache();
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge WP Engine cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },
            'wpcloud' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('Atomic_Persistent_Data');
                }

                public function onActivate(): void
                {
                    $muPluginDir = WPMU_PLUGIN_DIR; // Use a variable for better readability
                    if (!is_dir($muPluginDir)) {
                        mkdir($muPluginDir, 0755, true);
                    }
                    $advancedCacheLoaderPath = $muPluginDir . '/zizi-cache-helper.php';
                    $advancedCacheLoaderContent = "<?php
/*
Plugin Name: MU ZiziCache Host Helper
Description: MU ZiziCache Host Helper.
Author: ZiziCache
Version: 0.3
*/

if (function_exists('batcache_cancel')) { batcache_cancel(); }
if (file_exists(WP_CONTENT_DIR . '/zizi-cache-advanced-cache.php')) {
    require_once WP_CONTENT_DIR . '/zizi-cache-advanced-cache.php';
}
";
                    file_put_contents($advancedCacheLoaderPath, $advancedCacheLoaderContent);
                }

                public function onDeactivate(): void
                {
                    $advancedCacheLoaderPath = WPMU_PLUGIN_DIR . '/zizi-cache-helper.php';
                    if (file_exists($advancedCacheLoaderPath)) {
                        unlink($advancedCacheLoaderPath);
                    }
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!class_exists('Edge_Cache_Atomic')) {
                        $this->log('WP Cloud provider is not available', 'warning');
                        return false;
                    }
                    try {
                        $edge = new \Edge_Cache_Atomic();
                        if ($edge) {
                           $edge->purge_domain($edge->get_domain_name(), [
                                'wp_actions' => 'purge_cache',
                                'wp_action' => 'edge_cache_purge',
                                'batcache' => 'yes',
                            ]);
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge WP Cloud cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },

            // Additional hosting providers from Flying Press compatibility
            'flywheel' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return function_exists('fw_get_db_settings_option');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Flywheel provider is not available', 'warning');
                        return false;
                    }
                    try {
                        // Flywheel uses Varnish for caching
                        if (function_exists('wp_cache_flush')) {
                            wp_cache_flush();
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Flywheel cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },

            'pagely' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('PagelyCachePurge');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Pagely provider is not available', 'warning');
                        return false;
                    }
                    try {
                        $purger = new \PagelyCachePurge();
                        foreach ($urls as $url) {
                            $purger->purgeUrl($url);
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Pagely cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        $purger = new \PagelyCachePurge();
                        $purger->purgeAll();
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all Pagely cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },

            'presslabs' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return function_exists('pl_pre_cache_refresh') || defined('PL_CACHE_DIR');
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Presslabs provider is not available', 'warning');
                        return false;
                    }
                    try {
                        // Presslabs uses specific hooks for cache purging
                        foreach ($urls as $url) {
                            do_action('pl_pre_url_button_cache_refresh', $url);
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Presslabs cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        do_action('pl_pre_cache_refresh');
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all Presslabs cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },

            'wp_serveur' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return function_exists('rocket_wpserveur_varnish_field') || 
                           (defined('WPSERVEUR_PLUGIN') && WPSERVEUR_PLUGIN);
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('WP Serveur provider is not available', 'warning');
                        return false;
                    }
                    try {
                        // WP Serveur uses Varnish for caching
                        foreach ($urls as $url) {
                            if (function_exists('rocket_varnish_request')) {
                                rocket_varnish_request($url, 'PURGE');
                            }
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge WP Serveur cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        if (function_exists('rocket_varnish_request')) {
                            rocket_varnish_request(home_url('/.*'), 'PURGE');
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all WP Serveur cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },

            'nginx_compatibility' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    global $is_nginx;
                    return isset($is_nginx) && $is_nginx;
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('NGINX provider is not available', 'warning');
                        return false;
                    }
                    try {
                        // NGINX compatibility is mainly about URL handling
                        // but we can try to purge cache if FastCGI is available
                        foreach ($urls as $url) {
                            if (function_exists('fastcgi_finish_request')) {
                                fastcgi_finish_request();
                            }
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to handle NGINX cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    return $this->purgeUrls([]);
                }
            },

            // Premium hosting providers with specialized APIs
            'pantheon' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return function_exists('pantheon_clear_edge_all') || 
                           function_exists('pantheon_clear_edge_paths') ||
                           (defined('PANTHEON_ENVIRONMENT') && PANTHEON_ENVIRONMENT);
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Pantheon provider is not available', 'warning');
                        return false;
                    }
                    try {
                        if (function_exists('pantheon_clear_edge_paths') && !empty($urls)) {
                            // Convert URLs to paths for Pantheon API
                            $paths = [];
                            foreach ($urls as $url) {
                                $path = parse_url($url, PHP_URL_PATH) ?: '/';
                                $paths[] = $path;
                            }
                            pantheon_clear_edge_paths($paths);
                        } elseif (function_exists('pantheon_clear_edge_all')) {
                            // Fallback to clearing all if specific paths not available
                            pantheon_clear_edge_all();
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Pantheon cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        if (function_exists('pantheon_clear_edge_all')) {
                            pantheon_clear_edge_all();
                            return true;
                        }
                        return false;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all Pantheon cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },

            'godaddy_managed' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('WPaaS\\Cache') || 
                           (defined('GD_SYSTEM_PLUGIN_DIR') && GD_SYSTEM_PLUGIN_DIR);
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('GoDaddy Managed WordPress provider is not available', 'warning');
                        return false;
                    }
                    try {
                        if (class_exists('WPaaS\\Cache')) {
                            foreach ($urls as $url) {
                                \WPaaS\Cache::ban($url);
                            }
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge GoDaddy cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        if (class_exists('WPaaS\\Cache')) {
                            \WPaaS\Cache::ban(home_url());
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all GoDaddy cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },

            'bluehost' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return function_exists('the_cache_purger_purge_all_caches') ||
                           (defined('BLUEHOST_PLUGIN_VERSION') && BLUEHOST_PLUGIN_VERSION);
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Bluehost provider is not available', 'warning');
                        return false;
                    }
                    try {
                        // Bluehost cache purger typically handles all cache layers
                        if (function_exists('the_cache_purger_purge_all_caches')) {
                            the_cache_purger_purge_all_caches();
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Bluehost cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        if (function_exists('the_cache_purger_purge_all_caches')) {
                            the_cache_purger_purge_all_caches();
                            return true;
                        }
                        return false;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all Bluehost cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },

            'hostinger_litespeed' => new class extends AbstractHostProvider {
                public function isAvailable(): bool
                {
                    return class_exists('\\LiteSpeed\\Purge') || 
                           class_exists('LiteSpeed_Cache') ||
                           (defined('LSCWP_V') && LSCWP_V);
                }

                public function purgeUrls(array $urls): bool
                {
                    if (!$this->isAvailable()) {
                        $this->log('Hostinger LiteSpeed provider is not available', 'warning');
                        return false;
                    }
                    try {
                        if (class_exists('\\LiteSpeed\\Purge')) {
                            foreach ($urls as $url) {
                                \LiteSpeed\Purge::purge_url($url);
                            }
                        } elseif (function_exists('wp_cache_flush')) {
                            wp_cache_flush();
                        }
                        return true;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge Hostinger LiteSpeed cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }

                public function purgeAll(): bool
                {
                    if (!$this->isAvailable()) {
                        return false;
                    }
                    try {
                        if (class_exists('\\LiteSpeed\\Purge')) {
                            \LiteSpeed\Purge::purge_all();
                            return true;
                        } elseif (function_exists('wp_cache_flush')) {
                            wp_cache_flush();
                            return true;
                        }
                        return false;
                    } catch (\Exception $e) {
                        $this->log(sprintf('Failed to purge all Hostinger LiteSpeed cache: %s', $e->getMessage()), 'error');
                        return false;
                    }
                }
            },
        ];
    }

    /**
     * Get the active hosting provider if available.
     *
     * Returns the first available provider instance, or null if none are detected.
     *
     * @return HostProviderInterface|null The active provider or null if none found.
     */
    public static function getActiveProvider(): ?HostProviderInterface
    {
        foreach (self::getProviders() as $provider) {
            if ($provider->isAvailable()) {
                return $provider;
            }
        }
        return null;
    }

    /**
     * Purge specific URLs from the active provider's cache.
     *
     * Calls the purgeUrls method on the active provider, if available.
     *
     * @param string[] $urls Array of URLs to purge.
     * @return bool True if successful, false otherwise.
     */
    public static function purgeUrls(array $urls): bool
    {
        $provider = self::getActiveProvider();
        if ($provider === null) {
            return false;
        }
        return $provider->purgeUrls($urls);
    }

    /**
     * Purge all cache from the active provider.
     *
     * Calls the purgeAll method on the active provider, if available.
     *
     * @return bool True if successful, false otherwise.
     */
    public static function purgeAll(): bool
    {
        $provider = self::getActiveProvider();
        if ($provider === null) {
            return false;
        }
        return $provider->purgeAll();
    }

    /**
     * Run activation tasks for all providers.
     *
     * Calls the onActivate method for each available provider.
     *
     * @return void
     */
    public static function onActivate(): void
    {
        foreach (self::getProviders() as $provider) {
            if ($provider->isAvailable() && method_exists($provider, 'onActivate')) {
                $provider->onActivate();
            }
        }
    }

    /**
     * Run deactivation tasks for all providers.
     *
     * Calls the onDeactivate method for each available provider.
     *
     * @return void
     */
    public static function onDeactivate(): void
    {
        foreach (self::getProviders() as $provider) {
            if ($provider->isAvailable() && method_exists($provider, 'onDeactivate')) {
                $provider->onDeactivate();
            }
        }
    }

    /**
     * Initialize the HostProviders system.
     *
     * This can be used to perform any necessary setup or initialization for providers.
     *
     * @return void
     */
    public static function init(): void
    {
        // Initialize any provider-specific functionality
        foreach (self::getProviders() as $provider) {
            if ($provider->isAvailable() && method_exists($provider, 'init')) {
                $provider->init();
            }
        }
    }

    /**
     * Check if any provider is available.
     *
     * Returns true if at least one provider is detected as available.
     *
     * @return bool True if at least one provider is available.
     */
    public static function hasActiveProvider(): bool
    {
        return self::getActiveProvider() !== null;
    }

    /**
     * Get the name of the active provider.
     *
     * Returns the key (name) of the first available provider, or null if none are detected.
     *
     * @return string|null The name of the active provider or null if none found.
     */
    public static function getActiveProviderName(): ?string
    {
        foreach (self::getProviders() as $name => $provider) {
            if ($provider->isAvailable()) {
                return $name;
            }
        }
        return null;
    }

    /**
     * Register WordPress hooks for the active provider.
     *
     * Registers activation, deactivation, and cache purge hooks for the active provider.
     *
     * @return void
     */
    public static function registerWordPressHooks(): void
    {
        $provider = self::getActiveProvider();
        if ($provider === null) {
            return;
        }

        // Register activation/deactivation hooks if needed
        if (method_exists($provider, 'onActivate')) {
            if (empty(self::$pluginFileName)) {
                \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache: Plugin file name is not set. Please call HostProviders::setPluginFileName() on plugin load.');
            } else {
                register_activation_hook(self::$pluginFileName, [$provider, 'onActivate']);
            }
        }

        if (method_exists($provider, 'onDeactivate')) {
            if (empty(self::$pluginFileName)) {
                \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache: Plugin file name is not set. Please call HostProviders::setPluginFileName() on plugin load.');
            } else {
                register_deactivation_hook(self::$pluginFileName, [$provider, 'onDeactivate']);
            }
        }

        // Register cache purge actions
        add_action('zizi_cache_purge_everything:before', function () use ($provider) {
            $provider->purgeAll();
        });

        add_action('zizi_cache_purge_pages:before', function () use ($provider) {
            $provider->purgeAll();
        });

        add_action('zizi_cache_purge_urls:before', function ($urls) use ($provider) {
            if (is_array($urls)) {
                $provider->purgeUrls($urls);
            }
        });
    }
}
