<?php

declare(strict_types=1);

namespace ZiziCache\HostProviders;

/**
 * Interface for all hosting provider implementations.
 *
 * This interface defines the contract for hosting provider integrations in the ZiziCache plugin.
 * Implementing classes must provide methods for cache purging, availability checks, and lifecycle hooks.
 *
 * @package ZiziCache\HostProviders
 */
interface HostProviderInterface
{
    /**
     * Check if this provider is available or installed in the current environment.
     *
     * Implementations should detect whether the hosting provider's environment or API is accessible.
     *
     * @return bool True if the provider is available, false otherwise.
     */
    public function isAvailable(): bool;
    
    /**
     * Purge specific URLs from the cache.
     *
     * Implementations should remove the provided URLs from the hosting provider's cache.
     *
     * @param string[] $urls Array of URLs to purge from cache.
     * @return bool True on success, false on failure.
     */
    public function purgeUrls(array $urls): bool;
    
    /**
     * Purge the entire cache for the site.
     *
     * Implementations should clear all cached content for the site from the hosting provider.
     *
     * @return bool True on success, false on failure.
     */
    public function purgeAll(): bool;
    
    /**
     * Run logic on plugin activation.
     *
     * Implementations can perform provider-specific setup tasks when the plugin is activated.
     *
     * @return void
     */
    public function onActivate(): void;
    
    /**
     * Run logic on plugin deactivation.
     *
     * Implementations can perform provider-specific cleanup tasks when the plugin is deactivated.
     *
     * @return void
     */
    public function onDeactivate(): void;
}
