<?php

declare(strict_types=1);

namespace ZiziCache\HostProviders;

/**
 * Abstract base class for hosting providers.
 *
 * This class provides a template for implementing hosting provider-specific logic
 * in the ZiziCache plugin. It defines lifecycle hooks for activation and deactivation,
 * and provides a protected logging utility for subclasses. Concrete hosting provider
 * classes should extend this class and implement the HostProviderInterface.
 *
 * @package ZiziCache\HostProviders
 * @abstract
 */
abstract class AbstractHostProvider implements HostProviderInterface
{
    /**
     * Called when the plugin is activated.
     *
     * Hosting provider implementations can override this method to perform
     * provider-specific setup tasks, such as configuring environment variables,
     * writing files, or initializing resources. The default implementation does nothing.
     *
     * @return void
     */
    public function onActivate(): void
    {
        // Default implementation does nothing
    }
    
    /**
     * Called when the plugin is deactivated.
     *
     * Hosting provider implementations can override this method to perform
     * provider-specific cleanup tasks, such as removing files, resetting configuration,
     * or releasing resources. The default implementation does nothing.
     *
     * @return void
     */
    public function onDeactivate(): void
    {
        // Default implementation does nothing
    }
    
    /**
     * Log a message to the ZiziCache log file.
     *
     * This utility method writes a formatted log message to the ZiziCache log file,
     * including the log level and the calling class name. Logging only occurs if WP_DEBUG is enabled.
     *
     * @param string $message The message to log.
     * @param string $level The log level (e.g., 'info', 'warning', 'error'). Defaults to 'info'.
     *
     * @return void
     */
    protected function log(string $message, string $level = 'info'): void
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            \ZiziCache\CacheSys::writeLog(sprintf(
                '[%s][%s] %s',
                $level,
                static::class,
                $message
            ));
        }
    }
}
