<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Handles font monitoring functionality
 */
class FontMonitor
{
    /**
     * Initializes the font monitoring system
     */
    public static function init()
    {
        // EMERGENCY DISABLED: Font monitoring has been disabled due to memory issues
        // Original code has been preserved for reference but is not executed
        return;
        
        // @codeCoverageIgnoreStart
        if (empty(SysConfig::$config['fonts_monitor_variants'])) {
            return;
        }

        // Add monitoring action to WP shutdown hook to not affect page load time
        // EMERGENCY DISABLE - hook removed
          // Add admin notice for font warnings if available
        if (is_admin()) {
            add_action('admin_notices', ['\ZiziCache\Font', 'display_admin_warnings']);
        }
        // @codeCoverageIgnoreEnd
    }    /**
     * Check if the current page should be monitored for font issues
     */
    public static function maybe_monitor_page() {
        return; // EMERGENCY DISABLE
        // EMERGENCY DISABLED: Function completely disabled due to memory issues
        return;
        
        // @codeCoverageIgnoreStart
        // Only run on frontend pages
        if (is_admin()) {
            return;
        }

        // Skip if doing AJAX, REST or CRON
        if (wp_doing_ajax() || wp_doing_cron() || (defined('REST_REQUEST') && REST_REQUEST)) {
            return;
        }
        // @codeCoverageIgnoreEnd

        // Implement random sampling - only check 10% of pageviews for performance
        if (mt_rand(1, 10) !== 1) {
            return;
        }

        // Check if we should monitor this URL
        $current_url = self::get_current_page_url();
        if (self::is_excluded_url($current_url)) {
            return;
        }

        // Get page content from output buffer but limit the amount of data processed
        $html = ob_get_contents();
        if (empty($html)) {
            return;
        }
        
        // Safety check - set an upper limit of 500KB per page to analyze
        $max_size = 500 * 1024; // 500KB
        if (strlen($html) > $max_size) {
            $html = substr($html, 0, $max_size);
        }        // Process the page for font variants (with timeout protection)
        try {
            // Set time limit for processing to avoid long-running operations
            $prev_time_limit = ini_get('max_execution_time');
            set_time_limit(10); // 10 seconds max for font analysis
            
            \ZiziCache\Font::monitor_font_variants($html);
            
            // Restore original time limit
            set_time_limit($prev_time_limit);
        } catch (\Exception $e) {
            // Log error but don't crash the page
            \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Font Monitor Error: ' . $e->getMessage());
        }
    }

    /**
     * Check if a URL is excluded from font monitoring
     *
     * @param string $url The URL to check
     * @return bool Whether the URL is excluded
     */
    public static function is_excluded_url(string $url): bool
    {
        $excludes = SysConfig::$config['fonts_monitor_excludes'] ?? [];
        if (empty($excludes)) {
            return false;
        }

        foreach ($excludes as $pattern) {
            if (empty($pattern)) continue;
            
            $pattern = str_replace('*', '.*', preg_quote($pattern, '/'));
            if (preg_match('/' . $pattern . '/i', $url)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get current page URL
     *
     * @return string The current page URL
     */
    public static function get_current_page_url(): string
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $uri = $_SERVER['REQUEST_URI'] ?? '';

        if (empty($host)) {
            return '';
        }

        return $protocol . '://' . $host . $uri;
    }

    /**
     * Fetch content from a URL safely with size limits
     * 
     * @param string $url The URL to fetch
     * @param int $max_size Maximum size to process in bytes
     * @return string|false Content or false on failure
     */
    private static function fetch_content_safely(string $url, int $max_size = 300000): string|false
    {
        try {
            $args = [
                'timeout'     => 5, // Reduced timeout from 15 to 5 seconds
                'redirection' => 1, // Reduced redirection from 2 to 1
                'sslverify'   => false,
                'user-agent'  => 'ZiziCache Font Monitor v' . ZIZI_CACHE_VERSION,
                'httpversion' => '1.1',
                'blocking'    => true,
            ];

            // Set a time limit for the operation
            $prev_time_limit = ini_get('max_execution_time');
            set_time_limit(10); // 10 seconds max for each request
            
            $response = wp_remote_get($url, $args);
            
            // Restore original time limit
            set_time_limit($prev_time_limit);
            
            if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                return false;
            }
            
            $content = wp_remote_retrieve_body($response);
            
            // Limit content size to prevent memory issues
            if (strlen($content) > $max_size) {
                $content = substr($content, 0, $max_size);
            }
            
            return $content;
        } catch (\Exception $e) {
            \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Font Monitor Error fetching ' . $url . ': ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get all font warnings
     *
     * @return array List of warnings with URLs
     */
    public static function get_warnings(): array
    {
        // Check if we have a cached version of all warnings
        $cached_warnings = get_transient('zizi_all_font_warnings_cache');
        if ($cached_warnings !== false) {
            return $cached_warnings;
        }
        
        $warning_urls = get_option('zizi_font_warning_urls', []);
        $warnings = [];
        $max_warnings = 50; // Limit the total number of warnings to process
        $warning_count = 0;
        $current_time = time();
        $seven_days_ago = $current_time - (7 * 24 * 60 * 60);
        
        if (!empty($warning_urls)) {
            // Sort warnings by timestamp (newest first)
            uasort($warning_urls, function($a, $b) {
                return $b - $a; // Descending order
            });
            
            foreach ($warning_urls as $url => $timestamp) {
                // Skip warnings older than 7 days
                if ($timestamp < $seven_days_ago) {
                    continue;
                }
                
                $transient_key = 'zizi_font_warnings_' . md5($url);
                $url_warnings = get_transient($transient_key);
                
                if (!empty($url_warnings)) {
                    foreach ($url_warnings as $warning) {
                        if ($warning_count >= $max_warnings) {
                            // Add a note that we're only showing the most recent warnings
                            $warnings[] = '⚠️ Only showing the ' . $max_warnings . ' most recent warnings. Please fix these issues and clear warnings to see more.';
                            break 2; // Break out of both loops
                        }
                        
                        $warnings[] = $warning . ' (URL: ' . esc_url($url) . ')';
                        $warning_count++;
                    }
                }
            }
        }
        
        $result = [
            'status' => 'success',
            'warnings' => $warnings,
            'count' => $warning_count,
            'has_more' => count($warning_urls) > count($warnings)
        ];
        
        // Cache the result for 1 hour to reduce database queries
        set_transient('zizi_all_font_warnings_cache', $result, HOUR_IN_SECONDS);
        
        return $result;
    }

    /**
     * Clear all font warnings
     *
     * @return array Success response
     */
    public static function clear_warnings(): array
    {
        $warning_urls = get_option('zizi_font_warning_urls', []);
        
        if (!empty($warning_urls)) {
            foreach ($warning_urls as $url => $timestamp) {
                $transient_key = 'zizi_font_warnings_' . md5($url);
                delete_transient($transient_key);
            }
        }
        
        // Clear the aggregated warnings cache as well
        delete_transient('zizi_all_font_warnings_cache');
        
        delete_option('zizi_font_warning_urls');
        
        return [
            'status' => 'success',
            'message' => 'All font warnings have been cleared.'
        ];
    }    /**
     * Scan all pages for font issues
     * 
     * Uses a more memory-efficient approach with limited pages and content size
     *
     * @return array Scan results
     */
    public static function scan_all_pages(): array
    {
        $warnings = [];
        $max_sites = 3; // Limit multisite processing
        $max_posts_per_site = 5; // Reduce from 10 to 5 posts per site
        $max_content_size = 300 * 1024; // 300KB max per page
        
        // For multisite, limit the number of sites processed
        if (is_multisite()) {
            $sites = get_sites([
                'fields' => 'ids',
                'number' => $max_sites, // Process max 3 sites at once
                'orderby' => 'last_updated',
                'order' => 'DESC' // Most recently updated sites first
            ]);
        } else {
            $sites = [get_current_blog_id()];
        }
        
        // Process one site at a time to prevent memory issues
        foreach ($sites as $site_id) {
            if (is_multisite()) {
                switch_to_blog($site_id);
            }
            
            // Only get the most important post types
            $post_types = ['page', 'post']; // Limit to core post types
            $posts = get_posts([
                'post_type' => $post_types,
                'post_status' => 'publish',
                'posts_per_page' => $max_posts_per_site,
                'orderby' => 'modified',
                'order' => 'DESC' // Most recently modified first
            ]);
              // Process homepage first with memory safety checks
            $home_url = get_home_url();
            $html = self::fetch_content_safely($home_url, $max_content_size);
            if ($html) {
                try {
                    \ZiziCache\Font::monitor_font_variants($html);
                    $transient_key = 'zizi_font_warnings_' . md5($home_url);
                    $url_warnings = get_transient($transient_key);
                    if (!empty($url_warnings)) {
                        foreach ($url_warnings as $warning) {
                            $warnings[] = $warning . ' (URL: ' . esc_url($home_url) . ')';
                        }
                    }
                } catch (\Exception $e) {
                    \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Font Monitor Error on homepage: ' . $e->getMessage());
                }
            }
            
            // Clear memory between processing pages
            $html = null;
            if (function_exists('gc_collect_cycles')) {
                gc_collect_cycles();
            }
            
            // Process a limited number of posts with memory checks between each
            foreach ($posts as $post) {                $url = get_permalink($post->ID);
                $html = self::fetch_content_safely($url, $max_content_size);
                if ($html) {
                    try {
                        \ZiziCache\Font::monitor_font_variants($html);
                        $transient_key = 'zizi_font_warnings_' . md5($url);
                        $url_warnings = get_transient($transient_key);
                        if (!empty($url_warnings)) {
                            foreach ($url_warnings as $warning) {
                                $warnings[] = $warning . ' (URL: ' . esc_url($url) . ')';
                            }
                        }
                    } catch (\Exception $e) {
                        \ZiziCache\CacheSys::writeLog('[ERROR] ZiziCache Font Monitor Error on ' . $url . ': ' . $e->getMessage());
                    }
                }
                
                // Clear memory after each page
                $html = null;
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
            }
            
            if (is_multisite()) {
                restore_current_blog();
            }
        }
        
        return [
            'status' => 'success',
            'warnings' => $warnings,
            'message' => 'Scan completed. ' . count($warnings) . ' warnings found.'
        ];
    }
}
