<?php
declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\SysConfig;
use ZiziCache\SysTool;

/**
 * Font Optimization and Management
 * 
 * This class handles various font optimization techniques including:
 * - Google Fonts optimization and self-hosting
 * - Font display optimization with font-display: swap
 * - Preloading of critical fonts
 * - Font face optimization
 * - Local font hosting and optimization
 * 
 * @package ZiziCache
 */
class Font
{
  /**
   * Stores font statistics for the current page
   * @var array
   */
  private static $font_stats = [];
  
  /**
   * Adds font-display: swap to Google Fonts link tags
   * 
   * This method processes all Google Fonts links in the HTML and ensures they include
   * the display=swap parameter for better font loading performance.
   *
   * @param string $html The HTML content to process
   * @return string Modified HTML with updated Google Fonts links
   */
  public static function add_display_swap_to_google_fonts(string $html): string
  {
    // Check master font optimization switch first
    if (empty(SysConfig::$config['fonts_enabled']) || empty(SysConfig::$config['fonts_display_swap'])) {
      return $html;
    }

    // get all link tags with google fonts
    preg_match_all(
      '/<link[^>]+href=[\'"]https\:\/\/fonts\.googleapis\.com\/css[^\'"]+[\'"][^>]*>/i',
      $html,
      $googlefonts_matches
    );
    $googlefonts = $googlefonts_matches[0] ?? [];

    try {
      foreach ($googlefonts as $google_font_tag) {
        if(!is_string($google_font_tag)) continue;
        $google_font = new HTML($google_font_tag);

        $href = $google_font->href;
        if(empty($href) || !is_string($href)) continue;

        if (preg_match('/display=\w+/', $href)) {
            $new_href = preg_replace('/display=\w+/', 'display=swap', $href);
        } else {
            $new_href = $href . '&display=swap';
        }
          if ($new_href !== null) { // Check if preg_replace was successful
            $google_font->href = $new_href;
            $html = str_replace($google_font_tag, (string)$google_font, $html);
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Font Optimizer Error (optimize_google_fonts): ' . $e->getMessage(), 'Security');
    } 
    return $html;
  }

  /**
   * Adds font-display: swap to internal style tags
   * 
   * Processes all internal style tags and ensures @font-face rules include
   * font-display: swap for better font loading performance.
   *
   * @param string $html The HTML content to process
   * @return string Modified HTML with updated font-face rules
   */
  public static function add_display_swap_to_internal_styles(string $html): string
  {
    // Check master font optimization switch first
    if (empty(SysConfig::$config['fonts_enabled']) || empty(SysConfig::$config['fonts_display_swap'])) {
      return $html;
    }

    // get all style tags
    preg_match_all('/<style[^>]*>([^<]*)<\/style>/i', $html, $styles_matches);
    $style_tags = $styles_matches[0] ?? [];
    $style_contents = $styles_matches[1] ?? [];

    try {
      foreach ($style_tags as $key => $style_tag) {
        if(!is_string($style_tag)) continue;
        $style_content = $style_contents[$key] ?? '';
        if(empty($style_content) || !is_string($style_content)) continue;

        $modified_css = self::inject_display_swap($style_content);
        if($modified_css !== $style_content){            $new_style_tag = str_replace($style_content, $modified_css, $style_tag);
            $html = str_replace($style_tag, $new_style_tag, $html);
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Font Optimizer Error (add_font_display_swap): ' . $e->getMessage(), 'Security');
    }
    return $html;
  }

  /**
   * Optimizes Google Fonts by self-hosting them
   * 
   * Downloads Google Fonts CSS and font files to serve them locally,
   * reducing external dependencies and improving loading times.
   *
   * @param string $html The HTML content to process
   * @return string Modified HTML with self-hosted Google Fonts
   */  public static function optimize_google_fonts(string $html): string
  {
    // Check master font optimization switch first
    if (empty(SysConfig::$config['fonts_enabled'])) {
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('Font optimization disabled via master switch', 'info', 'ZiziCache');
      }
      return $html;
    }
    
    // Přísnější kontrola - optimalizace proběhne pouze pokud je explicitně zapnuta
    if (empty(SysConfig::$config['fonts_optimize_google_fonts']) || 
        SysConfig::$config['fonts_optimize_google_fonts'] !== true || 
        !empty(SysConfig::$config['bloat_remove_google_fonts'])) {
      
      // Logování kontroly pro debugging
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('Font optimization skipped: fonts_optimize_google_fonts=' . 
          (empty(SysConfig::$config['fonts_optimize_google_fonts']) ? 'false' : 'true') . 
          ', bloat_remove_google_fonts=' . 
          (!empty(SysConfig::$config['bloat_remove_google_fonts']) ? 'true' : 'false'), 
          'info', 'ZiziCache');
      }
      
      return $html;
    }

    // Remove all preconnect, preload, prefech, and dns-prefetch tags for Google Fonts
    $html = preg_replace(
      '/<link[^>]*(?:preload|preconnect|prefetch|dns-prefetch)[^>]*(?:fonts\.gstatic\.com|fonts\.googleapis\.com)[^>]*>/i',
      '',
      $html
    ) ?? $html; // Ensure $html is not null if preg_replace fails

    // Find all Google Fonts
    preg_match_all(
      '/<link[^>]+href=[\'"](https?:)?\/\/fonts\.googleapis\.com\/css[^\'"]+[\'"][^>]*>/i',
      $html,
      $googlefonts_matches
    );
    $googlefont_tags = $googlefonts_matches[0] ?? [];

    try {
      foreach ($googlefont_tags as $google_font_tag) {
        if(!is_string($google_font_tag)) continue;
        $google_font = new HTML($google_font_tag);
        $href = $google_font->href;

        if(empty($href) || !is_string($href)) continue;

        $hash = substr(md5($href), 0, 12);
        $file_name = "{$hash}.google-font.css";
        $file_path = ZIZI_CACHE_CACHE_DIR . $file_name;
        $file_url = ZIZI_CACHE_CACHE_URL . $file_name;
        
        // NEW: Robust validation before URL replacement
        $should_self_host = false;
        
        if (!is_file($file_path) || filesize($file_path) === 0) {
          // Try to download and validate
          if(self::self_host_style_sheet($href, $file_path)){
              $should_self_host = self::validate_self_hosted_fonts($file_path);
              if (!$should_self_host) {
                \ZiziCache\CacheSys::writeWarning("Font Optimizer Warning: Self-hosted fonts validation failed for {$href}. Keeping original URL.", 'Security');
                // Clean up incomplete files
                if (is_file($file_path)) {
                  unlink($file_path);
                }
              }
          }
        } else {
          // File exists, validate it's complete
          $should_self_host = self::validate_self_hosted_fonts($file_path);
        }
        
        // Only replace URL if self-hosting is successful and validated
        if ($should_self_host) {
          $google_font->href = $file_url;
          $html = str_replace($google_font_tag, (string)$google_font, $html);
        } else {
          // Fallback: Keep original URL but add display=swap if not present
          $original_href = $google_font->href;
          if (strpos($original_href, 'display=') === false) {
            $separator = strpos($original_href, '?') !== false ? '&' : '?';
            $google_font->href = $original_href . $separator . 'display=swap';
            $html = str_replace($google_font_tag, (string)$google_font, $html);
          }
        }
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Font Optimizer Error (self_host_google_fonts): ' . $e->getMessage(), 'Security');
    }
    return $html;
  }

  /**
   * Get font URLs for preloading from optimized CSS based on Font Statistics
   * 
   * This method extracts font URLs from locally cached Google Fonts CSS files
   * that have been optimized based on Font Statistics data.
   *
   * @return array Array of font URLs suitable for preloading
   */
  private static function get_preload_fonts_from_optimized_css(): array
  {
    $preload_urls = [];
    
    try {
      // Look for cached Google Fonts CSS files
      $cache_dir = ZIZI_CACHE_CACHE_DIR;
      if (!is_dir($cache_dir)) {
        return $preload_urls;
      }
      
      // Find CSS files that might contain font URLs
      $css_files = glob($cache_dir . '*.css');
      
      foreach ($css_files as $css_file) {
        if (!is_readable($css_file)) {
          continue;
        }
        
        $css_content = file_get_contents($css_file);
        if (empty($css_content)) {
          continue;
        }
        
        // Extract font URLs from CSS
        $font_urls = self::get_font_urls($css_content);
        
        if (!empty($font_urls)) {
          // Limit to prevent too many preloads
          $font_urls = array_slice($font_urls, 0, 6); // Max 6 font files
          $preload_urls = array_merge($preload_urls, $font_urls);
        }
      }
      
      // Remove duplicates and limit total
      $preload_urls = array_unique($preload_urls);
      $preload_urls = array_slice($preload_urls, 0, 4); // Max 4 total preloads
      
      if (!empty(SysConfig::$config['fonts_debug']) && !empty($preload_urls)) {
        \ZiziCache\CacheSys::writeLog(
          'Font preload: Found ' . count($preload_urls) . ' font URLs from optimized CSS: ' . 
          implode(', ', array_map('basename', $preload_urls)), 
          'info', 'ZiziCache'
        );
      }
      
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Font preload error: ' . $e->getMessage(), 'Security');
    }
    
    return $preload_urls;
  }

  /**
   * Adds preload links for critical fonts
   * 
   * Inserts preload links for fonts specified in the configuration or
   * automatically detected from Font Statistics for improved loading performance.
   *
   * @param string $html The HTML content to process
   * @return string Modified HTML with font preload links
   */  public static function preload_fonts(string $html): string
  {
    // Check master font optimization switch first
    if (empty(SysConfig::$config['fonts_enabled'])) {
      return $html;
    }
    
    // Get manually configured preload URLs
    $manual_font_urls = SysConfig::$config['fonts_preload_urls'] ?? [];
    
    // Get automatically detected font URLs from optimized CSS (based on Font Statistics)
    $auto_font_urls = [];
    if (!empty(SysConfig::$config['fonts_optimize_google_fonts'])) {
      $auto_font_urls = self::get_preload_fonts_from_optimized_css();
    }
    
    // Combine manual and automatic URLs
    $font_urls = array_merge($manual_font_urls, $auto_font_urls);
    $font_urls = array_unique($font_urls); // Remove duplicates
    
    if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog(
          sprintf('Font preloading: %d manual URLs, %d auto-detected URLs, %d total URLs', 
            count($manual_font_urls), count($auto_font_urls), count($font_urls)
          ), 'info', 'ZiziCache'
        );
        \ZiziCache\CacheSys::writeLog('Font preload URLs: ' . implode(', ', $font_urls), 'info', 'ZiziCache');
    }

    if (empty($font_urls)) {
        return $html;
    }

    $preload_tags_array = [];
    $only_woff2 = !empty(SysConfig::$config['fonts_preload_only_woff2']);
    
    try {
      foreach ($font_urls as $font_url) {
        if(empty($font_url) || !is_string($font_url) || !filter_var($font_url, FILTER_VALIDATE_URL)) continue;
        
        $path_info = pathinfo($font_url, PATHINFO_EXTENSION);
        if(empty($path_info)) continue;
        
        // Filter by extension if only WOFF2 is enabled
        if ($only_woff2 && strtolower($path_info) !== 'woff2') {
            continue;
        }
        
        $type = 'font/' . strtolower($path_info);
        
        // Determine fetchpriority: high for auto-detected (critical) fonts, normal for manual
        $is_auto_detected = in_array($font_url, $auto_font_urls);
        $fetchpriority = $is_auto_detected ? 'high' : 'normal';
        
        $preload_tags_array[] = "<link rel='preload' href='" . esc_url($font_url) . "' as='font' type='{$type}' fetchpriority='{$fetchpriority}' crossorigin='anonymous'>";
      }

      if(empty($preload_tags_array)) return $html;

      $preload_tags_string = implode(PHP_EOL, array_unique($preload_tags_array));      if (strpos($html, '</title>') !== false) {
        $html = SysTool::str_replace_first('</title>', '</title>' . PHP_EOL . $preload_tags_string, $html);
      } elseif (strpos($html, '<head>') !== false) {
         $html = SysTool::str_replace_first('<head>', '<head>' . PHP_EOL . $preload_tags_string, $html);
      }

    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Font Optimizer Error (add_preload_links): ' . $e->getMessage(), 'Security');
    }
    return $html;
  }

  /**
   * Injects font-display: swap into @font-face rules
   * 
   * Ensures all @font-face declarations include the font-display: swap
   * property for better font loading behavior.
   *
   * @param string $content The CSS content to process
   * @return string Modified CSS with font-display: swap
   */
  public static function inject_display_swap(string $content): string
  {
    if (empty(SysConfig::$config['fonts_display_swap']) || empty(trim($content))) {
      return $content;
    }

    // Remove existing font-display: xxx
    $content = preg_replace('/font-display:\s*(swap|block|fallback|optional)\s*;?/', '', $content) ?? $content;

    // Add font-display: swap
    return preg_replace('/(@font-face\s*{)/', '$1font-display:swap;', $content) ?? $content;
  }

  /**
   * Downloads multiple font files to the local filesystem
   * 
   * Efficiently downloads font files from remote URLs and saves them
   * to the specified directory with proper error handling.
   *
   * @param array $urls Array of font URLs to download
   * @param string $save_path Directory path where to save the fonts
   * @return bool True if all fonts were downloaded successfully, false otherwise
   */
  public static function download_fonts(array $urls, string $save_path): bool
  {
    if(empty($urls) || empty($save_path) || !is_dir($save_path) || !is_writable($save_path)){
        return false;
    }
    // Initialize a single cURL handle for reuse
    $curl_handle = curl_init();
    if($curl_handle === false) return false;

    curl_setopt($curl_handle, CURLOPT_USERAGENT, SysTool::$user_agent);
    curl_setopt($curl_handle, CURLOPT_HEADER, 0);
    curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 10); // Reduced timeout
    curl_setopt($curl_handle, CURLOPT_TIMEOUT, 30); // Added overall timeout    curl_setopt($curl_handle, CURLOPT_ENCODING, ''); // Accept encoding responses (gzip, deflate)
    curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, true); // Return content instead of direct output
    curl_setopt($curl_handle, CURLOPT_FOLLOWLOCATION, true); // Follow redirects
      // Kontrola, zda jsme v lokálním vývojovém prostředí
    if (defined('ZIZI_CACHE_LOCAL_ENV') && ZIZI_CACHE_LOCAL_ENV) {
        // Pro lokální vývojové prostředí
        $ca_bundle_path = ZIZI_CACHE_PLUGIN_DIR . 'includes/cacert.pem';
        if (file_exists($ca_bundle_path)) {
            curl_setopt($curl_handle, CURLOPT_CAINFO, $ca_bundle_path);
            curl_setopt($curl_handle, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($curl_handle, CURLOPT_SSL_VERIFYHOST, 2);
            \ZiziCache\CacheSys::writeLog("Using bundled CA certificates in local environment", 'Security');
        } else {
            curl_setopt($curl_handle, CURLOPT_SSL_VERIFYPEER, false); // Vypnout ověřování SSL certifikátu v lokálním prostředí
            curl_setopt($curl_handle, CURLOPT_SSL_VERIFYHOST, 0); // Vypnout ověřování hostitele v lokálním prostředí
            \ZiziCache\CacheSys::writeLog("SSL verification disabled in local environment - CA bundle not found", 'Security');
        }
    } else {
        curl_setopt($curl_handle, CURLOPT_SSL_VERIFYPEER, true); // Verify SSL cert
        curl_setopt($curl_handle, CURLOPT_SSL_VERIFYHOST, 2); // Check common name and verify server cert
    }

    $success = true;
    // Loop through each URL
    foreach ($urls as $url) {
      if(empty($url) || !is_string($url) || !filter_var($url, FILTER_VALIDATE_URL)) continue;
      $file_name = basename($url);
      if(empty($file_name)) continue;
      $file = rtrim($save_path, '/') . '/' . $file_name;

      // Set cURL options specific to this request
      curl_setopt($curl_handle, CURLOPT_URL, $url);
      
      $font_content = curl_exec($curl_handle);
      $http_code = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);      if ($font_content === false || $http_code !== 200) {
        \ZiziCache\CacheSys::writeError("Font Download Error: Failed to download {$url}. HTTP Code: {$http_code}. CURL Error: " . curl_error($curl_handle), 'Security');
        $success = false;
        continue;
      }
      if(file_put_contents($file, $font_content) === false){
          \ZiziCache\CacheSys::writeError("Font Download Error: Failed to write font to {$file}", 'Security');
          $success = false;
      }
    }

    // Close the cURL session
    curl_close($curl_handle);
    return $success;
  }

  /**
   * Self-hosts a remote stylesheet and its fonts
   * 
   * Downloads a remote stylesheet, extracts font URLs, downloads the fonts,
   * and updates the CSS to use local font files.
   *
   * @param string $url URL of the remote stylesheet
   * @param string $file_path Local path where to save the stylesheet
   * @return bool True if successful, false on failure
   */
  private static function self_host_style_sheet(string $url, string $file_path): bool
  {
    // If URL starts with "//", add https
    if (strpos($url, '//') === 0) {
      $url = 'https:' . $url;
    }
    if(!filter_var($url, FILTER_VALIDATE_URL)) return false;    // Decode URL (e.g. https://fonts.googleapis.com/css?display=swap&#038;family=Nunito%3A400%2C700&#038;ver=c63b)
    $decoded_url = html_entity_decode($url);    // Security: SSRF protection - validate URL doesn't point to private IP ranges
    $parsed_url = parse_url($decoded_url);
    if (isset($parsed_url['host'])) {
      $ip = gethostbyname($parsed_url['host']);
      if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
        \ZiziCache\CacheSys::writeWarning('Font Optimizer Error: SSRF protection - blocked private/reserved IP: ' . $ip . ' for URL: ' . $decoded_url, 'Security');
        return false;
      }
    }

    // Download style sheet
    $css_file_response = wp_remote_get($decoded_url, [
      'user-agent' => SysTool::$user_agent,
      'httpversion' => '2.0', // Consider making this conditional or configurable if issues arise
      'timeout' => 10, // Added timeout
    ]);    // Check Google Fonts returned response
    if (is_wp_error($css_file_response) || wp_remote_retrieve_response_code($css_file_response) !== 200) {
      \ZiziCache\CacheSys::writeError('Font Optimizer Error: Failed to download Google Fonts CSS: ' . $decoded_url . ' Error: ' . (is_wp_error($css_file_response) ? $css_file_response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($css_file_response)), 'Security');
      return false;
    }

    // Extract body (CSS)
    $css = wp_remote_retrieve_body($css_file_response);
    if(empty(trim($css))) return false;

    // NEW: Filter CSS by used fonts from statistics
    $used_fonts = self::get_used_fonts_from_stats();
    if (!empty($used_fonts)) {
        $original_size = strlen($css);
        $css = self::filter_css_by_used_fonts($css, $used_fonts);
        $filtered_size = strlen($css);
        
        if (!empty(SysConfig::$config['fonts_debug'])) {
            \ZiziCache\CacheSys::writeLog(
                sprintf('Font optimization: CSS filtered from %d to %d bytes (%.1f%% reduction)', 
                    $original_size, $filtered_size, 
                    $original_size > 0 ? (($original_size - $filtered_size) / $original_size) * 100 : 0
                ), 
                'info', 'ZiziCache'
            );
        }
    } else {
        if (!empty(SysConfig::$config['fonts_debug'])) {
            \ZiziCache\CacheSys::writeLog('Font optimization: No font statistics found - keeping full CSS', 'info', 'ZiziCache');
        }
    }

    // ATOMIC: Deduplicate @font-face rules while preserving unicode-range subsets
    if (!empty(SysConfig::$config['fonts_enabled'])) {
        $pre_dedup_size = strlen($css);
        $css = self::deduplicate_font_faces($css);
        $post_dedup_size = strlen($css);
        
        if (!empty(SysConfig::$config['fonts_debug']) && $pre_dedup_size !== $post_dedup_size) {
            \ZiziCache\CacheSys::writeLog(
                sprintf('Font deduplication: CSS size changed from %d to %d bytes', 
                    $pre_dedup_size, $post_dedup_size
                ), 
                'info', 'ZiziCache'
            );
        }
    }

    // Get list of fonts (woff2 files) inside the CSS
    $font_urls = self::get_font_urls($css);
    if(empty($font_urls)){
        // If no font URLs, just save the CSS
        return file_put_contents($file_path, $css) !== false;
    }    if(!self::download_fonts($font_urls, ZIZI_CACHE_CACHE_DIR)){
        \ZiziCache\CacheSys::writeError("Font Optimizer Error: Failed to download one or more fonts for {$decoded_url}", 'Security');
        // Return false instead of continuing with incomplete fonts
        return false;
    }

    foreach ($font_urls as $font_url) {
      if(empty($font_url) || !is_string($font_url)) continue;
      $font_basename = basename($font_url);
      if(empty($font_basename)) continue;

      $cached_font_path = ZIZI_CACHE_CACHE_DIR . $font_basename;
      $cached_font_url = ZIZI_CACHE_CACHE_URL . $font_basename;
        // Skip if file is too small or doesn't exist (meaning download failed for this specific font)
      if (!is_file($cached_font_path) || filesize($cached_font_path) < 100) {
        \ZiziCache\CacheSys::writeWarning("Font Optimizer Warning: Font file {$cached_font_path} is missing or too small. CSS will retain original URL for this font: {$font_url}", 'Security');
        continue;
      }
      $css = str_replace($font_url, $cached_font_url, $css);
    }

    return file_put_contents($file_path, $css) !== false;
  }

  /**
   * Validates that self-hosted font files are complete and accessible
   * 
   * Checks CSS file and all referenced font files to ensure they're properly downloaded
   * and accessible before switching from Google Fonts to local hosting.
   *
   * @param string $css_file_path Path to the local CSS file
   * @return bool True if all fonts are valid, false otherwise
   */
  private static function validate_self_hosted_fonts(string $css_file_path): bool
  {
    if (!is_file($css_file_path) || filesize($css_file_path) < 100) {
      return false;
    }
    
    try {
      $css_content = file_get_contents($css_file_path);
      if (empty($css_content)) {
        return false;
      }
      
      // Extract font URLs from CSS
      $font_urls = self::get_font_urls($css_content);
      
      // If no font URLs found, CSS might be invalid
      if (empty($font_urls)) {
        // Check if CSS contains @font-face rules at all
        if (strpos($css_content, '@font-face') === false) {
          \ZiziCache\CacheSys::writeWarning("Font Optimizer Warning: CSS file contains no @font-face rules: {$css_file_path}", 'Security');
          return false;
        }
        // CSS has @font-face but no URLs extracted - might be OK for some edge cases
        return true;
      }
      
      // Validate each referenced font file
      foreach ($font_urls as $font_url) {
        $font_basename = basename(parse_url($font_url, PHP_URL_PATH));
        if (empty($font_basename)) continue;
        
        $font_file_path = ZIZI_CACHE_CACHE_DIR . $font_basename;
        
        // Check if font file exists and has reasonable size
        if (!is_file($font_file_path)) {
          \ZiziCache\CacheSys::writeWarning("Font Optimizer Warning: Missing font file: {$font_file_path}", 'Security');
          return false;
        }
        
        if (filesize($font_file_path) < 100) {
          \ZiziCache\CacheSys::writeWarning("Font Optimizer Warning: Font file too small (likely incomplete): {$font_file_path}", 'Security');
          return false;
        }
        
        // Additional validation: check if file is readable
        if (!is_readable($font_file_path)) {
          \ZiziCache\CacheSys::writeWarning("Font Optimizer Warning: Font file not readable: {$font_file_path}", 'Security');
          return false;
        }
      }
      
      return true;
      
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError("Font Optimizer Error during validation: " . $e->getMessage(), 'Security');
      return false;
    }
  }

  /**
   * Extracts font URLs from CSS content
   * 
   * Parses CSS and extracts all font file URLs from @font-face rules.
   *
   * @param string $css The CSS content to parse
   * @return array Array of unique font file URLs
   */
  public static function get_font_urls(string $css): array
  {
    if(empty(trim($css))) return [];
    // Extract font urls like: https://fonts.gstatic.com/s/roboto/v20/KFOmCnqEu92Fr1Mu4mxKKTU1Kg.woff2
    // Regex updated for more robustness and to capture various quote types or no quotes.
    $regex = '/(?:@font-face\s*\{[^}]*?src\s*:[^;]*?)url\(\s*([\'"]?)((?:https?:)?\/\/[^\'"\)]+\.(?:woff2|woff|ttf|svg|otf|eot)(\?[^\'"\)]*)?)\1\s*\)/i';
    preg_match_all($regex, $css, $matches);
    return array_unique($matches[2] ?? []); // $matches[2] contains the full URL
  }
    /**
   * Detects fonts used in above-the-fold content and automatically adds them to preload list
   * This helps with Core Web Vitals and overall font performance
   * 
   * @param string $html The full HTML content of the page
   * @return string Modified HTML with preloaded critical fonts
   */




  
  /**
   * Get the current page URL
   * 
   * @return string The current page URL
   */
  private static function get_current_page_url()
  {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'] ?? '';
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    
    return $protocol . $host . $uri;
  }

    /**
   * Fetch CSS content from a local stylesheet URL
   * 
   * @param string $url URL to the local stylesheet
   * @return string CSS content or empty string on failure
   */
  private static function fetch_local_css(string $url): string {
    // EMERGENCY DISABLE - complete function removal
    return '';
  }
  
  /**
   * Fetch content from a URL
   * Optimized for performance and resource usage
   */
  public static function fetch_url_content(string $url, int $max_size = 204800)
  {
    if (empty($url)) {
      return false;
    }
    
    $args = [
      'timeout'     => 5,  // Reduced from 15 to 5 seconds
      'redirection' => 1,  // Reduced from 2 to 1
      'sslverify'   => false,
      'user-agent'  => 'ZiziCache Font Monitor v' . ZIZI_CACHE_VERSION,
      'blocking'    => true,
      'httpversion' => '1.1',
      'headers'     => [
        'Accept-Encoding' => 'gzip, deflate',
      ],
    ];

    $response = wp_remote_get($url, $args);
    
    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
      return false;
    }
    
    $content = wp_remote_retrieve_body($response);
    
    // Check content size
    $content_length = strlen($content);
    if ($content_length > $max_size) {
      return false;
    }
    
    return $content;
  }

  /**
   * Optimizes Google Fonts in inline style attributes
   * 
   * Processes all inline style attributes in HTML elements and optimizes Google Fonts URLs,
   * adds display=swap parameter, and optionally self-hosts them for better performance.
   * 
   * This method fills a gap that exists in most optimization plugins including FlyingPress,
   * which only process <style> blocks and <link> tags but ignore inline style attributes.
   *
   * @param string $html The HTML content to process
   * @return string Modified HTML with optimized inline Google Fonts
   */  public static function optimize_inline_google_fonts(string $html): string
  {
    // Simplified: This feature is disabled in the simplified version
    return $html;
  }

  /**
   * Process Google Fonts URLs within inline style content (simplified)
   * @param string $style_content The CSS content from style attribute
   * @return string Optimized CSS content
   */
  private static function process_inline_style_fonts(string $style_content): string
  {
    // Simplified: Return original content unchanged
    return $style_content;
  }

  private static function add_display_swap_to_url(string $url): string
  {
    // Remove existing display parameter if present
    $url = preg_replace('/[?&]display=[^&]*/', '', $url);
    
    // Add display=swap
    $separator = (strpos($url, '?') !== false) ? '&' : '?';
    return $url . $separator . 'display=swap';
  }
  
  /**
   * Convert Google Font URL to self-hosted version (simplified)
   * @param string $url Google Font URL
   * @return string URL with display=swap as fallback
   */
  private static function convert_to_self_hosted_url(string $url): string
  {
    // Simplified: Just add display=swap instead of self-hosting
    return self::add_display_swap_to_url($url);
  }
    /**
   * Validate inline style content for security
   * 
   * Checks for potentially dangerous CSS that should not be processed.
   * 
   * @param string $style_content CSS content to validate
   * @return bool True if safe to process, false otherwise
   */
  private static function is_safe_inline_style(string $style_content): bool
  {
    if (empty($style_content)) {
      return true;
    }

    // Define patterns for potentially dangerous CSS
    $dangerous_patterns = [
      '/javascript\s*:/i',           // javascript: URLs
      '/expression\s*\(/i',          // IE expression() - XSS vector
      '/behavior\s*:/i',             // IE behavior property
      '/binding\s*:/i',              // Mozilla binding
      '/-moz-binding/i',             // Mozilla binding
      '/vbscript\s*:/i',             // VBScript URLs
      '/data\s*:\s*text\/html/i',    // Data URLs with HTML
      '/}\s*[^}]*{/i',               // CSS injection (closing brace followed by new CSS rule)
      '/;\s*}\s*[^}]*{/i',           // CSS injection after semicolon
    ];
    
    foreach ($dangerous_patterns as $pattern) {
      if (preg_match($pattern, $style_content)) {
        return false;
      }
    }

    // Additional check: only allow Google Fonts domains in @import and url()
    if (preg_match('/@import|url\(/i', $style_content)) {      // Check if there are any non-Google Fonts URLs
      $url_pattern = '/(?:@import\s+)?url\(\s*[\'"]?(?!https?:\/\/fonts\.googleapis\.com\/)(https?:\/\/[^\)\'\"]+)/i';
      if (preg_match($url_pattern, $style_content)) {
        \ZiziCache\CacheSys::writeLog("Font Optimizer: Non-Google Fonts URL detected in inline style - skipping optimization", 'INFO');
        return false;
      }
    }
    
    // Additional security: check for CSS rule injection patterns
    // This catches patterns like "'; } body { display: none; } .malicious {"
    if (preg_match('/[\'"];?\s*}\s*[a-zA-Z#.][^}]*{/i', $style_content)) {
      return false;
    }
    
    return true;
  }

  /**
   * Logs diagnostic information about font optimization settings
   * 
   * This function helps debug font optimization settings by adding a comment to the HTML output
   * showing which font optimization features are enabled/disabled.
   *
   * @param string $html The HTML content to modify
   * @return string HTML with added diagnostic comment
   */
  public static function add_font_optimization_debug_info(string $html): string
  {
    // Only add debug info if specifically enabled
    if (empty(SysConfig::$config['fonts_debug'])) {
      return $html;
    }
    
    $config = SysConfig::$config;
    $debug_info = [];
    
    // Collect font optimization settings
    $debug_info[] = "Font Optimization Debug (Simplified):";
    $debug_info[] = "- fonts_optimize_google_fonts: " . (empty($config['fonts_optimize_google_fonts']) ? 'DISABLED' : 'ENABLED');
    $debug_info[] = "- fonts_display_swap: " . (empty($config['fonts_display_swap']) ? 'DISABLED' : 'ENABLED');
    $debug_info[] = "- fonts_preload_urls count: " . (count($config['fonts_preload_urls'] ?? []));
    $debug_info[] = "- fonts_allow_cdn_domains: " . (empty($config['fonts_allow_cdn_domains']) ? 'DISABLED' : 'ENABLED');
    
    // Create HTML comment
    $comment = "\n<!-- " . implode("\n", $debug_info) . " -->\n";
      // Add at the end of the body
    if (strpos($html, '</body>') !== false) {
      $html = str_replace('</body>', $comment . '</body>', $html);
    } else {
      // Only append if the content appears to be HTML (contains HTML tags)
      if (preg_match('/<[^>]+>/', $html)) {
        $html .= $comment;
      }
    }
    
    return $html;
  }

  /**
   * Reset all font statistics data
   * 
   * Clears all collected font statistics data from the database.
   * This includes both basic statistics and detailed font data collected from frontend.
   *
   * @return bool True if reset was successful, false otherwise
   */
  public static function reset_font_statistics(): bool
  {
    global $wpdb;
    
    try {
        // Clear basic font statistics
        delete_option('zizi_cache_font_stats');
        
        // Clear font data collected from frontend
        $deleted_count = $wpdb->query("
            DELETE FROM {$wpdb->options} 
            WHERE option_name LIKE 'zizi_cache_font_data_%'
        ");
        
        // Log the reset action
        \ZiziCache\CacheSys::writeLog(
            sprintf('Font statistics have been reset. Deleted %d records.', $deleted_count), 
            'INFO'
        );
        
        return true;
        
    } catch (\Exception $e) {
        \ZiziCache\CacheSys::writeError('Font Statistics Reset Error: ' . $e->getMessage(), 'Security');
        return false;
    }
  }

  /**
   * Injects optimized Google Fonts CSS into HTML head
   * 
   * Generates and injects Google Fonts CSS link based on Font Statistics.
   * This proactively loads only the fonts that are actually used on the site.
   *
   * @param string $html The HTML content to process
   * @return string Modified HTML with injected Google Fonts link
   */
  public static function inject_optimized_google_fonts(string $html): string
  {
    // Check master font optimization switch first
    if (empty(SysConfig::$config['fonts_enabled'])) {
      return $html;
    }
    
    // Only inject if optimization is enabled
    if (empty(SysConfig::$config['fonts_optimize_google_fonts'])) {
      return $html;
    }

    $optimized_url = self::generate_optimized_google_fonts_url();
    
    if (!$optimized_url) {
      return $html;
    }

    // Check if Google Fonts link already exists
    if (preg_match('/<link[^>]+fonts\.googleapis\.com[^>]*>/i', $html)) {
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('Google Fonts link already exists in HTML - skipping injection', 'info', 'ZiziCache');
      }
      return $html;
    }

    // Check if page builder is active - if so, load CSS synchronously for compatibility
    $is_page_builder_active = false;
    if (class_exists('ZiziCache\Plugins\Integrations\PageBuilders')) {
      $is_page_builder_active = \ZiziCache\Plugins\Integrations\PageBuilders::is_page_builder_active();
    }

    // Generate optimized link tag
    if ($is_page_builder_active) {
      // Page builder mode - load synchronously for compatibility
      $link_tag = sprintf(
        '<link rel="stylesheet" href="%s">',
        esc_url($optimized_url)
      );
      
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('Page builder detected - loading Google Fonts synchronously for compatibility', 'info', 'ZiziCache');
      }
    } else {
      // Normal mode - load asynchronously for performance
      $link_tag = sprintf(
        '<link rel="stylesheet" href="%s" media="print" onload="this.media=\'all\';this.onload=null;">',
        esc_url($optimized_url)
      );
    }

    // Add preconnect for performance
    $preconnect_tags = 
      '<link rel="preconnect" href="https://fonts.googleapis.com">' . PHP_EOL .
      '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>' . PHP_EOL;

    $full_injection = $preconnect_tags . $link_tag;

    // Inject before </head>
    if (strpos($html, '</head>') !== false) {
      $html = str_replace('</head>', $full_injection . PHP_EOL . '</head>', $html);
      
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('Injected optimized Google Fonts CSS into HTML head', 'info', 'ZiziCache');
      }
    }

    return $html;
  }

  /**
   * Generates Google Fonts CSS URL based on Font Statistics
   * 
   * Creates optimized Google Fonts URL containing only fonts detected
   * in Font Statistics data, significantly reducing CSS size.
   *
   * @return string|null Generated Google Fonts URL or null if no statistics
   */
  public static function generate_optimized_google_fonts_url(): ?string
  {
    $used_fonts = self::get_used_fonts_from_stats();
    
    if (empty($used_fonts)) {
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('Cannot generate optimized Google Fonts URL - no font statistics available', 'info', 'ZiziCache');
      }
      return null;
    }

    // Known Google Fonts that we can safely generate URLs for
    $google_fonts = [
      'Inter', 'Roboto', 'Open Sans', 'Poppins', 'Lato', 'Montserrat', 
      'Source Sans Pro', 'Raleway', 'PT Sans', 'Nunito', 'Playfair Display',
      'Merriweather', 'Lora', 'Source Serif Pro', 'Crimson Text'
    ];

    $font_families_params = [];
    $total_variants = 0;

    foreach ($used_fonts as $family => $weights) {
      // Only include known Google Fonts
      if (!in_array($family, $google_fonts)) {
        continue;
      }

      // Sort weights and limit to reasonable number
      sort($weights);
      $weights = array_slice($weights, 0, 4); // Max 4 weights per family
      
      $family_param = str_replace(' ', '+', $family);
      $weights_param = implode(';', $weights);
      
      $font_families_params[] = "{$family_param}:wght@{$weights_param}";
      $total_variants += count($weights);
    }

    if (empty($font_families_params)) {
      if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog('No Google Fonts found in statistics - available fonts: ' . implode(', ', array_keys($used_fonts)), 'info', 'ZiziCache');
      }
      return null;
    }

    // Generate URL
    $families_string = implode('&family=', $font_families_params);
    $optimized_url = "https://fonts.googleapis.com/css2?family={$families_string}&display=swap";

    if (!empty(SysConfig::$config['fonts_debug'])) {
      \ZiziCache\CacheSys::writeLog(
        sprintf('Generated optimized Google Fonts URL with %d families, %d variants: %s', 
          count($font_families_params), $total_variants, $optimized_url
        ), 
        'info', 'ZiziCache'
      );
    }

    return $optimized_url;
  }

  /**
   * Extracts only fonts that are actually detected and used on the frontend
   * according to Font Statistics data.
   *
   * @return array Array of ['font-family' => [weights]]
   */
  private static function get_used_fonts_from_stats(): array
  {
    global $wpdb;
    
    // Get all font data from frontend collection
    $options = $wpdb->get_results("
        SELECT option_name, option_value
        FROM {$wpdb->options} 
        WHERE option_name LIKE 'zizi_cache_font_data_%'
    ");
    
    if (empty($options)) {
        if (!empty(SysConfig::$config['fonts_debug'])) {
            \ZiziCache\CacheSys::writeLog('No font statistics found - CSS will not be filtered', 'info', 'ZiziCache');
        }
        return [];
    }

    $used_fonts = [];
    $total_variants = 0;
    
    foreach ($options as $option) {
        $data = maybe_unserialize($option->option_value);
        
        // Skip expired or invalid data (older than 7 days)
        if (!$data || empty($data['fonts']) || (isset($data['updated']) && (time() - $data['updated']) > 604800)) {
            continue;
        }
        
        foreach ($data['fonts'] as $font) {
            if (empty($font['family'])) continue;
            
            $family = trim($font['family'], '"\'');
            $weight = isset($font['weight']) ? (int) $font['weight'] : 400;
            
            if (!isset($used_fonts[$family])) {
                $used_fonts[$family] = [];
            }
            
            if (!in_array($weight, $used_fonts[$family])) {
              $used_fonts[$family][] = $weight;
              $total_variants++;
            }
        }
    }
    
    if (!empty(SysConfig::$config['fonts_debug'])) {
        \ZiziCache\CacheSys::writeLog("Font filtering: Found {$total_variants} font variants from " . count($used_fonts) . " families in statistics: " . implode(', ', array_keys($used_fonts)), 'info', 'ZiziCache');
    }
    
    return $used_fonts;
  }

  /**
   * Filter Google Fonts CSS content to only include rules for used fonts.
   * 
   * This method processes CSS from Google Fonts and removes all @font-face rules
   * for fonts that are not detected in the Font Statistics. This significantly
   * reduces the CSS file size and prevents loading of unused font variants.
   *
   * @param string $css The original CSS content from Google Fonts
   * @param array $used_fonts Array of used fonts ['font-family' => [weights]]
   * @return string The filtered CSS content containing only used fonts
   */
  private static function filter_css_by_used_fonts(string $css, array $used_fonts): string
  {
      if (empty($used_fonts)) {
          if (!empty(SysConfig::$config['fonts_debug'])) {
              \ZiziCache\CacheSys::writeLog('No used fonts data - returning original CSS', 'info', 'ZiziCache');
          }
          return $css;
      }

      // Extract all @font-face rules
      $font_face_regex = '/@font-face\s*\{(?:[^{}]|\{[^{}]*\})*\}/s';
      preg_match_all($font_face_regex, $css, $matches);

      $filtered_css = '';
      $included_count = 0;
      $excluded_count = 0;
      $font_face_map = []; // Track unique font-face combinations
      
      foreach ($matches[0] as $font_face_rule) {
          $font_family_regex = "/font-family:\s*['\"]?([^;'\"]+)['\"]?[;\s]/i";
          $font_weight_regex = '/font-weight:\s*(\d+)[;\s]/i';
          $unicode_range_regex = '/unicode-range:\s*([^;]+);?/i';

          preg_match($font_family_regex, $font_face_rule, $family_match);
          preg_match($font_weight_regex, $font_face_rule, $weight_match);
          preg_match($unicode_range_regex, $font_face_rule, $unicode_match);

          if (isset($family_match[1])) {
              $font_family = trim($family_match[1], '"\'');
              $font_weight = isset($weight_match[1]) ? (int) $weight_match[1] : 400;
              $unicode_range = isset($unicode_match[1]) ? trim($unicode_match[1]) : '';

              // Check if this font family and weight are in our used fonts
              $is_used = false;
              foreach ($used_fonts as $used_family => $used_weights) {
                  if (strcasecmp(trim($used_family, '"\''), $font_family) === 0) {
                      if (in_array($font_weight, $used_weights)) {
                          $is_used = true;
                          break;
                      }
                  }
              }

              if ($is_used) {
                  // Create unique key including unicode-range to preserve subsets
                  $unique_key = sprintf('%s_%d_%s', 
                      strtolower($font_family), 
                      $font_weight, 
                      md5($unicode_range)
                  );
                  
                  // Only include if not already processed (prevents duplicates while preserving unicode-range)
                  if (!isset($font_face_map[$unique_key])) {
                      $font_face_map[$unique_key] = true;
                      $filtered_css .= $font_face_rule . PHP_EOL;
                      $included_count++;
                  }
              } else {
                  $excluded_count++;
                  if (!empty(SysConfig::$config['fonts_debug'])) {
                      \ZiziCache\CacheSys::writeLog("Font filtering: Excluded {$font_family} weight {$font_weight} - not in statistics", 'info', 'ZiziCache');
                  }
              }
          } else {
              // If we can't parse font-family, include it to be safe
              $filtered_css .= $font_face_rule . PHP_EOL;
              $included_count++;
          }
      }

      // Add any remaining CSS that's not @font-face rules (e.g., @charset)
      $non_font_face_css = preg_replace($font_face_regex, '', $css);
      
      // Remove orphaned comments that are likely left after removing @font-face rules
      $non_font_face_css = preg_replace('/\/\*\s*(?:vietnamese|latin-ext|latin|cyrillic-ext|cyrillic|greek-ext|greek|math|symbols)\s*\*\/\s*(?=\/\*|$)/i', '', $non_font_face_css);
      
      // Remove multiple consecutive line breaks
      $non_font_face_css = preg_replace('/\n\s*\n\s*\n/', "\n\n", $non_font_face_css);
      $non_font_face_css = trim($non_font_face_css);
      
      if (!empty($non_font_face_css)) {
          $filtered_css = $non_font_face_css . PHP_EOL . $filtered_css;
      }

      if (!empty(SysConfig::$config['fonts_debug'])) {
          \ZiziCache\CacheSys::writeLog("Font filtering completed: included {$included_count} variants, excluded {$excluded_count} variants", 'info', 'ZiziCache');
      }

      return $filtered_css;
  }

  /**
   * Deduplicate @font-face rules while preserving unicode-range subsets
   * 
   * This method intelligently removes duplicate @font-face rules for the same font-family
   * and font-weight combination, while preserving different unicode-range subsets.
   * Essential for multilingual sites that need cyrillic, vietnamese, arabic, etc. subsets.
   *
   * @param string $css The CSS content containing @font-face rules
   * @return string Deduplicated CSS with preserved unicode-range functionality
   */
  private static function deduplicate_font_faces(string $css): string
  {
      if (empty(trim($css))) {
          return $css;
      }

      // Extract all @font-face rules
      $font_face_regex = '/@font-face\s*\{(?:[^{}]|\{[^{}]*\})*\}/s';
      preg_match_all($font_face_regex, $css, $matches);
      
      if (empty($matches[0])) {
          return $css;
      }

      $deduplicated_rules = [];
      $duplicate_count = 0;
      $preserved_count = 0;
      
      foreach ($matches[0] as $font_face_rule) {
          // Extract font properties
          $font_family = self::extract_css_property($font_face_rule, 'font-family');
          $font_weight = self::extract_css_property($font_face_rule, 'font-weight');
          $font_style = self::extract_css_property($font_face_rule, 'font-style');
          $unicode_range = self::extract_css_property($font_face_rule, 'unicode-range');
          $src_url = self::extract_font_src_url($font_face_rule);
          
          // Skip if can't extract essential properties
          if (empty($font_family)) {
              $deduplicated_rules[] = $font_face_rule;
              $preserved_count++;
              continue;
          }
          
          // Normalize values
          $font_family = trim($font_family, '"\'');
          $font_weight = !empty($font_weight) ? trim($font_weight) : '400';
          $font_style = !empty($font_style) ? trim($font_style) : 'normal';
          
          // Create unique key - CRITICAL: include unicode-range in key
          $unique_key = sprintf('%s|%s|%s|%s', 
              strtolower($font_family), 
              $font_weight, 
              $font_style,
              !empty($unicode_range) ? trim($unicode_range) : 'default'
          );
          
          // Check for exact duplicate (same URL and properties)
          $is_duplicate = false;
          foreach ($deduplicated_rules as $existing_rule) {
              $existing_family = trim(self::extract_css_property($existing_rule, 'font-family'), '"\'');
              $existing_weight = trim(self::extract_css_property($existing_rule, 'font-weight') ?: '400');
              $existing_style = trim(self::extract_css_property($existing_rule, 'font-style') ?: 'normal');
              $existing_unicode = self::extract_css_property($existing_rule, 'unicode-range');
              $existing_url = self::extract_font_src_url($existing_rule);
              
              // Check if this is exact duplicate
              if (strcasecmp($existing_family, $font_family) === 0 &&
                  $existing_weight === $font_weight &&
                  $existing_style === $font_style &&
                  trim($existing_unicode ?: 'default') === trim($unicode_range ?: 'default') &&
                  $existing_url === $src_url) {
                  
                  $is_duplicate = true;
                  break;
              }
          }
          
          if (!$is_duplicate) {
              $deduplicated_rules[] = $font_face_rule;
              $preserved_count++;
          } else {
              $duplicate_count++;
              if (!empty(SysConfig::$config['fonts_debug'])) {
                  \ZiziCache\CacheSys::writeLog(
                      sprintf('Font deduplication: Removed duplicate %s %s %s (unicode-range: %s)', 
                          $font_family, $font_weight, $font_style, $unicode_range ?: 'default'
                      ), 
                      'info', 'ZiziCache'
                  );
              }
          }
      }
      
      // Reconstruct CSS
      $deduplicated_css = '';
      
      // Add non-@font-face CSS first
      $non_font_face_css = preg_replace($font_face_regex, '', $css);
      $non_font_face_css = trim($non_font_face_css);
      if (!empty($non_font_face_css)) {
          $deduplicated_css .= $non_font_face_css . PHP_EOL;
      }
      
      // Add deduplicated @font-face rules
      if (!empty($deduplicated_rules)) {
          $deduplicated_css .= implode(PHP_EOL, $deduplicated_rules);
      }
      
      // Log results
      if (!empty(SysConfig::$config['fonts_debug']) && $duplicate_count > 0) {
          \ZiziCache\CacheSys::writeLog(
              sprintf('Font deduplication completed: %d duplicates removed, %d rules preserved', 
                  $duplicate_count, $preserved_count
              ), 
              'info', 'ZiziCache'
          );
      }
      
      return $deduplicated_css;
  }

  /**
   * Extract specific CSS property value from a CSS rule
   * 
   * @param string $css_rule CSS rule to parse
   * @param string $property Property name to extract
   * @return string|null Property value or null if not found
   */
  private static function extract_css_property(string $css_rule, string $property): ?string
  {
      // Create regex pattern for the property
      $pattern = '/' . preg_quote($property, '/') . '\s*:\s*([^;]+);?/i';
      
      if (preg_match($pattern, $css_rule, $matches)) {
          return trim($matches[1]);
      }
      
      return null;
  }

  /**
   * Extract font source URL from @font-face rule
   * 
   * @param string $font_face_rule @font-face CSS rule
   * @return string|null Font URL or null if not found
   */
  private static function extract_font_src_url(string $font_face_rule): ?string
  {
      // Extract URL from src property - handles various formats
      $src_pattern = '/src\s*:\s*[^;]*?url\(\s*[\'"]?([^\)\'\"]+)[\'"]?\s*\)[^;]*/i';
      
      if (preg_match($src_pattern, $font_face_rule, $matches)) {
          return trim($matches[1]);
      }
      
      return null;
  }

  /**
   * Get site-specific unicode-range priority based on language settings
   * 
   * This method determines which unicode-range subsets should be prioritized
   * based on the site's language configuration and user preferences.
   *
   * @return array Array of unicode-range priorities
   */
  public static function get_site_unicode_priority(): array
  {
      // Get language priority setting from config
      $language_priority = SysConfig::$config['fonts_language_priority'] ?? 'auto';
      
      if ($language_priority !== 'auto') {
          return self::get_unicode_priority_by_language($language_priority);
      }
      
      // Auto-detect based on WordPress locale
      $locale = get_locale();
      $detected_language = 'latin'; // Default fallback
      
      // Map WordPress locales to language categories
      $locale_map = [
          // Cyrillic languages
          'ru_RU' => 'cyrillic',
          'uk' => 'cyrillic', 
          'bg_BG' => 'cyrillic',
          'mk_MK' => 'cyrillic',
          'sr_RS' => 'cyrillic',
          
          // Greek
          'el' => 'greek',
          
          // Vietnamese  
          'vi' => 'vietnamese',
          
          // Arabic/RTL
          'ar' => 'arabic',
          'fa_IR' => 'arabic',
          'he_IL' => 'hebrew',
          
          // Asian languages
          'ja' => 'japanese',
          'ko_KR' => 'korean',
          'zh_CN' => 'chinese-simplified',
          'zh_TW' => 'chinese-traditional',
      ];
      
      // Check for exact locale match
      if (isset($locale_map[$locale])) {
          $detected_language = $locale_map[$locale];
      } else {
          // Check for language prefix match (e.g., 'ru' from 'ru_RU')
          $lang_prefix = substr($locale, 0, 2);
          foreach ($locale_map as $mapped_locale => $mapped_language) {
              if (substr($mapped_locale, 0, 2) === $lang_prefix) {
                  $detected_language = $mapped_language;
                  break;
              }
          }
      }
      
      if (!empty(SysConfig::$config['fonts_debug'])) {
          \ZiziCache\CacheSys::writeLog(
              sprintf('Unicode priority auto-detection: locale=%s, detected=%s', $locale, $detected_language), 
              'info', 'ZiziCache'
          );
      }
      
      return self::get_unicode_priority_by_language($detected_language);
  }

  /**
   * Get unicode-range priority array for specific language
   * 
   * @param string $language Language identifier
   * @return array Prioritized unicode-range array
   */
  private static function get_unicode_priority_by_language(string $language): array
  {
      $priorities = [
          'latin' => ['latin', 'latin-ext'],
          'cyrillic' => ['cyrillic', 'cyrillic-ext', 'latin', 'latin-ext'],
          'greek' => ['greek', 'greek-ext', 'latin', 'latin-ext'],
          'vietnamese' => ['vietnamese', 'latin', 'latin-ext'],
          'arabic' => ['arabic', 'latin', 'latin-ext'],
          'hebrew' => ['hebrew', 'latin', 'latin-ext'],
          'japanese' => ['japanese', 'latin', 'latin-ext'],
          'korean' => ['korean', 'latin', 'latin-ext'],
          'chinese-simplified' => ['chinese-simplified', 'latin', 'latin-ext'],
          'chinese-traditional' => ['chinese-traditional', 'latin', 'latin-ext'],
      ];
      
      return $priorities[$language] ?? $priorities['latin'];
  }
}