<?php
/**
 * FTP Image Auto-Registration System
 * Automatically register FTP uploaded images in WordPress Media Library
 */

namespace ZiziCache;

class FTPImageRegistration
{
    private static $instance = null;
    
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize FTP image auto-registration
     */
    public static function init()
    {
        $instance = self::get_instance();
        
        // Auto-scan when Media Library is opened
        add_action('load-upload.php', [$instance, 'auto_scan_on_media_library']);
        
        // Add admin notice for new FTP images found
        add_action('admin_notices', [$instance, 'admin_notices']);
        
        // Add manual scan button to ImageConverter page
        add_action('wp_ajax_zizi_scan_ftp_images', [$instance, 'ajax_scan_ftp_images']);
    }
    
    /**
     * Auto-scan when Media Library is opened
     */
    public function auto_scan_on_media_library()
    {
        // Only scan once per session to avoid multiple scans
        if (!get_transient('zizi_ftp_scan_done_this_session')) {
            $result = $this->scan_and_register_ftp_images();
            
            // Set transient for 10 minutes to avoid multiple scans in same session
            set_transient('zizi_ftp_scan_done_this_session', true, 600);
            
            if ($result['registered_count'] > 0) {
                set_transient('zizi_ftp_scan_result', [
                    'type' => 'success',
                    'message' => sprintf('Auto-scan: Registered %d new FTP images in Media Library.', $result['registered_count'])
                ], 300); // Show for 5 minutes
            }
        }
    }
    
    /**
     * Schedule periodic FTP scanning via WP Cron (REMOVED - now using auto-scan on media library access)
     */
    public function schedule_ftp_scan()
    {
        // Remove any existing scheduled events - we don't need them anymore
        wp_clear_scheduled_hook('zizi_ftp_image_scan');
    }
    
    /**
     * Scan uploads directory and register FTP uploaded images
     */
    public function scan_and_register_ftp_images()
    {
        $upload_dir = wp_upload_dir();
        $uploads_path = $upload_dir['basedir'];
        
        $registered_count = 0;
        $scanned_directories = [];
        
        // Scan recent directories (last 3 months)
        $current_year = date('Y');
        $current_month = date('m');
        
        for ($i = 0; $i < 3; $i++) {
            $year = $current_year;
            $month = $current_month - $i;
            
            if ($month <= 0) {
                $year--;
                $month += 12;
            }
            
            $month = str_pad($month, 2, '0', STR_PAD_LEFT);
            $dir_path = $uploads_path . "/$year/$month";
            
            if (is_dir($dir_path)) {
                $scanned_directories[] = "$year/$month";
                $registered_count += $this->scan_directory($dir_path, "$year/$month");
            }
        }
        
        // Log the scan result
        error_log("ZiZi Cache FTP Scan: Registered $registered_count images from directories: " . implode(', ', $scanned_directories));
        
        return [
            'registered_count' => $registered_count,
            'scanned_directories' => $scanned_directories
        ];
    }
    
    /**
     * Scan specific directory for unregistered images
     */
    private function scan_directory($dir_path, $relative_dir)
    {
        $registered_count = 0;
        
        if (!is_dir($dir_path)) {
            return 0;
        }
        
        $files = scandir($dir_path);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            
            $full_path = $dir_path . '/' . $file;
            if (!is_file($full_path)) continue;
            
            // Check if it's an image
            $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            if (!in_array($extension, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff'])) continue;
            
            // Skip converted files (they should not be registered as separate attachments)
            $filename_without_ext = pathinfo($file, PATHINFO_FILENAME);
            $original_extensions = ['jpg', 'jpeg', 'png', 'gif'];
            $is_converted_file = false;
            
            foreach ($original_extensions as $orig_ext) {
                if (file_exists($dir_path . '/' . $filename_without_ext . '.' . $orig_ext)) {
                    $is_converted_file = true;
                    break;
                }
            }
            
            if ($is_converted_file && in_array($extension, ['webp', 'avif'])) {
                continue; // Skip converted files
            }
            
            // Try to register the image
            if ($this->register_image_in_wp($full_path, $relative_dir . '/' . $file)) {
                $registered_count++;
            }
        }
        
        return $registered_count;
    }
    
    /**
     * Register image in WordPress database
     */
    private function register_image_in_wp($file_path, $relative_path)
    {
        if (!file_exists($file_path)) {
            return false;
        }
        
        // Normalize path separators
        $relative_path = str_replace('\\', '/', $relative_path);
        
        // Check if already exists
        global $wpdb;
        $existing = $wpdb->get_var($wpdb->prepare("
            SELECT post_id FROM {$wpdb->postmeta} 
            WHERE meta_key = '_wp_attached_file' 
            AND meta_value = %s
        ", $relative_path));
        
        if ($existing) {
            return false; // Already registered
        }
        
        // Get file info
        $filetype = wp_check_filetype($file_path);
        if (!$filetype['type']) {
            return false; // Unsupported file type
        }
        
        $filename = basename($file_path);
        $upload_dir = wp_upload_dir();
        
        // Create attachment
        $attachment = array(
            'guid'           => $upload_dir['baseurl'] . '/' . $relative_path,
            'post_mime_type' => $filetype['type'],
            'post_title'     => sanitize_file_name(pathinfo($filename, PATHINFO_FILENAME)),
            'post_content'   => '',
            'post_status'    => 'inherit'
        );
        
        // Insert the attachment
        $attach_id = wp_insert_attachment($attachment, $file_path);
        
        if (is_wp_error($attach_id)) {
            return false;
        }
        
        // Update attachment metadata
        add_post_meta($attach_id, '_wp_attached_file', $relative_path);
        
        // Generate metadata
        if (function_exists('wp_generate_attachment_metadata')) {
            require_once(ABSPATH . 'wp-admin/includes/image.php');
            $metadata = wp_generate_attachment_metadata($attach_id, $file_path);
            wp_update_attachment_metadata($attach_id, $metadata);
        }
        
        return $attach_id;
    }
    
    /**
     * AJAX handler for manual FTP scan
     */
    public function ajax_scan_ftp_images()
    {
        check_ajax_referer('zizi_ftp_scan', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $result = $this->scan_and_register_ftp_images();
        
        wp_send_json_success([
            'message' => sprintf('Scan completed. Registered %d new images.', $result['registered_count']),
            'registered_count' => $result['registered_count'],
            'scanned_directories' => $result['scanned_directories']
        ]);
    }
    
    /**
     * Admin notices
     */
    public function admin_notices()
    {
        // Show auto-scan results if available
        $scan_result = get_transient('zizi_ftp_scan_result');
        if ($scan_result && is_array($scan_result)) {
            $class = $scan_result['type'] === 'success' ? 'notice-success' : 'notice-info';
            echo '<div class="notice ' . $class . ' is-dismissible"><p>' . esc_html($scan_result['message']) . '</p></div>';
            delete_transient('zizi_ftp_scan_result');
        }
    }
    
    /**
     * Get manual scan URL
     */
    public static function get_manual_scan_url()
    {
        return wp_nonce_url(admin_url('admin.php?page=zizi-cache-settings&tab=image-converter&zizi_scan_ftp=1'), 'zizi_scan_ftp');
    }
}

// Initialize the FTP registration system
add_action('init', ['ZiziCache\\FTPImageRegistration', 'init']);