<?php
namespace ZiziCache;

/**
 * Class DatabaseStatus
 *
 * Provides methods to retrieve database status and optimization information.
 *
 * @package ZiziCache
 */
class DatabaseStatus
{
    /**
     * Initializes the DatabaseStatus class.
     *
     * This method is required for compatibility with other plugin classes,
     * but does not contain any initialization logic in this class.
     *
     * @return void
     */
    public static function init(): void
    {
        // No initialization required; method present for compatibility
    }
    
    /**
     * Returns database status and optimization information.
     *
     * @return array{
     *     status: 'ok',
     *     data: array{
     *         db_version: string,
     *         db_server: string,
     *         wp_version: string,
     *         db_size_mb: float,
     *         cleanable_items: array<string, int>,
     *         actionscheduler: array|null,
     *         clean_interval: string,
     *         optimization_active: bool
     *     }
     * }
     */
    public static function get_status()
    {
        global $wpdb;
        $config = SysConfig::$config;

        // 1. Database version
        $db_version = $wpdb->db_version();
        $db_server = $wpdb->db_server_info();
        $wp_version = get_bloginfo('version');

        // 2. Total database size in MB
        $db_size = $wpdb->get_var(
            "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) FROM information_schema.TABLES WHERE table_schema = '{$wpdb->dbname}'"
        );

        // 3. Cleanable items counts
        $items = [
            'revisions' => $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'revision'"),
            'auto_drafts' => $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'auto-draft'"),
            'trash_posts' => $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'trash'"),
            'spam_comments' => $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->comments WHERE comment_approved = 'spam'"),
            'trash_comments' => $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->comments WHERE comment_approved = 'trash'"),
            'transients' => $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->options WHERE option_name LIKE '%_transient_%'")
        ];

        // 4. Action Scheduler tables status and size
        $as_status = class_exists('ZiziCache\\ActionSchedulerCleanup') ? ActionSchedulerCleanup::get_status_info() : null;

        // 5. Database cleaning interval
        $interval = isset($config['db_schedule_clean']) ? $config['db_schedule_clean'] : 'never';
        // Map interval codes to English labels
        $interval_map = ['never' => 'Never', 'daily' => 'Daily', 'weekly' => 'Weekly'];
        $interval_label = isset($interval_map[$interval]) ? $interval_map[$interval] : $interval;

        // 6. Is optimization active?
        $optimization_active = !empty($config['db_optimize_tables']);

        return [
            'status' => 'ok',
            'data' => [
                'db_version' => $db_version,
                'db_server' => $db_server,
                'wp_version' => $wp_version,
                'db_size_mb' => $db_size,
                'cleanable_items' => $items,
                'actionscheduler' => $as_status,
                'clean_interval' => $interval_label,
                'optimization_active' => $optimization_active,
            ]
        ];
    }
}