<?php
namespace ZiziCache;

/**
 * Class for monitoring the database and its growth
 *
 * Provides functions for monitoring database growth and detecting abnormal situations
 * with the option of automatic administrator notifications.
 * 
 * Main functions:
 * 1. Monitoring database growth and detecting anomalies
 * 2. Sending email alerts for abnormal table growth
 * 3. Adaptive adjustment of database cleanup schedule based on growth rate
 * 4. Calculating growth statistics for display in the admin
 *
 * Note: The class uses a centralized logging system via CacheSys::writeLog()
 * to ensure consistent format and log rotation.
 *
 * @package ZiziCache
 */
class DatabaseMonitor
{
    /**
     * Initialize the DatabaseMonitor class
     *
     * @return void
     */
    public static function init(): void
    {
        // Register CRON event for regular database check
        add_action('zizi_cache_db_monitor', [__CLASS__, 'analyze_database_growth']);
        
        // Set up regular check on initialization
        add_action('init', [__CLASS__, 'setup_monitoring']);
        
        // Register filter for email alerts
        add_filter('zizi_cache_db_alert_recipients', [__CLASS__, 'get_alert_recipients'], 10, 1);
    }
    
    /**
     * Set up regular monitoring
     *
     * @return void
     */
    public static function setup_monitoring(): void
    {
        // Check if monitoring is enabled in config
        $config = SysConfig::$config;
        $monitoring_enabled = isset($config['db_growth_monitoring']) ? (bool)$config['db_growth_monitoring'] : true;
        
        if (!$monitoring_enabled) {
            // If monitoring is disabled, remove CRON event
            wp_clear_scheduled_hook('zizi_cache_db_monitor');
            return;
        }
        
        // Set up daily check if not already scheduled
        if (!wp_next_scheduled('zizi_cache_db_monitor')) {
            wp_schedule_event(time(), 'daily', 'zizi_cache_db_monitor');
        }
    }
    
    /**
     * Analyze database growth and detect abnormal situations
     *
     * @return array Analysis result
     */
    public static function analyze_database_growth(): array
    {
        // Načtení předchozích velikostí tabulek
        $prev_sizes = get_option('zizi_cache_db_table_sizes', []);
        $current_time = time();
        
        global $wpdb;
        // Získání aktuálních velikostí tabulek
        $tables = $wpdb->get_results("
            SELECT 
                TABLE_NAME as table_name, 
                TABLE_ROWS as table_rows,
                DATA_LENGTH as data_length, 
                INDEX_LENGTH as index_length,
                ROUND((DATA_LENGTH + INDEX_LENGTH) / 1024 / 1024, 2) as size_mb
            FROM information_schema.TABLES
            WHERE TABLE_SCHEMA = '" . DB_NAME . "'
        ");
        
        $current_sizes = [];
        $alerts = [];
        $alert_threshold = 30; // 30% increase as default alert threshold
        $min_size_for_alert = 5; // Minimum size for monitoring (MB)
        
        foreach ($tables as $table) {
            $table_name = $table->table_name;
            $current_sizes[$table_name] = [
                'size_mb' => $table->size_mb,
                'rows' => $table->table_rows,
                'timestamp' => $current_time
            ];
            
            // Check for size increase
            if (isset($prev_sizes[$table_name])) {
                $prev = $prev_sizes[$table_name];
                $growth_percent = 0;
                $days_diff = max(1, ($current_time - $prev['timestamp']) / DAY_IN_SECONDS);
                
                if ($prev['size_mb'] > 0) {
                    // Calculate percentage growth normalized per day
                    $growth_percent = (($table->size_mb - $prev['size_mb']) / $prev['size_mb']) * 100 / $days_diff;
                }
                
                // If the table grew significantly and is larger than the minimum size
                if ($growth_percent > $alert_threshold && $table->size_mb > $min_size_for_alert) {
                    $alerts[] = [
                        'table' => $table_name,
                        'previous_size' => $prev['size_mb'],
                        'current_size' => $table->size_mb,
                        'growth_mb' => round($table->size_mb - $prev['size_mb'], 2),
                        'growth_percent' => round($growth_percent, 2),
                        'time_period' => human_time_diff($prev['timestamp'], $current_time)
                    ];
                }
            }
        }
        
        // Calculate total database size
        $total_size = 0;
        foreach ($current_sizes as $data) {
            $total_size += $data['size_mb'];
        }
        
        // Save current sizes for next check
        update_option('zizi_cache_db_table_sizes', $current_sizes);
        
        // Calculate change in total database size
        $prev_total = get_option('zizi_cache_db_total_size', 0);
        $size_change = $total_size - $prev_total;
        update_option('zizi_cache_db_total_size', $total_size);
        
        // Save size history (last 30 records)
        $size_history = get_option('zizi_cache_db_size_history', []);
        $size_history[] = [
            'size' => $total_size,
            'timestamp' => $current_time,
            'change' => $size_change
        ];
          if (count($size_history) > 30) {
            array_shift($size_history);
        }
        
        update_option('zizi_cache_db_size_history', $size_history);
        
        // Send alert if issues were found
        if (!empty($alerts)) {
            // Save alerts for later use in API
            update_option('zizi_cache_db_alerts', $alerts);
            
            self::send_growth_alerts($alerts, $total_size);
        } else {
            // Clear existing alerts because there are none
            delete_option('zizi_cache_db_alerts');
            
            // Log regular check without issues
            self::log_database_event(
                'growth_analysis', 
                'Provedena běžná kontrola růstu databáze', 
                [
                    'total_size_mb' => round($total_size, 2),
                    'size_change_mb' => round($size_change, 2),
                    'tables_count' => count($tables)
                ]
            );
        }
        
        return [
            'analysis_time' => current_time('mysql'),
            'total_size_mb' => $total_size,
            'size_change_mb' => $size_change,
            'alerts' => $alerts,
            'tables_count' => count($tables)
        ];
    }
    
    /**
     * Send alerts for abnormal table growth
     *
     * @param array $alerts Alerts for rapidly growing tables
     * @param float $total_size Total database size in MB
     * @return void
     */
    private static function send_growth_alerts(array $alerts, float $total_size): void
    {
        // Get emails for alerts
        $recipients = apply_filters('zizi_cache_db_alert_recipients', [get_option('admin_email')]);
        
        // Create email subject
        $subject = sprintf(
            '[%s] Upozornění na rychlý růst databáze',
            wp_specialchars_decode(get_option('blogname'), ENT_QUOTES)
        );
        
        // Create message
        $message = "ZiziCache: Unusual database table growth\n\n";
        $message .= sprintf("Total database size: %.2f MB\n\n", $total_size);
        $message .= "The following tables show abnormal growth:\n\n";
        
        foreach ($alerts as $alert) {
            $message .= sprintf(
                "Table: %s\nSize: %.2f MB → %.2f MB (increase %.2f MB)\nIncrease: %.2f%%\nPeriod: %s\n\n",
                $alert['table'],
                $alert['previous_size'],
                $alert['current_size'],
                $alert['growth_mb'],
                $alert['growth_percent'],
                $alert['time_period']
            );
        }
        
        $message .= "\nThis alert was generated automatically by the ZiziCache plugin.\n";
        $message .= "To resolve the issue, we recommend checking the table ";
        $message .= "and possibly cleaning the database in the WordPress admin.\n";
        $message .= sprintf("\nDatabase management link: %s\n", admin_url('admin.php?page=zizi-cache#/database'));
        
        // Odeslání e-mailu
        foreach ($recipients as $recipient) {
            wp_mail(
                $recipient,
                $subject,
                $message
            );
        }
          // Log to centralized log using CacheSys::writeLog
        CacheSys::writeLog('[ZiziCache Database Growth Alert] ' . json_encode($alerts, JSON_UNESCAPED_UNICODE));
        
        // Action hook for integration with other systems
        do_action('zizi_cache_db_growth_alert', $alerts, $total_size);
    }
    
    /**
     * Get list of recipients for alerts
     *
     * @param array $default Default list of recipients
     * @return array List of emails for alerts
     */
    public static function get_alert_recipients(array $default): array
    {
        // Check if a custom recipient list is set in config
        $config = SysConfig::$config;
        if (!empty($config['db_growth_alert_emails'])) {
            $emails = explode(',', $config['db_growth_alert_emails']);
            $valid_emails = [];
            
            foreach ($emails as $email) {
                $email = trim($email);
                if (is_email($email)) {
                    $valid_emails[] = $email;
                }
            }
            
            if (!empty($valid_emails)) {
                return $valid_emails;
            }
        }
        
        return $default;
    }
    
    /**
     * Calculate the average database growth rate for the last period
     *
     * @param int $days Number of days for analysis (default 7)
     * @return float Average growth rate in % per day
     */    public static function calculate_growth_rate(int $days = 7): float
    {
        // Get database size history
        $history = get_option('zizi_cache_db_size_history', []);
        $entries = count($history);
        
        // Force refresh current database size to catch new posts
        global $wpdb;
        $current_size = $wpdb->get_var(
            "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) FROM information_schema.TABLES WHERE table_schema = '{$wpdb->dbname}'"
        );

        // Store current state for immediate use
        $current_time = time();
        $current = end($history);
        
        // If we have less than 2 entries and it's not a fresh install, force a new measurement
        if ($entries < 2 && $current_size > 0) {
            // Add a new measurement to history
            $history[] = [
                'size' => (float)$current_size,
                'timestamp' => $current_time,
                'change' => isset($current['size']) ? (float)$current_size - (float)$current['size'] : 0
            ];
            
            // Update the history option
            update_option('zizi_cache_db_size_history', $history);
            
            // Reload history after adding new entry
            $history = get_option('zizi_cache_db_size_history', []);
            $entries = count($history);
        }
        
        if ($entries < 2) {
            return 0; // Not enough data to calculate
        }
        
        // Sort history by timestamp to ensure correct order
        usort($history, function($a, $b) {
            return $a['timestamp'] - $b['timestamp'];
        });
        
        // Limit number of days to available data
        $days = min($days, $entries - 1);
        
        // Get most recent entry
        $current = end($history);
        $current_size = (float)$current['size'];
        $current_time = (int)$current['timestamp'];
        
        // Find entry closest to 'days' days ago
        $old = reset($history); // Default to oldest
        foreach ($history as $entry) {
            $days_old = ($current_time - (int)$entry['timestamp']) / DAY_IN_SECONDS;
            if ($days_old >= $days) {
                $old = $entry;
                break;
            }
        }
        
        $old_size = (float)$old['size'];
        $old_time = (int)$old['timestamp'];
        
        // Calculate number of days between records
        $days_diff = ($current_time - $old_time) / DAY_IN_SECONDS;
        
        // Debug information
        error_log(sprintf(
            '[ZiziCache] Growth rate calculation: Current: %.2f MB, Old: %.2f MB, Days diff: %.2f, Growth: %.2f%%',
            $current_size,
            $old_size,
            $days_diff,
            $old_size > 0 && $days_diff > 0 ? (($current_size - $old_size) / $old_size) * 100 : 0
        ));
        
        // Calculate growth - simplified for better display
        if ($old_size > 0 && $days_diff > 0) {
            // Weekly growth rate (normalize to 7 days)
            $growth_rate = (($current_size - $old_size) / $old_size) * 100 * (7 / $days_diff);
            return round($growth_rate, 2);
        }
        
        return 0;
    }
    
    /**
     * Adaptively adjust the cleanup schedule based on database growth rate
     *
     * @return array Information about the schedule adjustment
     */
    public static function adjust_cleanup_schedule(): array
    {
        $growth_rate = self::calculate_growth_rate();
        
        // Default schedule
        $config = SysConfig::$config;
        $schedule = $config['db_schedule_clean'] ?? 'weekly';
        
        // Adjust schedule based on growth rate
        if ($growth_rate > 10) { // Rychlý růst (10% za den)
            $new_schedule = 'daily';
        } elseif ($growth_rate > 5) { // Střední růst (5-10% za den)
            $new_schedule = 'twice_daily';
        } elseif ($growth_rate < 1) { // Pomalý růst (méně než 1% za den)
            $new_schedule = 'weekly';
        } else { // Normální růst (1-5% za den)
            $new_schedule = 'daily';
        }
        
        // If schedule needs to be changed and automatic adjustment is enabled
        if ($new_schedule !== $schedule && !empty($config['db_adaptive_cleaning'])) {
            $config_update = ['db_schedule_clean' => $new_schedule];
            SysConfig::update_config($config_update);
              CacheSys::writeLog(sprintf(
                '[ZiziCache DB Schedule] Adaptively changed cleanup schedule from %s to %s, growth rate: %s%%',
                $schedule,
                $new_schedule,
                $growth_rate
            ));
        }
        
        return [
            'previous_schedule' => $schedule,
            'recommended_schedule' => $new_schedule,
            'growth_rate' => $growth_rate,
            'adaptive_enabled' => !empty($config['db_adaptive_cleaning']),
            'schedule_changed' => ($new_schedule !== $schedule && !empty($config['db_adaptive_cleaning']))
        ];
    }
    
    /**
     * Log database events to the main log
     *
     * @param string $action Type of action
     * @param string $message Message to log
     * @param array $context Additional event context (optional)
     * @return void
     */
    public static function log_database_event(string $action, string $message, array $context = []): void
    {
        // Prepare context for logging
        $formatted_context = !empty($context) ? json_encode($context, JSON_UNESCAPED_UNICODE) : '{}';
        
        // Write to central log using CacheSys::writeLog with rotation
        CacheSys::writeLog(sprintf('[ZiziCache DB Event] Action: %s, Message: %s, Context: %s', 
            $action, 
            $message,
            $formatted_context
        ));
        
        // Call action for further integration
        do_action('zizi_cache_db_event_logged', $action, $message, $context);
    }
}
