<?php
/**
 * Database management class for ZiziCache
 * 
 * Handles database cleanup, optimization, and maintenance operations.
 * Implements batch processing for large datasets to prevent server overload.
 * Provides utilities for safe database maintenance and performance optimization.
 *
 * @package ZiziCache
 * @since 1.0.0
 */
namespace ZiziCache;

class Database
{
  /**
   * Initialize database cleanup hooks and schedules
   * 
   * Registers WordPress actions for scheduled and manual database cleanup tasks.
   * Sets up the core functionality for database maintenance operations.
   *
   * @return void
   */
  public static function init()
  {
    add_action('zizi_cache_clean_database', [__CLASS__, 'clean']);
    add_action('init', [__CLASS__, 'setup_scheduled_clean']);
  }

  /**
   * Setup scheduled cleanup events
   * 
   * Schedules or clears the database cleanup cron event based on configuration.
   * Creates WordPress scheduled events according to the defined cleaning frequency.
   *
   * @return void
   */
  public static function setup_scheduled_clean()
  {
    $schedule = SysConfig::$config['db_schedule_clean'];
    $action_name = 'zizi_cache_clean_database';

    if ($schedule == 'never') {
      wp_clear_scheduled_hook($action_name);
      return;
    }

    if (!wp_next_scheduled($action_name) || wp_get_schedule($action_name) != $schedule) {
      wp_clear_scheduled_hook($action_name);
      wp_schedule_event(time(), $schedule, $action_name);
    }
  }

  /**
   * Main cleanup method that handles database optimization and cleanup
   * 
   * Cleans up post revisions, auto-drafts, trashed posts, spam/trashed comments,
   * expired transients, and optimizes tables. Accepts an optional array of 
   * cleanup options to override default configuration.
   * 
   * @param array|null $options Optional array of cleanup options
   * @return array Results of the cleanup operation
   */
  public static function clean($options = null)
  {
    global $wpdb;

    $config = SysConfig::$config;

    // Manual cleanup flags: override config with user options
    $known = [
        'db_post_revisions','db_post_auto_drafts','db_post_trashed',
        'db_comments_spam','db_comments_trashed',
        'db_transients_expired','db_transients_all',
        'db_optimize_tables'
    ];
    $flags = [];
    if (is_array($options)) {
        foreach ($options as $opt) {
            if (in_array($opt, $known, true)) {
                $flags[$opt] = true;
            }
        }
    } else {
        foreach ($known as $opt) {
            if (!empty($config[$opt])) {
                $flags[$opt] = true;
            }
        }
    }

    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache DB: clean options: ' . print_r($options, true));
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache DB: clean flags: ' . print_r($flags, true));    if (!empty($flags['db_post_revisions'])) {
      // Use direct database query instead of N+1 approach for better performance
      $deleted = $wpdb->query("DELETE FROM $wpdb->posts WHERE post_type = 'revision'");
      if ($deleted) {
        \ZiziCache\CacheSys::writeLog("[INFO] ZiziCache DB: Deleted {$deleted} post revisions using optimized query");
      }
    }    if (!empty($flags['db_post_auto_drafts'])) {
      // Use direct database query instead of N+1 approach for better performance
      $deleted = $wpdb->query("DELETE FROM $wpdb->posts WHERE post_status = 'auto-draft'");
      if ($deleted) {
        \ZiziCache\CacheSys::writeLog("[INFO] ZiziCache DB: Deleted {$deleted} auto-draft posts using optimized query");
      }
    }    if (!empty($flags['db_post_trashed'])) {
      // Use direct database query instead of N+1 approach for better performance
      $deleted = $wpdb->query("DELETE FROM $wpdb->posts WHERE post_status = 'trash'");
      if ($deleted) {
        \ZiziCache\CacheSys::writeLog("[INFO] ZiziCache DB: Deleted {$deleted} trashed posts using optimized query");
      }
    }    if (!empty($flags['db_comments_spam'])) {
      // Use direct database query instead of N+1 approach for better performance
      $deleted = $wpdb->query("DELETE FROM $wpdb->comments WHERE comment_approved = 'spam'");
      if ($deleted) {
        \ZiziCache\CacheSys::writeLog("[INFO] ZiziCache DB: Deleted {$deleted} spam comments using optimized query");
      }
    }    if (!empty($flags['db_comments_trashed'])) {
      // Use direct database query instead of N+1 approach for better performance  
      $deleted = $wpdb->query("DELETE FROM $wpdb->comments WHERE comment_approved IN ('trash', 'post-trashed')");
      if ($deleted) {
        \ZiziCache\CacheSys::writeLog("[INFO] ZiziCache DB: Deleted {$deleted} trashed comments using optimized query");
      }
    }

    if (!empty($flags['db_transients_expired'])) {
      $time = isset($_SERVER['REQUEST_TIME']) ? (int) $_SERVER['REQUEST_TIME'] : time();
      $query = $wpdb->get_col(
        $wpdb->prepare(
          "SELECT option_name FROM $wpdb->options WHERE option_name LIKE %s AND option_value < %d",
          $wpdb->esc_like('_transient_timeout') . '%',
          $time
        )
      );
      if ($query) {
        foreach ($query as $transient) {
          $key = str_replace('_transient_timeout_', '', $transient);
          delete_transient($key);
        }
      }
    }

    if (!empty($flags['db_transients_all'])) {
      $query = $wpdb->get_col(
        $wpdb->prepare(
          "SELECT option_name FROM $wpdb->options WHERE option_name LIKE %s OR option_name LIKE %s",
          $wpdb->esc_like('_transient_') . '%',
          $wpdb->esc_like('_site_transient_') . '%'
        )
      );
      if ($query) {
        foreach ($query as $transient) {
          if (strpos($transient, '_site_transient_') !== false) {
            delete_site_transient(str_replace('_site_transient_', '', $transient));
          } else {
            delete_transient(str_replace('_transient_', '', $transient));
          }
        }
      }
    }

    if (!empty($flags['db_optimize_tables'])) {
      $query = $wpdb->get_results(
        "SELECT table_name, data_free FROM information_schema.tables WHERE table_schema = '" .
          DB_NAME .
          "' and Engine <> 'InnoDB' and data_free > 0"
      );
      if ($query) {
        foreach ($query as $table) {
          $wpdb->query("OPTIMIZE TABLE $table->table_name");
        }
      }
    }

// Trigger action for ActionSchedulerCleanup
do_action('zizi_cache_database_after_options'); 
    
// Invalidate cache counts after cleanup to ensure data freshness
delete_transient('zizi_cache_db_counts');
// Log cleanup for database growth monitoring
// Use CacheSys::writeLog instead of DatabaseMonitor::log_database_event
CacheSys::writeLog('[ZiziCache DB Event] Action: cleanup, Source: Database.php, Details: {"message":"Database cleanup performed"}');
  }

    /**
     * Get counts for manual cleanup options, with optional caching.
     * @param array|null $options
     * @return array
     */    public static function get_counts($options = null)
    {
        global $wpdb;
        $known = [
            'db_post_revisions','db_post_auto_drafts','db_post_trashed',
            'db_comments_spam','db_comments_trashed',
            'db_transients_expired','db_transients_all',
            'db_optimize_tables'
        ];
        $flags = [];
        
        // If specific items are provided for calculation, use them
        if (is_array($options) && !empty($options)) {
            foreach ($options as $opt) {
                if (in_array($opt, $known, true)) {
                    $flags[$opt] = true;
                }
            }
        } 
        // Otherwise, count all items
        else {
            foreach ($known as $opt) {
                $flags[$opt] = true;
            }
        }
        
        // Try to use cache first if no specific items are specified
        $cache_key = 'zizi_cache_db_counts';
        $use_cache = is_null($options);
        
        if ($use_cache) {
            $cached = get_transient($cache_key);
            if ($cached !== false) {
                return $cached;
            }
        }
        $counts = [];
        if (!empty($flags['db_post_revisions'])) {
            $counts['db_post_revisions'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type='revision'");
        }
        if (!empty($flags['db_post_auto_drafts'])) {
            $counts['db_post_auto_drafts'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status='auto-draft'");
        }
        if (!empty($flags['db_post_trashed'])) {
            $counts['db_post_trashed'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status='trash'");
        }
        if (!empty($flags['db_comments_spam'])) {
            $counts['db_comments_spam'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved='spam'");
        }
        if (!empty($flags['db_comments_trashed'])) {
            $counts['db_comments_trashed'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved IN ('trash','post-trashed')");
        }
        if (!empty($flags['db_transients_expired'])) {
            $time = isset($_SERVER['REQUEST_TIME']) ? (int) $_SERVER['REQUEST_TIME'] : time();
            $counts['db_transients_expired'] = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE %s AND option_value < %d",
                    $wpdb->esc_like('_transient_timeout_') . '%',
                    $time
                )
            );
        }
        if (!empty($flags['db_transients_all'])) {
            $counts['db_transients_all'] = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                    $wpdb->esc_like('_transient_') . '%',
                    $wpdb->esc_like('_site_transient_') . '%'
                )
            );
        }
        if (!empty($flags['db_optimize_tables'])) {
            $counts['db_optimize_tables'] = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM information_schema.tables WHERE table_schema='" . DB_NAME . "' AND Engine<>'InnoDB' AND data_free>0"
            );
        }
        if (is_null($options)) {
            set_transient($cache_key, $counts, 5 * MINUTE_IN_SECONDS);
        }
        return $counts;
    }

  /**
   * Clean database in batches to prevent server overload
   * 
   * Processes database cleanup operations in manageable chunks with memory and time limits.
   * Handles various cleanup tasks like post revisions, spam comments, transients, etc.
   *
   * @param array|null $options Cleanup options
   * @param int $batch_size Number of items to process in each batch
   * @return array Statistics about the cleanup operation
   */
  public static function clean_in_batches($options = null, $batch_size = 500)
  {
    global $wpdb;
    $config = SysConfig::$config;
    
    // Start time measurement for limit control
    if (!defined('ZIZI_BATCH_START_TIME')) {
      define('ZIZI_BATCH_START_TIME', microtime(true));
    }
    
    // Manual cleanup mode: take options from parameters
    $known = [
        'db_post_revisions','db_post_auto_drafts','db_post_trashed',
        'db_comments_spam','db_comments_trashed',
        'db_transients_expired','db_transients_all',
        'db_optimize_tables'
    ];
    $flags = [];
    if (is_array($options)) {
        foreach ($options as $opt) {
            if (in_array($opt, $known, true)) {
                $flags[$opt] = true;
            }
        }
    } else {
        foreach ($known as $opt) {
            if (!empty($config[$opt])) {
                $flags[$opt] = true;
            }
        }
    }
    
    // Statistics for return
    $stats = [
      'processed' => 0,
      'skipped' => 0,
      'errors' => 0,
      'completed' => true,
      'details' => []
    ];
    
    // Processing of revisions
    if (!empty($flags['db_post_revisions'])) {
      try {
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'revision'");
        $stats['details']['db_post_revisions'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE post_type = 'revision' LIMIT %d OFFSET %d", 
              $batch_size, $offset)
          );
          
          if ($query) {
            foreach ($query as $id) {
              if (wp_delete_post_revision(intval($id)) instanceof \WP_Post) {
                $stats['processed']++;
                $stats['details']['db_post_revisions']['deleted']++;
              } else {
                $stats['errors']++;
              }
            }
          }
          
          // Check if time or memory limit is exceeded
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_post_revisions']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_post_revisions']['error'] = $e->getMessage();
      }
    }
    
    // Processing of auto-drafts
    if (!empty($flags['db_post_auto_drafts']) && $stats['completed']) {
      try {
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'auto-draft'");
        $stats['details']['db_post_auto_drafts'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE post_status = 'auto-draft' LIMIT %d OFFSET %d", 
              $batch_size, $offset)
          );
          
          if ($query) {
            foreach ($query as $id) {
              if (wp_delete_post(intval($id), true) instanceof \WP_Post) {
                $stats['processed']++;
                $stats['details']['db_post_auto_drafts']['deleted']++;
              } else {
                $stats['errors']++;
              }
            }
          }
          
          // Check limits
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_post_auto_drafts']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_post_auto_drafts']['error'] = $e->getMessage();
      }
    }
    
    // Processing of trashed posts
    if (!empty($flags['db_post_trashed']) && $stats['completed']) {
      try {
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_status = 'trash'");
        $stats['details']['db_post_trashed'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE post_status = 'trash' LIMIT %d OFFSET %d", 
              $batch_size, $offset)
          );
          
          if ($query) {
            foreach ($query as $id) {
              if (wp_delete_post($id, true) instanceof \WP_Post) {
                $stats['processed']++;
                $stats['details']['db_post_trashed']['deleted']++;
              } else {
                $stats['errors']++;
              }
            }
          }
          
          // Check limits
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_post_trashed']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_post_trashed']['error'] = $e->getMessage();
      }
    }
    
    // Processing of spam comments
    if (!empty($flags['db_comments_spam']) && $stats['completed']) {
      try {
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->comments WHERE comment_approved = 'spam'");
        $stats['details']['db_comments_spam'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare("SELECT comment_ID FROM $wpdb->comments WHERE comment_approved = 'spam' LIMIT %d OFFSET %d",
              $batch_size, $offset)
          );
          
          if ($query) {
            foreach ($query as $id) {
              if (wp_delete_comment(intval($id), true)) {
                $stats['processed']++;
                $stats['details']['db_comments_spam']['deleted']++;
              } else {
                $stats['errors']++;
              }
            }
          }
          
          // Check limits
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_comments_spam']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_comments_spam']['error'] = $e->getMessage();
      }
    }
    
    // Processing of trashed comments
    if (!empty($flags['db_comments_trashed']) && $stats['completed']) {
      try {
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->comments WHERE (comment_approved = 'trash' OR comment_approved = 'post-trashed')");
        $stats['details']['db_comments_trashed'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare("SELECT comment_ID FROM $wpdb->comments WHERE (comment_approved = 'trash' OR comment_approved = 'post-trashed') LIMIT %d OFFSET %d",
              $batch_size, $offset)
          );
          
          if ($query) {
            foreach ($query as $id) {
              if (wp_delete_comment(intval($id), true)) {
                $stats['processed']++;
                $stats['details']['db_comments_trashed']['deleted']++;
              } else {
                $stats['errors']++;
              }
            }
          }
          
          // Check limits
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_comments_trashed']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_comments_trashed']['error'] = $e->getMessage();
      }
    }
    
    // Processing of expired transients
    if (!empty($flags['db_transients_expired']) && $stats['completed']) {
      try {
        $time = isset($_SERVER['REQUEST_TIME']) ? (int) $_SERVER['REQUEST_TIME'] : time();
        $total = $wpdb->get_var(
          $wpdb->prepare(
            "SELECT COUNT(*) FROM $wpdb->options WHERE option_name LIKE %s AND option_value < %d",
            $wpdb->esc_like('_transient_timeout_') . '%',
            $time
          )
        );
        
        $stats['details']['db_transients_expired'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare(
              "SELECT option_name FROM $wpdb->options WHERE option_name LIKE %s AND option_value < %d LIMIT %d OFFSET %d",
              $wpdb->esc_like('_transient_timeout_') . '%',
              $time,
              $batch_size,
              $offset
            )
          );
          
          if ($query) {
            foreach ($query as $transient) {
              $key = str_replace('_transient_timeout_', '', $transient);
              if (delete_transient($key)) {
                $stats['processed']++;
                $stats['details']['db_transients_expired']['deleted']++;
              } else {
                $stats['skipped']++;
              }
            }
          }
          
          // Check limits
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_transients_expired']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_transients_expired']['error'] = $e->getMessage();
      }
    }
    
    // Processing of all transients
    if (!empty($flags['db_transients_all']) && $stats['completed']) {
      try {
        $total = $wpdb->get_var(
          $wpdb->prepare(
            "SELECT COUNT(*) FROM $wpdb->options WHERE option_name LIKE %s OR option_name LIKE %s",
            $wpdb->esc_like('_transient_') . '%',
            $wpdb->esc_like('_site_transient_') . '%'
          )
        );
        
        $stats['details']['db_transients_all'] = ['total' => (int)$total, 'deleted' => 0];
        
        for ($offset = 0; $offset < $total; $offset += $batch_size) {
          $query = $wpdb->get_col(
            $wpdb->prepare(
              "SELECT option_name FROM $wpdb->options WHERE option_name LIKE %s OR option_name LIKE %s LIMIT %d OFFSET %d",
              $wpdb->esc_like('_transient_') . '%',
              $wpdb->esc_like('_site_transient_') . '%',
              $batch_size,
              $offset
            )
          );
          
          if ($query) {
            foreach ($query as $transient) {
              if (strpos($transient, '_site_transient_') !== false) {
                $key = str_replace('_site_transient_', '', $transient);
                if (delete_site_transient($key)) {
                  $stats['processed']++;
                  $stats['details']['db_transients_all']['deleted']++;
                } else {
                  $stats['skipped']++;
                }
              } else {
                $key = str_replace('_transient_', '', $transient);
                if (delete_transient($key)) {
                  $stats['processed']++;
                  $stats['details']['db_transients_all']['deleted']++;
                } else {
                  $stats['skipped']++;
                }
              }
            }
          }
          
          // Check limits
          if (self::should_abort_batch_processing()) {
            $stats['completed'] = false;
            $stats['details']['db_transients_all']['message'] = 'Operation was interrupted due to system limits being exceeded.';
            break;
          }
        }
      } catch (\Exception $e) {
        $stats['errors']++;
        $stats['details']['db_transients_all']['error'] = $e->getMessage();
      }
    }
    
    // Optimization of tables (if previous actions were completed)
    if (!empty($flags['db_optimize_tables']) && $stats['completed']) {
      $optimize_stats = self::optimize_tables_safely();
      $stats['details']['db_optimize_tables'] = $optimize_stats;
      $stats['processed'] += count($optimize_stats['optimized']);
      $stats['errors'] += count($optimize_stats['failed']);
    }

    // Trigger action for ActionSchedulerCleanup
    do_action('zizi_cache_database_after_options'); 
    
    // Log operation to the log
    self::log_database_operation('clean_in_batches', $options, $stats);
    
    return $stats;
  }

  /**
   * Safely optimize database tables using transaction processing
   * 
   * Ensures table optimization without database lock risks by processing
   * tables one at a time with time limits. Handles both InnoDB and MyISAM tables.
   *
   * @param array|null $tables List of tables to optimize (null = all tables)
   * @return array Optimization statistics
   */
  public static function optimize_tables_safely($tables = null)
  {
    global $wpdb;

    // Initialize operation statistics
    $stats = [
      'optimized' => 0,   // Count of successfully optimized tables
      'skipped' => 0,    // Count of skipped tables
      'errors' => 0,     // Count of errors during optimization
      'saved_bytes' => 0, // Total bytes saved by optimization
      'details' => []     // Detailed information about each table
    ];

    // Get list of tables to optimize (skip InnoDB tables unless explicitly requested or fragmented)
    if ($tables === null) {
      $tables = $wpdb->get_results(
        "SELECT table_name, engine, data_free, (data_length + index_length) as total_size
         FROM information_schema.tables
         WHERE table_schema = '" . DB_NAME . "'
         AND (Engine <> 'InnoDB' OR data_free > 0)"
      );
    } else {
      // Get information about user-defined tables
      $tables_list = implode("','", array_map('esc_sql', $tables));
      $tables = $wpdb->get_results(
        "SELECT table_name, engine, data_free, (data_length + index_length) as total_size
         FROM information_schema.tables
         WHERE table_schema = '" . DB_NAME . "'
         AND table_name IN ('$tables_list')"
      );
    }

    // Check if there are any tables to optimize
    if (empty($tables)) {
      $stats['message'] = 'No tables found for optimization';
      return $stats;
    }

    // Process tables one by one for optimization
    foreach ($tables as $table) {
      $start_time = microtime(true);
      $table_name = $table->table_name;

      // Skip InnoDB tables without fragmentation
      if ($table->engine === 'InnoDB' && (int)$table->data_free === 0) {
        $stats['skipped']++;
        $stats['details'][$table_name] = [
          'status' => 'skipped',
          'reason' => 'InnoDB without fragmentation'
        ];
        continue;
      }

      // Skip very large tables to prevent timeouts
      $table_size_mb = $table->total_size / 1024 / 1024;
      if ($table_size_mb > 500) { // tables larger than 500 MB will be processed carefully
        $stats['skipped']++;
        $stats['details'][$table_name] = [
          'status' => 'skipped',
          'reason' => 'Table is too large for safe optimization',
          'size_mb' => round($table_size_mb, 2)
        ];
        continue;
      }

      try {
        // Start transaction for MyISAM tables (not needed for InnoDB)
        if ($table->engine !== 'InnoDB') {
          $wpdb->query('START TRANSACTION');
        }

        // Log optimization start
        self::log_database_operation('optimize_start', "Starting optimization of table {$table_name}");

        // Record table size before optimization
        $before_size = $wpdb->get_var("SELECT data_length + index_length FROM information_schema.TABLES WHERE table_name = '$table_name' AND table_schema = '" . DB_NAME . "'");

        // Execute table optimization
        $result = $wpdb->query("OPTIMIZE TABLE {$table_name}");

        if ($result === false) {
          throw new \Exception("Optimization of table {$table_name} failed");
        }

        // Record table size after optimization
        $after_size = $wpdb->get_var("SELECT data_length + index_length FROM information_schema.TABLES WHERE table_name = '$table_name' AND table_schema = '" . DB_NAME . "'");
        $saved_bytes = max(0, $before_size - $after_size);
        
        // Commit transaction for MyISAM tables
        if ($table->engine !== 'InnoDB') {
          $wpdb->query('COMMIT');
        }

        // Update statistics
        $stats['optimized']++;
        $stats['saved_bytes'] += $saved_bytes;
        $stats['details'][$table_name] = [
          'status' => 'success',
          'saved_bytes' => $saved_bytes,
          'time_ms' => round((microtime(true) - $start_time) * 1000, 2)
        ];

        // Log successful optimization
        self::log_database_operation('optimize_success', "Table {$table_name} optimization completed, saved " . size_format($saved_bytes));
      } catch (\Exception $e) {
        // Rollback transaction for MyISAM tables in case of error
        if ($table->engine !== 'InnoDB') {
          $wpdb->query('ROLLBACK');
        }

        // Update statistics
        $stats['errors']++;
        $stats['details'][$table_name] = [
          'status' => 'error',
          'error' => $e->getMessage()
        ];

        // Log optimization error
        self::log_database_operation('optimize_error', "Error optimizing table {$table_name}: " . $e->getMessage());
      }

      // Short pause between table optimizations
      usleep(100000); // 100ms pause
    }

    return $stats;
  }

  /**
   * Kontroluje, zda by mělo být dávkové zpracování přerušeno z důvodu překročení limitů
   *
   * @return bool TRUE, pokud by mělo být zpracování přerušeno
   * Check if batch processing should be aborted due to system limits
   * 
   * Monitors execution time and memory usage to prevent timeouts and memory exhaustion.
   * This helps maintain server stability during intensive database operations.
   * 
   * @return bool True if processing should be aborted to prevent server overload
   */
  public static function should_abort_batch_processing()
  {
    // Check if maximum execution time has been reached (default: 20 seconds)
    $max_execution_time = 20;
    if (defined('ZIZI_BATCH_MAX_EXECUTION_TIME')) {
      $max_execution_time = ZIZI_BATCH_MAX_EXECUTION_TIME;
    }
    
    // Calculate elapsed time since batch processing started
    $elapsed_time = microtime(true) - ZIZI_BATCH_START_TIME;
    if ($elapsed_time > $max_execution_time) {
      return true;
    }

    // Check if memory usage is approaching the PHP memory limit (80% threshold)
    $memory_limit = self::get_memory_limit_bytes();
    $current_memory = memory_get_usage(true);
    $memory_threshold = $memory_limit * 0.8;

    if ($current_memory > $memory_threshold) {
      return true;
    }

    return false;
  }

  /**
   * Get PHP memory limit in bytes
   * 
   * Retrieves the PHP memory limit and converts it to bytes for consistent comparison.
   * Handles different units (G, M, K) and falls back to bytes if no unit is specified.
   * 
   * @return int Memory limit in bytes
   */
  private static function get_memory_limit_bytes()
  {
    $memory_limit = ini_get('memory_limit');

    // Convert to bytes based on the unit (G, M, K)
    $unit = strtolower(substr($memory_limit, -1));
    $value = (int) $memory_limit;

    switch ($unit) {
      case 'g': return $value * 1024 * 1024 * 1024;
      case 'm': return $value * 1024 * 1024;
      case 'k': return $value * 1024;
      default: return $value;
    }
  }

  /**
   * Log database operations for diagnostics and monitoring
   * 
   * Stores operation details in WordPress options and optionally in the debug log.
   * Maintains a rolling log of the last 100 operations to prevent excessive database growth.
   * Also logs to the standard PHP error log when WP_DEBUG is enabled.
   *
   * @param string $operation Type of operation being logged (e.g., 'cleanup', 'optimize')
   * @param string $message Descriptive message about the operation
   * @return void
   */
  public static function log_database_operation($operation, $message)
  {
    // Retrieve existing log from WordPress options
    $log = get_option('zizi_cache_db_operations_log', []);

    // Limit log size to the last 100 entries
    if (count($log) > 100) {
      $log = array_slice($log, -100);
    }

    // Add new log entry
    $log[] = [
      'timestamp' => time(),
      'operation' => $operation,
      'message' => $message,
      'user_id' => get_current_user_id()
    ];

    // Save updated log
    update_option('zizi_cache_db_operations_log', $log, false);

    // Log to standard error log if debug mode is enabled
    if (defined('WP_DEBUG') && WP_DEBUG) {
      \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache DB: ' . $message);
    }
  }

  /**
   * Retrieve database operation logs
   * 
   * Fetches the most recent database operation logs with optional limit.
   * Logs are returned in reverse chronological order (newest first).
   *
   * @param int $limit Maximum number of log entries to return (default: 50)
   * @return array Array of log entries, each containing timestamp, operation, message, and user_id
   */
  public static function get_database_operation_logs($limit = 50)
  {
    // Retrieve logs from WordPress options
    $logs = get_option('zizi_cache_db_operations_log', []);
    
    // Reverse to show newest entries first
    $logs = array_reverse($logs);
    
    // Apply limit if specified (and there are more logs than the limit)
    if ($limit > 0 && count($logs) > $limit) {
      $logs = array_slice($logs, 0, $limit);
    }
    
    return $logs;
  }
  
  /**
   * Database index recommendations system
   * 
   * Analyzes database performance and recommends indexes for optimization.
   * Safe implementation that only suggests improvements without making changes.
   *
   * @return array Array of index recommendations with detailed analysis
   */
  public static function get_index_recommendations()
  {
    global $wpdb;
    
    $recommendations = [
      'analyzed_tables' => [],
      'recommended_indexes' => [],
      'performance_issues' => [],
      'analysis_timestamp' => current_time('mysql'),
      'summary' => [
        'total_recommendations' => 0,
        'high_priority' => 0,
        'medium_priority' => 0,
        'low_priority' => 0
      ]
    ];
    
    try {
      // Analyze WordPress core tables that are frequently queried
      $core_tables = [
        $wpdb->posts => 'posts',
        $wpdb->postmeta => 'postmeta', 
        $wpdb->comments => 'comments',
        $wpdb->commentmeta => 'commentmeta',
        $wpdb->options => 'options',
        $wpdb->usermeta => 'usermeta',
        $wpdb->users => 'users'
      ];
      
      foreach ($core_tables as $table_name => $table_type) {
        $table_analysis = self::analyze_table_indexes($table_name, $table_type);
        if (!empty($table_analysis)) {
          $recommendations['analyzed_tables'][$table_name] = $table_analysis;
          
          // Add recommendations from this table
          if (!empty($table_analysis['recommendations'])) {
            foreach ($table_analysis['recommendations'] as $rec) {
              $recommendations['recommended_indexes'][] = $rec;
              $recommendations['summary']['total_recommendations']++;
              
              // Count by priority
              switch ($rec['priority']) {
                case 'high':
                  $recommendations['summary']['high_priority']++;
                  break;
                case 'medium':
                  $recommendations['summary']['medium_priority']++;
                  break;
                case 'low':
                  $recommendations['summary']['low_priority']++;
                  break;
              }
            }
          }
        }
      }
      
      // Analyze slow queries from performance schema if available
      $slow_queries = self::analyze_slow_queries();
      if (!empty($slow_queries)) {
        $recommendations['performance_issues'] = $slow_queries;
      }
      
      // Log analysis completion
      self::log_database_operation('index_analysis', 
        sprintf('Index analysis completed. Found %d recommendations (%d high priority, %d medium priority, %d low priority)',
          $recommendations['summary']['total_recommendations'],
          $recommendations['summary']['high_priority'],
          $recommendations['summary']['medium_priority'],
          $recommendations['summary']['low_priority']
        )
      );
      
    } catch (\Exception $e) {
      $recommendations['error'] = $e->getMessage();
      self::log_database_operation('index_analysis_error', 'Index analysis failed: ' . $e->getMessage());
    }
    
    return $recommendations;
  }
  
  /**
   * Analyze indexes for a specific table
   * 
   * @param string $table_name Full table name
   * @param string $table_type Type of table (posts, comments, etc.)
   * @return array Analysis results with recommendations
   */
  private static function analyze_table_indexes($table_name, $table_type)
  {
    global $wpdb;
    
    $analysis = [
      'table_name' => $table_name,
      'table_type' => $table_type,
      'current_indexes' => [],
      'recommendations' => [],
      'table_stats' => []
    ];
    
    try {
      // Get current indexes
      $indexes = $wpdb->get_results("SHOW INDEX FROM `{$table_name}`");
      foreach ($indexes as $index) {
        $analysis['current_indexes'][] = [
          'name' => $index->Key_name,
          'column' => $index->Column_name,
          'unique' => $index->Non_unique == 0,
          'type' => $index->Index_type
        ];
      }
      
      // Get table statistics
      $stats = $wpdb->get_row("SHOW TABLE STATUS LIKE '{$table_name}'");
      if ($stats) {
        $analysis['table_stats'] = [
          'rows' => $stats->Rows,
          'avg_row_length' => $stats->Avg_row_length,
          'data_length' => $stats->Data_length,
          'index_length' => $stats->Index_length
        ];
      }
      
      // Generate recommendations based on table type
      switch ($table_type) {
        case 'posts':
          $analysis['recommendations'] = array_merge(
            $analysis['recommendations'],
            self::get_posts_table_recommendations($table_name, $analysis['current_indexes'])
          );
          break;
          
        case 'postmeta':
          $analysis['recommendations'] = array_merge(
            $analysis['recommendations'],
            self::get_postmeta_table_recommendations($table_name, $analysis['current_indexes'])
          );
          break;
          
        case 'comments':
          $analysis['recommendations'] = array_merge(
            $analysis['recommendations'],
            self::get_comments_table_recommendations($table_name, $analysis['current_indexes'])
          );
          break;
          
        case 'options':
          $analysis['recommendations'] = array_merge(
            $analysis['recommendations'],
            self::get_options_table_recommendations($table_name, $analysis['current_indexes'])
          );
          break;
      }
      
    } catch (\Exception $e) {
      $analysis['error'] = $e->getMessage();
    }
    
    return $analysis;
  }
  
  /**
   * Get index recommendations for wp_posts table
   * 
   * @param string $table_name
   * @param array $current_indexes
   * @return array Recommendations
   */
  private static function get_posts_table_recommendations($table_name, $current_indexes)
  {
    $recommendations = [];
    $existing_index_columns = array_column($current_indexes, 'column');
    
    // Check for common query patterns and recommend indexes
    $recommended_indexes = [
      [
        'columns' => ['post_status', 'post_type', 'post_date'],
        'name' => 'idx_status_type_date',
        'priority' => 'high',
        'reason' => 'Optimizes queries filtering by status and type with date ordering (very common in WordPress)',
        'estimated_benefit' => 'High - used in most frontend queries'
      ],
      [
        'columns' => ['post_author', 'post_status'],
        'name' => 'idx_author_status',
        'priority' => 'medium',
        'reason' => 'Optimizes author archive pages and admin post listings',
        'estimated_benefit' => 'Medium - used in author pages and admin'
      ],
      [
        'columns' => ['post_parent', 'post_type'],
        'name' => 'idx_parent_type',
        'priority' => 'medium',
        'reason' => 'Optimizes queries for child posts (attachments, revisions)',
        'estimated_benefit' => 'Medium - used for hierarchical content'
      ],
      [
        'columns' => ['post_name', 'post_status'],
        'name' => 'idx_name_status',
        'priority' => 'low',
        'reason' => 'Optimizes permalink resolution for published posts',
        'estimated_benefit' => 'Low - only for specific URL patterns'
      ]
    ];
    
    foreach ($recommended_indexes as $rec_index) {
      // Check if this index already exists (by checking if all columns are covered)
      $exists = self::index_covers_columns($current_indexes, $rec_index['columns']);
      
      if (!$exists) {
        $recommendations[] = [
          'table' => $table_name,
          'index_name' => $rec_index['name'],
          'columns' => $rec_index['columns'],
          'priority' => $rec_index['priority'],
          'reason' => $rec_index['reason'],
          'estimated_benefit' => $rec_index['estimated_benefit'],
          'sql' => "ALTER TABLE `{$table_name}` ADD INDEX `{$rec_index['name']}` (`" . implode('`, `', $rec_index['columns']) . "`)"
        ];
      }
    }
    
    return $recommendations;
  }
  
  /**
   * Get index recommendations for wp_postmeta table
   * 
   * @param string $table_name
   * @param array $current_indexes
   * @return array Recommendations
   */
  private static function get_postmeta_table_recommendations($table_name, $current_indexes)
  {
    $recommendations = [];
    
    $recommended_indexes = [
      [
        'columns' => ['meta_key', 'meta_value'],
        'name' => 'idx_key_value',
        'priority' => 'high',
        'reason' => 'Optimizes meta queries with specific key-value lookups (very common)',
        'estimated_benefit' => 'High - used in most meta queries'
      ],
      [
        'columns' => ['post_id', 'meta_key'],
        'name' => 'idx_post_key',
        'priority' => 'medium',
        'reason' => 'Optimizes queries for all meta of a specific post',
        'estimated_benefit' => 'Medium - used in post meta retrieval'
      ]
    ];
    
    foreach ($recommended_indexes as $rec_index) {
      $exists = self::index_covers_columns($current_indexes, $rec_index['columns']);
      
      if (!$exists) {
        $recommendations[] = [
          'table' => $table_name,
          'index_name' => $rec_index['name'],
          'columns' => $rec_index['columns'],
          'priority' => $rec_index['priority'],
          'reason' => $rec_index['reason'],
          'estimated_benefit' => $rec_index['estimated_benefit'],
          'sql' => "ALTER TABLE `{$table_name}` ADD INDEX `{$rec_index['name']}` (`" . implode('`, `', $rec_index['columns']) . "`)"
        ];
      }
    }
    
    return $recommendations;
  }
  
  /**
   * Get index recommendations for wp_comments table
   * 
   * @param string $table_name
   * @param array $current_indexes
   * @return array Recommendations
   */
  private static function get_comments_table_recommendations($table_name, $current_indexes)
  {
    $recommendations = [];
    
    $recommended_indexes = [
      [
        'columns' => ['comment_approved', 'comment_date_gmt'],
        'name' => 'idx_approved_date',
        'priority' => 'high',
        'reason' => 'Optimizes queries for approved comments with date ordering',
        'estimated_benefit' => 'High - used in comment listings'
      ],
      [
        'columns' => ['comment_post_ID', 'comment_approved'],
        'name' => 'idx_post_approved',
        'priority' => 'medium',
        'reason' => 'Optimizes queries for comments on specific posts',
        'estimated_benefit' => 'Medium - used in post comment display'
      ],
      [
        'columns' => ['comment_author_email', 'comment_approved'],
        'name' => 'idx_author_email_approved',
        'priority' => 'low',
        'reason' => 'Optimizes queries for comment moderation by author email',
        'estimated_benefit' => 'Low - used in admin comment management'
      ]
    ];
    
    foreach ($recommended_indexes as $rec_index) {
      $exists = self::index_covers_columns($current_indexes, $rec_index['columns']);
      
      if (!$exists) {
        $recommendations[] = [
          'table' => $table_name,
          'index_name' => $rec_index['name'],
          'columns' => $rec_index['columns'],
          'priority' => $rec_index['priority'],
          'reason' => $rec_index['reason'],
          'estimated_benefit' => $rec_index['estimated_benefit'],
          'sql' => "ALTER TABLE `{$table_name}` ADD INDEX `{$rec_index['name']}` (`" . implode('`, `', $rec_index['columns']) . "`)"
        ];
      }
    }
    
    return $recommendations;
  }
  
  /**
   * Get index recommendations for wp_options table
   * 
   * @param string $table_name
   * @param array $current_indexes
   * @return array Recommendations
   */
  private static function get_options_table_recommendations($table_name, $current_indexes)
  {
    $recommendations = [];
    
    $recommended_indexes = [
      [
        'columns' => ['autoload', 'option_name'],
        'name' => 'idx_autoload_name',
        'priority' => 'high',
        'reason' => 'Optimizes autoload queries during WordPress initialization',
        'estimated_benefit' => 'High - used on every page load'
      ]
    ];
    
    foreach ($recommended_indexes as $rec_index) {
      $exists = self::index_covers_columns($current_indexes, $rec_index['columns']);
      
      if (!$exists) {
        $recommendations[] = [
          'table' => $table_name,
          'index_name' => $rec_index['name'],
          'columns' => $rec_index['columns'],
          'priority' => $rec_index['priority'],
          'reason' => $rec_index['reason'],
          'estimated_benefit' => $rec_index['estimated_benefit'],
          'sql' => "ALTER TABLE `{$table_name}` ADD INDEX `{$rec_index['name']}` (`" . implode('`, `', $rec_index['columns']) . "`)"
        ];
      }
    }
    
    return $recommendations;
  }
  
  /**
   * Check if existing indexes cover the required columns
   * 
   * @param array $current_indexes
   * @param array $required_columns
   * @return bool
   */
  private static function index_covers_columns($current_indexes, $required_columns)
  {
    // Group indexes by name to check compound indexes
    $indexes_by_name = [];
    foreach ($current_indexes as $index) {
      if (!isset($indexes_by_name[$index['name']])) {
        $indexes_by_name[$index['name']] = [];
      }
      $indexes_by_name[$index['name']][] = $index['column'];
    }
    
    // Check if any existing index covers all required columns in the same order
    foreach ($indexes_by_name as $index_name => $index_columns) {
      // Skip PRIMARY key
      if ($index_name === 'PRIMARY') {
        continue;
      }
      
      // Check if this index starts with our required columns in the same order
      if (count($required_columns) <= count($index_columns)) {
        $matches = true;
        for ($i = 0; $i < count($required_columns); $i++) {
          if ($index_columns[$i] !== $required_columns[$i]) {
            $matches = false;
            break;
          }
        }
        if ($matches) {
          return true;
        }
      }
    }
    
    return false;
  }
  
  /**
   * Analyze slow queries if performance schema is available
   * 
   * @return array Slow query analysis
   */
  private static function analyze_slow_queries()
  {
    global $wpdb;
    
    $slow_queries = [];
    
    try {
      // Check if performance schema is available
      $performance_schema = $wpdb->get_var("SHOW DATABASES LIKE 'performance_schema'");
      if (!$performance_schema) {
        return ['message' => 'Performance schema not available'];
      }
      
      // Get slow query statistics (simplified version)
      $slow_query_stats = $wpdb->get_results("
        SELECT 
          LEFT(digest_text, 100) as query_sample,
          count_star as exec_count,
          avg_timer_wait/1000000000 as avg_time_sec,
          sum_rows_examined/count_star as avg_rows_examined
        FROM performance_schema.events_statements_summary_by_digest
        WHERE avg_timer_wait/1000000000 > 1
        ORDER BY avg_timer_wait DESC
        LIMIT 10
      ");
      
      if ($slow_query_stats) {
        $slow_queries['slow_queries'] = $slow_query_stats;
      }
      
    } catch (\Exception $e) {
      $slow_queries['error'] = 'Could not analyze slow queries: ' . $e->getMessage();
    }
    
    return $slow_queries;
  }
  
  /**
   * Apply recommended indexes (DANGER: This actually modifies the database)
   * 
   * @param array $recommendations Array of recommendations to apply
   * @return array Results of index creation
   */
  public static function apply_index_recommendations($recommendations)
  {
    global $wpdb;
    
    $results = [
      'applied' => [],
      'failed' => [],
      'skipped' => []
    ];
      // Safety check - only allow specific high-priority indexes
    $allowed_indexes = [
      // Posts table indexes
      'idx_status_type_date',
      'idx_author_status',
      'idx_parent_type',
      'idx_name_status',
      // Postmeta table indexes
      'idx_key_value',
      'idx_post_key',
      // Comments table indexes
      'idx_approved_date',
      'idx_post_approved',
      'idx_author_email_approved',
      // Options table indexes
      'idx_autoload_name'
    ];
    
    foreach ($recommendations as $rec) {
      // Safety check
      if (!in_array($rec['index_name'], $allowed_indexes)) {
        $results['skipped'][] = [
          'index' => $rec['index_name'],
          'reason' => 'Index not in allowed list for safety'
        ];
        continue;
      }
      
      try {
        // Log before applying
        self::log_database_operation('apply_index', 
          "Applying index {$rec['index_name']} on {$rec['table']}"
        );
        
        // Apply the index
        $result = $wpdb->query($rec['sql']);
        
        if ($result !== false) {
          $results['applied'][] = [
            'index' => $rec['index_name'],
            'table' => $rec['table'],
            'sql' => $rec['sql']
          ];
          
          self::log_database_operation('apply_index_success', 
            "Successfully applied index {$rec['index_name']} on {$rec['table']}"
          );
        } else {
          throw new \Exception('SQL execution failed');
        }
        
      } catch (\Exception $e) {
        $results['failed'][] = [
          'index' => $rec['index_name'],
          'table' => $rec['table'],
          'error' => $e->getMessage(),
          'sql' => $rec['sql']
        ];
        
        self::log_database_operation('apply_index_error', 
          "Failed to apply index {$rec['index_name']} on {$rec['table']}: " . $e->getMessage()
        );
      }
    }
    
    return $results;
  }
}
