<?php

namespace ZiziCache;

/**
 * Class Dashboard
 *
 * Handles the creation and management of the ZiziCache admin dashboard page in WordPress.
 * Registers the admin menu, loads necessary assets, and renders the dashboard container.
 */
class Dashboard
{
    /**
     * Emoji icon for the dashboard menu (handled via CSS).
     * 
     * This property is intentionally left empty as the icon is added via CSS
     * in the admin_head action for better compatibility and styling control.
     *
     * @var string
     * @see Dashboard::add_menu()
     */
    public static $menu_icon = '';

    /**
     * Registers the dashboard menu in WordPress admin.
     *
     * @return void
     */
    public static function init()
    {
        add_action('admin_menu', [__CLASS__, 'add_menu']);
    }

    /**
     * Adds the ZiziCache dashboard menu page to the WordPress admin menu.
     * Checks user permissions before adding the menu.
     *
     * @return void
     */
    public static function add_menu()
    {
        if (!Authority::is_allowed()) {
            return;
        }

        $menu = add_menu_page(
            'ZiziCache',
            'ZiziCache',
            'edit_posts',
            'zizi-cache',
            [__CLASS__, 'render'],
            self::$menu_icon,
            59 // Just above Appearance
        );

        // Add emoji icon via CSS for accessibility and compatibility
        add_action('admin_head', function() {
            echo "<style>#adminmenu #toplevel_page_zizi-cache .wp-menu-image:before { content: '⚡'; font-size: 20px; line-height: 1; } </style>";
        });

        // Enqueue JS/CSS only on the dashboard page
        add_action('admin_print_scripts-' . $menu, [__CLASS__, 'add_assets']);
    }

    /**
     * Enqueues JavaScript and CSS assets for the dashboard page.
     * Passes configuration and REST data to the JS app via window.ziziCacheData.
     *
     * @return void
     */
    public static function add_assets()
    {
        $script_handle = 'zizi_cache_dashboard';
        $script_url = ZIZI_CACHE_PLUGIN_URL . 'assets/js/admin.js';
        $style_url = ZIZI_CACHE_PLUGIN_URL . 'assets/css/admin.css';
        $script_version = file_exists(ZIZI_CACHE_PLUGIN_DIR . 'assets/js/admin.js') ? filemtime(ZIZI_CACHE_PLUGIN_DIR . 'assets/js/admin.js') : false;
        $style_path = ZIZI_CACHE_PLUGIN_DIR . 'assets/css/admin.css';

        // Register and enqueue WordPress core scripts with proper dependencies
        wp_register_script('wp-i18n', includes_url('js/dist/i18n.min.js'), [], false, true);
        wp_register_script('wp-hooks', includes_url('js/dist/hooks.min.js'), [], false, true);

        // Enqueue CSS if it exists
        if (file_exists($style_path)) {
            $style_version = filemtime($style_path);
            wp_enqueue_style(
                $script_handle . '-style',
                $style_url,
                [],
                $style_version
            );
        }
        

        // Load webpack chunks in correct order for Vue 3 application
        
        // 1. Load vendor chunks first (Vue dependencies)
        $vue_vendor_assets = include ZIZI_CACHE_PLUGIN_DIR . 'assets/js/vue-vendor.asset.php';
        wp_enqueue_script(
            'zizi_cache_vue_vendor',
            ZIZI_CACHE_PLUGIN_URL . 'assets/js/vue-vendor.js',
            $vue_vendor_assets['dependencies'],
            $vue_vendor_assets['version'],
            true
        );
        
        // 2. Load charts vendor chunk
        $charts_vendor_assets = include ZIZI_CACHE_PLUGIN_DIR . 'assets/js/charts-vendor.asset.php';
        wp_enqueue_script(
            'zizi_cache_charts_vendor',
            ZIZI_CACHE_PLUGIN_URL . 'assets/js/charts-vendor.js',
            array_merge($charts_vendor_assets['dependencies'], ['zizi_cache_vue_vendor']),
            $charts_vendor_assets['version'],
            true
        );
        
        // 3. Load general vendors chunk (includes common dependencies)
        wp_enqueue_script(
            'zizi_cache_vendors',
            ZIZI_CACHE_PLUGIN_URL . 'assets/js/vendors.ed21b3ccfbdde755d9a8.js',
            ['zizi_cache_vue_vendor'],
            ZIZI_CACHE_VERSION,
            true
        );
        
        // 4. Finally load main admin script
        $admin_assets = include ZIZI_CACHE_PLUGIN_DIR . 'assets/js/admin.asset.php';
        wp_enqueue_script(
            $script_handle,
            $script_url,
            array_merge($admin_assets['dependencies'], ['zizi_cache_vue_vendor', 'zizi_cache_charts_vendor', 'zizi_cache_vendors', 'wp-element', 'wp-i18n', 'wp-hooks']),
            $admin_assets['version'],
            true // Load in footer
        );

        // Prepare settings for localization and pass data to JavaScript
        $plugin_dir_name = basename(dirname(ZIZI_CACHE_FILE));
        $problematic_preload_urls = get_option('zizi_cache_preload_problematic_urls', []);
        
        // Check Image Converter environment
        $image_converter_env = null;
        if (class_exists('\\ZiziCache\\ImageConverter')) {
            $image_converter_env = \ZiziCache\ImageConverter::get_environment_info();
        }
        
        $settings = [
            'restUrl' => rtrim(get_rest_url(null, $plugin_dir_name . '/v1'), '/'), // Dynamic REST API URL with /v1 namespace
            'nonce'   => wp_create_nonce('wp_rest'), // Security nonce for REST API
            'config'  => SysConfig::$config, // Plugin configuration
            'version' => defined('ZIZI_CACHE_VERSION') ? ZIZI_CACHE_VERSION : null, // Plugin version
            'adminUrl' => admin_url('admin.php?page=zizi-cache'), // Admin URL
            'pluginUrl' => ZIZI_CACHE_PLUGIN_URL, // Plugin directory URL
            'ajaxUrl' => admin_url('admin-ajax.php'), // AJAX endpoint URL
            'problematicPreloadUrls' => $problematic_preload_urls, // Pass problematic URLs
            'imageConverterEnv' => $image_converter_env, // Image Converter environment info
        ];

        // Localize the script with plugin data (available as window.ziziCacheData)
        wp_localize_script($script_handle, 'ziziCacheData', $settings);
        
        // Add Font Intelligence localization (available as window.ziziCacheFontIntelligence)
        $font_intelligence_settings = [
            'restUrl' => rtrim(get_rest_url(), '/') . '/' . $plugin_dir_name . '/v1/',
            'nonce' => wp_create_nonce('wp_rest'),
            'namespaces' => [$plugin_dir_name . '/v1'], // Dynamic namespace based on plugin directory with /v1
        ];
        wp_localize_script($script_handle, 'ziziCacheFontIntelligence', $font_intelligence_settings);

        // Set up script translations for internationalization
        wp_set_script_translations($script_handle, 'zizi-cache');
    }

    /**
     * Renders the dashboard container div for the JavaScript application.
     * 
     * This method outputs the root container element that will be used by the React/JavaScript
     * application to render the admin interface. The container uses a specific ID
     * (#zizi-cache-admin-app) to ensure isolation from other plugins.
     * 
     * The actual data and configuration are passed to the JavaScript via the
     * window.ziziCacheData object in the add_assets() method.
     *
     * @return void
     * @see Dashboard::add_assets()
     */
    public static function render()
    {
        // Output the container div - the JavaScript app will mount here
        // Data is passed via window.ziziCacheData in the add_assets() method
        echo '<div id="zizi-cache-admin-app" class="zizi-cache-container"></div>';
    }
}
