<?php
/**
 * Early Hints Implementation for ZiziCache v2.0.0
 * 
 * Implements HTTP 103 Early Hints for faster resource loading by sending
 * preload headers before the full response is ready. This can significantly
 * improve Core Web Vitals by allowing browsers to start downloading critical
 * resources earlier.
 *
 * Features:
 * - Early Hints for CSS, fonts, and critical images
 * - Crossorigin and fetchpriority attribute support
 * - Integration with existing preload system
 * - Security validation and error handling
 * - Performance monitoring and logging
 *
 * @package ZiziCache\Core
 * @version 2.0.0
 * @since 2.0.0
 * @author ZiziCache Development Team
 * @enterprise-grade true
 * @security-level high
 * @performance-impact minimal
 */

namespace ZiziCache\Core;

use ZiziCache\SysConfig;
use ZiziCache\CacheSys;

/**
 * Early Hints Class - Enterprise Implementation
 * 
 * Handles HTTP 103 Early Hints for optimal resource loading performance.
 * 
 * @since 2.0.0
 * @version 2.0.0
 */
class EarlyHints {
    
    /**
     * Performance metrics
     * @var array
     */
    private static $metrics = [
        'headers_sent' => 0,
        'css_hints' => 0,
        'font_hints' => 0,
        'image_hints' => 0,
        'script_hints' => 0,
        'errors' => 0,
        'init_time' => 0
    ];
    
    /**
     * Already sent hints to prevent duplicates
     * @var array
     */
    private static $sent_hints = [];
    
    /**
     * Initialize Early Hints functionality
     * 
     * @return void
     */
    public static function init(): void {
        $start_time = microtime(true);
        
        try {
            // Ensure SysConfig is loaded first
            if (!class_exists('\\ZiziCache\\SysConfig') || empty(SysConfig::$config)) {
                // Use simple debug check when SysConfig is not ready
                if (defined('WP_DEBUG') && WP_DEBUG && class_exists('\\ZiziCache\\CacheSys')) {
                    CacheSys::writeError('Early Hints: SysConfig not ready, skipping initialization');
                }
                return;
            }
            
            if (!self::is_enabled()) {
                return;
            }
            
            // Hook into output buffer processing (BEFORE HTML output)
            add_action('zizi_cache_before_optimizations', [__CLASS__, 'send_early_hints'], 1);
            
            // Hook into preload actions
            add_filter('zizi_cache_preload_css', [__CLASS__, 'add_css_early_hint'], 10, 2);
            add_filter('zizi_cache_preload_font', [__CLASS__, 'add_font_early_hint'], 10, 2);
            
            self::$metrics['init_time'] = microtime(true) - $start_time;
            
            if (self::is_debug_enabled()) {
                CacheSys::writeLog('Early Hints: Initialized successfully in ' . round(self::$metrics['init_time'] * 1000, 2) . 'ms');
            }
            
        } catch (\Exception $e) {
            self::$metrics['errors']++;
            CacheSys::writeError('Early Hints: Initialization failed - ' . $e->getMessage());
        }
    }
    
    /**
     * Check if Early Hints are enabled
     * 
     * @return bool
     */
    private static function is_enabled(): bool {
        // Ensure SysConfig is loaded
        if (empty(SysConfig::$config)) {
            return false;
        }
        
        return !empty(SysConfig::$config['early_hints_enabled']) &&
               !headers_sent() &&
               !is_admin() &&
               !wp_doing_ajax() &&
               !wp_doing_cron();
    }
    
    /**
     * Check if debug logging is enabled
     * 
     * @return bool
     */
    private static function is_debug_enabled(): bool {
        if (empty(SysConfig::$config)) {
            return defined('WP_DEBUG') && WP_DEBUG;
        }
        
        return !empty(SysConfig::$config['fonts_debug']) || defined('WP_DEBUG') && WP_DEBUG;
    }
    
    /**
     * Send Early Hints for preloaded resources
     * 
     * @return void
     */
    public static function send_early_hints(): void {
        try {
            if (!self::is_enabled() || headers_sent()) {
                return;
            }
            
            $config = SysConfig::$config ?? [];
            $hints_types = $config['early_hints_types'] ?? [];
            
            // Send hints for preloaded fonts
            if (!empty($hints_types['font']) && !empty($config['fonts_preload_urls'])) {
                self::send_font_hints($config['fonts_preload_urls']);
            }
            
            // Send hints for preloaded CSS from configuration
            if (!empty($hints_types['css']) && !empty($config['css_preload_urls'])) {
                self::send_css_hints_from_config($config['css_preload_urls']);
            }
            
            // Send hints for critical images from configuration
            if (!empty($hints_types['image']) && !empty($config['img_preload_urls'])) {
                self::send_image_hints($config['img_preload_urls']);
            }
            
            if (self::is_debug_enabled() && self::$metrics['headers_sent'] > 0) {
                CacheSys::writeError('Early Hints: Sent ' . self::$metrics['headers_sent'] . ' early hint headers');
            }
            
        } catch (\Exception $e) {
            self::$metrics['errors']++;
            CacheSys::writeError('Early Hints: Failed to send hints - ' . $e->getMessage());
        }
    }
    
    /**
     * Send Early Hints for fonts
     * 
     * @param array $font_urls Font URLs to preload
     * @return void
     */
    private static function send_font_hints(array $font_urls): void {
        foreach ($font_urls as $font_url) {
            if (self::send_early_hint($font_url, 'font', ['crossorigin' => true])) {
                self::$metrics['font_hints']++;
            }
        }
    }
    
    /**
     * Send Early Hints for CSS from configuration
     * 
     * @param array $css_urls CSS URLs to preload
     * @return void
     */
    private static function send_css_hints_from_config(array $css_urls): void {
        $critical_count = 0;
        $max_critical = 5; // Allow more CSS than before
        
        foreach ($css_urls as $css_url) {
            if ($critical_count >= $max_critical) {
                break;
            }
            
            if (self::send_early_hint($css_url, 'style')) {
                self::$metrics['css_hints']++;
                $critical_count++;
            }
        }
    }
    
    /**
     * Send Early Hints for images
     * 
     * @param array $image_urls Image URLs to preload
     * @return void
     */
    private static function send_image_hints(array $image_urls): void {
        $image_count = 0;
        $max_images = 2; // Limit critical images
        
        foreach ($image_urls as $image_url) {
            if ($image_count >= $max_images) {
                break;
            }
            
            if (self::send_early_hint($image_url, 'image', ['fetchpriority' => 'high'])) {
                self::$metrics['image_hints']++;
                $image_count++;
            }
        }
    }
    
    /**
     * Send individual Early Hint
     * 
     * @param string $url Resource URL
     * @param string $as Resource type (font, style, script, image)
     * @param array $attributes Additional attributes
     * @return bool Success status
     */
    private static function send_early_hint(string $url, string $as, array $attributes = []): bool {
        try {
            // Prevent duplicates
            $hint_key = md5($url . $as);
            if (isset(self::$sent_hints[$hint_key])) {
                return false;
            }
            
            // Validate URL
            if (!filter_var($url, FILTER_VALIDATE_URL) || !self::is_safe_url($url)) {
                return false;
            }
            
            // Build header value
            $header_value = "<{$url}>; rel=preload; as={$as}";
            
            $config = SysConfig::$config ?? [];
            
            // Add crossorigin for fonts
            if ($as === 'font' || (!empty($attributes['crossorigin']) && !empty($config['early_hints_crossorigin']))) {
                $header_value .= '; crossorigin';
            }
            
            // Add fetchpriority for critical resources or as specified
            if (!empty($attributes['fetchpriority'])) {
                $header_value .= '; fetchpriority=' . esc_attr($attributes['fetchpriority']);
            } elseif (!empty($config['early_hints_fetchpriority']) && in_array($as, ['style', 'font', 'image'])) {
                $header_value .= '; fetchpriority=high';
            }
            
            // Add type for fonts
            if ($as === 'font') {
                $font_type = self::detect_font_type($url);
                if ($font_type) {
                    $header_value .= "; type={$font_type}";
                }
            }
            
            // Send header
            if (!headers_sent()) {
                header("Link: {$header_value}", false);
                self::$sent_hints[$hint_key] = true;
                self::$metrics['headers_sent']++;
                
                if (self::is_debug_enabled()) {
                    CacheSys::writeError("Early Hints: Sent header - {$header_value}");
                }
                
                return true;
            }
            
        } catch (\Exception $e) {
            self::$metrics['errors']++;
            CacheSys::writeError('Early Hints: Failed to send hint for ' . $url . ' - ' . $e->getMessage());
        }
        
        return false;
    }
    
    /**
     * Detect font type from URL
     * 
     * @param string $url Font URL
     * @return string|null Font MIME type
     */
    private static function detect_font_type(string $url): ?string {
        $extension = strtolower(pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION));
        
        $font_types = [
            'woff2' => 'font/woff2',
            'woff' => 'font/woff',
            'ttf' => 'font/ttf',
            'otf' => 'font/otf',
            'eot' => 'application/vnd.ms-fontobject'
        ];
        
        return $font_types[$extension] ?? 'font/woff2'; // Fallback to woff2 (from Flying Press improvement)
    }
    
    /**
     * Check if URL is internal/safe
     * 
     * @param string $url URL to check
     * @return bool
     */
    private static function is_internal_url(string $url): bool {
        $site_url = site_url();
        $parsed_site = parse_url($site_url);
        $parsed_url = parse_url($url);
        
        // Allow relative URLs
        if (empty($parsed_url['host'])) {
            return true;
        }
        
        // Check if same domain
        return $parsed_url['host'] === $parsed_site['host'];
    }
    
    /**
     * Validate URL safety
     * 
     * @param string $url URL to validate
     * @return bool
     */
    private static function is_safe_url(string $url): bool {
        // Block dangerous schemes
        $dangerous_schemes = ['javascript:', 'data:', 'vbscript:', 'file:'];
        
        foreach ($dangerous_schemes as $scheme) {
            if (stripos($url, $scheme) === 0) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Add CSS Early Hint filter callback
     * 
     * @param string $css_url CSS URL
     * @param array $attributes CSS attributes
     * @return string Original CSS URL
     */
    public static function add_css_early_hint(string $css_url, array $attributes = []): string {
        if (self::is_enabled()) {
            self::send_early_hint($css_url, 'style', $attributes);
        }
        return $css_url;
    }
    
    /**
     * Add Font Early Hint filter callback
     * 
     * @param string $font_url Font URL
     * @param array $attributes Font attributes
     * @return string Original font URL
     */
    public static function add_font_early_hint(string $font_url, array $attributes = []): string {
        if (self::is_enabled()) {
            self::send_early_hint($font_url, 'font', ['crossorigin' => true]);
        }
        return $font_url;
    }
    
    /**
     * Get performance metrics
     * 
     * @return array Performance data
     */
    public static function get_metrics(): array {
        return self::$metrics;
    }
    
    /**
     * Log performance metrics on shutdown
     * 
     * @return void
     */
    public static function log_metrics(): void {
        if (!self::is_debug_enabled()) {
            return;
        }
        
        $metrics = self::$metrics;
        if ($metrics['headers_sent'] > 0 || $metrics['errors'] > 0) {
            $summary = sprintf(
                'Early Hints Metrics - Headers: %d, CSS: %d, Fonts: %d, Images: %d, Scripts: %d, Errors: %d, Init: %sms',
                $metrics['headers_sent'],
                $metrics['css_hints'],
                $metrics['font_hints'],
                $metrics['image_hints'],
                $metrics['script_hints'],
                $metrics['errors'],
                round($metrics['init_time'] * 1000, 2)
            );
            
            CacheSys::writeError($summary);
        }
    }
}
