<?php
/**
 * Converted Files Cleanup System
 * Automatically delete converted files (WebP/AVIF) when original is deleted
 */

namespace ZiziCache;

class ConvertedFilesCleanup
{
    private static $instance = null;
    
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize cleanup system
     */
    public static function init()
    {
        $instance = self::get_instance();
        
        // Hook into attachment deletion
        add_action('delete_attachment', [$instance, 'delete_converted_files'], 10, 1);
        
        // Hook into file deletion (when attachment is permanently deleted)
        add_action('wp_delete_file', [$instance, 'delete_converted_files_by_path'], 10, 1);
    }
    
    /**
     * Delete converted files when attachment is deleted
     */
    public function delete_converted_files($attachment_id)
    {
        // Only process image attachments
        if (!wp_attachment_is_image($attachment_id)) {
            return;
        }
        
        $original_file = get_attached_file($attachment_id);
        if (!$original_file || !file_exists($original_file)) {
            return;
        }
        
        $deleted_files = $this->delete_converted_versions($original_file);
        
        // Log the cleanup
        if (!empty($deleted_files)) {
            error_log("ZiZi Cache: Deleted converted files for attachment $attachment_id: " . implode(', ', $deleted_files));
        }
    }
    
    /**
     * Delete converted files by original file path
     */
    public function delete_converted_files_by_path($file_path)
    {
        // Check if this is an image file
        $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        if (!in_array($extension, ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'tiff'])) {
            return $file_path;
        }
        
        $this->delete_converted_versions($file_path);
        
        return $file_path;
    }
    
    /**
     * Delete all converted versions of an image file
     */
    private function delete_converted_versions($original_file)
    {
        $deleted_files = [];
        $path_info = pathinfo($original_file);
        
        // Get enabled formats from settings
        $config = get_option('ZIZI_CACHE_CONFIG', []);
        $enabled_formats = $config['image_converter_formats'] ?? ['avif', 'webp'];
        
        foreach ($enabled_formats as $format) {
            $converted_file = $path_info['dirname'] . '/' . $path_info['filename'] . '.' . $format;
            
            if (file_exists($converted_file)) {
                if (unlink($converted_file)) {
                    $deleted_files[] = basename($converted_file);
                } else {
                    error_log("ZiZi Cache: Failed to delete converted file: $converted_file");
                }
            }
        }
        
        // Also delete any thumbnail versions
        $this->delete_converted_thumbnails($original_file);
        
        return $deleted_files;
    }
    
    /**
     * Delete converted thumbnails
     */
    private function delete_converted_thumbnails($original_file)
    {
        $path_info = pathinfo($original_file);
        $dir = $path_info['dirname'];
        $filename = $path_info['filename'];
        
        // Get enabled formats
        $config = get_option('ZIZI_CACHE_CONFIG', []);
        $enabled_formats = $config['image_converter_formats'] ?? ['avif', 'webp'];
        
        // Look for thumbnail versions (WordPress naming pattern: filename-123x456.ext)
        foreach ($enabled_formats as $format) {
            $pattern = $dir . '/' . $filename . '-*x*.' . $format;
            $thumbnail_files = glob($pattern);
            
            foreach ($thumbnail_files as $thumbnail_file) {
                if (file_exists($thumbnail_file)) {
                    unlink($thumbnail_file);
                }
            }
        }
    }
    
    /**
     * Manual cleanup of orphaned converted files
     */
    public static function cleanup_orphaned_converted_files()
    {
        $upload_dir = wp_upload_dir();
        $uploads_path = $upload_dir['basedir'];
        
        $cleaned_count = 0;
        $cleaned_size = 0;
        
        // Get enabled formats
        $config = get_option('ZIZI_CACHE_CONFIG', []);
        $enabled_formats = $config['image_converter_formats'] ?? ['avif', 'webp'];
        
        // Scan recent directories
        $current_year = date('Y');
        $current_month = date('m');
        
        for ($i = 0; $i < 12; $i++) { // Last 12 months
            $year = $current_year;
            $month = $current_month - $i;
            
            if ($month <= 0) {
                $year--;
                $month += 12;
            }
            
            $month = str_pad($month, 2, '0', STR_PAD_LEFT);
            $dir_path = $uploads_path . "/$year/$month";
            
            if (is_dir($dir_path)) {
                $result = self::cleanup_directory_orphaned_files($dir_path, $enabled_formats);
                $cleaned_count += $result['count'];
                $cleaned_size += $result['size'];
            }
        }
        
        return [
            'cleaned_count' => $cleaned_count,
            'cleaned_size' => $cleaned_size
        ];
    }
    
    /**
     * Cleanup orphaned files in specific directory
     */
    private static function cleanup_directory_orphaned_files($dir_path, $enabled_formats)
    {
        $cleaned_count = 0;
        $cleaned_size = 0;
        
        foreach ($enabled_formats as $format) {
            $pattern = $dir_path . '/*.' . $format;
            $converted_files = glob($pattern);
            
            foreach ($converted_files as $converted_file) {
                // Check if original file exists
                $path_info = pathinfo($converted_file);
                $original_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'tiff'];
                $original_exists = false;
                
                foreach ($original_extensions as $ext) {
                    $original_file = $path_info['dirname'] . '/' . $path_info['filename'] . '.' . $ext;
                    if (file_exists($original_file)) {
                        $original_exists = true;
                        break;
                    }
                }
                
                // If original doesn't exist, delete the converted file
                if (!$original_exists) {
                    $file_size = filesize($converted_file);
                    if (unlink($converted_file)) {
                        $cleaned_count++;
                        $cleaned_size += $file_size;
                    }
                }
            }
        }
        
        return [
            'count' => $cleaned_count,
            'size' => $cleaned_size
        ];
    }
}

// Initialize the cleanup system
add_action('init', ['ZiziCache\\ConvertedFilesCleanup', 'init']);