<?php

namespace ZiziCache;

/**
 * Handles compatibility checks and notices for ZiziCache.
 * 
 * This class is responsible for detecting and managing plugin incompatibilities,
 * displaying admin notices, and verifying system requirements like permalink structure.
 */
class CompatLayer
{
  /**
   * List of detected active incompatible plugins.
   *
   * @var array<string> Array of plugin names that are incompatible with ZiziCache.
   */
  public static $active_incompatible_plugins = [];

  /**
   * Initializes compatibility checks and admin notices for the plugin.
   *
   * Registers hooks for finding incompatible plugins, displaying notices,
   * and checking for required permalink structure.
   *
   * @return void
   */
  public static function init()
  {
    add_action('admin_init', [__CLASS__, 'find_incompatible_plugins']);
    add_action('admin_notices', [__CLASS__, 'add_notice']);
    add_action('admin_notices', [__CLASS__, 'check_permalink_structure']);
  }

  /**
   * Detects and stores a list of active incompatible plugins.
   *
   * @return void
   */
  public static function find_incompatible_plugins()
  {
    $incompatible_plugins = [
      [
        'name' => 'Comet Cache',
        'file' => 'comet-cache/comet-cache.php',
      ],
      [
        'name' => 'Hyper Cache',
        'file' => 'hyper-cache/plugin.php',
      ],
      [
        'name' => 'Simple Cache',
        'file' => 'simple-cache/simple-cache.php',
      ],
      [
        'name' => 'Borlabs Cache',
        'file' => 'borlabs-cache/borlabs-cache.php',
      ],
      [
        'name' => 'Powered Cache',
        'file' => 'powered-cache/powered-cache.php',
      ],
      [
        'name' => 'Cachify',
        'file' => 'cachify/cachify.php',
      ],
      [
        'name' => 'Cache Master',
        'file' => 'cache-master/cache-master.php',
      ],
      [
        'name' => 'Lite Cache',
        'file' => 'lite-cache/lite-cache.php',
      ],
      [
        'name' => 'SG Optimizer',
        'file' => 'sg-cachepress/sg-cachepress.php',
      ],
      [
        'name' => 'WP Speed of Light',
        'file' => 'wp-speed-of-light/wp-speed-of-light.php',
      ],
      [
        'name' => 'WP Fast Cache',
        'file' => 'wp-fast-cache/wp-fast-cache.php',
      ],
      [
        'name' => 'Redis Object Cache',
        'file' => 'redis-cache/redis-cache.php',
      ],
      [
        'name' => 'Memcached Object Cache',
        'file' => 'memcached/memcached.php',
      ],
      [
        'name' => 'NitroPack',
        'file' => 'nitropack/main.php',
      ],
      [
        'name' => 'Swift Performance Pro',
        'file' => 'swift-performance-pro/performance.php',
      ],
      [
        'name' => 'WP Cloudflare Super Page Cache',
        'file' => 'wp-cloudflare-page-cache/wp-cloudflare-super-page-cache.php',
      ],
      [
        'name' => 'WP Engine System',
        'file' => 'wpengine-common/wpengine.php',
      ],
      [
        'name' => 'Hummingbird Pro',
        'file' => 'hummingbird-performance/wp-hummingbird.php',
      ],
      [
        'name' => 'Autocache',
        'file' => 'autocache/autocache.php',
      ],
      [
        'name' => 'WP Super Minify',
        'file' => 'wp-super-minify/wp-super-minify.php',
      ],
      [
        'name' => 'Better WordPress Minify',
        'file' => 'bwp-minify/bwp-minify.php',
      ],
      [
        'name' => 'Asset CleanUp',
        'file' => 'wp-asset-clean-up/wpacu.php',
      ],
      [
        'name' => 'Async JavaScript',
        'file' => 'async-javascript/async-javascript.php',
      ],
      [
        'name' => 'Cloudflare',
        'file' => 'cloudflare/cloudflare.php',
      ],
      [
        'name' => 'PageSpeed Ninja',
        'file' => 'pagespeedninja/pagespeedninja.php',
      ],
      [
        'name' => 'Autoptimize',
        'file' => 'autoptimize/autoptimize.php',
      ],
      [
        'name' => 'WP Rocket',
        'file' => 'wp-rocket/wp-rocket.php',
      ],
      [
        'name' => 'LiteSpeed Cache',
        'file' => 'litespeed-cache/litespeed-cache.php',
      ],
      [
        'name' => 'Swift Performance',
        'file' => 'swift-performance/performance.php',
      ],
      [
        'name' => 'Swift Performance Lite',
        'file' => 'swift-performance-lite/performance.php',
      ],
      [
        'name' => 'W3 Total Cache',
        'file' => 'w3-total-cache/w3-total-cache.php',
      ],
      [
        'name' => 'WP Fastest Cache',
        'file' => 'wp-fastest-cache/wpFastestCache.php',
      ],
      [
        'name' => 'WP Super Cache',
        'file' => 'wp-super-cache/wp-cache.php',
      ],
      [
        'name' => 'Hummingbird',
        'file' => 'hummingbird-performance/wp-hummingbird.php',
      ],
      [
        'name' => 'Cache Enabler',
        'file' => 'cache-enabler/cache-enabler.php',
      ],
      [
        'name' => 'Fast Velocity Minify',
        'file' => 'fast-velocity-minify/fvm.php',
      ],
      [
        'name' => 'WP Optimize',
        'file' => 'wp-optimize/wp-optimize.php',
      ],
      [
        'name' => 'instant.page',
        'file' => 'instant-page/instantpage.php',
      ],
      [
        'name' => 'WP Meteor',
        'file' => 'wp-meteor/wp-meteor.php',
      ],
    ];

    // Generate a list of incompatible active plugins
    foreach ($incompatible_plugins as $plugin) {
      if (\is_plugin_active($plugin['file'])) {
        array_push(self::$active_incompatible_plugins, $plugin['name']);
      }
    }
  }

  /**
   * Displays an admin notice if any incompatible plugins are active.
   * 
   * This method checks for any incompatible plugins that are currently active
   * and displays a warning notice to the administrator with a list of
   * conflicting plugins that should be deactivated.
   *
   * @return void
   */
  public static function add_notice()
  {
    // Return if there are no incompatible plugins active
    if (!count(self::$active_incompatible_plugins)) {
      return;
    }

    // Build the notice HTML with list of incompatible plugins
    $notice = '<div class="notice notice-error is-dismissible">';
    $notice .= '<p><b>ZiziCache</b> is not compatible with the following plugins:<br/><ul>';
    
    foreach (self::$active_incompatible_plugins as $plugin) {
      $notice .= sprintf('<li><b>• %s</b></li>', esc_html($plugin));
    }
    
    $notice .= '</ul>Please deactivate these plugins to prevent conflicts with ZiziCache.</p>';
    $notice .= '</div>';
    
    // Output the notice
    echo $notice;
  }

  /**
   * Verifies that permalinks are properly configured.
   * 
   * This method checks if the WordPress permalink structure is set to something
   * other than the default "Plain" setting. If not, it displays an admin notice
   * with a link to the permalinks settings page.
   *
   * @return void
   */
  public static function check_permalink_structure()
  {
    // Check if permalinks are set to something other than the default
    $permalink_structure = get_option('permalink_structure');
    if (!empty($permalink_structure)) {
      return;
    }
    
    // Build the notice with a link to the permalinks settings
    $configure_url = admin_url('options-permalink.php');
    $notice = sprintf(
      '<div class="notice notice-error">' .
      '<p><b>ZiziCache:</b> Pretty permalinks are required for proper functionality. ' .
      '<a href="%s">Configure now</a></p>' .
      '</div>',
      esc_url($configure_url)
    );
    
    // Output the notice
    echo $notice;
  }
}
