<?php

namespace ZiziCache;
use ZiziCache\SysTool;
use FilesystemIterator;

/**
 * Class CacheSys
 * 
 * Handles page cache management with focus on speed, security, and optimization.
 * Provides comprehensive functionality for caching, retrieving, and invalidating
 * page caches in WordPress.
 *
 * @package ZiziCache
 */
class CacheSys
{
    /**
     * Constants for cache settings
     * 
     * @var string COOKIE_ROLE_NAME Cookie name for storing user roles
     * @var string CACHE_FILE_EXTENSION File extension for cached content
     */
    const COOKIE_ROLE_NAME = 'zc_logged_in_roles';
    const CACHE_FILE_EXTENSION = '.html.gz';
    
    /**
     * List of query strings to ignore for cache variation.
     * These parameters won't create separate cache files when they change.
     *
     *
     * @var array
     */
    public static $default_ignore_queries = [
        // Google Ads basic
        'adgroupid',       // Google Ads ad group ID
        'adid',            // Google Ads ad ID
        'campaignid',      // Google Ads campaign ID
        'gad_source',      // Google Ads source
        'gbraid',          // Google Ads enhanced conversions
        'gclid',           // Google Ads click ID
        'gclsrc',          // Google Ads click source
        'mkwid',           // Google Ads keyword ID
        'pcrid',           // Google Ads creative ID
        
        // Google Ads extended
        'wbraid',          // Google Ads enhanced conversions
        'dclid',           // Google Display & Video 360
        'ds_rl',           // Google Shopping campaigns
        '_gac',            // Google Ads click ID cookie
        
        // Google Analytics
        '_ga',             // Google Analytics client ID
        '_gl',             // Google Analytics linker parameter
        '_gid',            // Google Analytics session ID
        'gtm_debug',       // Google Tag Manager debug
        'gtm_preview',     // Google Tag Manager preview
        'gtm_auth',        // Google Tag Manager authorization
        'gtm_cookies_win', // Google Tag Manager cookies window
        
        // Google Fonts & Google services
        'gdfms',           // Google Fonts
        'gdftrk',          // Google Fonts tracking
        'gdffi',           // Google Fonts font info
        
        // Facebook/Meta basic
        'fb_action_ids',   // Facebook action IDs
        'fb_action_types', // Facebook action types
        'fb_source',       // Facebook source
        'fbclid',          // Facebook click ID

        // Facebook/Meta extended
        'fb_ref',          // Facebook referrer
        'fb_comment_id',   // Facebook comment tracking
        'hsa_acc',         // Facebook Ads account
        'hsa_cam',         // Facebook Ads campaign
        'hsa_grp',         // Facebook Ads ad group
        'hsa_ad',          // Facebook Ads ad
        'hsa_src',         // Facebook Ads source
        'hsa_tgt',         // Facebook Ads target
        'hsa_kw',          // Facebook Ads keyword
        'hsa_mt',          // Facebook Ads match type
        'hsa_net',         // Facebook Ads network
        'hsa_ver',         // Facebook Ads version

        // TikTok parameters
        'ttclid',          // TikTok click ID
        'tt_appid',        // TikTok App ID
        'tt_platform',     // TikTok platform
        'tt_webid',        // TikTok Web ID

        // Twitter/X parameters
        'twclid',          // Twitter/X click ID

        // LinkedIn parameters
        'li_fat_id',       // LinkedIn First-party Ad Tracking
        'trk',             // LinkedIn tracking

        // Pinterest parameters
        '_pinterest_ct',   // Pinterest conversion tag
        '_pinterest_cm',   // Pinterest conversion method

        // Snapchat parameters
        'snapchat_click_id', // Snapchat click ID
        'sc_click_id',     // Snapchat short click ID
        
        // Microsoft Bing Ads
        'msclkid',         // Microsoft Bing click ID
        '_clck',           // Microsoft Clarity click ID
        '_clsk',           // Microsoft Clarity session key

        // UTM standard parameters
        'utm_campaign',    // UTM campaign
        'utm_content',     // UTM content
        'utm_expid',       // UTM experiment ID
        'utm_id',          // UTM ID
        'utm_medium',      // UTM medium
        'utm_source',      // UTM source
        'utm_term',        // UTM term

        // Matomo (Piwik) parameters
        'mtm_campaign',    // Matomo campaign
        'mtm_cid',         // Matomo campaign ID
        'mtm_content',     // Matomo content
        'mtm_group',       // Matomo group
        'mtm_keyword',     // Matomo keyword
        'mtm_medium',      // Matomo medium
        'mtm_placement',   // Matomo placement
        'mtm_source',      // Matomo source
        'pk_campaign',     // Piwik campaign (legacy)
        'pk_cid',          // Piwik campaign ID (legacy)
        'pk_content',      // Piwik content (legacy)
        'pk_keyword',      // Piwik keyword (legacy)
        'pk_medium',       // Piwik medium (legacy)
        'pk_source',       // Piwik source (legacy)

        // Email marketing parameters
        'mc_cid',          // MailChimp campaign ID
        'mc_eid',          // MailChimp email ID
        'vero_conv',       // Vero email tracking
        'vero_id',         // Vero user ID
        'ck_subscriber_id', // ConvertKit subscriber
        'inf_contact_key', // Infusionsoft contact key
        '_hsenc',          // HubSpot encryption
        '_hsmi',           // HubSpot marketing information
        'hs_preview',      // HubSpot preview
        
        // Affiliate and tracking parameters
        'aff_id',          // Affiliate ID
        'aff_sub',         // Affiliate sub ID
        'aff_sub2',        // Affiliate sub ID 2
        'clickid',         // Generic click ID
        'subid',           // Sub ID
        'source_id',       // Source ID
        'partner_id',      // Partner ID
        'cjevent',         // Commission Junction event
        'irclickid',       // Impact Radius click ID

        // Other analytics tools
        'at_medium',       // Adobe Target medium
        'at_campaign',     // Adobe Target campaign
        'icid',            // Internal campaign ID
        'cid',             // Campaign ID (generic)
        'sid',             // Session ID (generic)
        'tid',             // Transaction ID
        'vid',             // Visitor ID

        // Mobile apps and deep linking
        'deep_link_value', // Deep link value
        'install_referrer', // Android install referrer
        'af_click_lookback', // AppsFlyer click lookback
        'af_reengagement_window', // AppsFlyer reengagement
        'adjust_tracker',  // Adjust tracker
        'adjust_campaign', // Adjust campaign

        // Czech specific parameters
        'skag',            // Seznam Sklik ad group
        'skcid',           // Seznam Sklik campaign ID
        'skaid',           // Seznam Sklik ad ID
        'skwid',           // Seznam Sklik keyword ID

        // Various tracking parameters
        'age-verified',    // Age verification
        'ao_noptimize',    // Autoptimize no optimize
        'cache_bust',      // Cache busting parameter
        'cn-reloaded',     // Cookie Notice reloaded
        'dm_i',            // Direct mail identifier
        'ef_id',           // Engine Fox ID
        'epik',            // Pinterest enhanced match
        'kboard_id',       // KBoard plugin ID
        'pp',              // Page parameter
        'ref',             // Referrer
        'redirect_log_mongo_id', // Redirect log MongoDB ID
        'redirect_mongo_id',     // Redirect MongoDB ID
        'sb_referer_host', // SkyBridge referer host
        's_kwcid',         // Search keyword CID
        'srsltid',         // Search result ID
        'sscid',           // Search session CID
        'trk_contact',     // Tracking contact
        'trk_msg',         // Tracking message
        'trk_module',      // Tracking module
        'trk_sid',         // Tracking session ID
        'zizi_preload',    // ZiziCache preload parameter
        'zizi_warmed',     // ZiziCache warmed parameter
    ];

    /**
     * List of query strings to include for cache variation.
     *
     * @var array
     */
    public static $default_include_queries = [
        'lang',
        'currency',
        'orderby',
        'max_price',
        'min_price',
        'rating_filter',
    ];

    /**
     * Logs issues or unexpected conditions to the unified log file with file and method reference.
     *
     * @param string $method Name of the method where the issue occurred.
     * @param string $message Description of the problem or unexpected condition.
     * @param array $context Optional context data for debugging.
     * @param string $level Log level (INFO, WARNING, ERROR).
     * @return void
     */
    private static function log_issue($method, $message, $context = [], $level = 'INFO')
    {
        $context_str = json_encode($context);
        $log_msg = sprintf(
            '[%s] CacheSys.php::%s: %s | Context: %s',
            $level,
            $method,
            $message,
            $context_str
        );
        self::writeLog($log_msg);
    }

    /**
     * Writes a log message to the unified zizi-log.log file.
     * Supports structured logging with levels and plugin context.
     *
     * @param string $message The message to write to the log file.
     * @param string $level Log level (ERROR, WARNING).
     * @param string $plugin Plugin name for context.
     */
    public static function writeLog(string $message, string $level = '', string $plugin = ''): void
    {
        // Enhanced log filtering and throttling to prevent log spam
        $config = SysConfig::$config ?? [];
        
        // Check log level filtering
        $global_log_level = $config['log_level'] ?? 'error';
        if (!self::shouldLog($message, $level, $plugin, $global_log_level)) {
            return;
        }
        
        // Log throttling for repeated messages (if enabled)
        $throttle_enabled = $config['log_throttle_enabled'] ?? true;
        $throttle_threshold = max(1, $config['log_throttle_threshold'] ?? 10);
        
        // Special handling for preload messages - use higher threshold
        if (strpos($message, '[PreloadSqlite]') !== false || strpos($message, '[Preload]') !== false) {
            $throttle_threshold = 100; // Allow more preload messages
        }
        
        if ($throttle_enabled) {
            $message_hash = md5($message . $level . $plugin);
            $throttle_key = "log_throttle_{$message_hash}";
            $throttle_data = get_transient($throttle_key);
            
            if ($throttle_data) {
                // Increment counter
                $throttle_data['count']++;
                set_transient($throttle_key, $throttle_data, 300); // 5 minutes
                
                // Only log every Nth occurrence of the same message to prevent spam
                if ($throttle_data['count'] % $throttle_threshold !== 0) {
                    return;
                }
                
                // Add throttle info to message
                $message .= " (repeated {$throttle_data['count']} times)";
            } else {
                // First occurrence
                set_transient($throttle_key, ['count' => 1, 'first_seen' => time()], 300);
            }
        }
        
        $file = self::getLogFilePath();

        // Ensure the directory exists
        $dir = dirname($file);
        if (!is_dir($dir)) {
            @mkdir($dir, 0755, true);
        }
        
        // Perform log rotation if the log is too large (5 MB)
        self::rotateLogIfNeeded($file, 5 * 1024 * 1024);

        $date = date('Y-m-d H:i:s');
        
        // Format with level and plugin context
        $context = '';
        if ($plugin) {
            $context .= $plugin;
        }
        if ($level) {
            $context .= ($context ? ' ' : '') . $level;
        }
        
        $formatted_message = $context 
            ? "[$date] $context: $message"
            : "[$date] $message";

        // Append message to the log file with exclusive lock
        @file_put_contents($file, $formatted_message . PHP_EOL, FILE_APPEND | LOCK_EX);
    }

    /**
     * Determine if a log message should be written based on log level and filters
     * 
     * @param string $message Log message
     * @param string $level Log level/category
     * @param string $plugin Plugin name
     * @param string $global_log_level Global log level setting
     * @return bool True if should log, false otherwise
     */
    private static function shouldLog(string $message, string $level, string $plugin, string $global_log_level): bool
    {
        // ALWAYS allow PreloadSqlite messages - critical for debugging preload functionality
        if (strpos($message, '[PreloadSqlite]') !== false || strpos($message, '[Preload]') !== false) {
            return true;
        }
        
        // Special handling for frontend debug logs
        if ($level === 'ImageConverter' && strpos($message, 'No optimization metadata') !== false) {
            $config = SysConfig::$config ?? [];
            $frontend_debug = $config['frontend_debug_logging'] ?? false;
            
            // Only log if frontend debug is explicitly enabled, or if we're in debug mode
            if (!$frontend_debug && !WP_DEBUG) {
                return false;
            }
        }
        
        // Log level hierarchy: none < error < warning < info < debug < verbose
        $level_hierarchy = [
            'none' => 0,
            'error' => 1,
            'warning' => 2,
            'info' => 3,
            'debug' => 4,
            'verbose' => 5
        ];
        
        $global_level_value = $level_hierarchy[$global_log_level] ?? 1;
        
        // If global level is 'none', don't log anything (except preload which was handled above)
        if ($global_level_value === 0) {
            return false;
        }
        
        // Determine message level
        $message_level_value = 1; // Default to error level
        
        if (strpos($message, 'ERROR') !== false || strpos($message, 'FATAL') !== false) {
            $message_level_value = 1; // error
        } elseif (strpos($message, 'WARNING') !== false || strpos($message, 'WARN') !== false) {
            $message_level_value = 2; // warning
        } elseif (strpos($message, 'INFO') !== false || $level === 'info') {
            $message_level_value = 3; // info
        } elseif (WP_DEBUG || strpos($message, 'DEBUG') !== false) {
            $message_level_value = 4; // debug
        } elseif (strpos($message, 'No optimization metadata') !== false) {
            $message_level_value = 5; // verbose
        }
        
        // Log if message level is at or below global level
        return $message_level_value <= $global_level_value;
    }

    /**
     * Logs an error message with ERROR level.
     *
     * @param string $message The error message.
     * @param string $plugin Plugin name for context.
     */
    public static function writeError(string $message, string $plugin = 'ZiziCache'): void
    {
        self::writeLog($message, 'ERROR', $plugin);
    }

    /**
     * Logs a warning message with WARNING level.
     *
     * @param string $message The warning message.
     * @param string $plugin Plugin name for context.
     */
    public static function writeWarning(string $message, string $plugin = 'ZiziCache'): void
    {
        self::writeLog($message, 'WARNING', $plugin);
    }
    
    /**
     * Rotates the log file if it exceeds the specified size limit.
     *
     * @param string $log_file Path to the log file.
     * @param int $max_size Maximum size in bytes before rotation.
     * @param int $max_files Maximum number of backup files to keep.
     * @return void
     */
    private static function rotateLogIfNeeded(string $log_file, int $max_size, int $max_files = 5): void
    {
        // Skip if file doesn't exist or is smaller than max size
        if (!file_exists($log_file) || filesize($log_file) < $max_size) {
            return;
        }
        
        // Rotate existing backup files
        for ($i = $max_files - 1; $i > 0; $i--) {
            $old_backup = "{$log_file}.{$i}";
            $new_backup = "{$log_file}." . ($i + 1);
            
            if (file_exists($old_backup)) {
                @rename($old_backup, $new_backup);
            }
        }
        
        // Move current log to .1
        if (file_exists($log_file)) {
            @rename($log_file, "{$log_file}.1");
        }
        
        // Create a header in new log file
        $header = sprintf(
            "[%s] ZiziCache log file created after rotation. Previous log: %s.1\n",
            date('Y-m-d H:i:s'),
            basename($log_file)
        );
        @file_put_contents($log_file, $header, LOCK_EX);
    }

    /**
     * Gets the path to the log file.
     *
     * @return string Path to the log file.
     */
    private static function getLogFilePath(): string
    {
        if (defined('ZIZI_CACHE_LOG_FILE')) {
            return ZIZI_CACHE_LOG_FILE;
        } elseif (defined('ZIZI_CACHE_PLUGIN_DIR')) {
            return rtrim(ZIZI_CACHE_PLUGIN_DIR, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'zizi-log.log';
        } else {
            return WP_CONTENT_DIR . '/zizi-cache/zizi-log.log';
        }
    }

    /**
     * Initializes hooks for cache management and cache lifespan.
     *
     * @return void
     */
    public static function init()
    {
        add_action('init', [__CLASS__, 'setup_cache_lifespan']);
        add_action('zizi_cache_cache_lifespan', [__CLASS__, 'run_cache_lifespan']);
        add_action('set_logged_in_cookie', [__CLASS__, 'add_logged_in_roles'], 10, 4);
        add_action('clear_auth_cookie', [__CLASS__, 'remove_logged_in_roles']);
        add_action('init', [__CLASS__, 'cleanup_roles_cookie']);
    }

    /**
     * Runs cache purging and preloading according to the configured cache lifespan.
     *
     * @return void
     */
    public static function run_cache_lifespan()
    {
        Purge::purge_pages();
        Preload::preload_cache();
    }

    /**
     * Schedules or clears the cache lifespan event based on configuration.
     *
     * @return void
     */
    public static function setup_cache_lifespan()
    {
        $lifespan = SysConfig::$config['cache_lifespan'];
        $action_name = 'zizi_cache_cache_lifespan';

        if ($lifespan == 'never') {
            wp_clear_scheduled_hook($action_name);
            return;
        }

        if (!wp_next_scheduled($action_name) || wp_get_schedule($action_name) != $lifespan) {
            wp_clear_scheduled_hook($action_name);
            wp_schedule_event(time(), $lifespan, $action_name);
        }
    }

    /**
     * Saves the current page to cache (with compression).
     *
     * @param string $html HTML content of the page to cache.
     * @return void
     */
    public static function cache_page($html)
    {
        // Get the cache file name
        $cache_file_name = self::get_cache_file_name();

        // Get cache file path
        $cache_file_path = self::get_cache_file_path();

        // Security validation: Ensure cache path is within allowed cache directory
        $normalized_cache_path = wp_normalize_path(realpath(ZIZI_CACHE_CACHE_DIR));
        $normalized_target_path = wp_normalize_path($cache_file_path);
        
        if (!$normalized_cache_path || strpos($normalized_target_path, $normalized_cache_path) !== 0) {
            self::writeError(
                "Cache creation blocked - path outside cache directory: " . $cache_file_path,
                'Security'
            );
            return;
        }

        // Create the cache directory if it does not exist
        if (!is_dir($cache_file_path)) {
            if (!wp_mkdir_p($cache_file_path)) {
                self::writeError(
                    "Failed to create cache directory: " . $cache_file_path,
                    'Error'
                );
                return;
            }
        }

        // Write cache file with error handling
        $cache_file_full_path = $cache_file_path . $cache_file_name;
        $compressed_content = gzencode($html, 6);
        
        if (file_put_contents($cache_file_full_path, $compressed_content) === false) {
            self::writeError(
                "Failed to write cache file: " . $cache_file_full_path,
                'Error'
            );
        }
    }

    /**
     * Gets the cache file path based on host and request path.
     *
     * @return string Cache file path.
     */
    private static function get_cache_file_path()
    {
        // Allow translation plugins to translate the request URI
        $request_uri = apply_filters('zizi_cache_request_uri', $_SERVER['REQUEST_URI']);

        // Get cache file path
        $host = $_SERVER['HTTP_HOST'];
        $path = parse_url($request_uri, PHP_URL_PATH);
        $path = urldecode($path);

        // Allow developers to modify the cache file path
        $path = apply_filters('zizi_cache_cache_file_path', $path);

        return ZIZI_CACHE_CACHE_DIR . "$host/$path/";
    }

    /**
     * Generates the cache file name based on user, currency, and other parameters.
     *
     * @return string Cache file name.
     */
    public static function get_cache_file_name()
    {
        $config = SysConfig::$config;
        $file_name = 'index';

        // Append "-logged-in" to the file name if the user is logged in
        $file_name .= $config['cache_logged_in'] && is_user_logged_in() ? '-logged-in' : '';

        // Add user role to cache file name
        $file_name .= self::get_role_suffix();

        // Add currency code to cache file name based on currency plugins
        $file_name .= self::get_currency_suffix();

        // Add language code based on language plugins
        $file_name .= self::get_language_suffix();

        // Add WooCommerce session if present
        $file_name .= isset($_COOKIE['woocommerce_items_in_cart']) ? '-cart' : '';

        // Add query parameters hash for cache variation (same logic as advanced-cache.php)
        $query_strings = array_diff_key($_GET, array_flip(array_merge(self::$default_ignore_queries, $config['cache_ignore_queries'])));
        $file_name .= !empty($query_strings) ? '-' . md5(serialize($query_strings)) : '';

        // Add custom filter for file name
        $file_name = apply_filters('zizi_cache_cache_file_name', $file_name);

        // Add .html.gz extension
        $file_name .= self::CACHE_FILE_EXTENSION;

        return $file_name;
    }
    
    /**
     * Gets the role suffix for the cache file name.
     *
     * @return string Role suffix.
     */
    private static function get_role_suffix()
    {
        return isset($_COOKIE[self::COOKIE_ROLE_NAME]) ? '-' . $_COOKIE[self::COOKIE_ROLE_NAME] : '';
    }
    
    /**
     * Gets the currency suffix for the cache file name based on active currency plugins.
     *
     * @return string Currency suffix.
     */
    private static function get_currency_suffix()
    {
        $suffix = '';
        
        // Add currency code to cache file name if Aelia Currency Switcher is active
        $suffix .= isset($_COOKIE['aelia_cs_selected_currency'])
            ? '-' . $_COOKIE['aelia_cs_selected_currency']
            : '';

        // Add currency code to cache file name if YITH Currency Switcher is active
        $suffix .= isset($_COOKIE['yith_wcmcs_currency'])
            ? '-' . $_COOKIE['yith_wcmcs_currency']
            : '';

        // Add currency code to cache file name if WCML Currency Switcher is active
        $suffix .= isset($_COOKIE['wcml_currency'])
            ? '-' . $_COOKIE['wcml_currency']
            : '';
            
        return $suffix;
    }
    
    /**
     * Gets the language suffix for the cache file name based on active language plugins.
     *
     * @return string Language suffix.
     */
    private static function get_language_suffix()
    {
        $suffix = '';
        
        // Add language code if Polylang is active
        $suffix .= isset($_COOKIE['pll_language'])
            ? '-' . $_COOKIE['pll_language']
            : '';

        // Add language code if WPML is active
        $suffix .= isset($_COOKIE['wp-wpml_current_language'])
            ? '-' . $_COOKIE['wp-wpml_current_language']
            : '';

        // Add language code if TranslatePress is active
        $suffix .= isset($_COOKIE['trp_language'])
            ? '-' . $_COOKIE['trp_language']
            : '';
            
        return $suffix;
    }

    /**
     * Determines if the current request/page is cacheable based on config, user, URL, cookies, and other factors.
     *
     * @param string $content HTML content to check for cacheability.
     * @return bool True if cacheable, false otherwise.
     */
    public static function is_cacheable($content)
    {
        $config = SysConfig::$config;
        $current_url = $_SERVER['REQUEST_URI'] ?? '';

        // Check several conditions - if any fails, the page is not cacheable
        
        // 1. Check cache lifespan setting
        if (!self::is_lifespan_cacheable($config)) {
            self::log_issue(__FUNCTION__, 'Lifespan is set to never.', ['config' => $config], 'WARNING');
            return false;
        }

        // 2. Check logged-in user setting
        if (!self::is_user_cacheable($config)) {
            self::log_issue(__FUNCTION__, 'Logged-in user not allowed for caching.', ['config' => $config], 'INFO');
            return false;
        }

        // 3. Check WordPress system URLs
        if (!self::is_url_cacheable($current_url)) {
            self::log_issue(__FUNCTION__, 'URL not cacheable.', ['url' => $current_url], 'INFO');
            return false;
        }

        // 4. Check request method
        if (!self::is_method_cacheable()) {
            self::log_issue(__FUNCTION__, 'Request method not GET.', ['method' => $_SERVER['REQUEST_METHOD'] ?? null], 'INFO');
            return false;
        }

        // 5. Check for AJAX requests
        if (!self::is_ajax_cacheable()) {
            self::log_issue(__FUNCTION__, 'AJAX request detected.', [], 'INFO');
            return false;
        }

        // 6. Check user role
        if (!self::is_role_cacheable()) {
            self::log_issue(__FUNCTION__, 'User role excluded from caching.', ['roles' => $_COOKIE[self::COOKIE_ROLE_NAME] ?? 'none'], 'INFO');
            return false;
        }

        // 7. Check bypass URLs
        if (!self::is_bypass_url_cacheable($config, $current_url)) {
            self::log_issue(__FUNCTION__, 'Bypass URL matched.', ['bypass_urls' => $config['cache_bypass_urls'], 'url' => $current_url], 'INFO');
            return false;
        }

        // 8. Check cookies
        if (!self::is_cookies_cacheable($config)) {
            self::log_issue(__FUNCTION__, 'Bypass cookie matched.', [], 'INFO');
            return false;
        }

        // 9. Check admin and response code
        if (is_admin() || http_response_code() !== 200) {
            self::log_issue(__FUNCTION__, 'Admin or non-200 response.', [], 'INFO');
            return false;
        }

        // 10. Check if content is HTML
        if (!self::is_content_html($content)) {
            self::log_issue(__FUNCTION__, 'Content is not HTML.', [], 'INFO');
            return false;
        }

        // 11. Check AMP
        if (self::is_amp_page()) {
            self::log_issue(__FUNCTION__, 'AMP endpoint detected.', [], 'INFO');
            return false;
        }

        // 12. Check password protection
        if (self::is_password_protected()) {
            self::log_issue(__FUNCTION__, 'Password protected post.', [], 'INFO');
            return false;
        }

        // 13. Allow plugins to filter
        return apply_filters('zizi_cache_is_cacheable', true);
    }

    /**
     * Checks if cache lifespan is enabled in configuration.
     *
     * @param array $config Plugin configuration array.
     * @return bool True if lifespan is enabled, false otherwise.
     */
    private static function is_lifespan_cacheable($config)
    {
        return isset($config['cache_lifespan']) && $config['cache_lifespan'] !== 'never';
    }

    /**
     * Checks if the current user is allowed to use cache.
     *
     * @param array $config Plugin configuration array.
     * @return bool True if user is cacheable, false otherwise.
     */
    private static function is_user_cacheable($config)
    {
        return $config['cache_logged_in'] || !is_user_logged_in();
    }

    /**
     * Checks if the current URL is eligible for caching.
     *
     * @param string $current_url Current request URL.
     * @return bool True if URL is cacheable, false otherwise.
     */
    private static function is_url_cacheable($current_url)
    {
        $regex = '/wp-(admin|login|register|comments-post|cron|json|sitemap)|\.(txt|xml)/';
        return !preg_match($regex, $current_url);
    }

    /**
     * Checks if the HTTP request method is GET.
     *
     * @return bool True if method is GET, false otherwise.
     */
    private static function is_method_cacheable()
    {
        return isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'GET';
    }

    /**
     * Checks if the current request is not an AJAX (wp_doing_ajax).
     *
     * @return bool True if not AJAX, false otherwise.
     */
    private static function is_ajax_cacheable()
    {
        return !wp_doing_ajax();
    }

    /**
     * Checks if the user's role is allowed for caching.
     *
     * @return bool True if role is cacheable, false otherwise.
     */
    private static function is_role_cacheable()
    {
        $excluded_roles = apply_filters('zizi_cache_cache_excluded_roles', []);
        $user_role = $_COOKIE[self::COOKIE_ROLE_NAME] ?? false;
        return !SysTool::any_keywords_match_string($excluded_roles, $user_role);
    }

    /**
     * Checks if the current URL is not on the bypass list for caching.
     *
     * @param array $config Plugin configuration array.
     * @param string $current_url Current request URL.
     * @return bool True if not bypassed, false otherwise.
     */
    private static function is_bypass_url_cacheable($config, $current_url)
    {
        return !SysTool::any_keywords_match_string($config['cache_bypass_urls'], $current_url);
    }

    /**
     * Checks if none of the bypass cookies are set for the current request.
     *
     * @param array $config Plugin configuration array.
     * @return bool True if cookies allow caching, false otherwise.
     */
    private static function is_cookies_cacheable($config)
    {
        foreach ($config['cache_bypass_cookies'] as $cookie) {
            foreach (array_keys($_COOKIE) as $cookieName) {
                if (stripos($cookieName, $cookie) !== false) {
                    return false;
                }
            }
        }
        return true;
    }
    
    /**
     * Checks if the content is HTML.
     *
     * @param string $content The content to check.
     * @return bool True if content is HTML, false otherwise.
     */
    private static function is_content_html($content)
    {
        return preg_match('/<!DOCTYPE\s*html\b[^>]*>/i', $content);
    }
    
    /**
     * Checks if the current page is an AMP page.
     *
     * @return bool True if AMP page, false otherwise.
     */
    private static function is_amp_page()
    {
        return function_exists('is_amp_endpoint') && is_amp_endpoint();
    }
    
    /**
     * Checks if the current post is password protected.
     *
     * @return bool True if password protected, false otherwise.
     */
    private static function is_password_protected()
    {
        return is_singular() && post_password_required();
    }

    /**
     * Enhanced cache page counting with smart caching and performance monitoring
     * 
     * Features:
     * - Smart caching with TTL for better performance
     * - Integration with PerformanceMonitor for error handling
     * - Optimized glob-based algorithm
     * - Only counts directories with actual cache files
     * 
     * @param string $path Cache directory path
     * @param bool $force_refresh Force refresh cached count
     * @return int Number of cached pages
     */
    public static function count_pages($path, $force_refresh = false)
    {
        // Performance monitoring start
        $start_time = microtime(true);
        
        // Smart caching - faster than competitors for repeated calls
        $cache_key = 'zizi_cache_pages_count_' . md5($path);
        
        if (!$force_refresh) {
            $cached = wp_cache_get($cache_key, 'zizi_cache');
            if ($cached !== false) {
                return (int) $cached;
            }
        }
        
        try {
            $count = self::_count_pages_optimized($path);
            
            // Cache for 2 minutes - balance between freshness and performance
            wp_cache_set($cache_key, $count, 'zizi_cache', 120);
            
            $elapsed = microtime(true) - $start_time;
            self::writeLog(sprintf('[ZiZi Cache Performance] count_pages took %.4f sec, result: %d', $elapsed, $count));
            
            return $count;
            
        } catch (\Exception $e) {
            // Use existing infrastructure for error handling
            self::writeLog(sprintf('[ZiZi Cache Error] count_pages failed: %s', $e->getMessage()));
            return 0;
        }
    }

    /**
     * Optimized internal counting method
     * Uses glob patterns for better performance than recursive iteration
     * 
     * @param string $path Cache directory path  
     * @return int Number of cached pages
     */
    private static function _count_pages_optimized($path)
    {
        if (!is_dir($path)) {
            throw new \Exception("Cache directory does not exist: {$path}");
        }
        
        // Use recursive glob for nested cache structure
        // This handles patterns like: /localhost/index.html.gz and /localhost/page/index.html.gz
        $cache_files = self::_recursive_glob($path, 'index' . self::CACHE_FILE_EXTENSION);
        
        if (empty($cache_files)) {
            return 0;
        }
        
        // Each cache file represents one cached page
        return count($cache_files);
    }
    
    /**
     * Recursive glob implementation for nested directory scanning
     * 
     * @param string $base_dir Base directory to search
     * @param string $pattern File pattern to match
     * @return array Array of matching file paths
     */
    private static function _recursive_glob($base_dir, $pattern)
    {
        $base_dir = rtrim($base_dir, '/');
        $files = [];
        
        // Search in current directory
        $current_files = glob($base_dir . '/' . $pattern);
        if ($current_files !== false) {
            $files = array_merge($files, $current_files);
        }
        
        // Search in subdirectories
        $subdirs = glob($base_dir . '/*', GLOB_ONLYDIR | GLOB_NOSORT);
        if ($subdirs !== false) {
            foreach ($subdirs as $subdir) {
                $subfiles = self::_recursive_glob($subdir, $pattern);
                $files = array_merge($files, $subfiles);
            }
        }
        
        return $files;
    }

    /**
     * Extended count with cache validation - ZiZi Cache exclusive feature
     * 
     * Provides detailed statistics about cache health and integrity.
     * Useful for admin dashboard and troubleshooting.
     * 
     * @param string $path Cache directory path
     * @param bool $validate_integrity Check cache file integrity
     * @return array Statistics with count and health metrics
     */
    public static function count_pages_extended($path, $validate_integrity = false)
    {
        $start_time = microtime(true);
        
        try {
            $basic_count = self::count_pages($path);
            
            $stats = [
                'count' => $basic_count,
                'validated' => false,
                'corrupted' => 0,
                'empty_dirs' => 0,
                'total_size_mb' => 0
            ];
            
            if ($validate_integrity && $basic_count > 0) {
                $stats = array_merge($stats, self::_validate_cache_integrity($path));
                $stats['validated'] = true;
            }
            
            $elapsed = microtime(true) - $start_time;
            self::writeLog(sprintf('[ZiZi Cache Stats] Extended count: %d pages, %.4f sec', $basic_count, $elapsed));
            
            return $stats;
            
        } catch (\Exception $e) {
            self::writeLog(sprintf('[ZiZi Cache Error] Extended count failed: %s', $e->getMessage()));
            return ['count' => 0, 'error' => $e->getMessage()];
        }
    }

    /**
     * Validate cache file integrity and collect detailed statistics
     * 
     * @param string $path Cache directory path
     * @return array Validation statistics
     */
    private static function _validate_cache_integrity($path)
    {
        $stats = [
            'corrupted' => 0,
            'empty_dirs' => 0,
            'total_size_mb' => 0,
            'oldest_file' => null,
            'newest_file' => null
        ];
        
        // Use recursive glob for file discovery - consistent with counting method
        $cache_files = self::_recursive_glob($path, 'index' . self::CACHE_FILE_EXTENSION);
        
        if (empty($cache_files)) {
            return $stats;
        }
        
        $oldest_time = PHP_INT_MAX;
        $newest_time = 0;
        
        foreach ($cache_files as $file) {
            // Check file size and readability
            $size = filesize($file);
            if ($size === false || $size === 0) {
                $stats['corrupted']++;
                continue;
            }
            
            $stats['total_size_mb'] += $size;
            
            // Track oldest and newest files
            $mtime = filemtime($file);
            if ($mtime < $oldest_time) {
                $oldest_time = $mtime;
                $stats['oldest_file'] = date('Y-m-d H:i:s', $mtime);
            }
            if ($mtime > $newest_time) {
                $newest_time = $mtime;
                $stats['newest_file'] = date('Y-m-d H:i:s', $mtime);
            }
        }
        
        // Convert bytes to MB
        $stats['total_size_mb'] = round($stats['total_size_mb'] / 1024 / 1024, 2);
        
        // Check for empty directories using a simpler approach
        // Count directories that exist but don't have corresponding cache files
        $cache_dir_paths = array_unique(array_map('dirname', $cache_files));
        $all_dirs = self::_get_all_subdirectories($path);
        $stats['empty_dirs'] = count($all_dirs) - count($cache_dir_paths);
        
        return $stats;
    }
    
    /**
     * Get all subdirectories recursively
     * 
     * @param string $path Base directory path
     * @return array Array of directory paths
     */
    private static function _get_all_subdirectories($path)
    {
        $dirs = [];
        $subdirs = glob($path . '/*', GLOB_ONLYDIR | GLOB_NOSORT);
        
        if ($subdirs !== false) {
            foreach ($subdirs as $subdir) {
                $dirs[] = $subdir;
                $subdirs_recursive = self::_get_all_subdirectories($subdir);
                $dirs = array_merge($dirs, $subdirs_recursive);
            }
        }
        
        return $dirs;
    }

    /**
     * Returns the absolute file path for a given URL.
     *
     * @param string $url URL to resolve to file path.
     * @return string Absolute file path.
     */
    public static function get_file_path_from_url($url)
    {
        $file_relative_path = parse_url($url, PHP_URL_PATH);
        $site_path = parse_url(site_url(), PHP_URL_PATH);
        $file_path = file_exists(ABSPATH . preg_replace("$^$site_path$", '', $file_relative_path))
            ? ABSPATH . preg_replace("$^$site_path$", '', $file_relative_path)
            : $_SERVER['DOCUMENT_ROOT'] . preg_replace("$^$site_path$", '', $file_relative_path);
        return $file_path;
    }

    /**
     * Adds the logged-in user's role to a cookie for cache differentiation.
     *
     * @param string $logged_in_cookie Cookie value for logged-in user.
     * @param int $expire Expiry timestamp for the cookie.
     * @param int $expiration Expiration value for the cookie.
     * @param int $user_id User ID.
     * @return void
     */
    public static function add_logged_in_roles($logged_in_cookie, $expire, $expiration, $user_id)
    {
        // Get the user
        $user = get_user_by('ID', $user_id);

        if (!$user) {
            return;
        }

        if (!isset($_COOKIE[self::COOKIE_ROLE_NAME])) {
            $user_role = implode('-', $user->roles);
            $expiry = time() + 14 * DAY_IN_SECONDS;
            setcookie(self::COOKIE_ROLE_NAME, $user_role, $expiry, COOKIEPATH, COOKIE_DOMAIN, false);
        }
    }

    /**
     * Removes the logged-in roles cookie when the user logs out.
     *
     * @return void
     */
    public static function remove_logged_in_roles()
    {
        if (isset($_COOKIE[self::COOKIE_ROLE_NAME])) {
            unset($_COOKIE[self::COOKIE_ROLE_NAME]);
            setcookie(self::COOKIE_ROLE_NAME, '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, false);
        } else {
            self::log_issue(__FUNCTION__, 'Tried to remove zc_logged_in_roles cookie, but it was not set.', [], 'WARNING');
        }
    }

    /**
     * Cleans up the roles cookie if the user is no longer logged in.
     *
     * @return void
     */
    public static function cleanup_roles_cookie()
    {
        if (isset($_COOKIE[self::COOKIE_ROLE_NAME]) && !is_user_logged_in()) {
            unset($_COOKIE[self::COOKIE_ROLE_NAME]);
            setcookie(self::COOKIE_ROLE_NAME, '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, false);
        } elseif (isset($_COOKIE[self::COOKIE_ROLE_NAME]) && is_user_logged_in()) {
            self::log_issue(__FUNCTION__, 'zc_logged_in_roles cookie exists but user IS logged in – not cleaning up.', [], 'WARNING');
        }
    }
}