<?php

declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\CacheSys;
use ZiziCache\SysConfig;
use ZiziCache\SysTool;
use ZiziCache\HTML;
use ZiziCache\Font;
use Sabberworm\CSS\CSSList\AtRuleBlockList;
use Sabberworm\CSS\CSSList\CSSBlockList;
use Sabberworm\CSS\CSSList\Document;
use Sabberworm\CSS\OutputFormat;
use Sabberworm\CSS\Parser as CSSParser;
use Sabberworm\CSS\Property\Charset;
use Sabberworm\CSS\RuleSet\DeclarationBlock;
use Sabberworm\CSS\Settings;
use Sabberworm\CSS\Value\URL;

/**
 * Enhanced CSS Optimization and Management with Sabberworm Parser
 * 
 * This class provides comprehensive CSS optimization features including:
 * - Advanced CSS parsing using Sabberworm CSS Parser
 * - Critical CSS generation and injection
 * - Removal of unused CSS (RUCSS)
 * - CSS file minification and combination
 * - Google Fonts optimization with critical CSS exclusion
 * 
 * @package ZiziCache
 */
class CSSEnhanced
{
  /**
   * Used selectors found in HTML
   */
  private static $used_selectors = [];

  /**
   * Excluded selectors that should always be kept
   */
  private static $excluded_selectors = [];

  /**
   * Current URL type for caching
   */
  private static $url_type = '';

  /**
   * Critical CSS content
   */
  private static $critical_css = '';

  /**
   * Initialize the enhanced CSS optimization features
   * 
   * @return void
   */
  public static function init(): void
  {
    // Initialize filters and actions
    add_filter('zizi_cache_download_external_file:before', [__CLASS__, 'self_host_third_party_fonts'], 10, 3);
    
    // Add action for critical CSS injection
    if (!empty(SysConfig::$config['css_critical'])) {
      add_action('wp_head', [__CLASS__, 'inject_critical_css'], 1);
    }
  }

  /**
   * Main CSS optimization method with enhanced Sabberworm parsing
   * 
   * @param string $html The HTML content to optimize
   * @return string Optimized HTML
   */
  public static function optimize(string $html): string
  {
    if (empty($html)) {
      return $html;
    }

    // Skip if CSS optimization is disabled
    if (empty(SysConfig::$config['css_minify']) && empty(SysConfig::$config['css_rucss']) && empty(SysConfig::$config['css_critical'])) {
      return $html;
    }

    // Get URL type for caching
    self::$url_type = self::get_url_type();
    
    // Strip comments before processing
    $html_no_comments = preg_replace('/<!--(.*)-->/Uis', '', $html);
    
    // Match all stylesheets
    preg_match_all('#<link(\s[^>]*?href=[\'"]([^\'"]+?\.css.*?)[\'"][^>]*?)\/?>#i', $html_no_comments, $stylesheets, PREG_SET_ORDER);

    if (empty($stylesheets)) {
      return $html;
    }

    // Create cache directory
    $cache_dir = ZIZI_CACHE_CACHE_DIR . 'css/';
    if (!is_dir($cache_dir)) {
      wp_mkdir_p($cache_dir);
    }

    // Get used selectors from HTML
    self::get_used_selectors($html);
    self::get_excluded_selectors();

    $critical_css_content = '';
    $processed_stylesheets = [];

    // Process each stylesheet
    foreach ($stylesheets as $stylesheet) {
      $atts_array = SysTool::get_atts_array($stylesheet[1] ?? '');
      
      // Skip non-stylesheet links
      if (empty($atts_array['rel']) || $atts_array['rel'] !== 'stylesheet') {
        continue;
      }

      $href = $atts_array['href'] ?? '';
      if (empty($href)) {
        continue;
      }

      // Skip excluded stylesheets
      if (self::is_stylesheet_excluded($href)) {
        continue;
      }

      // Process critical CSS
      if (!empty(SysConfig::$config['css_critical'])) {
        $critical_css = self::extract_critical_css($href, $atts_array);
        if (!empty($critical_css)) {
          $critical_css_content .= $critical_css;
        }
      }

      // Process RUCSS
      if (!empty(SysConfig::$config['css_rucss'])) {
        $html = self::process_rucss_stylesheet($html, $stylesheet[0], $atts_array);
      }

      $processed_stylesheets[] = $href;
    }

    // Inject critical CSS
    if (!empty($critical_css_content) && !empty(SysConfig::$config['css_critical'])) {
      self::$critical_css = $critical_css_content;
      $html = self::inject_critical_css_inline($html, $critical_css_content);
    }

    return $html;
  }

  /**
   * Extract critical CSS from stylesheet using Sabberworm parser
   * 
   * @param string $href Stylesheet URL
   * @param array $atts_array Stylesheet attributes
   * @return string Critical CSS content
   */
  private static function extract_critical_css(string $href, array $atts_array): string
  {
    // Skip Google Fonts if configured
    if (!empty(SysConfig::$config['css_critical_exclude_google_fonts']) && self::is_google_font($href)) {
      return '';
    }

    // Get local file path
    $file_path = CacheSys::get_file_path_from_url($href);
    if (empty($file_path) || !is_file($file_path)) {
      return '';
    }

    // Security: Check file size
    $file_size = filesize($file_path);
    if ($file_size === false || $file_size > 2 * 1024 * 1024) {
      CacheSys::writeWarning("CSS file too large for critical CSS: {$file_path} ({$file_size} bytes)");
      return '';
    }

    $css_content = file_get_contents($file_path);
    if ($css_content === false || empty($css_content)) {
      return '';
    }

    try {
      // Parse CSS with Sabberworm
      $critical_css = self::parse_critical_css_with_sabberworm($css_content, $href);
      
      // Wrap in media query if needed
      if (!empty($atts_array['media']) && $atts_array['media'] !== 'all') {
        $critical_css = "@media {$atts_array['media']} { {$critical_css} }";
      }

      return $critical_css;
      
    } catch (\Exception $e) {
      CacheSys::writeError('Critical CSS parsing error: ' . $e->getMessage() . ' for URL: ' . $href);
      return '';
    }
  }

  /**
   * Parse CSS with Sabberworm parser to extract critical CSS
   * 
   * @param string $css_content Raw CSS content
   * @param string $stylesheet_url Stylesheet URL for relative path resolution
   * @return string Critical CSS
   */
  private static function parse_critical_css_with_sabberworm(string $css_content, string $stylesheet_url): string
  {
    // Remove BOM if present
    $css_content = preg_replace('/^\xEF\xBB\xBF/', '', $css_content);

    // Setup CSS parser with settings
    $settings = Settings::create()
      ->withMultibyteSupport(false)
      ->withLenientParsing(true);
    
    $parser = new CSSParser($css_content, $settings);
    $parsed_css = $parser->parse();

    // Fix relative URLs
    self::fix_relative_urls($stylesheet_url, $parsed_css);

    // Prepare CSS data for processing
    $css_data = self::prep_css_data($parsed_css);

    // Extract only critical CSS rules
    return self::extract_critical_selectors($css_data);
  }

  /**
   * Fix relative URLs in parsed CSS
   * 
   * @param string $stylesheet_url Base stylesheet URL
   * @param Document $parsed_css Parsed CSS document
   */
  private static function fix_relative_urls(string $stylesheet_url, Document $parsed_css): void
  {
    $base_url = preg_replace('#[^/]+(\?.*)?$#', '', $stylesheet_url);
    
    $values = $parsed_css->getAllValues();
    foreach ($values as $value) {
      if (!($value instanceof URL)) {
        continue;
      }

      $url = $value->getURL()->getString();

      // Skip absolute URLs and data URIs
      if (preg_match('/^(https?|data):/', $url)) {
        continue;
      }

      $parsed_url = parse_url($url);

      // Skip if already absolute or invalid
      if (!empty($parsed_url['host']) || empty($parsed_url['path']) || $parsed_url['path'][0] === '/') {
        continue;
      }

      // Convert to absolute URL
      $new_url = $base_url . $url;
      $value->getUrl()->setString($new_url);
    }
  }

  /**
   * Prepare parsed CSS data for processing
   * 
   * @param CSSBlockList $data Parsed CSS data
   * @return array Prepared CSS data
   */
  private static function prep_css_data(CSSBlockList $data): array
  {
    $items = [];

    foreach ($data->getContents() as $content) {
      // Skip charset declarations for inline CSS
      if ($content instanceof Charset) {
        continue;
      }

      if ($content instanceof AtRuleBlockList) {
        $items[] = [
          'rulesets' => self::prep_css_data($content),
          'at_rule' => "@{$content->atRuleName()} {$content->atRuleArgs()}",
        ];
      } else {
        $item = ['css' => $content->render(OutputFormat::createCompact())];

        if ($content instanceof DeclarationBlock) {
          $item['selectors'] = self::sort_selectors($content->getSelectors());
        }

        $items[] = $item;
      }
    }

    return $items;
  }

  /**
   * Sort selectors into categories for analysis
   * 
   * @param array $selectors CSS selectors
   * @return array Sorted selectors
   */
  private static function sort_selectors(array $selectors): array
  {
    $selectors = array_map(function($sel) {
      return $sel->__toString();
    }, $selectors);

    $selectors_data = [];
    foreach ($selectors as $selector) {
      $data = [
        'selector' => trim($selector),
        'classes' => [],
        'ids' => [],
        'tags' => [],
        'atts' => []
      ];

      // Remove pseudo-classes and pseudo-elements for analysis
      $clean_selector = preg_replace('/(?<!\\\\)::?[a-zA-Z0-9_-]+(\(.+?\))?/', '', $selector);

      // Extract attributes
      $clean_selector = preg_replace_callback(
        '/\[([A-Za-z0-9_:-]+)(\W?=[^\]]+)?\]/',
        function($matches) use (&$data) {
          $data['atts'][] = $matches[1];
          return '';
        },
        $clean_selector
      );

      // Extract classes
      $clean_selector = preg_replace_callback(
        '/\.((?:[a-zA-Z0-9_-]+|\\\\.)+)/',
        function($matches) use (&$data) {
          $data['classes'][] = stripslashes($matches[1]);
          return '';
        },
        $clean_selector
      );

      // Extract IDs
      $clean_selector = preg_replace_callback(
        '/#([a-zA-Z0-9_-]+)/',
        function($matches) use (&$data) {
          $data['ids'][] = $matches[1];
          return '';
        },
        $clean_selector
      );

      // Extract tags
      $clean_selector = preg_replace_callback(
        '/[a-zA-Z0-9_-]+/',
        function($matches) use (&$data) {
          $data['tags'][] = $matches[0];
          return '';
        },
        $clean_selector
      );

      $selectors_data[] = array_filter($data);
    }

    return array_filter($selectors_data);
  }

  /**
   * Extract critical CSS selectors (above-the-fold content)
   * 
   * @param array $css_data Prepared CSS data
   * @return string Critical CSS
   */
  private static function extract_critical_selectors(array $css_data): string
  {
    $critical_css = [];

    foreach ($css_data as $item) {
      if (isset($item['css'])) {
        // Check if this selector is critical (above-the-fold)
        $is_critical = !isset($item['selectors']) || count(array_filter($item['selectors'], function($selector) {
          return self::is_selector_critical($selector);
        })) > 0;

        if ($is_critical) {
          $critical_css[] = $item['css'];
        }
        continue;
      }

      // Handle nested rulesets (media queries, etc.)
      if (!empty($item['rulesets'])) {
        $child_critical = self::extract_critical_selectors($item['rulesets']);
        if ($child_critical) {
          $critical_css[] = sprintf('%s{%s}', $item['at_rule'], $child_critical);
        }
      }
    }

    return implode('', $critical_css);
  }

  /**
   * Check if a selector is critical (likely above-the-fold)
   * 
   * @param array $selector Selector data
   * @return bool True if critical
   */
  private static function is_selector_critical(array $selector): bool
  {
    // Always include :root and other foundational selectors
    if ($selector['selector'] === ':root' || $selector['selector'] === 'html' || $selector['selector'] === 'body') {
      return true;
    }

    // Critical tags (typically visible above-the-fold)
    $critical_tags = ['html', 'body', 'header', 'nav', 'h1', 'h2', 'h3', 'main', 'section', 'article'];
    
    // Critical classes (common above-the-fold patterns)
    $critical_class_patterns = [
      'header', 'nav', 'menu', 'logo', 'hero', 'banner', 'intro', 'featured',
      'top', 'above', 'fold', 'primary', 'main', 'content', 'title', 'heading'
    ];

    // Check for critical tags
    if (!empty($selector['tags'])) {
      foreach ($selector['tags'] as $tag) {
        if (in_array(strtolower($tag), $critical_tags)) {
          return true;
        }
      }
    }

    // Check for critical class patterns
    if (!empty($selector['classes'])) {
      foreach ($selector['classes'] as $class) {
        foreach ($critical_class_patterns as $pattern) {
          if (stripos($class, $pattern) !== false) {
            return true;
          }
        }
      }
    }

    // Check if selector is used in the current HTML
    return self::is_selector_used($selector);
  }

  /**
   * Check if selector is used in current HTML
   * 
   * @param array $selector Selector data
   * @return bool True if used
   */
  private static function is_selector_used(array $selector): bool
  {
    // Check excluded selectors first
    if (!empty(self::$excluded_selectors)) {
      foreach (self::$excluded_selectors as $excluded) {
        if (preg_match('#(' . preg_quote($excluded) . ')(?=\s|\.|\:|,|\[|$)#', $selector['selector'])) {
          return true;
        }
      }
    }

    // Check each selector component
    foreach (['classes', 'ids', 'tags'] as $type) {
      if (!empty($selector[$type])) {
        $targets = (array)$selector[$type];
        foreach ($targets as $target) {
          if (!isset(self::$used_selectors[$type][$target])) {
            return false;
          }
        }
      }
    }

    return true;
  }

  /**
   * Process RUCSS for a stylesheet
   * 
   * @param string $html HTML content
   * @param string $stylesheet_tag Full stylesheet tag
   * @param array $atts_array Stylesheet attributes
   * @return string Modified HTML
   */
  private static function process_rucss_stylesheet(string $html, string $stylesheet_tag, array $atts_array): string
  {
    $href = $atts_array['href'] ?? '';
    
    // Skip if excluded or Google Fonts
    if (self::is_stylesheet_excluded($href) || self::is_google_font($href)) {
      return $html;
    }

    // Check if page builder is active - if so, skip async loading for compatibility
    $is_page_builder_active = false;
    if (class_exists('ZiziCache\Plugins\Integrations\PageBuilders')) {
      $is_page_builder_active = \ZiziCache\Plugins\Integrations\PageBuilders::is_page_builder_active();
    }

    if ($is_page_builder_active) {
      // Page builder mode - keep CSS loading synchronously for compatibility
      return $html;
    }

    // Modify stylesheet loading based on method
    $method = SysConfig::$config['css_rucss_method'] ?? 'async';
    
    switch ($method) {
      case 'async':
        // Load asynchronously
        $atts_array['media'] = 'print';
        $atts_array['onload'] = "this.media='all';this.onload=null;";
        break;
        
      case 'interaction':
        // Load on user interaction
        $atts_array['data-href'] = $atts_array['href'];
        unset($atts_array['href']);
        $atts_array['rel'] = 'preload';
        $atts_array['as'] = 'style';
        $atts_array['class'] = trim(($atts_array['class'] ?? '') . ' zizi-deferred-css');
        break;
        
      case 'domcontentloaded':
        // Load on DOMContentLoaded
        $atts_array['data-dcl-href'] = $atts_array['href'];
        unset($atts_array['href']);
        $atts_array['rel'] = 'preload';
        $atts_array['as'] = 'style';
        $atts_array['class'] = trim(($atts_array['class'] ?? '') . ' zizi-dcl-css');
        break;
        
      case 'remove':
        // Remove stylesheet entirely
        return str_replace($stylesheet_tag, '', $html);
    }

    // Replace stylesheet tag
    $new_atts_string = SysTool::get_atts_string($atts_array);
    $new_link = sprintf('<link %s>', $new_atts_string);
    return str_replace($stylesheet_tag, $new_link, $html);
  }

  /**
   * Inject critical CSS inline into HTML head
   * 
   * @param string $html HTML content
   * @param string $critical_css Critical CSS content
   * @return string Modified HTML
   */
  private static function inject_critical_css_inline(string $html, string $critical_css): string
  {
    if (empty($critical_css)) {
      return $html;
    }

    // Minify critical CSS
    $critical_css = self::minify_css($critical_css);
    
    // Create critical CSS style tag
    $critical_style = sprintf(
      '<style id="zizi-critical-css">%s</style>',
      $critical_css
    );

    // Inject right after <head> tag
    $pos = strpos($html, '<head>');
    if ($pos !== false) {
      $pos += 6; // Length of '<head>'
      $html = substr_replace($html, $critical_style, $pos, 0);
    }

    return $html;
  }

  /**
   * Get used selectors from HTML
   * 
   * @param string $html HTML content
   */
  private static function get_used_selectors(string $html): void
  {
    if (empty($html)) {
      return;
    }

    $libxml_previous = libxml_use_internal_errors(true);
    $dom = new \DOMDocument();
    $result = $dom->loadHTML($html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOWARNING | LIBXML_NOERROR);
    libxml_clear_errors();
    libxml_use_internal_errors($libxml_previous);

    if (!$result) {
      return;
    }

    self::$used_selectors = ['tags' => [], 'ids' => [], 'classes' => []];

    foreach ($dom->getElementsByTagName('*') as $tag) {
      // Add tag
      self::$used_selectors['tags'][$tag->tagName] = 1;

      // Add ID
      if ($tag->hasAttribute('id')) {
        self::$used_selectors['ids'][$tag->getAttribute('id')] = 1;
      }

      // Add classes
      if ($tag->hasAttribute('class')) {
        $classes = preg_split('/\s+/', $tag->getAttribute('class'));
        foreach ($classes as $class) {
          if (!empty($class)) {
            self::$used_selectors['classes'][$class] = 1;
          }
        }
      }
    }
  }

  /**
   * Get excluded selectors that should always be preserved
   */
  private static function get_excluded_selectors(): void
  {
    // Base excluded selectors
    self::$excluded_selectors = [
      '.ast-header-break-point', // Astra theme
      '.elementor-popup-modal', // Elementor
      '.elementor-has-item-ratio',
      '#elementor-device-mode',
      '.elementor-sticky--active',
      '.dialog-type-lightbox',
      '.dialog-widget-content',
      '.lazyloaded',
      '.elementor-nav-menu',
      '.elementor-motion-effects-container',
      '.elementor-motion-effects-layer',
      '.animated',
      '.elementor-animated-content',
      '.splide-initialized', // Splide slider
      '.splide',
      '.splide-slider',
      '.kb-splide', // Kadence
      '.dropdown-nav-special-toggle',
      'rs-fw-forcer', // Revolution Slider
      '#altEmail_container', // WP Armour
      // Enhanced exclusions inspired by Flying Press 5.0.6 and Perfmatters improvements
      '.bricks-popup', // Bricks Builder popups
      '.brxe-popup', // Bricks element popup
      '.breakdance-popup', // Breakdance popups
      '.brizy-popup', // Brizy popups
      '.thrive-lightbox', // Thrive Architect lightboxes
      '.thrv_responsive_video', // Thrive video elements
      '.tcb-lightbox', // Thrive Architect lightbox
      '.zion-popup', // Zion Builder popups
      '.lc-popup', // Live Composer popups
      '.tb-popup', // Themify Builder popups
      '.sow-carousel', // SiteOrigin widgets carousel
      '.wp-block-cover', // WordPress cover blocks
      '.wp-block-group', // WordPress group blocks
      '.wp-block-columns', // WordPress columns
      '.is-style-', // WordPress block styles
      '.has-', // WordPress block color/typography classes
      '.wc-block-', // WooCommerce blocks
      '.widget-', // Widget classes
      '.menu-item-', // Navigation menu items
      '.current-menu-', // Current menu states
      '.sticky', // Sticky elements
      '.fixed', // Fixed positioning
      '.absolute', // Absolute positioning
      '.hidden', // Hidden elements
      '.invisible', // Invisible elements
      '.opacity-', // Opacity utilities
      '.z-', // Z-index utilities
      '.transform', // Transform utilities
      '.transition', // Transition utilities
      '.duration-', // Animation duration utilities
      '.ease-', // Animation easing utilities
      '.animate-', // Animation utilities
      '.hover\\:', // Hover state utilities
      '.focus\\:', // Focus state utilities
      '.active\\:', // Active state utilities
      '.group-hover\\:', // Group hover utilities
      '.dark\\:', // Dark mode utilities
      '.sm\\:', // Small screen utilities
      '.md\\:', // Medium screen utilities
      '.lg\\:', // Large screen utilities
      '.xl\\:', // Extra large screen utilities
      '.2xl\\:', // 2XL screen utilities
      '@media', // Media queries
      '@keyframes', // Keyframe animations
      '@supports', // Feature queries
      '[data-', // Data attribute selectors
      '[aria-', // ARIA attribute selectors
      ':nth-child', // Pseudo-class selectors
      ':nth-of-type',
      ':first-child',
      ':last-child',
      ':only-child',
      ':hover',
      ':focus',
      ':active',
      ':visited',
      ':checked',
      ':disabled',
      ':invalid',
      ':valid',
      '::before',
      '::after',
      '::placeholder'
    ];

    // Add custom excluded selectors from config
    if (!empty(SysConfig::$config['css_rucss_include_selectors'])) {
      self::$excluded_selectors = array_merge(
        self::$excluded_selectors,
        SysConfig::$config['css_rucss_include_selectors']
      );
    }

    // Apply filter
    self::$excluded_selectors = apply_filters('zizi_cache_css_excluded_selectors', self::$excluded_selectors);
  }

  /**
   * Check if stylesheet should be excluded from processing
   * 
   * @param string $href Stylesheet URL
   * @return bool True if excluded
   */
  private static function is_stylesheet_excluded(string $href): bool
  {
    $excluded_patterns = [
      'dashicons.min.css', // WordPress core
      '/uploads/elementor/css/post-', // Elementor
      'animations.min.css',
      '/animations/',
      '/uploads/oxygen/css/', // Oxygen
      '/uploads/bb-plugin/cache/', // Beaver Builder
      '/uploads/generateblocks/', // GenerateBlocks
      '/uploads/bricks/css/', // Bricks Builder CSS (Enhanced)
      '/uploads/breakdance/css/', // Breakdance Builder CSS
      '/uploads/brizy/', // Brizy Builder  
      '/uploads/thrive/', // Thrive Architect
      '/et-cache/', // Divi
      '/widget-google-reviews/assets/css/public-main.css',
      '/wp-content/uploads/bricks/css/post-', // Bricks (legacy)
      // Theme-specific exclusions (NEW - Enhanced)
      '/themes/astra/assets/fonts/', // Astra local fonts (from Perfmatters changelog)
      '/themes/elementor/assets/fonts/', // Elementor local fonts (from Perfmatters changelog)
      '/themes/generatepress/assets/fonts/', // GeneratePress fonts
      '/themes/oceanwp/assets/fonts/', // OceanWP fonts
      '/themes/neve/assets/fonts/', // Neve fonts
      '/themes/kadence/assets/fonts/', // Kadence fonts
      '/themes/blocksy/static/fonts/', // Blocksy fonts
      // Additional competitive exclusions inspired by Flying Press 5.0.6
      '/themes/hello-elementor/', // Hello Elementor theme
      '/themes/storefront/', // WooCommerce Storefront
      '/themes/twentytwentyone/', // WordPress default themes
      '/themes/twentytwentytwo/',
      '/themes/twentytwentythree/',
      '/themes/twentytwentyfour/',
      '/plugins/woocommerce/assets/css/', // WooCommerce core styles
      '/plugins/contact-form-7/includes/css/', // Contact Form 7
      '/plugins/wp-rocket/', // WP Rocket compatibility
      '/plugins/litespeed-cache/', // LiteSpeed Cache compatibility
      '/plugins/w3-total-cache/', // W3 Total Cache compatibility
      '/plugins/wp-super-cache/', // WP Super Cache compatibility
      '/plugins/autoptimize/', // Autoptimize compatibility
      '/plugins/fast-velocity-minify/', // Fast Velocity Minify compatibility
      '/plugins/wp-fastest-cache/', // WP Fastest Cache compatibility
      '/plugins/sg-cachepress/', // SiteGround caching compatibility
      '/plugins/swift-performance/', // Swift Performance compatibility
      '/plugins/perfmatters/', // Perfmatters compatibility
      '/plugins/flying-press/', // Flying Press compatibility
      // Critical CSS patterns (never optimize these)
      'critical.css',
      'above-the-fold.css',
      'hero.css',
      'header.css',
      'navigation.css',
      'menu.css',
      'logo.css',
      // Font loading exclusions
      'font-display.css',
      'fonts.css',
      'typography.css',
      'google-fonts.css',
      'typekit.css',
      'adobe-fonts.css',
      // Animation and interaction exclusions
      'gsap.css',
      'swiper.css',
      'slick.css',
      'carousel.css',
      'slider.css',
      'modal.css',
      'popup.css',
      'tooltip.css',
      'dropdown.css'
    ];

    // Add custom exclusions
    if (!empty(SysConfig::$config['css_rucss_exclude_stylesheets'])) {
      $excluded_patterns = array_merge($excluded_patterns, SysConfig::$config['css_rucss_exclude_stylesheets']);
    }

    // Add critical CSS exclusions
    if (!empty(SysConfig::$config['css_critical_exclude_stylesheets'])) {
      $excluded_patterns = array_merge($excluded_patterns, SysConfig::$config['css_critical_exclude_stylesheets']);
    }

    foreach ($excluded_patterns as $pattern) {
      if (strpos($href, $pattern) !== false) {
        return true;
      }
    }

    return false;
  }

  /**
   * Check if URL is a Google Font
   * 
   * @param string $href URL to check
   * @return bool True if Google Font
   */
  private static function is_google_font(string $href): bool
  {
    $google_font_patterns = [
      'fonts.googleapis.com',
      'fonts.gstatic.com',
      '.google-fonts.',
      '.google-font.'
    ];

    foreach ($google_font_patterns as $pattern) {
      if (strpos($href, $pattern) !== false) {
        return true;
      }
    }

    return false;
  }

  /**
   * Get URL type for caching
   * 
   * @return string URL type
   */
  private static function get_url_type(): string
  {
    global $wp_query;

    if (!isset($wp_query)) {
      return 'unknown';
    }

    if ($wp_query->is_page) {
      return is_front_page() ? 'front' : (!empty($wp_query->post) ? 'page-' . $wp_query->post->ID : 'page');
    } elseif ($wp_query->is_home) {
      return 'home';
    } elseif ($wp_query->is_single) {
      return get_post_type() !== false ? get_post_type() : 'single';
    } elseif ($wp_query->is_category) {
      return 'category';
    } elseif ($wp_query->is_tag) {
      return 'tag';
    } elseif ($wp_query->is_tax) {
      return 'tax';
    } elseif ($wp_query->is_archive) {
      return $wp_query->is_post_type_archive() ? 'archive-' . get_post_type() : 'archive';
    } elseif ($wp_query->is_search) {
      return 'search';
    } elseif ($wp_query->is_404) {
      return '404';
    }

    return 'unknown';
  }

  /**
   * Minify CSS content
   * 
   * @param string $css CSS content
   * @return string Minified CSS
   */
  private static function minify_css(string $css): string
  {
    // Remove comments
    $css = preg_replace('!/\*.*?\*/!s', '', $css);
    
    // Remove unnecessary whitespace
    $css = preg_replace('/\s+/', ' ', $css);
    
    // Remove whitespace around specific characters
    $css = preg_replace('/\s*([{}:;,>+~])\s*/', '$1', $css);
    
    // Remove trailing semicolon before closing brace
    $css = str_replace(';}', '}', $css);
    
    return trim($css);
  }

  /**
   * Action hook to inject critical CSS
   */
  public static function inject_critical_css(): void
  {
    if (!empty(self::$critical_css)) {
      echo '<style id="zizi-critical-css">' . self::$critical_css . '</style>';
    }
  }

  /**
   * Clear critical CSS cache
   */
  public static function clear_critical_css_cache(): void
  {
    $cache_dir = ZIZI_CACHE_CACHE_DIR . 'css/';
    if (is_dir($cache_dir)) {
      $files = glob($cache_dir . '*.critical.css');
      foreach ($files as $file) {
        if (is_file($file)) {
          unlink($file);
        }
      }
    }
  }

  /**
   * Self-host third-party fonts (inherited method)
   */
  public static function self_host_third_party_fonts(string $content, string $url, string $extension): string
  {
    if ($extension !== 'css' || empty($content)) {
      return $content;
    }

    // Use existing implementation from original CSS class
    return CSS::self_host_third_party_fonts($content, $url, $extension);
  }
}
