<?php

declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\CacheSys;
use ZiziCache\SysConfig;
use ZiziCache\SysTool;
use Wa72\Url\Url;
use MatthiasMullie\Minify;

/**
 * CSS Optimization and Management
 * 
 * This class provides comprehensive CSS optimization features including:
 * - Minification of CSS files
 * - Removal of unused CSS (RUCSS - Remove Unused CSS)
 * - Critical CSS generation and injection
 * - CSS file combination and minification
 * - Media query optimization
 * - Font display optimization
 * 
 * @package ZiziCache
 */
class CSS
{
  /**
   * Initialize the CSS optimization features
   * 
   * Sets up the necessary filters and actions for CSS optimization.
   * 
   * @return void
   */
  public static function init(): void
  {
    add_filter(
      'zizi_cache_download_external_file:before',
      [__CLASS__, 'self_host_third_party_fonts'],
      10,
      3
    );

    // Initialize enhanced CSS optimization if Sabberworm is available
    if (class_exists('Sabberworm\CSS\Parser')) {
      CSSEnhanced::init();
    }
  }

  /**
   * Minifies CSS files in the provided HTML
   * 
   * Processes all linked stylesheets in the HTML, minifies them, and updates the references.
   * 
   * @param string $html The HTML content containing CSS files to minify
   * @return string The HTML with minified CSS files
   * @throws \Exception If minification fails
   */
  public static function minify(string $html): string
  {
    if (!SysConfig::$config['css_minify']) {
      return $html;
    }

        // Find all <link rel="stylesheet"> tags in the HTML
        preg_match_all("/<link[^>]*\srel=['\"]stylesheet['\"][^>]*>/", $html, $stylesheets);

        if (empty($stylesheets[0])) {
            return $html;
        }

        // Get excluded keywords from filter
        $exclude_keywords = apply_filters('zizi_cache_exclude_from_minify:css', []);
        try {
            foreach ($stylesheets[0] as $stylesheet_tag) {
                // Skip if any of the exclude keywords are in the tag
                if (!empty($exclude_keywords) && SysTool::any_keywords_match_string($exclude_keywords, $stylesheet_tag)) {
                    continue;
                }

                $stylesheet = new HTML($stylesheet_tag);
                $href = $stylesheet->href;

                if (empty($href)) {
                    continue;
                }
                // Convert relative path to absolute path
                $file_path = CacheSys::get_file_path_from_url($href);
                if (empty($file_path) || !is_file($file_path)) {
                    continue;
                }
                
                // Security: Check file size to prevent memory exhaustion attacks
                $file_size = filesize($file_path);
                if ($file_size === false || $file_size > 2 * 1024 * 1024) { // 2MB limit
                    CacheSys::writeWarning("CSS file too large or unreadable: {$file_path} ({$file_size} bytes)");
                    continue;
                }
                
                $css = file_get_contents($file_path);
                if ($css === false || empty($css)) {
                    continue;
                }
                // Generate hash based on the CSS content and CDN URL
                // If CDN URL changes, a new hash will be generated
                $hash =
                    !empty(SysConfig::$config['cdn']) && SysConfig::$config['cdn_type'] === 'custom' && !empty(SysConfig::$config['cdn_url'])
                        ? md5($css . SysConfig::$config['cdn_url'])
                        : md5($css);

                $file_name = substr($hash, 0, 12) . '.' . basename($file_path);
                $minified_path = ZIZI_CACHE_CACHE_DIR . $file_name;
                $minified_url = ZIZI_CACHE_CACHE_URL . $file_name;

                if (!is_file($minified_path)) {
                    $minifier = new Minify\CSS($css);
                    $minified_css = $minifier->minify();
                    $minified_css = self::rewrite_absolute_urls($minified_css, $href);
                    $minified_css = Font::inject_display_swap($minified_css);
                    // CDN rewrite should happen after all other modifications
                    // $minified_css = CDN::rewrite($minified_css); // This is handled by OptiCore globally or should be applied to the final URL
                    
                    // Save minified CSS
                    file_put_contents($minified_path, $minified_css);
                    
                    // Create gzipped version for OLS server compatibility
                    $gzipped_path = $minified_path . '.gz';
                    if (function_exists('gzencode')) {
                        $gzipped_css = gzencode($minified_css, 9);
                        file_put_contents($gzipped_path, $gzipped_css);
                    }
                }

                $html = str_replace($href, $minified_url, $html);
            }
        } catch (\Exception $e) {
            // Log CSS minification error
            \ZiziCache\CacheSys::writeError('CSS Minify Error: ' . $e->getMessage(), 'Security');
        }
        return $html;
    }

  /**
   * Removes unused CSS from the page
   * 
   * Analyzes the HTML to find used CSS selectors and removes unused ones from stylesheets.
   * 
   * @param string $html The HTML content to process
   * @return string The HTML with unused CSS removed
   */
  public static function remove_unused_css(string $html): string
  {
    $config = SysConfig::$config;

    if (!$config['css_rucss']) {
      return $html;
    }

    // Skip optimization if page builder editor is active
    if (class_exists('ZiziCache\Plugins\Integrations\PageBuilders') && 
        \ZiziCache\Plugins\Integrations\PageBuilders::is_page_builder_active()) {
      return $html;
    }

    // Find used selectors in HTML
    $html_selectors = self::get_used_html_selectors($html);

    // Find stylesheets in HTML
    preg_match_all("/<link[^>]*\srel=['\"]stylesheet['\"][^>]*>/", $html, $stylesheet_matches);

    if (empty($stylesheet_matches[0])) {
        return $html;
    }
    $stylesheet_tags = $stylesheet_matches[0];

    // Filter out excluded stylesheets
    $excludes = (array) ($config['css_rucss_exclude_stylesheets'] ?? []);
    if (!empty($excludes)) {
        $stylesheet_tags = array_filter($stylesheet_tags, function (string $stylesheet_tag) use ($excludes): bool {
            return !SysTool::any_keywords_match_string($excludes, $stylesheet_tag);
        });
    }
    
    if (empty($stylesheet_tags)) {
        return $html;
    }

    // Force include selectors
    $include_selectors_config = (array) ($config['css_rucss_include_selectors'] ?? []);
    $js_selectors = self::get_js_selectors($html);
    $merged_include_selectors = array_filter(array_unique([...$include_selectors_config, ...$js_selectors]));
    
    $include_selectors_regex = '';
    if(!empty($merged_include_selectors)) {
        $escaped_selectors = array_map('preg_quote', $merged_include_selectors, array_fill(0, count($merged_include_selectors), '/'));
        $include_selectors_regex = implode('|', $escaped_selectors);
    }


    foreach ($stylesheet_tags as $stylesheet_tag) {
      $stylesheet = new HTML($stylesheet_tag);

      // Skip print stylesheets
      if ($stylesheet->media === 'print') {
        continue;
      }

      $href = $stylesheet->href;
      if (empty($href)) {
        continue;
      }
      $css_file_path = CacheSys::get_file_path_from_url($href);

      // Skip if file doesn't exist or is not readable
      if (empty($css_file_path) || !is_file($css_file_path) || !is_readable($css_file_path)) {
        continue;
      }

      // Security: Check file size to prevent memory exhaustion attacks
      $file_size = filesize($css_file_path);
      if ($file_size === false || $file_size > 2 * 1024 * 1024) { // 2MB limit
        CacheSys::writeWarning("CSS file too large or unreadable: {$css_file_path} ({$file_size} bytes)");
        continue;
      }

      // Parse and remove unused CSS
      $css_content = file_get_contents($css_file_path);
      if ($css_content === false || empty($css_content)) {
          continue;
      }

      // Convert media queries to @media if present
      $media = $stylesheet->media;
      if (!empty($media) && $media !== 'all') {
        $css_content = "@media {$media} { {$css_content} }";
      }

      // Parse CSS and get used css
      $used_css = self::get_used_css_blocks($html_selectors, $css_content, $include_selectors_regex);

      if (!$config['css_minify']) { // If global minification is off, but RUCSS is on, we might still want to apply these
        $used_css = self::rewrite_absolute_urls($used_css, $href);
        $used_css = Font::inject_display_swap($used_css);
      }
      
      if (empty(trim($used_css))) { // If RUCSS resulted in empty CSS, don't add an empty style tag
          // Decide how to handle the original stylesheet: remove it or leave it to be async loaded
          if ($config['css_rucss_method'] === 'async' || $config['css_rucss_method'] === 'interaction') {
            // proceed to async/interaction loading of original stylesheet
          } else {
            // Potentially remove $stylesheet_tag if no RUCSS method is set to load it later
            // $html = str_replace($stylesheet_tag, '', $html); // Be cautious with this
            continue; 
          }
      } else {
        // Add used CSS to HTML, right after the link tag
        $used_styletag = "<style class='zizi-cache-used-css' data-original-href='" . esc_attr($href) . "'>{$used_css}</style>";
        $html = str_replace($stylesheet_tag, $used_styletag . PHP_EOL . $stylesheet_tag, $html);
      }

      // Load unused CSS based on method selected (async, interaction, domcontentloaded)
      switch ($config['css_rucss_method']) {
        case 'async':
          // Set media=print and onload, set media to original value
          $original_media = $stylesheet->media ?: 'all';
          $js_onload = "this.onload=null;this.rel='stylesheet';this.media='" . $original_media . "';";
          $stylesheet->onload = $js_onload;
          $stylesheet->media = 'print';
          // Ensure rel=stylesheet is present for this technique to work
          $stylesheet->rel = 'stylesheet';
          $html = str_replace($stylesheet_tag, (string)$stylesheet, $html);
          break;
        case 'interaction':
          // Use core.js to load stylesheet on interaction
          $stylesheet->{'data-href'} = $stylesheet->href;
          unset($stylesheet->href);
          // Add a class or attribute for core.js to target
          $current_class = $stylesheet->class ?? '';
          $stylesheet->class = trim($current_class . ' zizi-deferred-css');
          $stylesheet->rel = 'preload'; // Change rel to preload, as=style
          $stylesheet->as = 'style';
          $html = str_replace($stylesheet_tag, (string)$stylesheet, $html);
          break;
        case 'domcontentloaded':
          // Store CSS URL for DOMContentLoaded loading via JavaScript
          $stylesheet->{'data-dcl-href'} = $stylesheet->href;
          unset($stylesheet->href);
          $current_class = $stylesheet->class ?? '';
          $stylesheet->class = trim($current_class . ' zizi-dcl-css');
          $stylesheet->rel = 'preload'; // Change rel to preload, as=style for resource hint
          $stylesheet->as = 'style';
          $html = str_replace($stylesheet_tag, (string)$stylesheet, $html);
          break;
      }
    }

    return $html;
  }

  /**
   * Extracts all used CSS selectors from HTML content
   * 
   * Parses the HTML and collects all used tags, classes, IDs, and attributes.
   * 
   * @param string $html The HTML content to analyze
   * @return array Array of used selectors grouped by type (tags, classes, ids, attributes)
   */
  private static function get_used_html_selectors(string $html): array
  {
    if (empty($html)) {
        return [
            'tags' => [], 'classes' => [], 'ids' => [], 'attributes' => [],
        ];
    }
    libxml_use_internal_errors(true);
    $dom = new \DOMDocument();
    // Suppress warnings for invalid HTML by adding options
    $result = $dom->loadHTML($html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOWARNING | LIBXML_NOERROR);
    libxml_clear_errors();

    $html_selectors = [
      'tags' => [],
      'classes' => [],
      'ids' => [],
      'attributes' => [],
    ];

    if ($result === false) {
        return $html_selectors; // Return empty if DOM loading failed
    }

    foreach ($dom->getElementsByTagName('*') as $tag) {
      $html_selectors['tags'][$tag->tagName] = 1;

      if ($tag->hasAttribute('class')) {
        $classes_attr = $tag->getAttribute('class');
        // Escape special characters that might conflict with regex in is_selector_used or CSS parsing
        $escaped_classes = str_replace([':', '/'], ['\:', '\/'], $classes_attr);
        // Split classes to array
        $classes_array = preg_split('/\s+/', $escaped_classes);
        if ($classes_array) {
            foreach ($classes_array as $class_name) {
              if (!empty($class_name)) {
                $html_selectors['classes'][$class_name] = 1;
              }
            }
        }
      }

      if ($tag->hasAttribute('id')) {
        $id = $tag->getAttribute('id');
        if (!empty($id)) {
            $html_selectors['ids'][$id] = 1;
        }
      }

      if ($tag->hasAttributes()) { // Check if there are any attributes at all
        foreach ($tag->attributes as $attribute) {
          $html_selectors['attributes'][$attribute->name] = 1;
        }
      }
    }
    // Ensure keys exist even if empty
    $html_selectors['tags'] = array_keys($html_selectors['tags']);
    $html_selectors['classes'] = array_keys($html_selectors['classes']);
    $html_selectors['ids'] = array_keys($html_selectors['ids']);
    $html_selectors['attributes'] = array_keys($html_selectors['attributes']);

    return $html_selectors;
  }

  /**
   * Filters CSS blocks based on used selectors
   * 
   * Processes CSS content and keeps only the rules that match used selectors in the HTML.
   * 
   * @param array $html_selectors Array of selectors used in HTML
   * @param string $css The CSS content to filter
   * @param string $include_selectors_regex Regular expression for selectors to always include
   * @return string Filtered CSS content with only used rules
   */
  private static function get_used_css_blocks(array $html_selectors, string $css, string $include_selectors_regex): string
  {
    if (empty($css)) {
        return '';
    }
    $parsed_css_blocks = self::parse_css($css);
    $original_css_copy = $css; // Work on a copy

    foreach ($parsed_css_blocks as $css_block) {
      $selectors_string = $css_block['selectors'];

      if (!empty($include_selectors_regex) && preg_match("/$include_selectors_regex/i", $selectors_string)) { // Added /i for case-insensitive
        continue;
      }

      $individual_selectors = explode(',', $selectors_string);
      $is_any_selector_used = false;
      foreach($individual_selectors as $selector_item) {
          if (self::is_selector_used(trim($selector_item), $html_selectors)) {
              $is_any_selector_used = true;
              break;
          }
      }

      if (!$is_any_selector_used) {
        // More robustly remove the specific block. str_replace_first might be problematic if blocks are identical.
        // This requires a more sophisticated CSS parser or very careful regex.
        // For now, keeping SysTool::str_replace_first but acknowledging its limitation.
        $original_css_copy = SysTool::str_replace_first($css_block['css'], '', $original_css_copy);
      }
    }

    return $original_css_copy;
  }

  /**
   * Checks if a CSS selector is used in the HTML
   * 
   * @param string $selector The CSS selector to check
   * @param array $html_selectors Array of selectors extracted from HTML
   * @return bool True if the selector is used, false otherwise
   */
  private static function is_selector_used(string $selector, array $html_selectors): bool
  {
    if (empty(trim($selector))) {
        return false; // An empty selector string is not used
    }
    // Eliminate false negatives (:not(), pseudo-elements/classes, etc...)
    // This regex is broad; specific pseudo-classes like :hover, :focus might be desirable to keep in some RUCSS strategies
    $selector_check = preg_replace('/(?<!\\\\)::?[a-zA-Z0-9_-]+(?:\([^)]*\))?/', '', $selector);


    // Attributes
    preg_match_all('/\[([A-Za-z0-9_:-]+)(?:\W?=([^\]]+))?\]/', $selector_check, $attr_matches, PREG_SET_ORDER);
    foreach ($attr_matches as $match) {
      if (!in_array($match[1], $html_selectors['attributes'])) {
        return false;
      }
    }
    $selector_check = preg_replace('/\[([A-Za-z0-9_:-]+)(?:\W?=([^\]]+))?\]/', '', $selector_check);

    // Classes
    preg_match_all('/\.((?:[a-zA-Z0-9_-]+|\\\\.)+)/', $selector_check, $class_matches);
    foreach ($class_matches[1] as $class) {
      // Need to unescape class names if they were escaped, e.g., '\:' back to ':' for lookup
      $class_lookup = str_replace(['\:', '\/'], [':', '/'], $class);
      if (!in_array($class_lookup, $html_selectors['classes'])) {
        return false;
      }
    }
    $selector_check = preg_replace('/\.((?:[a-zA-Z0-9_-]+|\\\\.)+)/', '', $selector_check);

    // IDs
    preg_match_all('/#([a-zA-Z0-9_-]+)/', $selector_check, $id_matches);
    foreach ($id_matches[1] as $id) {
      if (!in_array($id, $html_selectors['ids'])) {
        return false;
      }
    }
    $selector_check = preg_replace('/#([a-zA-Z0-9_-]+)/', '', $selector_check);

    // Tags (element selectors)
    // This regex needs to be careful not to match parts of remaining pseudo-classes or other constructs
    // It should match valid tag names, considering universal selector *
    $selector_check = trim($selector_check);
    if ($selector_check === '*' && (!empty($html_selectors['tags']) || !empty($html_selectors['classes']) || !empty($html_selectors['ids']))) { // Universal selector
        return true;
    }
    // Match word characters, hyphens, possibly namespace prefixes (though rare in HTML directly)
    preg_match_all('/(?<![.#\[])\b([a-zA-Z0-9_-]+)\b(?![\(\]])/', $selector_check, $tag_matches);
     if (!empty($tag_matches[0])) {
        $all_tags_found = true;
        foreach ($tag_matches[0] as $tag) {
            if (strtolower($tag) === 'body' || strtolower($tag) === 'html') { // Always consider body/html used
                continue;
            }
            if (!in_array(strtolower($tag), $html_selectors['tags'])) {
                 $all_tags_found = false;
                 break;
            }
        }
        if ($all_tags_found && !empty($tag_matches[0])) return true; // If only tags were present and all found
        if (!$all_tags_found && !empty($tag_matches[0])) return false; // If some tag was not found
    }


    // If after removing all recognized parts, there's still something left that looks like a selector part,
    // it might be an unsupported complex selector, or a part of a valid one we didn't fully parse.
    // For safety, if $selector_check is not empty and not just whitespace or combinators like >, +, ~, it might be used.
    // This is a simplification; true accuracy needs a full CSS parser.
    $remaining_selector_symbols = trim(preg_replace('/[\s>+~]+/', '', $selector_check));
    if (!empty($remaining_selector_symbols)) {
        // If there are still characters left that are not combinators,
        // it implies a more complex selector structure we haven't fully parsed.
        // To be safe (avoid removing used styles), assume it IS used if we can't definitively prove it's NOT.
        // However, if all specific parts (tags, classes, ids, attrs) were checked and failed,
        // then even if something remains, it shouldn't validate.
        // The current logic returns false if any specific part is not found.
        // If we reach here, it means all specific parts *were* found or weren't present in the selector.
        // Example: `div > p` - if div and p are found, it's true.
        // Example: `.a .b` - if .a and .b are found, it's true.
        // If selector was just `[attr]`, and `attr` was found, it's true.
        // If selector was `.class[attr]`, and both found, it's true.
        // If we are here, it means all components of the selector that we checked were present in the HTML.
        return true; // Fallback: if we couldn't disprove it, assume used.
    }

    // If $selector_check is empty, it means all parts of the selector were recognized and accounted for.
    // The individual checks for tags, classes, IDs, attributes would have returned false if any part was missing.
    // So, if we reached here, it implies all parts were found.
    return true;
  }

  /**
   * Converts relative URLs in CSS to absolute URLs
   * 
   * Processes CSS content and updates all relative URLs to be absolute based on the provided base URL.
   * 
   * @param string $content The CSS content to process
   * @param string $base_url The base URL to use for relative URLs
   * @return string CSS with absolute URLs
   */
  private static function rewrite_absolute_urls(string $content, string $base_url): string
  {
    if (empty($content) || empty($base_url)) {
        return $content;
    }
    // Regex to find url(...) and @import "..."
    // It now handles optional quotes and whitespace more flexibly.
    $regex = '/url\(\s*([\'"]?)(?!data:)([^\'")]+?)\1\s*\)|@import\s+(?:url\(\s*([\'"]?)([^\'")]+?)\3\s*\)|([\'"])([^\'"]+\.[^\s,;{}()]+)\5)/i';

    $content = preg_replace_callback(
      $regex,
      function (array $match) use ($base_url): string {
        $url_string_original = $match[0];
        // Determine the actual matched URL:
        // $match[2] is for url(path)
        // $match[4] is for @import url(path)
        // $match[6] is for @import "path"
        $relative_url = $match[2] ?: ($match[4] ?: ($match[6] ?? ''));

        if (empty($relative_url) || strpos($relative_url, '#') === 0 || strpos($relative_url, 'data:') === 0) {
          return $url_string_original; // Keep fragment identifiers and data URIs as is
        }

        try {
          $absolute_url_obj = Url::parse($relative_url);
          if (empty($absolute_url_obj->getScheme())) { // Check if scheme is empty, meaning it's not an absolute URL
            $base_url_obj = Url::parse($base_url);
            $absolute_url_obj->makeAbsolute($base_url_obj);
          }
          $new_url_string = (string) $absolute_url_obj;
          // Replace only the URL part, keeping the original "url(...)" or "@import" structure
          return str_replace($relative_url, $new_url_string, $url_string_original);
        } catch (\Exception $e) {
          // Log error or handle - invalid URL encountered
          \ZiziCache\CacheSys::writeError('CSS Rewrite URL Error: ' . $e->getMessage() . ' for URL: ' . $relative_url . ' with base: ' . $base_url, 'Security');
          return $url_string_original; // Return original on error
        }
      },
      $content
    );

    return $content;
  }

  /**
   * Parses CSS into an array of rules
   * 
   * Splits CSS into individual rules while preserving @media queries and other at-rules.
   * 
   * @param string $css The CSS content to parse
   * @return array Array of parsed CSS rules with selectors and properties
   */
  private static function parse_css(string $css): array
  {
    if (empty($css)) {
        return [];
    }
    // More robust comment removal
    $css = preg_replace('!/\*.*?\*/!s', '', $css);
    $css = preg_replace('![ \t]*//.*[\r\n]!', '', $css); // Remove // comments too

    // Preserve @media blocks by replacing their content temporarily
    $media_blocks = [];
    $media_block_index = 0;
    $css = preg_replace_callback('/(@media[^{]+{\s*)(.*?\s*})(?=((?!}).)*@media|((?!}).)*$)/s', function($matches) use (&$media_blocks, &$media_block_index) {
        $placeholder = '__MEDIA_BLOCK_PLACEHOLDER_' . $media_block_index . '__';
        $media_blocks[$placeholder] = $matches[1] . $matches[2]; // Store the full @media block rule
        $media_block_index++;
        return $placeholder;
    }, $css);


    // Remove specific @rules that don't typically contain selectors RUCSS would act upon directly
    $css = preg_replace('/@(import|charset|namespace)[^;]+;/i', '', $css);
    // Remove @font-face and @keyframes entirely as RUCSS usually doesn't optimize inside them
    $css = preg_replace('/@font-face\s*\{[^}]*\}/is', '', $css);
    $css = preg_replace('/@(?:-webkit-|-moz-|-o-|-ms-)?keyframes\s*[^\{]+\s*\{[^}]*\}/is', '', $css);
    
    // Match selector blocks: selectors { properties }
    // This regex is a simplification. Complex nested structures or escaped characters in selectors can break it.
    preg_match_all('/([^\s{};][^{}]*?)\s*\{([^}]*)\}/s', $css, $matches, PREG_SET_ORDER);

    $parsed_css = [];
    foreach ($matches as $match) {
      $selector_group = trim($match[1]);
      $properties = trim($match[2]);
      
      // Skip empty selectors or properties (e.g. from removed blocks)
      if(empty($selector_group) || empty($properties) || strpos($selector_group, '__MEDIA_BLOCK_PLACEHOLDER_') !== false) {
          continue;
      }

      $parsed_css[] = [
        'selectors' => $selector_group,
        'css' => $selector_group . ' { ' . $properties . ' }', // Store the full rule
      ];
    }
    
    // Restore @media blocks and parse their content recursively or handle them as whole blocks
    foreach($media_blocks as $placeholder => $media_block_content) {
        // Option 1: Add media block as a whole (if RUCSS should not look inside media queries)
        // $parsed_css[] = ['selectors' => '@media ...', 'css' => $media_block_content];
        // Option 2: Recursively parse CSS inside media queries (more complex)
        // For simplicity here, we'll assume RUCSS operates on selectors outside media queries,
        // or that media queries are handled by `get_used_css_blocks` if it receives them.
        // The current `get_used_css_blocks` wraps content in @media if stylesheet had it.
        // So, we should probably pass these through.
         $parsed_css[] = [
            'selectors' => '@media', // Special marker or extract actual media query
            'css' => $media_block_content,
        ];
    }


    return $parsed_css;
  }

  /**
   * Extracts CSS selectors used in JavaScript
   * 
   * Analyzes JavaScript in the page to find dynamically added/removed classes.
   * 
   * @param string $html The HTML content to analyze
   * @return array Array of CSS selectors found in JavaScript
   */
  private static function get_js_selectors(string $html): array
  {
    if (empty($html)) {
        return [];
    }
    $htmlObj = new HTML($html);
    $scripts = $htmlObj->getElementsBySelector('script');

    if (empty($scripts)) {
        return [];
    }

    $js_content = '';
    foreach ($scripts as $script_node) {
      $script_html_obj = new HTML($script_node); // Process each script node as HTML object
      $src_attr = $script_html_obj->src; // Access attributes via property

      if (!empty($src_attr)) {
        $file_path = CacheSys::get_file_path_from_url($src_attr);
        if (!empty($file_path) && is_file($file_path) && is_readable($file_path)) {
          // Security: Check file size to prevent memory exhaustion attacks
          $file_size = filesize($file_path);
          if ($file_size !== false && $file_size <= 2 * 1024 * 1024) { // 2MB limit
            $js_content .= file_get_contents($file_path);
          } else {
            CacheSys::writeWarning("JS file too large or unreadable: {$file_path} ({$file_size} bytes)");
          }
        }
      } else {
        $js_content .= $script_html_obj->getContent();
      }
    }

    if (empty($js_content)) {
        return [];
    }

    // Regex to find class names in common JS class manipulation functions
    // This can be expanded for other libraries/methods (e.g., jQuery's addClass, etc.)
    // Matches: .addClass('class1 class2'), .classList.add("class3", "class4") etc.
    $regex = '/(?:classList\.add|classList\.remove|classList\.toggle|addClass|removeClass|toggleClass|setAttribute\(\s*([\'"])class\1\s*,)\(\s*([\'"])(.*?)\2/i';

    preg_match_all($regex, $js_content, $matches);
    
    $found_selectors = [];
    if (!empty($matches[3])) { // classes are in capture group 3
        foreach($matches[3] as $match_group) {
            $potential_classes = explode(' ', $match_group);
            foreach($potential_classes as $class) {
                $trimmed_class = trim($class);
                if (!empty($trimmed_class) && strlen($trimmed_class) >= 2 && preg_match('/^[a-zA-Z0-9_-]+$/', $trimmed_class)) { // Basic validation
                    $found_selectors[] = $trimmed_class;
                }
            }
        }
    }
    
    // Remove selectors with less than 2 characters (configurable if needed)
    // $selectors = array_filter($found_selectors, function (string $selector): bool {
    //   return strlen($selector) >= 2; // Adjusted from 4 to 2, common for utility classes
    // });

    return array_values(array_unique($found_selectors));
  }

  /**
   * Downloads and self-hosts third-party CSS files
   * 
   * References to external CSS files are replaced with local copies.
   * 
   * @param string $html The HTML content to process
   * @return string HTML with self-hosted CSS files
   */
  public static function self_host_third_party_css(string $html): string
  {
    if (empty(SysConfig::$config['self_host_third_party_css_js'])) { // Ensure it's not empty
      return $html;
    }

    try {
      preg_match_all("/<link[^>]*\srel=['\"]stylesheet['\"][^>]*>/i", $html, $stylesheets_matches); // Added /i

      if (empty($stylesheets_matches[0])) {
          return $html;
      }
      $stylesheet_tags = $stylesheets_matches[0];

      foreach ($stylesheet_tags as $stylesheet_tag) {
        $stylesheet = new HTML($stylesheet_tag);
        $original_href = $stylesheet->href;

        if(empty($original_href) || !SysTool::is_external_url($original_href, home_url())) {
            continue; // Skip local or empty hrefs
        }

        // Download the external file if allowed
        $downloaded_url = SysTool::download_external_file($original_href, 'css');

        if (empty($downloaded_url)) {
          continue;
        }

        // Remove resource hints for the original URL
        $html = SysTool::remove_resource_hints($original_href, $html);

        // Create a new link tag string to replace the old one
        $new_stylesheet_tag = new HTML($stylesheet_tag); // Create a fresh object for modification
        $new_stylesheet_tag->{'data-original-href'} = $original_href; // Use data-original-href
        $new_stylesheet_tag->href = $downloaded_url;

        unset($new_stylesheet_tag->integrity);
        unset($new_stylesheet_tag->crossorigin);
        
        $html = str_replace($stylesheet_tag, (string)$new_stylesheet_tag, $html);
      }
    } catch (\Exception $e) {
      \ZiziCache\CacheSys::writeError('Self-host CSS Error: ' . $e->getMessage(), 'Security');
    }
    return $html;
  }

  /**
   * Processes and self-hosts third-party fonts referenced in CSS
   * 
   * Downloads font files referenced in CSS and updates the URLs to point to local copies.
   * 
   * @param string $content The CSS content to process
   * @param string $url The URL of the CSS file being processed
   * @param string $extension The file extension (should be 'css' for this method to process)
   * @return string Processed CSS with self-hosted font URLs
   */
  public static function self_host_third_party_fonts(string $content, string $url, string $extension): string
  {
    // This function is a filter for 'zizi_cache_download_external_file:before'
    // It processes the content of a downloaded CSS file to self-host fonts referenced within it.

    if ($extension !== 'css' || empty($content)) {
      return $content;
    }

    // Convert relative font URLs within the CSS to absolute URLs, using the CSS file's URL as base
    $processed_content = self::rewrite_absolute_urls($content, $url);

    // Get a list of the font file URLs from the CSS content
    $font_urls = Font::get_font_urls($processed_content);

    if (empty($font_urls)) {
      return $processed_content; // Return content with absolute URLs if no fonts found or to download
    }

    // Download the font files
    $downloaded_font_map = Font::download_fonts($font_urls, ZIZI_CACHE_CACHE_DIR);

    // Replace the original font URLs in the CSS content with the new local URLs
    foreach ($downloaded_font_map as $original_font_url => $local_font_filename) {
      if (empty($local_font_filename)) continue;

      $local_font_url = ZIZI_CACHE_CACHE_URL . $local_font_filename;
      // Ensure we are replacing the absolute URL we created earlier
      $processed_content = str_replace($original_font_url, $local_font_url, $processed_content);
    }

    return $processed_content;
  }
}
