<?php

declare(strict_types=1);

namespace ZiziCache;

use ZiziCache\SysConfig;
use ZiziCache\SysTool;
use ZiziCache\CacheSys;

/**
 * CDN (Content Delivery Network) Management and Optimization
 * 
 * This class handles the integration and optimization of CDN functionality
 * for WordPress sites. It supports various CDN features including:
 * - Force HTTPS for CDN URLs
 * - Relative path handling
 * - Advanced regex pattern matching
 * - Comprehensive logging
 * - URL validation
 * - CDN availability verification
 * - Efficient URL rewriting with caching
 *
 * @package ZiziCache
 */
class CDN
{
    /**
     * Constants for logging and settings
     */
    const LOG_PREFIX = '[CDN] ';
    const DEFAULT_TIMEOUT = 5; // Timeout for CDN verification in seconds
    const CACHE_MAX_SIZE = 1000; // Maximum number of items in URL cache
    const URL_VALIDATE_PATTERN = '/^(https?:)?\/\/[a-z0-9]+([\-\.]{1}[a-z0-9]+)*(\.[a-z]{2,}|localhost|\.local|\.test|\.dev)(:[0-9]{1,5})?(\/.*)?$/i';
    
    /**
     * Internal static cache for optimization
     */
    private static $url_cache = [];
    private static $cdn_status = null;
    private static $cache_hits = 0;
    private static $cache_misses = 0;
    private static $start_time = 0;
    
    /**
     * Adds preconnect header for CDN domain to improve loading performance
     *
     * @param string $html The HTML content of the page
     * @return string Modified HTML with preconnect header
     */
    public static function add_preconnect(string $html): string
    {
        $config = SysConfig::$config;

        if (!($config['cdn'] ?? false) || empty($config['cdn_url']) || ($config['cdn_type'] ?? '') !== 'custom') {
            return $html;
        }

        $cdn_url = self::normalize_cdn_url((string) $config['cdn_url'], ($config['cdn_force_https'] ?? false));
        
        try {
            // Added crossorigin attribute for best practices and better compatibility
            $preconnect_link = '<link rel="preconnect" href="' . esc_url($cdn_url) . '" crossorigin/>'; 

            // Add preconnect link after <title> or at the beginning of <head>
            if (strpos($html, '</title>') !== false) {
                $html = SysTool::str_replace_first(
                    '</title>',
                    '</title>' . PHP_EOL . $preconnect_link,
                    $html
                );
            } elseif (strpos($html, '<head>') !== false) {
                $html = SysTool::str_replace_first(
                    '<head>',
                    '<head>' . PHP_EOL . $preconnect_link,
                    $html
                );
            } else {
                // Fallback: Add at the beginning of HTML if <head> or <title> are not found (less ideal)
                $html = $preconnect_link . PHP_EOL . $html;
                CacheSys::writeLog(self::LOG_PREFIX . 'Unusual HTML structure - preconnect added at the beginning of the document.');
            }
            
            self::log_debug('Přidána preconnect hlavička pro: ' . $cdn_url);
        } catch (\Exception $e) {
            CacheSys::writeLog(self::LOG_PREFIX . 'Chyba při přidávání preconnect: ' . $e->getMessage());
        }

        return $html;
    }

    /**
     * Rewrites URLs in the page to point to CDN instead of the original server
     *
     * @param string $html The HTML content of the page
     * @return string Modified HTML with CDN links
     */
    public static function rewrite(string $html): string
    {
        self::$start_time = microtime(true);
        $config = SysConfig::$config;

        // Check if CDN is enabled and properly configured
        if (!($config['cdn'] ?? false) || empty($config['cdn_url']) || ($config['cdn_type'] ?? '') !== 'custom') {
            return $html;
        }

        // Verify and initialize CDN on first call
        if (self::$cdn_status === null) {
            self::$cdn_status = self::verify_cdn_setup($config);
            
            // If CDN is in non-functional state, return original HTML and skip rewriting
            if (self::$cdn_status === false) {
                self::log_debug('CDN je v nefunkčním stavu, přeskakuji přepisování URL.');
                return $html;
            }
        }

        // Parse website and CDN URLs
        $site_url = site_url();
        $cdn_url = self::normalize_cdn_url((string) $config['cdn_url'], ($config['cdn_force_https'] ?? false));
        $site_url_parts = wp_parse_url($site_url);
        $cdn_url_parts = wp_parse_url($cdn_url);

        if (!is_array($site_url_parts) || !isset($site_url_parts['host']) || !is_array($cdn_url_parts) || !isset($cdn_url_parts['host'])) {
            self::log_error('Invalid website or CDN URL', ['site_url' => $site_url, 'cdn_url' => $cdn_url]);
            return $html; // Invalid website or CDN URL
        }

        // Normalize domains with support for both scheme and relative URLs
        $site_scheme = isset($site_url_parts['scheme']) ? $site_url_parts['scheme'] . '://' : '//';
        $cdn_scheme = isset($cdn_url_parts['scheme']) ? $cdn_url_parts['scheme'] . '://' : '//';
        
        if (($config['cdn_force_https'] ?? false)) {
            $cdn_scheme = 'https://';
        }
        
        $site_domain = $site_scheme . $site_url_parts['host'];
        $cdn_domain = $cdn_scheme . $cdn_url_parts['host'];
        
        // Add port if specified
        if (isset($site_url_parts['port'])) {
            $site_domain .= ':' . $site_url_parts['port'];
        }
        if (isset($cdn_url_parts['port'])) {
            $cdn_domain .= ':' . $cdn_url_parts['port'];
        }

        if ($site_domain === $cdn_domain) {
            self::log_warning('Website and CDN domains are the same, no rewrite needed');
            return $html; // Website and CDN domains are the same, no rewrite needed
        }

        // Define file types for CDN
        $file_types = [
            'all' => 'css|js|eot|otf|ttf|woff|woff2|gif|jpeg|jpg|png|svg|webp|avif|jxl|ico|webm|mp4|ogg',
            'css_js' => 'css|js|eot|otf|ttf|woff|woff2',
            'images' => 'gif|jpeg|jpg|png|svg|webp|avif|jxl|ico',
            'fonts' => 'eot|otf|ttf|woff|woff2',
            'videos' => 'webm|mp4|ogg',
            'static' => 'css|js|eot|otf|ttf|woff|woff2|gif|jpeg|jpg|png|svg|webp|ico',
        ];

        $cdn_file_types_key = $config['cdn_file_types'] ?? 'all';
        // Ensures that if the key is invalid, the default value 'all' is used
        $file_types_regex_part = $file_types[$cdn_file_types_key] ?? $file_types['all'];
        
        // Check if any files/patterns are excluded
        $excludes = [];
        if (!empty($config['cdn_exclusions'])) {
            if (is_array($config['cdn_exclusions'])) {
                $excludes = $config['cdn_exclusions'];
            } elseif (is_string($config['cdn_exclusions'])) {
                $excludes = array_filter(array_map('trim', explode("\n", $config['cdn_exclusions'])));
            }
        }

        // Cache size management - if too large, reset it
        if (count(self::$url_cache) > self::CACHE_MAX_SIZE) {
            self::log_debug('Cache URL překročila maximální velikost, resetuji...');
            self::$url_cache = [];
            self::$cache_hits = 0;
            self::$cache_misses = 0;
        }

        // Escape website domain for regex. Using preg_quote with '/' as delimiter.
        $site_domain_escaped = preg_quote($site_domain, '/');

        // Improved regex for better processing
        // 1. Basic URL from domain
        // 2. Supports relative paths (starting with /)
        // 3. Extended escaping for JSON/JS/CSS compatibility
        // 4. Better end-of-URL detection
        
        $patterns = [
            // Main regex for URLs with domain
            "/(?<=['\"\\s(])(" . $site_domain_escaped . ")([^'\"\\s)]*?\\.(" . $file_types_regex_part . "))(?=['\"\\s)>,;\\\])/i",
            
            // Regex for relative URLs (if enabled)
            "/(?<=['\"\\s(])(\\/[^'\"\\s)]*?\\.(" . $file_types_regex_part . "))(?=['\"\\s)>,;\\\])/i",
        ];

        // Cache URL count for statistics
        $replaced_count = 0;
        $excluded_count = 0;
        
        try {
            // First, process URLs with full domain
            $html = preg_replace_callback($patterns[0], function ($matches) use ($cdn_domain, $site_domain, $excludes, &$replaced_count, &$excluded_count) {
                // Check if URL is not in the exclusions list
                $full_url = $matches[0];
                
                foreach ($excludes as $exclude) {
                    if (!empty($exclude) && strpos($full_url, $exclude) !== false) {
                        $excluded_count++;
                        return $full_url; // Return original URL if excluded
                    }
                }
                
                // If this URL was processed before, use cache
                $cache_key = md5($full_url);
                if (isset(self::$url_cache[$cache_key])) {
                    self::$cache_hits++;
                    return self::$url_cache[$cache_key];
                }
                
                // Replace domain in URL
                $cdn_url = str_replace($site_domain, $cdn_domain, $full_url);
                $replaced_count++;
                self::$cache_misses++;
                
                // Save to cache for future use
                self::$url_cache[$cache_key] = $cdn_url;
                
                return $cdn_url;
            }, $html);
            
            // Process relative URLs if this feature is enabled
            if ($config['cdn_include_relative_paths'] ?? false) {
                $html = preg_replace_callback($patterns[1], function ($matches) use ($cdn_domain, $excludes, &$replaced_count, &$excluded_count) {
                    $rel_url = $matches[0];
                    
                    foreach ($excludes as $exclude) {
                        if (!empty($exclude) && strpos($rel_url, $exclude) !== false) {
                            $excluded_count++;
                            return $rel_url;
                        }
                    }
                    
                    // Cache processing for relative URLs
                    $cache_key = md5($rel_url);
                    if (isset(self::$url_cache[$cache_key])) {
                        self::$cache_hits++;
                        return self::$url_cache[$cache_key];
                    }
                    
                    // Convert relative URL to absolute URL with CDN domain
                    // Remove leading / from relative path
                    $path = ltrim($matches[1], '/');
                    $cdn_url = rtrim($cdn_domain, '/') . '/' . $path;
                    $replaced_count++;
                    self::$cache_misses++;
                    
                    self::$url_cache[$cache_key] = $cdn_url;
                    
                    return $cdn_url;
                }, $html);
            }
            
            // Log statistics for diagnostics
            $elapsed = microtime(true) - self::$start_time;
            self::log_debug(sprintf(
                'CDN rewrite: processed %d URLs (%d excluded) in %.4f seconds, cache hits: %d, misses: %d',
                $replaced_count, 
                $excluded_count, 
                $elapsed,
                self::$cache_hits,
                self::$cache_misses
            ));
            
        } catch (\Exception $e) {
            self::log_error('Error rewriting URLs: ' . $e->getMessage());
            // In case of error, return original HTML
            return $html;
        }

        return $html;
    }
    
    /**
     * Normalizes URL for CDN with optional HTTPS enforcement
     *
     * @param string $url CDN URL to normalize
     * @param bool $force_https Whether to force HTTPS regardless of original URL
     * @return string Normalized URL
     */
    public static function normalize_cdn_url(string $url, bool $force_https = false): string
    {
        // Validate URL format
        if (empty($url)) {
            return '';
        }
        
        // Check if this is a local domain - extended detection
        $is_local = preg_match('~(localhost|127\.0\.0\.1|::1|.*\.localhost|.*\.local|.*\.test|.*\.dev)~i', $url);
        
        // Special handling for local development - respect HTTP
        if ($is_local && !$force_https) {
            // Remove existing protocol for local URLs if HTTPS is not required
            $url = preg_replace('~^https?://~i', '', $url);
            
            // Explicitly set HTTP for local domains if we don't want to force HTTPS
            $url = 'http://' . $url;
            self::log_debug('Detected local domain, using HTTP protocol: ' . $url);
        } else {
            // URL format validation for non-empty addresses
            if (!preg_match(self::URL_VALIDATE_PATTERN, $url) && !preg_match(self::URL_VALIDATE_PATTERN, '//' . $url)) {
                self::log_warning('Invalid CDN URL format: ' . $url);
                // Attempt to fix URL - add protocol and check basic format
                if (strpos($url, '.') !== false) {
                    $url = '//' . preg_replace('/^(https?:)?\/\//', '', $url);
                }
            }
            
            // Add protocol if missing
            if (strpos($url, '//') !== 0 && strpos($url, 'http') !== 0) {
                $url = '//' . $url;
            }
        }
        
        // Force HTTPS if required
        if ($force_https) {
            // Convert protocol-relative URL to HTTPS
            if (strpos($url, '//') === 0) {
                $url = 'https:' . $url;
            }
            // Convert HTTP to HTTPS
            elseif (strpos($url, 'http://') === 0) {
                $url = 'https://' . substr($url, 7);
            }
        }
        
        // Remove trailing slash for consistent format
        return rtrim($url, '/');
    }
    
    /**
     * Verifies CDN settings and its availability
     * 
     * @param array $config Plugin configuration
     * @return bool True if CDN is properly set up and functional
     */
    private static function verify_cdn_setup(array $config): bool
    {
        $cdn_url = self::normalize_cdn_url((string) ($config['cdn_url'] ?? ''), ($config['cdn_force_https'] ?? false));
        
        // Basic URL validation
        if (empty($cdn_url)) {
            self::log_error('CDN URL is not set or is empty');
            return false;
        }
        
        // CDN test is optional - skip if not enabled
        if (!($config['cdn_verify'] ?? false)) {
            return true;
        }
        
        // Check if the domain is available
        $test_url = $cdn_url . '/favicon.ico';
        $timeout = $config['cdn_verify_timeout'] ?? self::DEFAULT_TIMEOUT;
        
        self::log_debug('Verifying CDN availability: ' . $test_url);
        
        try {
            $response = wp_remote_head($test_url, [
                'timeout' => $timeout,
                'sslverify' => false, // Skip SSL verification for diagnostics
                'user-agent' => 'ZiziCache CDN Verifier/1.0',
                'headers' => [
                    'Cache-Control' => 'no-cache',
                ],
            ]);
            
            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                self::log_error('CDN test failed: ' . $error_message, ['url' => $test_url]);
                return false;
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            
            // Success codes or expected errors (404 is OK, means CDN is responding)
            $acceptable_codes = [200, 201, 204, 301, 302, 304, 307, 308, 404, 403];
            
            if (!in_array($response_code, $acceptable_codes)) {
                self::log_error('CDN test returned unexpected status code: ' . $response_code, ['url' => $test_url]);
                return false;
            }
            
            // Check response speed - warning if too slow
            $ttfb = (float) wp_remote_retrieve_header($response, 'x-ttfb') ?: 0;
            if ($ttfb > 0.5) { // More than 500ms is considered slow
                self::log_warning(sprintf('CDN response is slow: %.3f seconds', $ttfb));
            }
            
            self::log_debug('CDN test successful! Status code: ' . $response_code);
            return true;
            
        } catch (\Exception $e) {
            self::log_error('Exception while testing CDN: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Logs debug information if debug is enabled
     *
     * @param string $message Message to log
     * @param array $context Optional debug context
     * @return void
     */
    private static function log_debug(string $message, array $context = []): void
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $debug_info = empty($context) ? '' : ' | ' . json_encode($context);
            CacheSys::writeLog(self::LOG_PREFIX . 'Debug: ' . $message . $debug_info);
        }
    }
    
    /**
     * Logs a warning
     *
     * @param string $message Warning message
     * @param array $context Optional debug context
     * @return void
     */
    private static function log_warning(string $message, array $context = []): void
    {
        $warning_info = empty($context) ? '' : ' | ' . json_encode($context);
        CacheSys::writeLog(self::LOG_PREFIX . 'Warning: ' . $message . $warning_info);
    }
    
    /**
     * Logs errors
     *
     * @param string $message Error message
     * @param array $context Optional debug context
     * @return void
     */
    private static function log_error(string $message, array $context = []): void
    {
        $error_info = empty($context) ? '' : ' | ' . json_encode($context);
        CacheSys::writeLog(self::LOG_PREFIX . 'ERROR: ' . $message . $error_info);
    }
    
    /**
     * Clears internal URL cache
     *
     * @return void
     */
    public static function clear_cache(): void
    {
        self::$url_cache = [];
        self::$cache_hits = 0;
        self::$cache_misses = 0;
        self::$cdn_status = null;
        self::log_debug('CDN cache has been cleared.');
    }
    
    /**
     * Dynamic detection of recommended CDN settings for the current server
     * 
     * @return array Recommended CDN settings
     */
    public static function get_recommended_settings(): array
    {
        $recommendations = [
            'cdn_force_https' => is_ssl(),
            'cdn_include_relative_paths' => true,
            'cdn_verify' => true,
            'cdn_verify_timeout' => 3,
        ];
        
        // Get default value from HTTP_HOST
        $host = $_SERVER['HTTP_HOST'] ?? '';
        if (!empty($host)) {
            if (strpos($host, 'www.') === 0) {
                // Recommendation: use cdn. subdomain instead of www
                $recommendations['cdn_url_suggestion'] = 'cdn.' . substr($host, 4);
            } else {
                // Recommendation: add cdn. prefix
                $recommendations['cdn_url_suggestion'] = 'cdn.' . $host;
            }
        }
        
        return $recommendations;
    }
}
