<?php

namespace ZiziCache;

/**
 * Handles automatic cache purging and preloading on post and comment updates.
 * 
 * This class is responsible for automatically clearing and preloading cache
 * when content is updated, ensuring that users always see the most recent version
 * of the site. It handles various content types and their associated URLs.
 * 
 * The class has been refactored for better structure, separation of concerns,
 * and extensibility.
 */
class AutomaticallyClear
{
  /**
   * Initializes hooks for automatic purging and preloading.
   * 
   * Sets up WordPress action hooks to trigger cache purging and preloading
   * when posts are published, updated, or when comments are made.
   * 
   * @return void
   */
  public static function init()
  {
    // Purge when scheduled post is published
    add_action('future_to_publish', function ($post) {
      self::post_updated($post->ID, $post, $post);
    });

    // Purge cache on updating post
    add_action('post_updated', [__CLASS__, 'post_updated'], 10, 3);

    // Preload post URL when comment count is updated
    add_action('wp_update_comment_count', [__CLASS__, 'preload_on_comment']);
  }

  /**
   * Purge and preload URLs when a post is updated.
   *
   * @param int $post_id
   * @param WP_Post $post_after
   * @param WP_Post $post_before
   */
  /**
   * Handles cache purging and preloading when a post is updated.
   *
   * This method is triggered when a post is updated and determines if the cache
   * should be purged and preloaded based on the post's status and type.
   *
   * @param int $post_id The ID of the post being updated.
   * @param WP_Post $post_after The post object after the update.
   * @param WP_Post $post_before The post object before the update.
   * @return void
   */
  public static function post_updated($post_id, $post_after, $post_before)
  {
    if (!self::should_purge($post_id, $post_after, $post_before)) {
      self::log_issue('Skipped purge: should_purge() returned false', __FILE__, __LINE__);
      return;
    }
    $urls = self::collect_purge_urls($post_id, $post_after, $post_before);
    if (empty($urls)) {
      self::log_issue('Skipped purge: No URLs collected', __FILE__, __LINE__);
      return;
    }
    Purge::purge_urls($urls);
    Preload::preload_urls($urls);
  }

  /**
   * Determines if a post update should trigger cache purging and preloading.
   *
   * This method checks if the post is being published or was previously published,
   * and ensures navigation menu items are excluded from purging.
   *
   * @param int $post_id The ID of the post being checked.
   * @param WP_Post $post_after The post object after the update.
   * @param WP_Post $post_before The post object before the update.
   * @return bool True if the cache should be purged, false otherwise.
   */
  private static function should_purge($post_id, $post_after, $post_before): bool
  {
    $post_after_status = get_post_status($post_after);
    $post_before_status = get_post_status($post_before);
    if (!in_array('publish', [$post_after_status, $post_before_status])) {
      return false;
    }
    $post_type = get_post_type($post_id);
    if ($post_type === 'nav_menu_item') {
      return false;
    }
    return true;
  }

  /**
   * Collects all URLs that should be purged and preloaded after a post update.
   *
   * This method gathers all relevant URLs that might be affected by a post update,
   * including the post's URL, homepage, archive pages, author pages, and taxonomy terms.
   *
   * @param int $post_id The ID of the post being updated.
   * @param WP_Post $post_after The post object after the update.
   * @param WP_Post $post_before The post object before the update.
   * @return array An array of URLs to be purged and preloaded.
   */
  private static function collect_purge_urls($post_id, $post_after, $post_before): array
  {
    $urls = [
      self::get_post_url($post_id),
      home_url(),
    ];
    $post_type = get_post_type($post_id);
    if ($post_type === 'post') {
      $posts_page = get_option('page_for_posts');
      if ($posts_page) {
        $urls[] = get_permalink($posts_page);
      }
    }
    $archive_url = get_post_type_archive_link($post_type);
    if ($archive_url) {
      $urls[] = $archive_url;
    }
    $author_id = get_post_field('post_author', $post_id);
    if ($author_id) {
      $urls[] = get_author_posts_url($author_id);
    }
    $urls = array_merge($urls, self::get_post_taxonomy_urls($post_id));
    $urls = apply_filters('zizi_cache_auto_purge_urls', $urls, $post_id);
    return array_unique(array_filter($urls));
  }

  /**
   * Returns the permalink for a given post ID.
   *
   * @param int $post_id
   * @return string|null
   */
  private static function get_post_url($post_id)
  {
    $url = get_permalink($post_id);
    return $url ? $url : null;
  }

  /**
   * Purge and preload post URL when comment count is updated.
   *
   * @param int $post_id
   */
  /**
   * Handles cache purging and preloading when a post's comment count is updated.
   *
   * This method is triggered when a comment is added, updated, or deleted,
   * and ensures the associated post's cache is refreshed.
   *
   * @param int $post_id The ID of the post whose comment count was updated.
   * @return void
   */
  public static function preload_on_comment($post_id)
  {
    $url = self::get_post_url($post_id);
    if (!$url) {
      self::log_issue('Skipped preload_on_comment: No URL for post', __FILE__, __LINE__);
      return;
    }
    Purge::purge_urls([$url]);
    Preload::preload_url($url);
  }

  /**
   * Get taxonomy archive and parent URLs for a post.
   *
   * @param int $post_id
   * @return array
   */
  /**
   * Collects taxonomy archive and ancestor term URLs for a post.
   *
   * This method gathers all taxonomy term archive URLs associated with a post,
   * including parent terms, to ensure all related archive pages are refreshed.
   *
   * @param int $post_id The ID of the post to get taxonomy URLs for.
   * @return array An array of taxonomy term archive URLs.
   */
  public static function get_post_taxonomy_urls($post_id)
  {
    $urls = [];
    $taxonomies = get_object_taxonomies(get_post_type($post_id), 'objects');
    // Only include taxonomies with public archive
    $taxonomies = array_filter($taxonomies, function ($taxonomy) {
      return $taxonomy->publicly_queryable;
    });
    foreach ($taxonomies as $taxonomy) {
      $terms = get_the_terms($post_id, $taxonomy->name);
      if (!is_array($terms) || is_wp_error($terms) || empty($terms)) {
        continue;
      }
      foreach ($terms as $term) {
        if (!is_object($term)) {
          continue;
        }
        $urls[] = get_term_link($term);
        if ($term->parent == 0) {
          continue;
        }
        $parent_terms = get_ancestors($term->term_id, $taxonomy->name);
        foreach ($parent_terms as $parent_term) {
          $urls[] = get_term_link($parent_term);
        }
      }
    }
    return $urls;
  }

  /**
   * Logs issues or unexpected conditions to the unified log file.
   *
   * This helper method formats and logs messages with file and line number context
   * to assist with debugging and monitoring.
   *
   * @param string $message The message to log.
   * @param string $file The file where the issue occurred.
   * @param string $line The line number where the issue occurred.
   * @return void
   */
  private static function log_issue(string $message, string $file, string $line): void
  {
    $log_message = sprintf(
        '[%s:%s] Issue: %s',
        basename($file), // Use basename for brevity
        $line,
        $message
    );
    // Use the central logging method
    CacheSys::writeLog($log_message);
  }
}
