<?php

namespace ZiziCache;

/**
 * Class Authority
 * Handles permission checks for ZiziCache plugin actions.
 */
class Authority
{
    /**
     * Checks if the given user (or current user) is allowed to access ZiziCache features.
     *
     * @param \WP_User|null $user Optional. User object. Defaults to current user.
     * @return bool True if allowed, false otherwise.
     */
    public static function isAllowed($user = null): bool
    {
        $user = $user ?: wp_get_current_user();
        $result = self::isSuperAdmin($user) || self::hasAllowedRole($user);
        if (!$result) {
            self::logDeniedAccess($user);
        }
        return $result;
    }    /**
     * Backwards compatibility: legacy method name.
     *
     * @return bool
     */
    public static function is_allowed()
    {
        return self::isAllowed();
    }

    /**
     * REST API permission callback with CSRF protection.
     * Validates both user permissions and nonce tokens for state-changing operations.
     *
     * @return bool True if allowed and nonce is valid, false otherwise.
     */
    public static function rest_api_permission_callback()
    {
        // First check user permissions
        if (!self::isAllowed()) {
            return false;
        }
        
        // For state-changing operations (POST, PUT, DELETE), verify nonce
        $request_method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        if (in_array($request_method, ['POST', 'PUT', 'DELETE'])) {
            // Get nonce from X-WP-Nonce header or _wpnonce parameter
            $nonce = $_SERVER['HTTP_X_WP_NONCE'] ?? $_REQUEST['_wpnonce'] ?? null;
            
            if (!$nonce || !wp_verify_nonce($nonce, 'wp_rest')) {
                // Log CSRF attempt for security monitoring
                \ZiziCache\CacheSys::writeError(
                    "CSRF attempt blocked - invalid or missing nonce for {$request_method} request from IP: " . 
                    ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
                    'Security'
                );
                return false;
            }
            
            // Log successful nonce verification for security audit
            \ZiziCache\CacheSys::writeLog(
                "REST API {$request_method} request authorized with valid nonce from IP: " . 
                ($_SERVER['REMOTE_ADDR'] ?? 'unknown'), 
                'INFO'
            );
        }
        
        return true;
    }

    /**
     * Checks if the given user (or current user) is allowed to access ZiziCache features
     * without logging access denied errors (silent check).
     *
     * @param \WP_User|null $user Optional. User object. Defaults to current user.
     * @return bool True if allowed, false otherwise.
     */
    public static function isAllowedSilent($user = null): bool
    {
        $user = $user ?: wp_get_current_user();
        return self::isSuperAdmin($user) || self::hasAllowedRole($user);
    }

    /**
     * Returns the list of allowed roles (filterable).
     *
     * @return array
     */
    private static function getAllowedRoles(): array
    {
        return apply_filters('zizi_cache_allowed_roles', ['administrator', 'editor']);
    }

    /**
     * Checks if user is super admin.
     *
     * @param \WP_User $user
     * @return bool
     */
    private static function isSuperAdmin($user): bool
    {
        return function_exists('is_super_admin') && is_super_admin($user->ID);
    }

    /**
     * Checks if user has any of the allowed roles.
     *
     * @param \WP_User $user
     * @return bool
     */
    private static function hasAllowedRole($user): bool
    {
        $allowed = self::getAllowedRoles();
        return is_array($user->roles) && count(array_intersect($user->roles, $allowed)) > 0;
    }

    /**
     * Logs denied access or unexpected input to the unified log file with file reference.
     *
     * @param string $message The message to log.
     * @param string $file The file where the check failed.
     * @param string $line The line number where the check failed.
     */
    private static function log_denied_access(string $message, string $file, string $line): void
    {        $log_message = sprintf(
            '[WARNING] %s:%s Access Denied/Unexpected Input: %s',
            basename($file), // Use basename for brevity
            $line,
            $message
        );
        // Use the central logging method
        CacheSys::writeLog($log_message);
    }

    /**
     * Logs denied access or unexpected input to zizi-log.log with file reference.
     *
     * @param \WP_User $user
     * @return void
     */
    private static function logDeniedAccess($user): void
    {
        // Get backtrace to identify the caller
        $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 3);
        $caller = isset($backtrace[1]) ? $backtrace[1]['function'] : 'unknown';
        $caller_file = isset($backtrace[1]) ? basename($backtrace[1]['file']) : 'unknown';
        $caller_line = isset($backtrace[1]) ? $backtrace[1]['line'] : 0;
        
        // Include URL and request data for better context
        $request_url = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : 'unknown';
        $request_method = isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : 'unknown';
        
        $message = sprintf(
            'Access denied for user ID %d, roles: %s | Called from: %s:%s:%s | URL: %s | Method: %s',
            isset($user->ID) ? $user->ID : 0,
            isset($user->roles) ? implode(',', (array)$user->roles) : 'none',
            $caller_file,
            $caller,
            $caller_line,
            $request_url,
            $request_method
        );
        
        // Only log if WP_DEBUG is enabled or if it's not a typical anonymous frontend request
        if (defined('WP_DEBUG') && WP_DEBUG || $user->ID !== 0 || is_admin() || wp_doing_ajax()) {
            self::log_denied_access($message, __FILE__, __LINE__);        }
    }
}
