<?php
/**
 * ZiZi Cache - AJAX Handler
 *
 * Handles AJAX requests for the plugin
 */

namespace ZiziCache;

class AjaxHandler {
    
    /**
     * Initialize AJAX handlers
     */
    public static function init() {
        // Register AJAX endpoints
        add_action('wp_ajax_zizi_cache_auto_preload_fonts', [__CLASS__, 'handle_auto_preload_fonts']);
        add_action('wp_ajax_zizi_cache_reset_font_stats', [__CLASS__, 'handle_reset_font_statistics']);
        add_action('wp_ajax_zizi_cache_test_redis', [__CLASS__, 'handle_test_redis_connection']);
    }
    
    /**
     * Handle auto preload fonts request (simplified)
     */
    public static function handle_auto_preload_fonts() {
        // Check nonce for security
        check_admin_referer('wp_rest');
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'message' => __('You do not have permission to perform this action.', 'zizi-cache')
            ], 403);
        }
        
        // Simplified response - auto preload is not implemented in simplified version
        wp_send_json_success([
            'message' => __('Auto preload feature has been simplified. Please use manual font preload URLs in settings.', 'zizi-cache'),
            'timestamp' => current_time('timestamp')
        ]);
    }

    /**
     * Handle reset font statistics request
     */
    public static function handle_reset_font_statistics() {
        // Check nonce for security - proper AJAX nonce check
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'wp_rest')) {
            wp_send_json_error([
                'message' => __('Security check failed.', 'zizi-cache')
            ], 403);
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'message' => __('You do not have permission to perform this action.', 'zizi-cache')
            ], 403);
            return;
        }
        
        $success = \ZiziCache\Font::reset_font_statistics();
        
        if ($success) {
            wp_send_json_success([
                'message' => __('Font statistics have been reset successfully.', 'zizi-cache'),
                'timestamp' => current_time('timestamp')
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Failed to reset font statistics. Please check the logs for more details.', 'zizi-cache')
            ], 500);
        }
    }

    /**
     * Handle Redis connection test request
     */
    public static function handle_test_redis_connection() {
        // Check nonce for security
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'wp_rest')) {
            wp_send_json_error([
                'message' => __('Security check failed.', 'zizi-cache')
            ], 403);
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'message' => __('You do not have permission to perform this action.', 'zizi-cache')
            ], 403);
            return;
        }
        
        // Test Redis connection
        $result = ObjectCacheHandler::check_redis_connection();
        
        if ($result['status']) {
            wp_send_json_success([
                'message' => $result['message'],
                'diagnostics' => $result['diagnostics'],
                'timestamp' => current_time('timestamp')
            ]);
        } else {
            wp_send_json_error([
                'message' => $result['message'],
                'diagnostics' => $result['diagnostics'],
                'timestamp' => current_time('timestamp')
            ], 500);
        }
    }
}
