<?php
declare(strict_types=1);

namespace ZiziCache;

/**
 * Advanced Usage Analyzer - Optimized thumbnail usage analysis for large websites
 * 
 * Provides comprehensive and efficient analysis of thumbnail usage across:
 * - Post content, meta fields, and custom fields
 * - Page builders (Elementor, Divi, Beaver Builder, etc.)
 * - Theme customizer and widgets
 * - WooCommerce products and variations
 * - Shortcodes and custom content
 * 
 * Optimized for sites with hundreds of thousands of images and posts
 * 
 * @package ZiziCache
 * @since 0.4.8-beta.5
 */
class AdvancedUsageAnalyzer
{
    /**
     * Maximum memory usage threshold (in bytes)
     */
    private const MEMORY_THRESHOLD = 128 * 1024 * 1024; // 128MB

    /**
     * Default batch size for large datasets
     */
    private const DEFAULT_BATCH_SIZE = 100;

    /**
     * Cache duration for expensive operations (in seconds)
     */
    private const CACHE_DURATION = 3600; // 1 hour

    /**
     * Initialize Advanced Usage Analyzer
     */
    public static function init(): void
    {
        // Register cleanup for cache on post updates
        add_action('save_post', [__CLASS__, 'invalidate_cache']);
        add_action('delete_post', [__CLASS__, 'invalidate_cache']);
        
        if (class_exists('\\ZiziCache\\CacheSys')) {
            \ZiziCache\CacheSys::writeLog('AdvancedUsageAnalyzer initialized', 'AdvancedUsageAnalyzer');
        }
    }

    /**
     * Comprehensive thumbnail usage analysis optimized for large datasets
     */
    public static function analyze_comprehensive_usage(array $options = []): array
    {
        $defaults = [
            'batch_size' => self::DEFAULT_BATCH_SIZE,
            'include_content' => true,
            'include_meta' => true,
            'include_builders' => true,
            'include_woocommerce' => true,
            'include_theme_settings' => true,
            'use_cache' => true,
            'max_memory_mb' => 256
        ];
        
        $options = array_merge($defaults, $options);
        
        // Check cache first
        if ($options['use_cache']) {
            $cache_key = 'zizi_advanced_usage_' . md5(serialize($options));
            $cached_result = get_transient($cache_key);
            if ($cached_result !== false) {
                return $cached_result;
            }
        }

        $start_time = microtime(true);
        $start_memory = memory_get_usage();
        
        $results = [
            'analysis_metadata' => [
                'start_time' => date('Y-m-d H:i:s'),
                'options' => $options,
                'total_processing_time' => 0,
                'memory_usage' => 0,
                'batches_processed' => 0
            ],
            'usage_summary' => [
                'total_images_analyzed' => 0,
                'used_images' => 0,
                'unused_images' => 0,
                'usage_sources' => []
            ],
            'size_analysis' => [],
            'detailed_usage' => [],
            'recommendations' => []
        ];

        try {
            // Phase 1: Analyze content usage
            if ($options['include_content']) {
                $content_results = self::analyze_content_usage($options['batch_size']);
                self::merge_analysis_results($results, $content_results, 'content');
            }

            // Phase 2: Analyze meta field usage  
            if ($options['include_meta']) {
                $meta_results = self::analyze_meta_usage($options['batch_size']);
                self::merge_analysis_results($results, $meta_results, 'meta');
            }

            // Phase 3: Analyze page builder usage
            if ($options['include_builders']) {
                $builder_results = self::analyze_builder_usage($options['batch_size']);
                self::merge_analysis_results($results, $builder_results, 'builders');
            }

            // Phase 4: Analyze WooCommerce usage
            if ($options['include_woocommerce'] && class_exists('WooCommerce')) {
                $wc_results = self::analyze_woocommerce_usage($options['batch_size']);
                self::merge_analysis_results($results, $wc_results, 'woocommerce');
            }

            // Phase 5: Analyze theme settings usage
            if ($options['include_theme_settings']) {
                $theme_results = self::analyze_theme_usage();
                self::merge_analysis_results($results, $theme_results, 'theme');
            }

            // Generate comprehensive recommendations
            $results['recommendations'] = self::generate_comprehensive_recommendations($results);

        } catch (\Exception $e) {
            $results['error'] = $e->getMessage();
            
            if (class_exists('\\ZiziCache\\CacheSys')) {
                \ZiziCache\CacheSys::writeError(
                    "AdvancedUsageAnalyzer error: " . $e->getMessage(),
                    'AdvancedUsageAnalyzer'
                );
            }
        }

        // Calculate final metrics
        $results['analysis_metadata']['total_processing_time'] = microtime(true) - $start_time;
        $results['analysis_metadata']['memory_usage'] = memory_get_usage() - $start_memory;

        // Cache results if requested
        if ($options['use_cache'] && !isset($results['error'])) {
            set_transient($cache_key, $results, self::CACHE_DURATION);
        }

        return $results;
    }

    /**
     * Analyze content usage (posts, pages)
     */
    private static function analyze_content_usage(int $batch_size): array
    {
        global $wpdb;
        
        $results = [
            'images_found' => [],
            'posts_scanned' => 0,
            'patterns_matched' => []
        ];

        $offset = 0;
        $patterns = [
            'img_src' => '/src=["\']([^"\']*wp-content\/uploads[^"\']*)["\']/',
            'wp_image_class' => '/wp-image-(\d+)/',
            'attachment_shortcode' => '/\[gallery.*?ids=["\']([^"\']*)["\']/',
            'background_url' => '/background(?:-image)?:\s*url\(["\']?([^"\']*wp-content\/uploads[^"\']*)["\']?\)/',
            'data_src' => '/data-src=["\']([^"\']*wp-content\/uploads[^"\']*)["\']/'
        ];

        do {
            $posts = $wpdb->get_results($wpdb->prepare(
                "SELECT ID, post_content FROM {$wpdb->posts} 
                 WHERE post_status = 'publish' 
                 AND post_type IN ('post', 'page', 'product')
                 AND post_content LIKE '%wp-content/uploads%'
                 LIMIT %d OFFSET %d",
                $batch_size,
                $offset
            ));

            foreach ($posts as $post) {
                $results['posts_scanned']++;
                
                foreach ($patterns as $pattern_name => $pattern) {
                    if (preg_match_all($pattern, $post->post_content, $matches)) {
                        $results['patterns_matched'][$pattern_name] = 
                            ($results['patterns_matched'][$pattern_name] ?? 0) + count($matches[0]);
                        
                        foreach ($matches[1] as $match) {
                            if (is_numeric($match)) {
                                $results['images_found'][] = (int)$match;
                            } elseif (strpos($match, ',') !== false) {
                                // Gallery IDs
                                $ids = array_map('trim', explode(',', $match));
                                foreach ($ids as $id) {
                                    if (is_numeric($id)) {
                                        $results['images_found'][] = (int)$id;
                                    }
                                }
                            } else {
                                // URL
                                $attachment_id = self::get_attachment_id_from_url($match);
                                if ($attachment_id) {
                                    $results['images_found'][] = $attachment_id;
                                }
                            }
                        }
                    }
                }
            }

            $offset += $batch_size;
            
            // Memory check
            if (memory_get_usage() > self::MEMORY_THRESHOLD) {
                break;
            }
            
        } while (count($posts) === $batch_size);

        $results['images_found'] = array_unique($results['images_found']);
        return $results;
    }

    /**
     * Analyze meta field usage
     */
    private static function analyze_meta_usage(int $batch_size): array
    {
        global $wpdb;
        
        $results = [
            'images_found' => [],
            'meta_entries_scanned' => 0,
            'meta_keys_found' => []
        ];

        $offset = 0;
        
        do {
            $meta_entries = $wpdb->get_results($wpdb->prepare(
                "SELECT meta_key, meta_value FROM {$wpdb->postmeta} 
                 WHERE meta_value LIKE '%wp-content/uploads%'
                 AND meta_value != ''
                 LIMIT %d OFFSET %d",
                $batch_size,
                $offset
            ));

            foreach ($meta_entries as $meta) {
                $results['meta_entries_scanned']++;
                $results['meta_keys_found'][$meta->meta_key] = 
                    ($results['meta_keys_found'][$meta->meta_key] ?? 0) + 1;
                
                // Extract images from meta value
                $images = self::extract_images_from_content($meta->meta_value);
                $results['images_found'] = array_merge($results['images_found'], $images);
            }

            $offset += $batch_size;
            
        } while (count($meta_entries) === $batch_size && memory_get_usage() < self::MEMORY_THRESHOLD);

        $results['images_found'] = array_unique($results['images_found']);
        return $results;
    }

    /**
     * Analyze page builder usage
     */
    private static function analyze_builder_usage(int $batch_size): array
    {
        if (!class_exists('\\ZiziCache\\PageBuilderThumbnailDetector')) {
            return ['images_found' => [], 'builders_scanned' => 0];
        }

        $detector_results = PageBuilderThumbnailDetector::scan_builder_images($batch_size);
        
        return [
            'images_found' => $detector_results['images_found'] ?? [],
            'builders_scanned' => count($detector_results['builder_usage'] ?? []),
            'builder_details' => $detector_results['builder_usage'] ?? []
        ];
    }

    /**
     * Analyze WooCommerce usage
     */
    private static function analyze_woocommerce_usage(int $batch_size): array
    {
        if (!class_exists('\\ZiziCache\\WooCommerceThumbnailManager')) {
            return ['images_found' => [], 'products_scanned' => 0];
        }

        $wc_results = WooCommerceThumbnailManager::analyze_woocommerce_usage($batch_size);
        
        return [
            'images_found' => [], // WC manager doesn't return specific image IDs in current implementation
            'products_scanned' => $wc_results['total_products'] ?? 0,
            'wc_details' => $wc_results
        ];
    }

    /**
     * Analyze theme and customizer usage
     */
    private static function analyze_theme_usage(): array
    {
        $results = [
            'images_found' => [],
            'settings_scanned' => 0
        ];

        // Theme mods
        $theme_mods = get_theme_mods();
        if ($theme_mods) {
            $images = self::extract_images_from_content(json_encode($theme_mods));
            $results['images_found'] = array_merge($results['images_found'], $images);
            $results['settings_scanned']++;
        }

        // Options with theme prefix
        $theme_slug = get_option('stylesheet');
        $theme_options = get_option("theme_mods_{$theme_slug}");
        if ($theme_options) {
            $images = self::extract_images_from_content(json_encode($theme_options));
            $results['images_found'] = array_merge($results['images_found'], $images);
            $results['settings_scanned']++;
        }

        // Widgets
        $widgets = get_option('widget_media_image', []);
        if ($widgets) {
            foreach ($widgets as $widget) {
                if (isset($widget['attachment_id']) && is_numeric($widget['attachment_id'])) {
                    $results['images_found'][] = (int)$widget['attachment_id'];
                }
            }
            $results['settings_scanned'] += count($widgets);
        }

        $results['images_found'] = array_unique($results['images_found']);
        return $results;
    }

    /**
     * Extract images from any content
     */
    private static function extract_images_from_content(string $content): array
    {
        $images = [];
        
        $patterns = [
            '/wp-image-(\d+)/',
            '/"id":(\d+)/',
            '/"attachment_id":"?(\d+)"?/',
            '/ids="([^"]*)"/',
            '/gallery_ids="([^"]*)"/',
            '/src="([^"]*wp-content\/uploads[^"]*)"/',
            '/url":"([^"]*wp-content\/uploads[^"]*)"/'
        ];

        foreach ($patterns as $pattern) {
            if (preg_match_all($pattern, $content, $matches)) {
                foreach ($matches[1] as $match) {
                    if (is_numeric($match)) {
                        $images[] = (int)$match;
                    } elseif (strpos($match, ',') !== false) {
                        // Multiple IDs
                        $ids = array_map('trim', explode(',', $match));
                        foreach ($ids as $id) {
                            if (is_numeric($id)) {
                                $images[] = (int)$id;
                            }
                        }
                    } else {
                        // URL
                        $attachment_id = self::get_attachment_id_from_url($match);
                        if ($attachment_id) {
                            $images[] = $attachment_id;
                        }
                    }
                }
            }
        }

        return array_unique($images);
    }

    /**
     * Get attachment ID from URL
     */
    private static function get_attachment_id_from_url(string $url): ?int
    {
        // Use WordPress function first
        $attachment_id = attachment_url_to_postid($url);
        if ($attachment_id) {
            return $attachment_id;
        }

        // Fallback to database search
        global $wpdb;
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} 
             WHERE post_type = 'attachment' 
             AND guid LIKE %s 
             LIMIT 1",
            '%' . basename($url)
        ));

        return $attachment_id ? (int)$attachment_id : null;
    }

    /**
     * Merge analysis results
     */
    private static function merge_analysis_results(array &$main_results, array $new_results, string $source): void
    {
        // Merge images found
        if (isset($new_results['images_found'])) {
            $main_results['usage_summary']['usage_sources'][$source] = count($new_results['images_found']);
            $main_results['detailed_usage'][$source] = $new_results;
        }

        $main_results['analysis_metadata']['batches_processed']++;
    }

    /**
     * Generate comprehensive recommendations
     */
    private static function generate_comprehensive_recommendations(array $results): array
    {
        $recommendations = [];
        
        // Analyze overall usage efficiency
        $total_sources = count($results['usage_summary']['usage_sources']);
        $total_images_used = array_sum($results['usage_summary']['usage_sources']);

        if ($total_images_used === 0) {
            $recommendations[] = [
                'type' => 'warning',
                'priority' => 'high',
                'title' => 'No Image Usage Detected',
                'description' => 'Analysis found no images being used in content. This might indicate a scanning issue.',
                'action' => 'verify_analysis_settings'
            ];
        }

        // Memory optimization recommendations
        $memory_mb = ($results['analysis_metadata']['memory_usage'] ?? 0) / 1024 / 1024;
        if ($memory_mb > 100) {
            $recommendations[] = [
                'type' => 'performance',
                'priority' => 'medium',
                'title' => 'High Memory Usage During Analysis',
                'description' => sprintf('Analysis used %.1f MB of memory. Consider reducing batch size.', $memory_mb),
                'action' => 'reduce_batch_size'
            ];
        }

        // Processing time recommendations
        $processing_time = $results['analysis_metadata']['total_processing_time'] ?? 0;
        if ($processing_time > 30) {
            $recommendations[] = [
                'type' => 'performance',
                'priority' => 'medium',
                'title' => 'Long Processing Time',
                'description' => sprintf('Analysis took %.1f seconds. Consider using background processing.', $processing_time),
                'action' => 'use_background_processing'
            ];
        }

        return $recommendations;
    }

    /**
     * Invalidate cache on content changes
     */
    public static function invalidate_cache(int $post_id): void
    {
        global $wpdb;
        
        // Delete all analysis cache entries
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_zizi_advanced_usage_%'"
        );
    }

    /**
     * Get quick usage statistics (optimized for dashboard)
     */
    public static function get_quick_statistics(): array
    {
        $cache_key = 'zizi_quick_stats';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }

        global $wpdb;
        
        $stats = [
            'total_attachments' => $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->posts} 
                 WHERE post_type = 'attachment' 
                 AND post_mime_type LIKE 'image/%'"
            ),
            'total_posts_with_images' => $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->posts} 
                 WHERE post_content LIKE '%wp-content/uploads%' 
                 AND post_status = 'publish'"
            ),
            'total_meta_with_images' => $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->postmeta} 
                 WHERE meta_value LIKE '%wp-content/uploads%'"
            ),
            'last_updated' => current_time('mysql')
        ];

        set_transient($cache_key, $stats, 300); // 5 minutes cache
        return $stats;
    }
}
