<?php

namespace ZiziCache;

/**
 * Class AdvancedCache
 *
 * Manages the generation and writing of the advanced-cache.php drop-in file based on the current plugin configuration.
 * Ensures proper integration with the WordPress page cache drop-in system.
 */
class AdvancedCache
{
    /**
     * Initializes hooks for managing the advanced-cache.php drop-in file.
     * Should be called during plugin bootstrap.
     *
     * @return void
     */
    public static function init(): void
    {
        register_activation_hook(ZIZI_CACHE_FILE_NAME, [__CLASS__, 'addAdvancedCache']);
        register_deactivation_hook(ZIZI_CACHE_FILE_NAME, [__CLASS__, 'removeAdvancedCache']);
        add_action('zizi_cache_update_config:after', [__CLASS__, 'addAdvancedCache']);
    }

    /**
     * Generates and writes the current advanced-cache.php drop-in file based on configuration.
     *
     * @return void
     */
    public static function addAdvancedCache(): void
    {
        $config = self::buildConfigArray();
        $content = self::generateAdvancedCacheContent($config);
        self::writeAdvancedCacheFile($content);
    }

    /**
     * Removes the advanced-cache.php drop-in file (or its variant).
     *
     * @return void
     */
    public static function removeAdvancedCache(): void
    {
        $file = self::getAdvancedCacheFileName();
        if (is_file($file)) {
            unlink($file);
        }
    }

    /**
     * Builds the current configuration array for the drop-in file.
     *
     * @return array
     */
    private static function buildConfigArray(): array
    {
        $config = SysConfig::$config;
        // Přidá výchozí ignorované query parametry
        $config['cache_ignore_queries'] = [
            ...CacheSys::$default_ignore_queries,
            ...$config['cache_ignore_queries'],
        ];
        return $config;
    }

    /**
     * Generates the content of the drop-in file from the template and provided configuration.
     *
     * @param array $config
     * @return string
     */
    private static function generateAdvancedCacheContent(array $config): string
    {
        $template = file_get_contents(ZIZI_CACHE_PLUGIN_DIR . 'assets/templates/advanced-cache.php');
        $configExport = var_export($config, true);
        $content = str_replace('CONFIG_TO_REPLACE', $configExport, $template);
        // Umožní pluginům modifikovat obsah
        return apply_filters('zizi_cache_advanced_cache', $content);
    }

    /**
     * Writes the content of the drop-in file to the correct location in wp-content.
     *
     * @param string $content
     * @return void
     */
    private static function writeAdvancedCacheFile(string $content): void
    {
        $file = self::getAdvancedCacheFileName();
        file_put_contents($file, $content);
    }

    /**
     * Returns the full path to the drop-in file depending on the environment (Atomic/classic).
     *
     * @return string
     */
    private static function getAdvancedCacheFileName(): string
    {
        return WP_CONTENT_DIR . '/' . (class_exists('Atomic_Persistent_Data') ? 'zizi-cache-advanced-cache.php' : 'advanced-cache.php');
    }
}
